/* 
 * 
 * This file is part of "Ngraph for Windows".
 * 
 * Copyright (C) 2015, Satoshi ISHIZAKA. isizaka@msa.biglobe.ne.jp
 * 
 * "Ngraph for Windows" is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * "Ngraph for Windows" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

/*
 *
 * ogra2win.c
 *
 */

#include <windows.h>
#include <stdlib.h>
#include <stdarg.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include "ngraph.h"
#include "object.h"
#include "ioutil.h"
#include "gra.h"
#include "nstring.h"
#include "jnstring.h"
#include "mathfn.h"
#include "config.h"

#define NAME "gra2win"
#define PARENT "gra2"
#define VERSION  "1.00.00"
#define G2WINCONF "[gra2win]"
#define LINETOLIMIT 500
#define TRUE  1
#define FALSE 0

extern HINSTANCE hInstance;

#define ERRNUM 1

#define ANSI2_CHARSET -1

char *g2winerrorlist[ERRNUM]={
  "",
};

#define FONT_ROMAN 0
#define FONT_BOLD 1
#define FONT_ITALIC 2
#define FONT_BOLDITALIC 3
#define FONT_UNKNOWN 4

struct fontmap;

struct fontmap {
  char *fontalias;
  char *fontname;
  int charset;
  int italic;
  struct fontmap *next;
};

struct g2wlocal {
  HWND hWnd;
  int scrollx,scrolly;
  char *title;
  unsigned int winwidth,winheight,windpi;
  int autoredraw;
  int storememory;
  HENHMETAFILE hMF;
  int mfdpi;
  int needredraw;
  HDC DC;
  HPEN ThePen,OrgPen;
  HBRUSH TheBrush,OrgBrush;
  HFONT TheFont,OrgFont;
  COLORREF Col;
  int bgR,bgG,bgB;
  int style,width,dashn;
  DWORD *dashlist;
  int x0,y0;
  int dashi,dotf;
  double dashlen;
  double pixel_dot;
  int offsetx,offsety;
  int cpx,cpy;
  struct fontmap *fontmaproot;
  int loadfontf;
  int charset;
  char *fontalias;
  double fontspace,fontcos,fontsin;
  int linetonum;
  int saveDC;
  int PaperWidth,PaperHeight;
  int minus_hyphen;
  int MakeMetaFile;
};

void g2wsaveDC(HDC dc,struct g2wlocal *g2wlocal,struct g2wlocal *g2wsave)
{
  memcpy(g2wsave,g2wlocal,sizeof(struct g2wlocal));
  g2wlocal->saveDC=SaveDC(dc);
  g2wlocal->DC=dc;
  g2wlocal->hMF=NULL;
  g2wlocal->style=PS_SOLID;
  g2wlocal->width=1;
  g2wlocal->Col=RGB(0,0,0);
  g2wlocal->dashlist=NULL;
  g2wlocal->dashn=0;
  g2wlocal->dashlen=0;
  g2wlocal->dashi=0;
  g2wlocal->dotf=TRUE;
  g2wlocal->x0=0;
  g2wlocal->y0=0;
  g2wlocal->offsetx=0;
  g2wlocal->offsety=0;
  g2wlocal->cpx=0;
  g2wlocal->cpy=0;
  g2wlocal->fontalias=NULL;
  g2wlocal->loadfontf=FALSE;
  g2wlocal->ThePen=(HPEN)GetStockObject(BLACK_PEN);
  g2wlocal->TheBrush=(HBRUSH)GetStockObject(BLACK_BRUSH);
  g2wlocal->TheFont=(HFONT)GetStockObject(SYSTEM_FONT);
  g2wlocal->OrgPen=(HPEN)SelectObject(dc,g2wlocal->ThePen);
  g2wlocal->OrgBrush=(HBRUSH)SelectObject(dc,g2wlocal->TheBrush);
  g2wlocal->OrgFont=(HFONT)SelectObject(dc,g2wlocal->TheFont);
  g2wlocal->linetonum=0;
}

void g2wrestoreDC(struct g2wlocal *g2wlocal,struct g2wlocal *g2wsave)
{
  if (g2wlocal->linetonum!=0) {
    EndPath(g2wlocal->DC);
    StrokePath(g2wlocal->DC);
    g2wlocal->linetonum=0;
  }
  SelectObject(g2wlocal->DC,g2wlocal->OrgPen);
  SelectObject(g2wlocal->DC,g2wlocal->OrgBrush);
  SelectObject(g2wlocal->DC,g2wlocal->OrgFont);
  RestoreDC(g2wlocal->DC,g2wlocal->saveDC);
  DeleteObject(g2wlocal->ThePen);
  DeleteObject(g2wlocal->TheBrush);
  DeleteObject(g2wlocal->TheFont);
  memfree(g2wlocal->dashlist);
  memfree(g2wlocal->fontalias);
  memcpy(g2wlocal,g2wsave,sizeof(struct g2wlocal));
}


int g2winloadconfig(struct g2wlocal *g2wlocal)
{
  FILE *fp;
  char *tok,*str,*s2;
  char *f1,*f2,*f3,*f4;
  struct fontmap *fcur,*fnew;
  int val;
  char *endptr;
  int len;

  if ((fp=openconfig(G2WINCONF))==NULL) return 0;
  fcur=g2wlocal->fontmaproot;
  while ((tok=getconfig(fp,&str))!=NULL) {
    s2=str;
    if (strcmp(tok,"font_map")==0) {
      f1=getitok2(&s2,&len," \x09,");
      f3=getitok2(&s2,&len," \x09,");
      f4=getitok2(&s2,&len," \x09,");
      for (;(s2[0]!='\0') && (strchr(" \x09,",s2[0])!=NULL);s2++);
      f2=getitok2(&s2,&len,"");
      if ((f1!=NULL) && (f2!=NULL) && (f3!=NULL) && (f4!=NULL)) {
        if ((fnew=memalloc(sizeof(struct fontmap)))==NULL) {
          memfree(tok);
          memfree(f1);
          memfree(f2);
          memfree(f3);
          memfree(f4);
          closeconfig(fp);
          return 1;
        }
        if (fcur==NULL) g2wlocal->fontmaproot=fnew;
        else fcur->next=fnew;
        fcur=fnew;
        fcur->next=NULL;
        fcur->fontalias=f1;
        fcur->fontname=f2;
        if (strcmp(f4,"roman")==0) fcur->italic=FONT_ROMAN;
        else if (strcmp(f4,"italic")==0) fcur->italic=FONT_ITALIC;
        else if (strcmp(f4,"bold")==0) fcur->italic=FONT_BOLD;
        else if (strcmp(f4,"bold_italic")==0) fcur->italic=FONT_BOLDITALIC;
        else fcur->italic=FONT_UNKNOWN;
        if (strcmp(f3,"shiftjis")==0) fcur->charset=SHIFTJIS_CHARSET;
        else if (strcmp(f3,"symbol")==0) fcur->charset=SYMBOL_CHARSET;
        else if (strcmp(f3,"ansi")==0) fcur->charset=ANSI_CHARSET;
        else if (strcmp(f3,"ansi2")==0) fcur->charset=ANSI2_CHARSET;
        else if (strcmp(f3,"oem")==0) fcur->charset=OEM_CHARSET;
        else if (strcmp(f3,"hangeul")==0) fcur->charset=HANGEUL_CHARSET;
        else if (strcmp(f3,"chinesebig5")==0) fcur->charset=CHINESEBIG5_CHARSET;
        else fcur->charset=DEFAULT_CHARSET;
        memfree(f3);
        memfree(f4);
      } else {
        memfree(f1);
        memfree(f2);
        memfree(f3);
        memfree(f4);
      }
    } else if (strcmp(tok,"win_dpi")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      if (endptr[0]=='\0')  g2wlocal->windpi=abs(val);
      memfree(f1);
    } else if (strcmp(tok,"win_width")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      if ((endptr[0]=='\0') && (val!=0)) g2wlocal->winwidth=val;
      memfree(f1);
    } else if (strcmp(tok,"win_height")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      if ((endptr[0]=='\0') && (val!=0)) g2wlocal->winheight=val;
      memfree(f1);
    } else if (strcmp(tok,"auto_redraw")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      if (endptr[0]=='\0') {
        if (val==0) g2wlocal->autoredraw=FALSE;
        else g2wlocal->autoredraw=TRUE;
      }
      memfree(f1);
	} else if (strcmp(tok,"store_in_memory")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      if (endptr[0]=='\0') {
        if (val==0) g2wlocal->storememory=FALSE;
        else g2wlocal->storememory=TRUE;
	  }
      memfree(f1);
    } else if (strcmp(tok,"backgroundR")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      if (endptr[0]=='\0') g2wlocal->bgR=val;
      memfree(f1);
    } else if (strcmp(tok,"backgroundG")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      if (endptr[0]=='\0') g2wlocal->bgG=val;
      memfree(f1);
    } else if (strcmp(tok,"backgroundB")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      if (endptr[0]=='\0') g2wlocal->bgB=val;
      memfree(f1);
    } else if (strcmp(tok,"minus_hyphen")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      if (endptr[0]=='\0') g2wlocal->minus_hyphen=val;
      memfree(f1);
    }
    memfree(tok);
    memfree(str);
  }
  closeconfig(fp);
  return 0;
}

extern int globallock;
int g2windone(struct objlist *obj,char *inst,char *rval,int argc,char **argv);

int g2wgetinst(HWND hWnd,
                struct objlist **obj,char **inst,struct g2wlocal **g2wlocal)
{
  int i,lastinst;

  *obj=getobject("gra2win");
  lastinst=chkobjlastinst(*obj);
  for (i=0;i<=lastinst;i++) {
    *inst=chkobjinst(*obj,i);
    _getobj(*obj,"_local",*inst,g2wlocal);
    if ((*g2wlocal)->hWnd==hWnd) return TRUE;
  }
  return FALSE;
}

void MakeRuler(HDC dc,struct g2wlocal *g2wlocal);

LRESULT CALLBACK Gra2WinProc(HWND hWnd,UINT iMessage,WPARAM wParam,LPARAM lParam)
{
  struct objlist *obj;
  struct g2wlocal *g2wlocal;
  char *inst;
  RECT rc;
  PAINTSTRUCT ps;
  int ScPos,ScMin,ScMax,Sc;
  HDC dc;
  int id;
  ENHMETAHEADER emh;
  RECT mfrect;

  double dpiratio;

  if (globallock) {
    if (g2wgetinst(hWnd,&obj,&inst,&g2wlocal))
      g2wlocal->needredraw=TRUE;
    return DefWindowProc(hWnd,iMessage,wParam,lParam);
  }
  switch (iMessage) {
  case WM_DESTROY:
  case WM_NCDESTROY:
    return 0;
  case WM_CLOSE:
    if (g2wgetinst(hWnd,&obj,&inst,&g2wlocal)) {
      _getobj(obj,"id",inst,&id);
      delobj(obj,id);
      return 0;
    }
    break;
  case WM_PAINT:
    if (GetUpdateRect(hWnd,&rc,FALSE)) {
      dc=BeginPaint(hWnd,&ps);
      if (g2wgetinst(hWnd,&obj,&inst,&g2wlocal)) {
        if (g2wlocal->autoredraw) {
          if (!g2wlocal->storememory || (g2wlocal->hMF==NULL)) {
            g2wlocal->needredraw=TRUE;
          } else {
            if (g2wlocal->hMF!=NULL) {
              dpiratio=((double )(g2wlocal->windpi))/g2wlocal->mfdpi;
              GetEnhMetaFileHeader(g2wlocal->hMF,sizeof(ENHMETAHEADER),&emh);
              mfrect.left=emh.rclBounds.left*dpiratio-g2wlocal->scrollx;
              mfrect.top=emh.rclBounds.top*dpiratio-g2wlocal->scrolly;
              mfrect.right=emh.rclBounds.right*dpiratio-g2wlocal->scrollx;
              mfrect.bottom=emh.rclBounds.bottom*dpiratio-g2wlocal->scrolly;
              MakeRuler(dc,g2wlocal);
              PlayEnhMetaFile(dc,g2wlocal->hMF,&mfrect);
            }
          }
        }
      }
      EndPaint(hWnd,&ps);
    }
    break;
  case WM_HSCROLL:
    if (g2wgetinst(hWnd,&obj,&inst,&g2wlocal)) {
      GetScrollRange(hWnd,SB_HORZ,&ScMin,&ScMax);
      switch (LOWORD(wParam)) {
      case SB_PAGEUP:
        Sc=-50;
        break;
      case SB_PAGEDOWN:
        Sc=50;
        break;
      case SB_LINEUP:
        Sc=-10;
        break;
      case SB_LINEDOWN:
        Sc=10;
        break;
      case SB_THUMBPOSITION:
        Sc=HIWORD(wParam)-g2wlocal->scrollx;
        break;
      default:
        Sc=0;
        break;
      }
      ScPos=g2wlocal->scrollx+Sc;
      ScPos=max(ScMin,ScPos);
      ScPos=min(ScMax,ScPos);
      Sc=ScPos-g2wlocal->scrollx;
      g2wlocal->scrollx=ScPos;
      if (Sc==0) break;
      SetScrollPos(hWnd,SB_HORZ,ScPos,TRUE);
      ScrollWindow(hWnd,-Sc,0,(CONST RECT *)NULL,(CONST RECT *)NULL);
      UpdateWindow(hWnd);
    }
    break;
  case WM_VSCROLL:
    if (g2wgetinst(hWnd,&obj,&inst,&g2wlocal)) {
      GetScrollRange(hWnd,SB_VERT,&ScMin,&ScMax);
      switch (LOWORD(wParam)) {
      case SB_PAGEUP:
        Sc=-50;
        break;
      case SB_PAGEDOWN:
        Sc=50;
        break;
      case SB_LINEUP:
        Sc=-10;
        break;
      case SB_LINEDOWN:
        Sc=10;
        break;
      case SB_THUMBPOSITION:
        Sc=HIWORD(wParam)-g2wlocal->scrolly;
        break;
      default:
        Sc=0;
        break;
      }
      ScPos=g2wlocal->scrolly+Sc;
      ScPos=max(ScMin,ScPos);
      ScPos=min(ScMax,ScPos);
      Sc=ScPos-g2wlocal->scrolly;
      g2wlocal->scrolly=ScPos;
      if (Sc==0) break;
      SetScrollPos(hWnd,SB_VERT,ScPos,TRUE);
      ScrollWindow(hWnd,0,-Sc,(CONST RECT *)NULL,(CONST RECT *)NULL);
      UpdateWindow(hWnd);
    }
    break;
  default:
    return DefWindowProc(hWnd,iMessage,wParam,lParam);
  }
  return 0L;
}

int setwndclass=FALSE;
char szClassName[]="GRA2WIN";
WNDCLASS wc;

int g2windoneproc(struct objlist *obj,void *local)
{
  if (setwndclass) UnregisterClass(szClassName,hInstance);
  return 0;
}

int g2wininit(struct objlist *obj,char *inst,char *rval,int argc,char **argv)
{
  struct g2wlocal *g2wlocal;
  struct objlist *robj;
  int oid,idn;
  struct fontmap *fcur,*fdel;
  HWND hWnd;
  HBRUSH brush;

  if (_exeparent(obj,(char *)argv[1],inst,rval,argc,argv)) return 1;
  if ((g2wlocal=memalloc(sizeof(struct g2wlocal)))==NULL) return 1;
  g2wlocal->fontmaproot=NULL;
  g2wlocal->title=NULL;
  if (_putobj(obj,"_local",inst,g2wlocal)) goto errexit;
  _getobj(obj,"oid",inst,&oid);
  g2wlocal->winwidth=0;
  g2wlocal->winheight=0;
  g2wlocal->windpi=70;
  g2wlocal->autoredraw=TRUE;
  g2wlocal->storememory=TRUE;
  g2wlocal->bgR=GetSysColor(COLOR_WINDOW) & 255;
  g2wlocal->bgG=(GetSysColor(COLOR_WINDOW) >> 8) & 255;
  g2wlocal->bgB=(GetSysColor(COLOR_WINDOW) >> 16) & 255;
  g2wlocal->minus_hyphen=TRUE;
  g2wlocal->MakeMetaFile=FALSE;
  if (g2winloadconfig(g2wlocal)) goto errexit;
  if (g2wlocal->windpi<1) g2wlocal->windpi=70;
  if (g2wlocal->windpi>2540) g2wlocal->windpi=2540;
  if (g2wlocal->winwidth<1) g2wlocal->winwidth=CW_USEDEFAULT;
  if (g2wlocal->winwidth>10000) g2wlocal->winwidth=CW_USEDEFAULT;
  if (g2wlocal->winheight<1) g2wlocal->winheight=CW_USEDEFAULT;
  if (g2wlocal->winheight>10000) g2wlocal->winheight=CW_USEDEFAULT;
  if (_putobj(obj,"dpi",inst,&(g2wlocal->windpi))) goto errexit;
  if (_putobj(obj,"auto_redraw",inst,&(g2wlocal->autoredraw))) goto errexit;
  if (_putobj(obj,"store_in_memory",inst,&(g2wlocal->storememory))) goto errexit;
  if (!setwndclass) {
    wc.style      =CS_HREDRAW|CS_VREDRAW;
    wc.lpfnWndProc=Gra2WinProc;
    wc.cbClsExtra =0;
    wc.cbWndExtra =0;
    wc.hInstance  =hInstance;
    wc.hIcon      =LoadIcon(hInstance,MAKEINTRESOURCE(2));
    wc.hCursor    =LoadCursor(NULL,IDC_CROSS);
    brush=CreateSolidBrush(RGB(g2wlocal->bgR,g2wlocal->bgG,g2wlocal->bgB));
    wc.hbrBackground=brush;
    wc.lpszMenuName=szClassName;
    wc.lpszClassName=szClassName;
    if (!RegisterClass(&wc)) goto errexit;
    setwndclass=TRUE;
  }
  g2wlocal->title=mkobjlist(obj,NULL,oid,NULL,TRUE);
  g2wlocal->hWnd=NULL;
  g2wlocal->DC=NULL;
  hWnd=CreateWindowEx(0L,szClassName,g2wlocal->title,
                    WS_OVERLAPPEDWINDOW | WS_CAPTION | WS_VSCROLL | WS_HSCROLL,
                    CW_USEDEFAULT,CW_USEDEFAULT,
                    g2wlocal->winwidth,g2wlocal->winheight,
                    NULL,NULL,hInstance,NULL);
  if (hWnd==NULL) goto errexit;
  g2wlocal->hWnd=hWnd;
  SetScrollRange(hWnd,SB_HORZ,0,0,TRUE);
  SetScrollRange(hWnd,SB_VERT,0,0,TRUE);
  ShowWindow(hWnd,SW_SHOW);
  UpdateWindow(hWnd);
  SetForegroundWindow(hWnd);
  if (chkobjfield(obj,"_evloop")) goto errexit;
  else {
    if ((idn=getobjtblpos(obj,"_evloop",&robj))==-1) goto errexit;
  }
  registerevloop(chkobjectname(obj),"_evloop",robj,idn,inst,g2wlocal);
  g2wlocal->DC=GetDC(hWnd);
  g2wlocal->style=PS_SOLID;
  g2wlocal->width=1;
  g2wlocal->Col=RGB(0,0,0);
  g2wlocal->dashlist=NULL;
  g2wlocal->dashn=0;
  g2wlocal->dashlen=0;
  g2wlocal->dashi=0;
  g2wlocal->dotf=TRUE;
  g2wlocal->x0=0;
  g2wlocal->y0=0;
  g2wlocal->offsetx=0;
  g2wlocal->offsety=0;
  g2wlocal->cpx=0;
  g2wlocal->cpy=0;
  g2wlocal->pixel_dot=g2wlocal->windpi/25.4/100;
  g2wlocal->fontalias=NULL;
  g2wlocal->loadfontf=FALSE;
  g2wlocal->ThePen=(HPEN)GetStockObject(BLACK_PEN);
  g2wlocal->TheBrush=(HBRUSH)GetStockObject(BLACK_BRUSH);
  g2wlocal->TheFont=(HFONT)GetStockObject(SYSTEM_FONT);
  g2wlocal->OrgPen=(HPEN)SelectObject(g2wlocal->DC,g2wlocal->ThePen);
  g2wlocal->OrgBrush=(HBRUSH)SelectObject(g2wlocal->DC,g2wlocal->TheBrush);
  g2wlocal->OrgFont=(HFONT)SelectObject(g2wlocal->DC,g2wlocal->TheFont);
  g2wlocal->linetonum=0;
  g2wlocal->scrollx=0;
  g2wlocal->scrolly=0;
  g2wlocal->hMF=NULL;
  return 0;

errexit:
  fcur=g2wlocal->fontmaproot;
  while (fcur!=NULL) {
    fdel=fcur;
    fcur=fcur->next;
    memfree(fdel->fontalias);
    memfree(fdel->fontname);
    memfree(fdel);
  }
  memfree(g2wlocal->title);
  memfree(g2wlocal);
  return 1;
}


int g2windone(struct objlist *obj,char *inst,char *rval,int argc,char **argv)
{
  struct g2wlocal *g2wlocal;
  struct objlist *robj;
  struct fontmap *fcur,*fdel;
  int idn;

  if (_exeparent(obj,(char *)argv[1],inst,rval,argc,argv)) return 1;
  _getobj(obj,"_local",inst,&g2wlocal);
  SelectObject(g2wlocal->DC,g2wlocal->OrgPen);
  SelectObject(g2wlocal->DC,g2wlocal->OrgBrush);
  SelectObject(g2wlocal->DC,g2wlocal->OrgFont);
  ReleaseDC(g2wlocal->hWnd,g2wlocal->DC);
  DeleteObject(g2wlocal->ThePen);
  DeleteObject(g2wlocal->TheBrush);
  DeleteObject(g2wlocal->TheFont);
  if (g2wlocal->hMF!=NULL) {
    DeleteEnhMetaFile(g2wlocal->hMF);
    g2wlocal->hMF=NULL;
  }
  fcur=g2wlocal->fontmaproot;
  while (fcur!=NULL) {
    fdel=fcur;
    fcur=fcur->next;
    memfree(fdel->fontalias);
    memfree(fdel->fontname);
    memfree(fdel);
  }
  memfree(g2wlocal->title);
  memfree(g2wlocal->fontalias);
  memfree(g2wlocal->dashlist);
  if ((idn=getobjtblpos(obj,"_evloop",&robj))!=-1)
    unregisterevloop(robj,idn,inst);
  DestroyWindow(g2wlocal->hWnd);
  return 0;
}

int g2wflush(struct objlist *obj,char *inst,char *rval,int argc,char **argv)
{
  struct g2wlocal *g2wlocal;

  _getobj(obj,"_local",inst,&g2wlocal);
  if (g2wlocal->linetonum!=0) {
    EndPath(g2wlocal->DC);
    StrokePath(g2wlocal->DC);
    g2wlocal->linetonum=0;
  }
  return 0;
}


int g2winclear(struct objlist *obj,char *inst,char *rval,int argc,char **argv)
{
  struct g2wlocal *g2wlocal;

  if (_exeparent(obj,(char *)argv[1],inst,rval,argc,argv)) return 1;
  _getobj(obj,"_local",inst,&g2wlocal);
  if (g2wlocal->linetonum!=0) {
    EndPath(g2wlocal->DC);
    StrokePath(g2wlocal->DC);
    g2wlocal->linetonum=0;
  }
  if (g2wlocal->hMF!=NULL) {
    DeleteEnhMetaFile(g2wlocal->hMF);
    g2wlocal->hMF=NULL;
  }
  InvalidateRect(g2wlocal->hWnd,NULL,TRUE);
  UpdateWindow(g2wlocal->hWnd);
  SetScrollRange(g2wlocal->hWnd,SB_HORZ,0,0,TRUE);
  SetScrollRange(g2wlocal->hWnd,SB_VERT,0,0,TRUE);
  g2wlocal->scrollx=0;
  g2wlocal->scrolly=0;
  return 0;
}

int g2windisconnect(struct objlist *obj,char *inst,char *rval,int argc,char **argv)
{
  struct g2wlocal *g2wlocal;

  if (_exeparent(obj,(char *)argv[1],inst,rval,argc,argv)) return 1;
  _getobj(obj,"_local",inst,&g2wlocal);
  if (g2wlocal->linetonum!=0) {
    EndPath(g2wlocal->DC);
    StrokePath(g2wlocal->DC);
    g2wlocal->linetonum=0;
  }
  if (!g2wlocal->storememory || !g2wlocal->autoredraw) {
    InvalidateRect(g2wlocal->hWnd,NULL,TRUE);
    UpdateWindow(g2wlocal->hWnd);
    SetScrollRange(g2wlocal->hWnd,SB_HORZ,0,0,TRUE);
    SetScrollRange(g2wlocal->hWnd,SB_VERT,0,0,TRUE);
    g2wlocal->scrollx=0;
    g2wlocal->scrolly=0;
  }
  return 0;
}

int g2windpi(struct objlist *obj,char *inst,char *rval,int argc,char **argv)
{
  struct g2wlocal *g2wlocal;
  int dpi;

  _getobj(obj,"_local",inst,&g2wlocal);
  dpi=abs(*(int *)argv[2]);
  if (dpi<1) dpi=1;
  if (dpi>2540) dpi=2540;
  g2wlocal->windpi=dpi;
  g2wlocal->pixel_dot=dpi/25.4/100;
  *(int *)argv[2]=dpi;
  InvalidateRect(g2wlocal->hWnd,NULL,TRUE);
  SetScrollRange(g2wlocal->hWnd,SB_HORZ,0,0,TRUE);
  SetScrollRange(g2wlocal->hWnd,SB_VERT,0,0,TRUE);
  g2wlocal->scrollx=0;
  g2wlocal->scrolly=0;
  g2wlocal->needredraw=TRUE;
  return 0;
}

int g2winsize(struct objlist *obj,char *inst,char *rval,int argc,char **argv)
{
  struct g2wlocal *g2wlocal;
  int width,height,x,y;
  RECT rect;
  char *field;

  _getobj(obj,"_local",inst,&g2wlocal);
  field=(char *)argv[1];
  if (field[0]=='w') {
    width=abs(*(int *)argv[2]);
    _getobj(obj,"height",inst,&height);
  } else {
    height=abs(*(int *)argv[2]);
    _getobj(obj,"width",inst,&width);
  }
  GetWindowRect(g2wlocal->hWnd,&rect);
  x=rect.left;
  y=rect.top;
  if (width<1) width=rect.right-rect.left;
  if (height<1) height=rect.bottom-rect.top;
  MoveWindow(g2wlocal->hWnd,x,y,width,height,TRUE);
  return 0;
}

int g2winautoredraw(struct objlist *obj,char *inst,char *rval,
                    int argc,char **argv)
{
  struct g2wlocal *g2wlocal;

  _getobj(obj,"_local",inst,&g2wlocal);
  if (!(g2wlocal->autoredraw) && (*(int *)argv[2])) g2wlocal->needredraw=TRUE;
  g2wlocal->autoredraw=*(int *)argv[2];
  return 0;
}

int g2winstorememory(struct objlist *obj,char *inst,char *rval,
                    int argc,char **argv)
{
  struct g2wlocal *g2wlocal;

  _getobj(obj,"_local",inst,&g2wlocal);
  if ((g2wlocal->storememory) && !(*(int *)argv[2])) {
    if (g2wlocal->hMF!=NULL) {
      DeleteEnhMetaFile(g2wlocal->hMF);
      g2wlocal->hMF=NULL;
    }
  }
  g2wlocal->storememory=*(int *)argv[2];
  return 0;
}


int g2winredraw(struct objlist *obj,char *inst,char *rval,int argc,char **argv)
{
  struct g2wlocal *g2wlocal;

  _getobj(obj,"_local",inst,&g2wlocal);
  g2wlocal->needredraw=TRUE;
  return 0;
}

int g2win_evloop(struct objlist *obj,char *inst,char *rval,int argc,char **argv)
{
  MSG msg;
  struct g2wlocal *g2wlocal;

  g2wlocal=(struct g2wlocal *)(argv[2]);
  if (g2wlocal->needredraw) {
    MakeRuler(g2wlocal->DC,g2wlocal);
    GRAredraw(obj,inst,FALSE,FALSE);
    g2wflush(obj,inst,NULL,0,NULL);
    g2wlocal->needredraw=FALSE;
  }
  if (PeekMessage(&msg,g2wlocal->hWnd,0,0,PM_NOREMOVE)) {
    GetMessage(&msg,g2wlocal->hWnd,0,0);
    TranslateMessage(&msg);
    DispatchMessage(&msg);
 }
  return 0;
}

int dot2pixel(struct g2wlocal *g2wlocal,int r)
{
  return nround(r*g2wlocal->pixel_dot);
}

int dot2pixelx(struct g2wlocal *g2wlocal,int x)
{
  return nround(x*g2wlocal->pixel_dot+g2wlocal->offsetx-g2wlocal->scrollx);
}

int dot2pixely(struct g2wlocal *g2wlocal,int y)
{
  return nround(y*g2wlocal->pixel_dot+g2wlocal->offsety-g2wlocal->scrolly);
}

int pixel2dot(struct g2wlocal *g2wlocal,int r)
{
  return nround(r/g2wlocal->pixel_dot);
}

void MakeRuler(HDC dc,struct g2wlocal *g2wlocal)
{
  int Width,Height;
  HPEN orgpen,pen;
  int smode;

  smode=SetROP2(dc,R2_XORPEN);
  SelectClipRgn(g2wlocal->DC,NULL);
  Width=g2wlocal->PaperWidth;
  Height=g2wlocal->PaperHeight;
  pen=CreatePen(PS_SOLID,0,RGB(128,128,128));
  orgpen=(HPEN)SelectObject(dc,pen);
  BeginPath(dc);
  MoveToEx(dc,dot2pixel(g2wlocal,0)-g2wlocal->scrollx,dot2pixel(g2wlocal,0)-g2wlocal->scrolly,NULL);
  LineTo(dc,dot2pixel(g2wlocal,Width)-g2wlocal->scrollx,dot2pixel(g2wlocal,0)-g2wlocal->scrolly);
  LineTo(dc,dot2pixel(g2wlocal,Width)-g2wlocal->scrollx,dot2pixel(g2wlocal,Height)-g2wlocal->scrolly);
  LineTo(dc,dot2pixel(g2wlocal,0)-g2wlocal->scrollx,dot2pixel(g2wlocal,Height)-g2wlocal->scrolly);
  LineTo(dc,dot2pixel(g2wlocal,0)-g2wlocal->scrollx,dot2pixel(g2wlocal,0)-g2wlocal->scrolly);
  CloseFigure(dc);
  EndPath(dc);
  StrokePath(dc);
  DeleteObject(SelectObject(dc,orgpen));
  SetROP2(dc,smode);

}

void g2w_moveto(struct g2wlocal *g2wlocal,int x0,int y0)
{
  g2wlocal->dashlen=0;
  g2wlocal->dashi=0;
  g2wlocal->dotf=TRUE;
  g2wlocal->x0=x0;
  g2wlocal->y0=y0;
  MoveToEx(g2wlocal->DC,x0,y0,NULL);
}

void g2w_lineto(struct g2wlocal *g2wlocal,int x,int y)
{
  double dd,len,len2;
  double dx,dy;
  int gx,gy,gx1,gy1,gx2,gy2;

  gx1=g2wlocal->x0;
  gy1=g2wlocal->y0;
  gx2=x;
  gy2=y;
  if (g2wlocal->dashn==0) LineTo(g2wlocal->DC,gx2,gy2);
  else {
    dx=(gx2-gx1);
    dy=(gy2-gy1);
    len2=len=sqrt(dx*dx+dy*dy);
    while (len2>((g2wlocal->dashlist)[g2wlocal->dashi]-g2wlocal->dashlen)) {
      dd=(len-len2+(g2wlocal->dashlist)[g2wlocal->dashi]-g2wlocal->dashlen)/len;
      gx=gx1+nround(dx*dd);
      gy=gy1+nround(dy*dd);
      if (g2wlocal->dotf) LineTo(g2wlocal->DC,gx,gy);
      else MoveToEx(g2wlocal->DC,gx,gy,NULL);
      g2wlocal->dotf=g2wlocal->dotf ? FALSE : TRUE;
      len2-=((g2wlocal->dashlist)[g2wlocal->dashi]-g2wlocal->dashlen);
      g2wlocal->dashlen=0;
      g2wlocal->dashi++;
      if (g2wlocal->dashi>=g2wlocal->dashn) {
        g2wlocal->dashi=0;
        g2wlocal->dotf=TRUE;
      }
    }
    if (g2wlocal->dotf) LineTo(g2wlocal->DC,gx2,gy2);
    g2wlocal->dashlen+=len2;
  }
  g2wlocal->x0=x;
  g2wlocal->y0=y;
}

int g2win_output(struct objlist *obj,char *inst,char *rval,int argc,char **argv)
{
  struct g2wlocal *g2wlocal;
  char code;
  int *cpar;
  char *cstr,ch[1],*s,*sw;
  HPEN TmpPen;
  HBRUSH TmpBrush;
  HFONT TmpFont;
  HRGN hClip;
  unsigned int style,cap,join;
  int i,R,G,B;
  double Theta1,Theta2;
  POINT *Points;
  LOGFONT IDFont;
  LOGBRUSH lBrush;
  SIZE lpsize;
  HDC DC;
  int charset,italic;
  double fontsize,fontspace,fontdir;
  char *fontname;
  struct fontmap *fcur;
  double x0,y0,fontwidth;
  HCURSOR hcrWait,hcrOld;
  HDC dc;
  struct g2wlocal g2wsave;
  RECT rect;
  OSVERSIONINFO VersionInformation;

  g2wlocal=(struct g2wlocal *)argv[2];
  code=*(char *)(argv[3]);
  cpar=(int *)argv[4];
  cstr=argv[5];
  DC=g2wlocal->DC;
  if (DC==NULL) return 0;
  if (g2wlocal->hWnd==NULL) return 0;
  if (g2wlocal->linetonum!=0) {
    if ((code!='T') || (g2wlocal->linetonum>=LINETOLIMIT)) {
      EndPath(DC);
      StrokePath(DC);
      g2wlocal->linetonum=0;
    }
  }
  switch (code) {
  case 'I':
    if (g2wlocal->hMF!=NULL) {
	   DeleteEnhMetaFile(g2wlocal->hMF);
	   g2wlocal->hMF=NULL;
	 }
	 SetScrollRange(g2wlocal->hWnd,SB_HORZ,0,dot2pixel(g2wlocal,cpar[3]),TRUE);
	 SetScrollRange(g2wlocal->hWnd,SB_VERT,0,dot2pixel(g2wlocal,cpar[4]),TRUE);
    g2wlocal->PaperWidth=cpar[3];
    g2wlocal->PaperHeight=cpar[4];
    MakeRuler(DC,g2wlocal);
	 break;
  case 'E':
	 if (g2wlocal->autoredraw && g2wlocal->storememory
    && (!g2wlocal->MakeMetaFile)) {
      g2wlocal->MakeMetaFile=TRUE;
	   hcrWait=LoadCursor(NULL,IDC_WAIT);
	   hcrOld=SetCursor(hcrWait);

      if (g2wlocal->hMF!=NULL) {

        DeleteEnhMetaFile(g2wlocal->hMF);

	     g2wlocal->hMF=NULL;

      }

      dc=CreateEnhMetaFile(DC,NULL,NULL,"Ngraph\0Graph\0\0");

      g2wsaveDC(dc,g2wlocal,&g2wsave);
      g2wlocal->autoredraw=FALSE;
      GRAredraw(obj,inst,FALSE,FALSE);
      g2wflush(obj,inst,NULL,0,NULL);
      g2wrestoreDC(g2wlocal,&g2wsave);
      g2wlocal->hMF=CloseEnhMetaFile(dc);
      g2wlocal->mfdpi=g2wlocal->windpi;
      SetCursor(hcrOld);
      g2wlocal->MakeMetaFile=FALSE;
    }
    break;
  case '%': case 'X':
    break;
  case 'V':
    g2wlocal->offsetx=dot2pixel(g2wlocal,cpar[1]);
    g2wlocal->offsety=dot2pixel(g2wlocal,cpar[2]);
    g2wlocal->cpx=0;
    g2wlocal->cpy=0;
    if (cpar[5]==1) {
      hClip=CreateRectRgn(dot2pixel(g2wlocal,cpar[1])-g2wlocal->scrollx,
                          dot2pixel(g2wlocal,cpar[2])-g2wlocal->scrolly,
                          dot2pixel(g2wlocal,cpar[3])-g2wlocal->scrollx,
                          dot2pixel(g2wlocal,cpar[4])-g2wlocal->scrolly);
      SelectClipRgn(DC,hClip);
      DeleteObject(hClip);
    } else {
      SelectClipRgn(DC,NULL);
    }
    break;
   case 'A':
    if (cpar[1]==0) {
      style=PS_SOLID;
      g2wlocal->dashn=0;
      memfree(g2wlocal->dashlist);
      g2wlocal->dashlist=NULL;
    } else {
      memfree(g2wlocal->dashlist);
      style=PS_USERSTYLE;
      if ((g2wlocal->dashlist=memalloc(sizeof(int)*cpar[1]))==NULL) break;
      for (i=0;i<cpar[1];i++)
       if ((g2wlocal->dashlist[i]=dot2pixel(g2wlocal,cpar[6+i]))<=0)
         g2wlocal->dashlist[i]=1;
      g2wlocal->dashn=cpar[1];
    }
    g2wlocal->width=dot2pixel(g2wlocal,cpar[2]);
    if (cpar[3]==2) cap=PS_ENDCAP_SQUARE;
    else if (cpar[3]==1) cap=PS_ENDCAP_ROUND;
    else cap=PS_ENDCAP_FLAT;
    if (cpar[4]==2) join=PS_JOIN_BEVEL;
    else if (cpar[4]==1) join=PS_JOIN_ROUND;
    else join=PS_JOIN_MITER;
    g2wlocal->style= PS_GEOMETRIC | style | cap | join;
    lBrush.lbStyle=BS_SOLID;
    lBrush.lbColor=g2wlocal->Col;
    VersionInformation.dwOSVersionInfoSize=sizeof(OSVERSIONINFO);
    GetVersionEx(&VersionInformation);
    if (VersionInformation.dwPlatformId==VER_PLATFORM_WIN32_NT) {
      TmpPen=ExtCreatePen(g2wlocal->style,
                          g2wlocal->width,&lBrush,
                          g2wlocal->dashn,g2wlocal->dashlist);
    } else {
      style=PS_SOLID;
      g2wlocal->style= PS_GEOMETRIC | style | cap | join;
      TmpPen=ExtCreatePen(g2wlocal->style,g2wlocal->width,&lBrush,0,NULL);
    }
    SelectObject(DC,TmpPen);
    DeleteObject(g2wlocal->ThePen);
    g2wlocal->ThePen=TmpPen;
    SetMiterLimit(DC,cpar[5]/100,NULL);
    break;
  case 'G':
    R=cpar[1];
    if (R>255) R=255;
    else if (R<0) R=0;
    G=cpar[2];
    if (G>255) G=255;
    else if (G<0) G=0;
    B=cpar[3];
    if (B>255) B=255;
    else if (B<0) B=0;
    g2wlocal->Col=RGB(R,G,B);
    lBrush.lbStyle=BS_SOLID;
    lBrush.lbColor=g2wlocal->Col;
    if (g2wlocal->style & PS_USERSTYLE) {
      TmpPen=ExtCreatePen(g2wlocal->style,
                          g2wlocal->width,&lBrush,
                          g2wlocal->dashn,g2wlocal->dashlist);
    } else {
      TmpPen=ExtCreatePen(g2wlocal->style,
                          g2wlocal->width,&lBrush,0,NULL);
    }
    SelectObject(DC,TmpPen);
    DeleteObject(g2wlocal->ThePen);
    g2wlocal->ThePen=TmpPen;
    TmpBrush=CreateSolidBrush(g2wlocal->Col);
    SelectObject(DC,TmpBrush);
    DeleteObject(g2wlocal->TheBrush);
    g2wlocal->TheBrush=TmpBrush;
    SetTextColor(DC,g2wlocal->Col);
    break;
  case 'M':
    g2wlocal->cpx=cpar[1];
    g2wlocal->cpy=cpar[2];
    break;
  case 'N':
    g2wlocal->cpx+=cpar[1];
    g2wlocal->cpy+=cpar[2];
    break;
  case 'L':
    BeginPath(DC);
    if ((g2wlocal->dashn!=0) && ((g2wlocal->style & PS_USERSTYLE)==0)) {
      g2w_moveto(g2wlocal,dot2pixelx(g2wlocal,cpar[1]),dot2pixely(g2wlocal,cpar[2]));
      g2w_lineto(g2wlocal,dot2pixelx(g2wlocal,cpar[3]),dot2pixely(g2wlocal,cpar[4]));
    } else {
      MoveToEx(DC,dot2pixelx(g2wlocal,cpar[1]),dot2pixely(g2wlocal,cpar[2]),NULL);
      LineTo(DC,dot2pixelx(g2wlocal,cpar[3]),dot2pixely(g2wlocal,cpar[4]));
    }
    EndPath(DC);
    StrokePath(DC);
    break;
  case 'T':
    if (g2wlocal->linetonum==0) {
      BeginPath(DC);
      if ((g2wlocal->dashn!=0) && ((g2wlocal->style & PS_USERSTYLE)==0))
        g2w_moveto(g2wlocal,dot2pixelx(g2wlocal,g2wlocal->cpx),dot2pixely(g2wlocal,g2wlocal->cpy));
      else
        MoveToEx(DC,dot2pixelx(g2wlocal,g2wlocal->cpx),
                  dot2pixely(g2wlocal,g2wlocal->cpy),NULL);
    }
    if ((g2wlocal->dashn!=0) && ((g2wlocal->style & PS_USERSTYLE)==0))
      g2w_lineto(g2wlocal,dot2pixelx(g2wlocal,cpar[1]),dot2pixely(g2wlocal,cpar[2]));
    else
      LineTo(DC,dot2pixelx(g2wlocal,cpar[1]),dot2pixely(g2wlocal,cpar[2]));
    g2wlocal->linetonum++;
    g2wlocal->cpx=cpar[1];
    g2wlocal->cpy=cpar[2];
    break;
  case 'C':
    if (cpar[6]==36000) {
      if (cpar[7]==0) {
        TmpBrush=(HBRUSH)GetStockObject(NULL_BRUSH);
        SelectObject(DC,TmpBrush);
        Ellipse(DC,
               dot2pixelx(g2wlocal,cpar[1]-cpar[3]),
               dot2pixely(g2wlocal,cpar[2]-cpar[4]),
               dot2pixelx(g2wlocal,cpar[1]+cpar[3]),
               dot2pixely(g2wlocal,cpar[2]+cpar[4]));
        SelectObject(DC,g2wlocal->TheBrush);
        DeleteObject(TmpBrush);
      } else {
        TmpPen=(HPEN)GetStockObject(NULL_PEN);
        SelectObject(DC,TmpPen);
        Ellipse(DC,
               dot2pixelx(g2wlocal,cpar[1]-cpar[3]),
               dot2pixely(g2wlocal,cpar[2]-cpar[4]),
               dot2pixelx(g2wlocal,cpar[1]+cpar[3]),
               dot2pixely(g2wlocal,cpar[2]+cpar[4]));
        SelectObject(DC,g2wlocal->ThePen);
        DeleteObject(TmpPen);
      }
    } else {
      Theta1=cpar[5]*MPI/18000.0;
      Theta2=Theta1+cpar[6]*MPI/18000.0;
      if (cpar[7]==0) {
        Arc(DC,dot2pixelx(g2wlocal,cpar[1]-cpar[3]),
               dot2pixely(g2wlocal,cpar[2]-cpar[4]),
               dot2pixelx(g2wlocal,cpar[1]+cpar[3]),
               dot2pixely(g2wlocal,cpar[2]+cpar[4]),
               dot2pixelx(g2wlocal,cpar[1]+nround(cpar[3]*cos(Theta1)))-1,
               dot2pixely(g2wlocal,cpar[2]-nround(cpar[4]*sin(Theta1)))-1,
               dot2pixelx(g2wlocal,cpar[1]+nround(cpar[3]*cos(Theta2)))-1,
               dot2pixely(g2wlocal,cpar[2]-nround(cpar[4]*sin(Theta2)))-1);
      } else {
        TmpPen=(HPEN)GetStockObject(NULL_PEN);
        SelectObject(DC,TmpPen);
        if (cpar[7]==1) {
          if ((dot2pixel(g2wlocal,cpar[3])<2) && (dot2pixel(g2wlocal,cpar[4])<2)) {
            SetPixel(DC,dot2pixelx(g2wlocal,cpar[1]),dot2pixely(g2wlocal,cpar[2]),g2wlocal->Col);
          }
          Pie(DC,dot2pixelx(g2wlocal,cpar[1]-cpar[3]),
                 dot2pixely(g2wlocal,cpar[2]-cpar[4]),
                 dot2pixelx(g2wlocal,cpar[1]+cpar[3]),
                 dot2pixely(g2wlocal,cpar[2]+cpar[4]),
                 dot2pixelx(g2wlocal,cpar[1]+nround(cpar[3]*cos(Theta1)))-1,
                 dot2pixely(g2wlocal,cpar[2]-nround(cpar[4]*sin(Theta1)))-1,
                 dot2pixelx(g2wlocal,cpar[1]+nround(cpar[3]*cos(Theta2)))-1,
                 dot2pixely(g2wlocal,cpar[2]-nround(cpar[4]*sin(Theta2)))-1);
        } else {
          Chord(DC,dot2pixelx(g2wlocal,cpar[1]-cpar[3]),
                   dot2pixely(g2wlocal,cpar[2]-cpar[4]),
                   dot2pixelx(g2wlocal,cpar[1]+cpar[3]),
                   dot2pixely(g2wlocal,cpar[2]+cpar[4]),
                   dot2pixelx(g2wlocal,cpar[1]+nround(cpar[3]*cos(Theta1)))-1,
                   dot2pixely(g2wlocal,cpar[2]-nround(cpar[4]*sin(Theta1)))-1,
                   dot2pixelx(g2wlocal,cpar[1]+nround(cpar[3]*cos(Theta2)))-1,
                   dot2pixely(g2wlocal,cpar[2]-nround(cpar[4]*sin(Theta2)))-1);
        }
        SelectObject(DC,g2wlocal->ThePen);
        DeleteObject(TmpPen);
      }
    }
    break;
  case 'B':
    if (cpar[5]==0) {
      TmpBrush=(HBRUSH)GetStockObject(NULL_BRUSH);
      SelectObject(DC,TmpBrush);
      Rectangle(DC,dot2pixelx(g2wlocal,cpar[1]),dot2pixely(g2wlocal,cpar[2]),
                   dot2pixelx(g2wlocal,cpar[3]),dot2pixely(g2wlocal,cpar[4]));
      SelectObject(DC,g2wlocal->TheBrush);
      DeleteObject(TmpBrush);
    } else {
      rect.left=dot2pixelx(g2wlocal,cpar[1]);
      rect.top=dot2pixely(g2wlocal,cpar[2]);
      rect.right=dot2pixelx(g2wlocal,cpar[3]);
      rect.bottom=dot2pixely(g2wlocal,cpar[4]);
      FillRect(DC,&rect,g2wlocal->TheBrush);
    }
    break;
  case 'P':
    SetPixel(DC,dot2pixelx(g2wlocal,cpar[1]),dot2pixely(g2wlocal,cpar[2]),
             g2wlocal->Col);
    break;
  case 'R':
    if (cpar[1]==0) break;
    BeginPath(DC);
    MoveToEx(DC,dot2pixelx(g2wlocal,cpar[2]),dot2pixely(g2wlocal,cpar[3]),NULL);
    for (i=1;i<cpar[1];i++) {
      LineTo(DC,dot2pixelx(g2wlocal,cpar[i*2+2]),dot2pixely(g2wlocal,cpar[i*2+3]));
    }
    EndPath(DC);
    StrokePath(DC);
    break;
  case 'D':
    if (cpar[1]==0) break;
    if ((Points=(POINT *)memalloc(sizeof(POINT)*cpar[1]))==NULL) break;
    for (i=0;i<cpar[1];i++) {
      Points[i].x=dot2pixelx(g2wlocal,cpar[i*2+3]);
      Points[i].y=dot2pixely(g2wlocal,cpar[i*2+4]);
    }
    if (cpar[2]==0) {
      TmpBrush=(HBRUSH)GetStockObject(NULL_BRUSH);
      SelectObject(DC,TmpBrush);
      BeginPath(DC);
      MoveToEx(DC,dot2pixelx(g2wlocal,cpar[3]),dot2pixely(g2wlocal,cpar[4]),NULL);
      for (i=1;i<cpar[1];i++) {
        LineTo(DC,dot2pixelx(g2wlocal,cpar[i*2+3]),dot2pixely(g2wlocal,cpar[i*2+4]));
      }
      CloseFigure(DC);
      EndPath(DC);
      StrokePath(DC);
      SelectObject(DC,g2wlocal->TheBrush);
      DeleteObject(TmpBrush);
    } else {
      TmpPen=(HPEN)GetStockObject(NULL_PEN);
      SelectObject(DC,TmpPen);
      if (cpar[2]==1) {
        SetPolyFillMode(DC,ALTERNATE);
        BeginPath(DC);
        Polygon(DC,Points,cpar[1]);
        CloseFigure(DC);
        EndPath(DC);
        FillPath(DC);
      } else {
        SetPolyFillMode(DC,WINDING);
        BeginPath(DC);
        Polygon(DC,Points,cpar[1]);
        CloseFigure(DC);
        EndPath(DC);
        FillPath(DC);
      }
      SelectObject(DC,g2wlocal->ThePen);
      DeleteObject(TmpPen);
    }
    memfree(Points);
    break;
  case 'F':
    memfree(g2wlocal->fontalias);
    if ((g2wlocal->fontalias=memalloc(strlen(cstr)+1))==NULL) break;
    strcpy(g2wlocal->fontalias,cstr);
    break;
  case 'H':
    g2wlocal->loadfontf=FALSE;
    fontspace=cpar[2]/72.0*25.4;
    g2wlocal->fontspace=fontspace;
    fontsize=cpar[1]/72.0*25.4;
    fontdir=cpar[3]/100.0;
    g2wlocal->fontsin=sin(fontdir/180*MPI);
    g2wlocal->fontcos=cos(fontdir/180*MPI);
    fcur=g2wlocal->fontmaproot;
    fontname=NULL;
    while (fcur!=NULL) {
      if (strcmp(g2wlocal->fontalias,fcur->fontalias)==0) {
        fontname=fcur->fontname;
        charset=fcur->charset;
        italic=fcur->italic;
        break;
      }
      fcur=fcur->next;
    }
    if (fontname==NULL) {
      g2wlocal->loadfontf=FALSE;
      break;
    }
    IDFont.lfHeight=-dot2pixel(g2wlocal,fontsize);
    IDFont.lfWidth=0;
    IDFont.lfEscapement=IDFont.lfOrientation=nround(fontdir*10);
    IDFont.lfUnderline=0;
    IDFont.lfStrikeOut=0;
    IDFont.lfPitchAndFamily=0;
    if (charset==ANSI2_CHARSET) IDFont.lfCharSet=ANSI_CHARSET;
    else IDFont.lfCharSet=charset;
    IDFont.lfOutPrecision=0;
    IDFont.lfClipPrecision=0;
    IDFont.lfQuality=0;
    if (italic==FONT_ROMAN) {
      IDFont.lfWeight=400;
      IDFont.lfItalic=FALSE;
    } else if (italic==FONT_ITALIC) {
      IDFont.lfWeight=400;
      IDFont.lfItalic=TRUE;
    } else if (italic==FONT_BOLD) {
      IDFont.lfWeight=700;
      IDFont.lfItalic=FALSE;
    } else if (italic==FONT_BOLDITALIC) {
      IDFont.lfWeight=700;
      IDFont.lfItalic=TRUE;
    } else {
      IDFont.lfWeight=0;
      IDFont.lfItalic=0;
    }
    strcpy(IDFont.lfFaceName,fontname);
    if ((TmpFont=CreateFontIndirect(&IDFont))!=NULL) {
      SelectObject(DC,TmpFont);
      DeleteObject(g2wlocal->TheFont);
      g2wlocal->TheFont=TmpFont;
      SetTextAlign(DC,TA_BASELINE | TA_LEFT);
      SetTextCharacterExtra(DC,dot2pixel(g2wlocal,g2wlocal->fontspace));
      SetBkMode(DC,TRANSPARENT);
      g2wlocal->loadfontf=TRUE;
      g2wlocal->charset=charset;
    }
    break;
  case 'S':
    if (!g2wlocal->loadfontf) break;
    if ((s=nstrnew())==NULL) break;
    i=0;
    while (i<strlen(cstr)) {
      if (cstr[i]=='\\') {
        if (cstr[i+1]=='x') {
          if (toupper(cstr[i+2])>='A') ch[0]=toupper(cstr[i+2])-'A'+10;
          else ch[0]=cstr[i+2]-'0';
          if (toupper(cstr[i+3])>='A') ch[0]=ch[0]*16+toupper(cstr[i+3])-'A'+10;
          else ch[0]=ch[0]*16+cstr[i+3]-'0';
          i+=4;
        } else if (cstr[i+1]=='\\') {
          ch[0]=cstr[i+1];
          i+=2;
        } else {
          ch[0]=cstr[i];
          i++;
        }
      } else {
        ch[0]=cstr[i];
        i++;
      }
      s=nstrccat(s,ch[0]);
    }
    if (s==NULL) break;
    x0=g2wlocal->cpx;
    y0=g2wlocal->cpy;
    if ((g2wlocal->charset==ANSI_CHARSET) && (!g2wlocal->minus_hyphen)) {
      if ((sw=(char *)memalloc(strlen(s)*2))!=NULL) {
        for (i=0;i<strlen(s);i++) {
          if (s[i]=='-') {
            sw[2*i]='\x12';
            sw[2*i+1]='\x22';
          } else {
            sw[2*i]=s[i];
            sw[2*i+1]='\0';
          }
        }
        ExtTextOutW(DC,dot2pixelx(g2wlocal,nround(x0)),
                  dot2pixely(g2wlocal,nround(y0)),NULL,NULL,(wchar_t *)sw,strlen(s),NULL);
        GetTextExtentPoint32W(DC,(wchar_t *)sw,strlen(s),&lpsize);
      }
      memfree(sw);
    } else {
      if ((g2wlocal->charset==ANSI_CHARSET) && (g2wlocal->minus_hyphen)) {
        for (i=0;i<strlen(s);i++) {
          if (s[i]=='-') s[i]='\x96';
          else if (s[i]=='\x96') s[i]='-';
        }
      }
      ExtTextOut(DC,dot2pixelx(g2wlocal,nround(x0)),
                  dot2pixely(g2wlocal,nround(y0)),NULL,NULL,s,strlen(s),NULL);
      GetTextExtentPoint32(DC,s,strlen(s),&lpsize);
    }
    memfree(s);
    fontwidth=pixel2dot(g2wlocal,lpsize.cx);
    x0+=fontwidth*g2wlocal->fontcos;
    y0-=fontwidth*g2wlocal->fontsin;
    g2wlocal->cpx=nround(x0);
    g2wlocal->cpy=nround(y0);
    break;
  case 'K':
    if (!g2wlocal->loadfontf) break;
    x0=g2wlocal->cpx;
    y0=g2wlocal->cpy;
    ExtTextOut(DC,dot2pixelx(g2wlocal,nround(x0)),
               dot2pixely(g2wlocal,nround(y0)),NULL,NULL,cstr,strlen(cstr),NULL);
    GetTextExtentPoint32(DC,cstr,strlen(cstr),&lpsize);
    fontwidth=pixel2dot(g2wlocal,lpsize.cx);
    x0+=fontwidth*g2wlocal->fontcos;
    y0-=fontwidth*g2wlocal->fontsin;
    g2wlocal->cpx=nround(x0);
    g2wlocal->cpy=nround(y0);
    break;
  default: break;
  }
  return 0;
}

int g2win_charwidth(struct objlist *obj,char *inst,char *rval,
                    int argc,char **argv)
{
  struct g2wlocal *g2wlocal;
  char ch[3];
  char *font;
  double size;
  char *fontname;
  struct fontmap *fcur;
  HFONT TmpFont;
  LOGFONT IDFont;
  SIZE lpsize;
  HDC DC;
  int charset,italic;

  ch[0]=(*(unsigned int *)(argv[3]) & 0xff);
  ch[1]=(*(unsigned int *)(argv[3]) & 0xff00) >> 8;
  ch[2]='\0';
  size=(*(int *)(argv[4]))/72.0*25.4;
  font=(char *)(argv[5]);
  _getobj(obj,"_local",inst,&g2wlocal);
  fcur=g2wlocal->fontmaproot;
  fontname=NULL;
  while (fcur!=NULL) {
    if (strcmp(font,fcur->fontalias)==0) {
      fontname=fcur->fontname;
      charset=fcur->charset;
      italic=fcur->italic;
      break;
    }
    fcur=fcur->next;
  }
  if (fontname==NULL) {
    *(int *)rval=nround(size*0.600);
  } else {
    IDFont.lfHeight=-size;
    IDFont.lfWidth=0;
    IDFont.lfEscapement=0;
    IDFont.lfOrientation=0;
    IDFont.lfUnderline=0;
    IDFont.lfStrikeOut=0;
    IDFont.lfPitchAndFamily=0;
    if (charset==ANSI2_CHARSET) IDFont.lfCharSet=ANSI_CHARSET;
    else IDFont.lfCharSet=charset;
    IDFont.lfOutPrecision=0;
    IDFont.lfClipPrecision=0;
    IDFont.lfQuality=0;
    if (italic==FONT_ROMAN) {
      IDFont.lfWeight=400;
      IDFont.lfItalic=FALSE;
    } else if (italic==FONT_ITALIC) {
      IDFont.lfWeight=400;
      IDFont.lfItalic=TRUE;
    } else if (italic==FONT_BOLD) {
      IDFont.lfWeight=700;
      IDFont.lfItalic=FALSE;
    } else if (italic==FONT_BOLDITALIC) {
      IDFont.lfWeight=700;
      IDFont.lfItalic=TRUE;
    } else {
      IDFont.lfWeight=0;
      IDFont.lfItalic=0;
    }
    strcpy(IDFont.lfFaceName,fontname);
    if ((TmpFont=CreateFontIndirect(&IDFont))!=NULL) {
      DC=GetDC(g2wlocal->hWnd);
      SelectObject(DC,TmpFont);
      SetTextAlign(DC,TA_BASELINE | TA_LEFT);
      if (IDFont.lfCharSet==SHIFTJIS_CHARSET)
        GetTextExtentPoint32(DC,ch,2,&lpsize);
      else if ((charset==ANSI_CHARSET) && (!g2wlocal->minus_hyphen)) {
        if (ch[0]=='-') {
          ch[0]='\x12';
          ch[1]='\x22';
        } else ch[1]='\0';
        GetTextExtentPoint32W(DC,(wchar_t *)ch,1,&lpsize);
      } else {
         if (charset==ANSI_CHARSET) {
           if (ch[0]=='-') ch[0]='\x96';
           else if (ch[0]=='\x96') ch[0]='-';
         }
         GetTextExtentPoint32(DC,ch,1,&lpsize);
      }
      *(int *)rval=lpsize.cx;
      ReleaseDC(g2wlocal->hWnd,DC);
      DeleteObject(TmpFont);
    } else *(int *)rval=nround(size*0.600);
  }
  return 0;
}

int g2win_charheight(struct objlist *obj,char *inst,char *rval,
                     int argc,char **argv)
{
  struct g2wlocal *g2wlocal;
  char *font;
  double size;
  char *fontname;
  struct fontmap *fcur;
  HFONT TmpFont;
  LOGFONT IDFont;
  TEXTMETRIC tm;
  HDC DC;
  int charset,italic;
  char *func;
  int height;

  func=(char *)argv[1];
  if (strcmp0(func,"_charascent")==0) height=TRUE;
  else height=FALSE;
  size=(*(int *)(argv[3]))/72.0*25.4;
  font=(char *)(argv[4]);
  _getobj(obj,"_local",inst,&g2wlocal);
  fcur=g2wlocal->fontmaproot;
  fontname=NULL;
  while (fcur!=NULL) {
    if (strcmp(font,fcur->fontalias)==0) {
      fontname=fcur->fontname;
      charset=fcur->charset;
      italic=fcur->italic;
      break;
    }
    fcur=fcur->next;
  }
  if (fontname==NULL) {
    if (height) *(int *)rval=nround(size*0.562);
    else *(int *)rval=nround(size*0.250);
  } else {
    IDFont.lfHeight=-size;
    IDFont.lfWidth=0;
    IDFont.lfEscapement=0;
    IDFont.lfOrientation=0;
    IDFont.lfUnderline=0;
    IDFont.lfStrikeOut=0;
    IDFont.lfPitchAndFamily=0;
    if (charset==ANSI2_CHARSET) IDFont.lfCharSet=ANSI_CHARSET;
    else IDFont.lfCharSet=charset;
    IDFont.lfOutPrecision=0;
    IDFont.lfClipPrecision=0;
    IDFont.lfQuality=0;
    if (italic==FONT_ROMAN) {
      IDFont.lfWeight=400;
      IDFont.lfItalic=FALSE;
    } else if (italic==FONT_ITALIC) {
      IDFont.lfWeight=400;
      IDFont.lfItalic=TRUE;
    } else if (italic==FONT_BOLD) {
      IDFont.lfWeight=700;
      IDFont.lfItalic=FALSE;
    } else if (italic==FONT_BOLDITALIC) {
      IDFont.lfWeight=700;
      IDFont.lfItalic=TRUE;
    } else {
      IDFont.lfWeight=0;
      IDFont.lfItalic=0;
    }
    strcpy(IDFont.lfFaceName,fontname);
    if ((TmpFont=CreateFontIndirect(&IDFont))!=NULL) {
      DC=GetDC(g2wlocal->hWnd);
      SelectObject(DC,TmpFont);
      GetTextMetrics(DC,&tm);
      if (height) *(int *)rval=tm.tmAscent;
      else *(int *)rval=tm.tmDescent;
      ReleaseDC(g2wlocal->hWnd,DC);
	  DeleteObject(TmpFont);
    } else {
      if (height) *(int *)rval=nround(size*0.562);
      else *(int *)rval=nround(size*0.250);
    }
  }
  return 0;
}

#define TBLNUM 18

struct objtable gra2win[TBLNUM] = {
  {"init",NVFUNC,NEXEC,g2wininit,NULL,0},
  {"done",NVFUNC,NEXEC,g2windone,NULL,0},
  {"next",NPOINTER,0,NULL,NULL,0},
  {"dpi",NINT,NREAD|NWRITE,g2windpi,NULL,0},
  {"width",NINT,NREAD|NWRITE,g2winsize,NULL,0},
  {"height",NINT,NREAD|NWRITE,g2winsize,NULL,0},
  {"auto_redraw",NBOOL,NREAD|NWRITE,g2winautoredraw,NULL,0},
  {"store_in_memory",NBOOL,NREAD|NWRITE,g2winstorememory,NULL,0},
  {"redraw",NVFUNC,NREAD|NEXEC,g2winredraw,"",0},
  {"flush",NVFUNC,NREAD|NEXEC,g2wflush,"",0},
  {"clear",NVFUNC,NREAD|NEXEC,g2winclear,"",0},
  {"_output",NVFUNC,0,g2win_output,NULL,0},
  {"_charwidth",NIFUNC,0,g2win_charwidth,NULL,0},
  {"_charascent",NIFUNC,0,g2win_charheight,NULL,0},
  {"_chardescent",NIFUNC,0,g2win_charheight,NULL,0},
  {"_evloop",NVFUNC,0,g2win_evloop,NULL,0},
  {"_local",NPOINTER,0,NULL,NULL,0},
  {"disconnect",NVFUNC,NREAD|NEXEC,g2windisconnect,"",0},
};

void *addgra2win()
/* addgra2win() returns NULL on error */
{
  return addobject(NAME,NULL,PARENT,VERSION,TBLNUM,gra2win,ERRNUM,g2winerrorlist,NULL,g2windoneproc);
}
