/* 
 * 
 * This file is part of "Ngraph for Windows".
 * 
 * Copyright (C) 2015, Satoshi ISHIZAKA. isizaka@msa.biglobe.ne.jp
 * 
 * "Ngraph for Windows" is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * "Ngraph for Windows" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

/*
 *
 * winaxis.cpp
 *
 */

#include <owl\pch.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

extern "C" {
#include "ngraph.h"
#include "object.h"
#include "nstring.h"
#include "ioutil.h"
#include "mathfn.h"
#include "owinmenu.h"
}

#include "bitmap.rh"
#include "winmenu.rh"
#include "winmenu.h"
#include "windialg.h"
#include "wincommn.h"
#include "winaxis.h"

DEFINE_RESPONSE_TABLE1(AxisComboBox,TComboBox)
  EV_WM_DRAWITEM,
END_RESPONSE_TABLE;

void AxisComboBox::EvDrawItem(UINT ctrlId,DRAWITEMSTRUCT far& draw)
{
  int x,y;
  TEXTMETRIC tm;
  COLORREF clrBackground;
  COLORREF clrForeground;
  HDC DC;
  struct objlist *aobj;
  char *name;

  TComboBox::EvDrawItem(ctrlId,draw);
  clrForeground=::SetTextColor(draw.hDC,
                ::GetSysColor(draw.itemState & ODS_SELECTED ?
                COLOR_HIGHLIGHTTEXT:COLOR_WINDOWTEXT));
  clrBackground=::SetBkColor(draw.hDC,
                ::GetSysColor(draw.itemState & ODS_SELECTED ?
                COLOR_HIGHLIGHT:COLOR_WINDOW));
  DC=draw.hDC;
  GetTextMetrics(DC,&tm);
  y=(draw.rcItem.bottom+draw.rcItem.top-tm.tmHeight) / 2;
  x=LOWORD(GetDialogBaseUnits())/4;
  aobj=getobject("axis");
  getobj(aobj,"group",draw.itemID,0,NULL,&name);
  if (name==NULL) name="";
  ExtTextOut(DC,x,y,
             ETO_CLIPPED | ETO_OPAQUE,&(draw.rcItem),
             name,lstrlen(name),NULL);
  SetTextColor(DC,clrForeground);
  SetBkColor(DC,clrBackground);
  if (draw.itemState & ODS_FOCUS) DrawFocusRect(DC,&(draw.rcItem));
}

char *AxisCB(struct objlist *obj,int id)
{
  char *s;
  int dir;
  char *valstr;
  char *name;

  if ((s=(char *)memalloc(128))==NULL) return NULL;
  getobj(obj,"direction",id,0,NULL,&dir);
  getobj(obj,"group",id,0,NULL,&name);
  if (name==NULL) name="";
  sgetobjfield(obj,id,"type",NULL,&valstr,FALSE,FALSE,FALSE);
  sprintf(s,"%-5d %-10s %.6s dir:%d",id,name,valstr,dir);
  memfree(valstr);
  return s;
}

char *GridCB(struct objlist *obj,int id)
{
  char *s,*s1,*s2;

  if ((s=(char *)memalloc(128))==NULL) return NULL;
  getobj(obj,"axis_x",id,0,NULL,&s1);
  getobj(obj,"axis_y",id,0,NULL,&s2);
  sprintf(s,"%-5d %.8s %.8s",id,s1,s2);
  return s;
}

class ZoomDialog  : public TMyDialog {
  public:
    ZoomDialog(TWindow *parent,TResId resID);
    int zoom;
  protected:
    void SetupWindow() {
      TMyDialog::SetupWindow();
      SetupItem();
    }
    void SetupItem();
    void CloseWindow(int retVal);
};

ZoomDialog::ZoomDialog(TWindow *parent,TResId resID):TMyDialog(parent,resID)
{
  zoom=20000;
}

void ZoomDialog::SetupItem()
{
  char buf[64];

  sprintf(buf,"%d",zoom);
  SetDlgItemText(IDZMZOOM,buf);
}

void ZoomDialog::CloseWindow(int retVal)
{
  int len,a;
  char *buf;
  char *endptr;

  len=SendDlgItemMessage(IDZMZOOM,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)memalloc(len+1))!=NULL) {
    GetDlgItemText(IDZMZOOM,buf,len+1);
    a=strtol(buf,&endptr,10);
    if (endptr[0]=='\0') zoom=a;
    memfree(buf);
  }
  TMyDialog::CloseWindow(retVal);
}


DEFINE_RESPONSE_TABLE1(CrossDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDACBX,BN_CLICKED,AxisXClicked),
  EV_CHILD_NOTIFY(IDACBY,BN_CLICKED,AxisYClicked),
  EV_CHILD_NOTIFY(IDDELETE,BN_CLICKED,DeleteClicked),
END_RESPONSE_TABLE;

CrossDialog::CrossDialog(TWindow *parent,TResId resID,int x,int y,int lenx,int leny,
                         struct objlist *obj,int idx,int idy)
:TMyDialog(parent,resID)
{
  X0=X=x;
  Y0=Y=y;
  LenX0=LenX=lenx;
  LenY0=LenY=leny;
  Obj=obj;
  IDX=idx;
  IDY=idy;
  OX=new TStatic(this,IDACOX);
  OY=new TStatic(this,IDACOY);
}

CrossDialog::~CrossDialog()
{
  delete OX;
  delete OY;
}

void CrossDialog::SetupItem()
{
  char buf[256];

  sprintf(buf,"%d",X);
  SetDlgItemText(IDACX,buf);
  sprintf(buf,"%d",Y);
  SetDlgItemText(IDACY,buf);
  sprintf(buf,"%d",LenX);
  SetDlgItemText(IDACW,buf);
  sprintf(buf,"%d",LenY);
  SetDlgItemText(IDACH,buf);
  if (IDX!=-1) sprintf(buf,"id:%d",IDX);
  else buf[0]='\0';
  OX->SetText(buf);
  if (IDY!=-1) sprintf(buf,"id:%d",IDY);
  else buf[0]='\0';
  OY->SetText(buf);
}

void CrossDialog::CloseWindow(int retVal)
{
  struct narray group;
  char *argv[2];
  int type,len,a;
  char *buf,*endptr;

  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  len=SendDlgItemMessage(IDACX,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)memalloc(len+1))!=NULL) {
    GetDlgItemText(IDACX,buf,len+1);
    a=strtol(buf,&endptr,10);
    if (endptr[0]!='\0') {
      memfree(buf);
      return;
    }
    X=a;
    memfree(buf);
  }
  len=SendDlgItemMessage(IDACY,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)memalloc(len+1))!=NULL) {
    GetDlgItemText(IDACY,buf,len+1);
    a=strtol(buf,&endptr,10);
    if (endptr[0]!='\0') {
      memfree(buf);
      return;
    }
    Y=a;
    memfree(buf);
  }
  len=SendDlgItemMessage(IDACW,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)memalloc(len+1))!=NULL) {
    GetDlgItemText(IDACW,buf,len+1);
    a=strtol(buf,&endptr,10);
    if (endptr[0]!='\0') {
      memfree(buf);
      return;
    }
    LenX=a;
    memfree(buf);
  }
  len=SendDlgItemMessage(IDACH,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)memalloc(len+1))!=NULL) {
    GetDlgItemText(IDACH,buf,len+1);
    a=strtol(buf,&endptr,10);
    if (endptr[0]!='\0') {
      memfree(buf);
      return;
    }
    LenY=a;
    memfree(buf);
  }
  if ((X!=X0) || (Y!=Y0) || (LenX!=LenX0) || (LenY!=LenY0)) {
    arrayinit(&group,sizeof(int));
    type=3;
    arrayadd(&group,&type);
    arrayadd(&group,&IDX);
    arrayadd(&group,&IDY);
    arrayadd(&group,&X);
    arrayadd(&group,&Y);
    arrayadd(&group,&LenX);
    arrayadd(&group,&LenY);
    argv[0]=(char *)&group;
    argv[1]=NULL;
    exeobj(Obj,"group_position",IDX,1,argv);
    arraydel(&group);
  }
  if ((IDX!=-1) && (IDY!=-1)) {
    exeobj(Obj,"adjust",IDX,0,NULL);
    exeobj(Obj,"adjust",IDY,0,NULL);
  }
  TMyDialog::CloseWindow(retVal);
}

void CrossDialog::AxisXClicked()
{
  AxisDialog *dlg;

  if (IDX>=0) {
    dlg=new AxisDialog(this,DIALOG_AXIS,Obj,IDX,FALSE);
    dlg->Execute();
    delete dlg;
  }
}

void CrossDialog::AxisYClicked()
{
  AxisDialog *dlg;

  if (IDY>=0) {
    dlg=new AxisDialog(this,DIALOG_AXIS,Obj,IDY,FALSE);
    dlg->Execute();
    delete dlg;
  }
}


DEFINE_RESPONSE_TABLE1(SectionDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDASBX,BN_CLICKED,AxisXClicked),
  EV_CHILD_NOTIFY(IDASBY,BN_CLICKED,AxisYClicked),
  EV_CHILD_NOTIFY(IDASBU,BN_CLICKED,AxisUClicked),
  EV_CHILD_NOTIFY(IDASBR,BN_CLICKED,AxisRClicked),
  EV_CHILD_NOTIFY(IDASBGRID,BN_CLICKED,GridClicked),
  EV_CHILD_NOTIFY(IDDELETE,BN_CLICKED,DeleteClicked),
END_RESPONSE_TABLE;

SectionDialog::SectionDialog(TWindow *parent,TResId resID,int x,int y,int lenx,int leny,
                           struct objlist *obj,int idx,int idy,int idu,int idr,
                           struct objlist *obj2,int *idg,int section)
:TMyDialog(parent,resID)
{
  X0=X=x;
  Y0=Y=y;
  LenX0=LenX=lenx;
  LenY0=LenY=leny;
  Obj=obj;
  Obj2=obj2;
  IDX=idx;
  IDY=idy;
  IDU=idu;
  IDR=idr;
  IDG=idg;
  Section=section;
  OX=new TStatic(this,IDASOX);
  OY=new TStatic(this,IDASOY);
  OU=new TStatic(this,IDASOU);
  OR=new TStatic(this,IDASOR);
  OG=new TStatic(this,IDASOGRID);
}

SectionDialog::~SectionDialog()
{
  delete OX;
  delete OY;
  delete OU;
  delete OR;
  delete OG;
}

void SectionDialog::SetupItem()
{
  char buf[256];

  sprintf(buf,"%d",X);
  SetDlgItemText(IDASX,buf);
  sprintf(buf,"%d",Y);
  SetDlgItemText(IDASY,buf);
  sprintf(buf,"%d",LenX);
  SetDlgItemText(IDASW,buf);
  sprintf(buf,"%d",LenY);
  SetDlgItemText(IDASH,buf);
  if (IDX!=-1) sprintf(buf,"id:%d",IDX);
  else buf[0]='\0';
  OX->SetText(buf);
  if (IDY!=-1) sprintf(buf,"id:%d",IDY);
  else buf[0]='\0';
  OY->SetText(buf);
  if (IDU!=-1) sprintf(buf,"id:%d",IDU);
  else buf[0]='\0';
  OU->SetText(buf);
  if (IDR!=-1) sprintf(buf,"id:%d",IDR);
  else buf[0]='\0';
  OR->SetText(buf);
  if (*IDG!=-1) sprintf(buf,"id:%d",*IDG);
  else buf[0]='\0';
  OG->SetText(buf);
}

void SectionDialog::CloseWindow(int retVal)
{
  struct narray group;
  char *argv[2];
  int type,len,a;
  char *buf,*endptr;

  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  len=SendDlgItemMessage(IDASX,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)memalloc(len+1))!=NULL) {
    GetDlgItemText(IDASX,buf,len+1);
    a=strtol(buf,&endptr,10);
    if (endptr[0]!='\0') {
      memfree(buf);
      return;
    }
    X=a;
    memfree(buf);
  }
  len=SendDlgItemMessage(IDASY,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)memalloc(len+1))!=NULL) {
    GetDlgItemText(IDASY,buf,len+1);
    a=strtol(buf,&endptr,10);
    if (endptr[0]!='\0') {
      memfree(buf);
      return;
    }
    Y=a;
    memfree(buf);
  }
  len=SendDlgItemMessage(IDASW,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)memalloc(len+1))!=NULL) {
    GetDlgItemText(IDASW,buf,len+1);
    a=strtol(buf,&endptr,10);
    if (endptr[0]!='\0') {
      memfree(buf);
      return;
    }
    LenX=a;
    memfree(buf);
  }
  len=SendDlgItemMessage(IDASH,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)memalloc(len+1))!=NULL) {
    GetDlgItemText(IDASH,buf,len+1);
    a=strtol(buf,&endptr,10);
    if (endptr[0]!='\0') {
      memfree(buf);
      return;
    }
    LenY=a;
    memfree(buf);
  }
  if ((X!=X0) || (Y!=Y0) || (LenX0!=LenX) || (LenY0!=LenY)) {
    arrayinit(&group,sizeof(int));
    if (Section) type=2;
    else type=1;
    arrayadd(&group,&type);
    arrayadd(&group,&IDX);
    arrayadd(&group,&IDY);
    arrayadd(&group,&IDU);
    arrayadd(&group,&IDR);
    arrayadd(&group,&X);
    arrayadd(&group,&Y);
    arrayadd(&group,&LenX);
    arrayadd(&group,&LenY);
    argv[0]=(char *)&group;
    argv[1]=NULL;
    exeobj(Obj,"group_position",IDX,1,argv);
    arraydel(&group);
  }
  TMyDialog::CloseWindow(retVal);
}

void SectionDialog::AxisXClicked()
{
  AxisDialog *dlg;

  if (IDX>=0) {
    dlg=new AxisDialog(this,DIALOG_AXIS,Obj,IDX,FALSE);
    dlg->Execute();
    delete dlg;
  }
}

void SectionDialog::AxisYClicked()
{
  AxisDialog *dlg;

  if (IDY>=0) {
    dlg=new AxisDialog(this,DIALOG_AXIS,Obj,IDY,FALSE);
    dlg->Execute();
    delete dlg;
  }
}

void SectionDialog::AxisUClicked()
{
  AxisDialog *dlg;

  if (IDU>=0) {
    dlg=new AxisDialog(this,DIALOG_AXIS,Obj,IDU,FALSE);
    dlg->Execute();
    delete dlg;
  }
}

void SectionDialog::AxisRClicked()
{
  AxisDialog *dlg;

  if (IDR>=0) {
    dlg=new AxisDialog(this,DIALOG_AXIS,Obj,IDR,FALSE);
    dlg->Execute();
    delete dlg;
  }
}

void SectionDialog::GridClicked()
{
  GridDialog *dlg;
  char *ref;
  int oidx,oidy;

  if (*IDG==-1) {
    if ((*IDG=newobj(Obj2))>=0) {
      if ((ref=(char *)memalloc(15))!=NULL) {
        getobj(Obj,"oid",IDX,0,NULL,&oidx);
        sprintf(ref,"axis:^%d",oidx);
        putobj(Obj2,"axis_x",*IDG,ref);
      }
      if ((ref=(char *)memalloc(15))!=NULL) {
        getobj(Obj,"oid",IDY,0,NULL,&oidy);
        sprintf(ref,"axis:^%d",oidy);
        putobj(Obj2,"axis_y",*IDG,ref);
      }
    }
  }
  if (*IDG>=0) {
    dlg=new GridDialog(this,DIALOG_GRID,Obj2,*IDG,TRUE);
    if (dlg->Execute()==IDDELETE) {
      delobj(Obj2,*IDG);
      *IDG=-1;
    }
    delete dlg;
  }
  SetupItem();
  NgraphApp->Changed=TRUE;
}

DEFINE_RESPONSE_TABLE1(GridDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
  EV_CHILD_NOTIFY(IDDELETE,BN_CLICKED,DeleteClicked),
END_RESPONSE_TABLE;

GridDialog::GridDialog(TWindow *parent,TResId resID,struct objlist *obj,int id,int candel)
:TMyDialog(parent,resID)
{
  Obj=obj;
  Id=id;
  CanDel=candel;
  Col=new ColorButton(this,IDAGCOLD);
  ColB=new ColorButton(this,IDAGBACKCOL);
  AxisX=new AxisComboBox(this,IDAGAX);
  AxisY=new AxisComboBox(this,IDAGAY);
}

GridDialog::~GridDialog()
{
  delete Col;
  delete ColB;
  delete AxisX;
  delete AxisY;
}

void GridDialog::SetupWindow()
{
  char *title;
  int len;

  TMyDialog::SetupWindow();
  if (!CanDel) ::EnableWindow(GetDlgItem(IDDELETE),FALSE);
  len=GetWindowTextLength();
  if ((title=(char *)memalloc(len+10))!=NULL) {
    GetWindowText(title,len+1);
    sprintf(title+len," %d",Id);
    SetCaption(title);
    memfree(title);
  }
  SetupItem(Id);
}

void GridDialog::SetupItem(int id)
{
  char *valstr;
  int i,j,R,G,B;
  int lastinst,oid;
  char buf[10];
  struct objlist *aobj;

  aobj=getobject("axis");
  lastinst=chkobjlastinst(aobj);
  SendDlgItemMessage(IDAGAX,CB_RESETCONTENT,0,0);
  SendDlgItemMessage(IDAGAY,CB_RESETCONTENT,0,0);
  for (j=0;j<=lastinst;j++) {
    getobj(aobj,"oid",j,0,NULL,&oid);
    sprintf(buf,"^%d",oid);
    SendDlgItemMessage(IDAGAX,CB_INSERTSTRING,j,(LPARAM)buf);
    SendDlgItemMessage(IDAGAY,CB_INSERTSTRING,j,(LPARAM)buf);
  }
  sgetobjfield(Obj,id,"axis_x",NULL,&valstr,FALSE,FALSE,FALSE);
  for (i=0;(valstr[i]!='\0') && (valstr[i]!=':');i++);
  if (valstr[i]==':') i++;
  SetDlgItemText(IDAGAX,valstr+i); memfree(valstr);
  sgetobjfield(Obj,id,"axis_y",NULL,&valstr,FALSE,FALSE,FALSE);
  for (i=0;(valstr[i]!='\0') && (valstr[i]!=':');i++);
  if (valstr[i]==':') i++;
  SetDlgItemText(IDAGAY,valstr+i); memfree(valstr);
  SetStyleFromObjField(Handle,IDAGSTYLE1,Obj,id,"style1");
  SetStyleFromObjField(Handle,IDAGSTYLE2,Obj,id,"style2");
  SetStyleFromObjField(Handle,IDAGSTYLE3,Obj,id,"style3");
  SetComboList(Handle,IDAGWIDTH1,cblinewidth,CBLINEWIDTH);
  SetTextFromObjField(Handle,IDAGWIDTH1,Obj,id,"width1");
  SetComboList(Handle,IDAGWIDTH2,cblinewidth,CBLINEWIDTH);
  SetTextFromObjField(Handle,IDAGWIDTH2,Obj,id,"width2");
  SetComboList(Handle,IDAGWIDTH3,cblinewidth,CBLINEWIDTH);
  SetTextFromObjField(Handle,IDAGWIDTH3,Obj,id,"width3");
  SetToggleFromObjField(Handle,IDAGBACK,Obj,id,"background");
  getobj(Obj,"R",id,0,NULL,&R);
  getobj(Obj,"G",id,0,NULL,&G);
  getobj(Obj,"B",id,0,NULL,&B);
  Col->SetColor(R,G,B);
  getobj(Obj,"BR",id,0,NULL,&R);
  getobj(Obj,"BG",id,0,NULL,&G);
  getobj(Obj,"BB",id,0,NULL,&B);
  ColB->SetColor(R,G,B);
}

void GridDialog::CloseWindow(int retVal)
{
  int len;
  char *buf;
  int R,G,B;

  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  len=SendDlgItemMessage(IDAGAX,WM_GETTEXTLENGTH,0,0);
  if ((len>0) && ((buf=(char *)memalloc(len+6))!=NULL)) {
    strcpy(buf,"axis:");
    GetDlgItemText(IDAGAX,buf+5,len+1);
    if (sputobjfield(Obj,Id,"axis_x",buf)!=0) {
      memfree(buf);
      return;
    }
    memfree(buf);
  } else putobj(Obj,"axis_x",Id,NULL);
  len=SendDlgItemMessage(IDAGAY,WM_GETTEXTLENGTH,0,0);
  if ((len>0) && ((buf=(char *)memalloc(len+6))!=NULL)) {
    strcpy(buf,"axis:");
    GetDlgItemText(IDAGAY,buf+5,len+1);
    if (sputobjfield(Obj,Id,"axis_y",buf)!=0) {
      memfree(buf);
      return;
    }
    memfree(buf);
  } else putobj(Obj,"axis_y",Id,NULL);
  if (SetObjFieldFromStyle(Handle,IDAGSTYLE1,Obj,Id,"style1")) return;
  if (SetObjFieldFromStyle(Handle,IDAGSTYLE2,Obj,Id,"style2")) return;
  if (SetObjFieldFromStyle(Handle,IDAGSTYLE3,Obj,Id,"style3")) return;
  if (SetObjFieldFromText(Handle,IDAGWIDTH1,Obj,Id,"width1")) return;
  if (SetObjFieldFromText(Handle,IDAGWIDTH2,Obj,Id,"width2")) return;
  if (SetObjFieldFromText(Handle,IDAGWIDTH3,Obj,Id,"width3")) return;
  if (SetObjFieldFromToggle(Handle,IDAGBACK,Obj,Id,"background")) return;
  Col->GetColor(&R,&G,&B);
  if (putobj(Obj,"R",Id,&R)==-1) return;
  if (putobj(Obj,"G",Id,&G)==-1) return;
  if (putobj(Obj,"B",Id,&B)==-1) return;
  ColB->GetColor(&R,&G,&B);
  if (putobj(Obj,"BR",Id,&R)==-1) return;
  if (putobj(Obj,"BG",Id,&G)==-1) return;
  if (putobj(Obj,"BB",Id,&B)==-1) return;
  TMyDialog::CloseWindow(retVal);
}



class GaugeDialog  : public TMyDialog {
  public:
    GaugeDialog(TWindow *parent,TResId resID,struct objlist *obj,int id);
    ~GaugeDialog();
  protected:
    struct objlist *Obj;
    int Id;
    ColorButton *Col;
    void SetupWindow();
    void SetupItem(int id);
    void CloseWindow(int retVal);
    void CopyClicked() {
      int sel;
      if ((sel=CopyClick(this,Obj,Id,AxisCB))!=-1) SetupItem(sel);
    }
    DECLARE_RESPONSE_TABLE(GaugeDialog);
};

DEFINE_RESPONSE_TABLE1(GaugeDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
END_RESPONSE_TABLE;

GaugeDialog::GaugeDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
:TMyDialog(parent,resID)
{
  Obj=obj;
  Id=id;
  Col=new ColorButton(this,IDGCOLD);
}

GaugeDialog::~GaugeDialog()
{
  delete Col;
}

void GaugeDialog::SetupWindow()
{
  TMyDialog::SetupWindow();
  SetupItem(Id);
}

void GaugeDialog::SetupItem(int id)
{
  int R,G,B;

  SetListFromObjField(Handle,IDGGAUGE,Obj,id,"gauge");
  SetTextFromObjField(Handle,IDGMIN,Obj,id,"gauge_min");
  SetTextFromObjField(Handle,IDGMAX,Obj,id,"gauge_max");
  SetStyleFromObjField(Handle,IDGSTYLE,Obj,id,"gauge_style");
  SetTextFromObjField(Handle,IDGLENGTH1,Obj,id,"gauge_length1");
  SetComboList(Handle,IDGWIDTH1,cblinewidth,CBLINEWIDTH);
  SetTextFromObjField(Handle,IDGWIDTH1,Obj,id,"gauge_width1");
  SetTextFromObjField(Handle,IDGLENGTH2,Obj,id,"gauge_length2");
  SetComboList(Handle,IDGWIDTH2,cblinewidth,CBLINEWIDTH);
  SetTextFromObjField(Handle,IDGWIDTH2,Obj,id,"gauge_width2");
  SetTextFromObjField(Handle,IDGLENGTH3,Obj,id,"gauge_length3");
  SetComboList(Handle,IDGWIDTH3,cblinewidth,CBLINEWIDTH);
  SetTextFromObjField(Handle,IDGWIDTH3,Obj,id,"gauge_width3");
  getobj(Obj,"gauge_R",id,0,NULL,&R);
  getobj(Obj,"gauge_G",id,0,NULL,&G);
  getobj(Obj,"gauge_B",id,0,NULL,&B);
  Col->SetColor(R,G,B);
}

void GaugeDialog::CloseWindow(int retVal)
{
  int R,G,B;

  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  if (SetObjFieldFromList(Handle,IDGGAUGE,Obj,Id,"gauge")) return;
  if (SetObjFieldFromText(Handle,IDGMIN,Obj,Id,"gauge_min")) return;
  if (SetObjFieldFromText(Handle,IDGMAX,Obj,Id,"gauge_max")) return;
  if (SetObjFieldFromStyle(Handle,IDGSTYLE,Obj,Id,"gauge_style")) return;
  if (SetObjFieldFromText(Handle,IDGLENGTH1,Obj,Id,"gauge_length1")) return;
  if (SetObjFieldFromText(Handle,IDGWIDTH1,Obj,Id,"gauge_width1")) return;
  if (SetObjFieldFromText(Handle,IDGLENGTH2,Obj,Id,"gauge_length2")) return;
  if (SetObjFieldFromText(Handle,IDGWIDTH2,Obj,Id,"gauge_width2")) return;
  if (SetObjFieldFromText(Handle,IDGLENGTH3,Obj,Id,"gauge_length3")) return;
  if (SetObjFieldFromText(Handle,IDGWIDTH3,Obj,Id,"gauge_width3")) return;
  Col->GetColor(&R,&G,&B);
  if (putobj(Obj,"gauge_R",Id,&R)==-1) return;
  if (putobj(Obj,"gauge_G",Id,&G)==-1) return;
  if (putobj(Obj,"gauge_B",Id,&B)==-1) return;
  TMyDialog::CloseWindow(retVal);
}


class AxisFontDialog  : public TMyDialog {
  public:
    AxisFontDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
    :TMyDialog(parent,resID) {
      Obj=obj;
      Id=id;
      Col=new ColorButton(this,IDNCOLB);
    }
    ~AxisFontDialog() {
      delete Col;
    }
  protected:
    int pt;
    ColorButton *Col;
    struct objlist *Obj;
    int Id;
    void SetupWindow();
    void SetupItem(int id);
    void CopyClicked() {
      int sel;
      if ((sel=CopyClick(this,Obj,Id,AxisCB))!=-1)
        SetupItem(sel);
    }
    void CloseWindow(int retVal);
    DECLARE_RESPONSE_TABLE(AxisFontDialog);
};

DEFINE_RESPONSE_TABLE1(AxisFontDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
END_RESPONSE_TABLE;

void AxisFontDialog::SetupWindow()
{
  TMyDialog::SetupWindow();
  SetupItem(Id);
}

void AxisFontDialog::SetupItem(int id)
{
  int j,selfont,R,G,B;
  struct fontmap *fcur;
  char *font;

  SetTextFromObjField(Handle,IDNSPACE,Obj,id,"num_space");
  SetComboList(Handle,IDNPT,cbtextpt,CBTEXTPT);
  SetTextFromObjField(Handle,IDNPT,Obj,id,"num_pt");
  SetTextFromObjField(Handle,IDNSCRIPT,Obj,id,"num_script_size");
  getobj(Obj,"num_font",id,0,NULL,&font);
  SendDlgItemMessage(IDNFONT,CB_RESETCONTENT,0,0);
  fcur=mwlocal->fontmaproot;
  j=0;
  selfont=-1;
  while (fcur!=NULL) {
    if (fcur->charset!=SHIFTJIS_CHARSET) {
      SendDlgItemMessage(IDNFONT,CB_INSERTSTRING,-1,(LPARAM)fcur->fontalias);
      if (strcmp(font,fcur->fontalias)==0) selfont=j;
      j++;
    }
    fcur=fcur->next;
  }
  if (selfont!=-1) SendDlgItemMessage(IDNFONT,CB_SETCURSEL,selfont,0);
#ifndef ENGLISH
  getobj(Obj,"num_jfont",id,0,NULL,&font);
  SendDlgItemMessage(IDNJFONT,CB_RESETCONTENT,0,0);
  fcur=mwlocal->fontmaproot;
  j=0;
  selfont=-1;
  while (fcur!=NULL) {
    if (fcur->charset==SHIFTJIS_CHARSET) {
      SendDlgItemMessage(IDNJFONT,CB_INSERTSTRING,-1,(LPARAM)fcur->fontalias);
      if (strcmp(font,fcur->fontalias)==0) selfont=j;
      j++;
    }
    fcur=fcur->next;
  }
  if (selfont!=-1) SendDlgItemMessage(IDNJFONT,CB_SETCURSEL,selfont,0);
#endif
  getobj(Obj,"num_R",id,0,NULL,&R);
  getobj(Obj,"num_G",id,0,NULL,&G);
  getobj(Obj,"num_B",id,0,NULL,&B);
  Col->SetColor(R,G,B);
}

void AxisFontDialog::CloseWindow(int retVal)
{
  int R,G,B;

  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  if (SetObjFieldFromText(Handle,IDNSPACE,Obj,Id,"num_space")) return;
  if (SetObjFieldFromText(Handle,IDNPT,Obj,Id,"num_pt")) return;
  if (SetObjFieldFromText(Handle,IDNSCRIPT,Obj,Id,"num_script_size")) return;
  if (SetObjFieldFromText(Handle,IDNFONT,Obj,Id,"num_font")) return;
#ifndef ENGLISH
  if (SetObjFieldFromText(Handle,IDNJFONT,Obj,Id,"num_jfont")) return;
#endif
  Col->GetColor(&R,&G,&B);
  if (putobj(Obj,"num_R",Id,&R)==-1) return;
  if (putobj(Obj,"num_G",Id,&G)==-1) return;
  if (putobj(Obj,"num_B",Id,&B)==-1) return;
  TMyDialog::CloseWindow(retVal);
}

class NumDialog  : public TMyDialog {
  public:
    NumDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
    :TMyDialog(parent,resID) {
      Obj=obj;
      Id=id;
    }
    ~NumDialog() {
    }
  protected:
    int pt;
    struct objlist *Obj;
    int Id;
    void SetupWindow();
    void SetupItem(int id);
    void CopyClicked() {
      int sel;
      if ((sel=CopyClick(this,Obj,Id,AxisCB))!=-1)
        SetupItem(sel);
    }
    void CloseWindow(int retVal);
    DECLARE_RESPONSE_TABLE(NumDialog);
};

DEFINE_RESPONSE_TABLE1(NumDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
END_RESPONSE_TABLE;

void NumDialog::SetupWindow()
{
  TMyDialog::SetupWindow();
  SetupItem(Id);
}

void NumDialog::SetupItem(int id)
{
  char *format,*endptr;
  int j,a;

  SetListFromObjField(Handle,IDNNUMBERING,Obj,id,"num");
  SetTextFromObjField(Handle,IDNBEGIN,Obj,id,"num_begin");
  SetTextFromObjField(Handle,IDNSTEP,Obj,id,"num_step");
  SetTextFromObjField(Handle,IDNNUM,Obj,id,"num_num");
  SetTextFromObjField(Handle,IDNHEAD,Obj,id,"num_head");
  SendDlgItemMessage(IDNFFRAC,CB_RESETCONTENT,0,0);
  for (j=0;j<CNUMSTYLE;j++)
    SendDlgItemMessage(IDNFFRAC,CB_INSERTSTRING,j,(LPARAM)fwnumstyle[j]);
  getobj(Obj,"num_format",id,0,NULL,&format);
  if (strchr(format,'+')!=NULL) SendDlgItemMessage(IDNFZERO,BM_SETCHECK,1,0);
  else SendDlgItemMessage(IDNFZERO,BM_SETCHECK,0,0);
  if ((strchr(format,'f')==NULL) || (strchr(format,'.')==NULL)) a=0;
  else a=strtol(strchr(format,'.')+1,&endptr,10)+1;
  if (a<0) a=0;
  else if (a>10) a=10;
  SendDlgItemMessage(IDNFFRAC,CB_SETCURSEL,a,0);
  SetTextFromObjField(Handle,IDNTAIL,Obj,id,"num_tail");
  SetListFromObjField(Handle,IDNALIGN,Obj,id,"num_align");
  SetListFromObjField(Handle,IDNDIR,Obj,id,"num_direction");
  SetTextFromObjField(Handle,IDNSHIFTP,Obj,id,"num_shift_p");
  SetTextFromObjField(Handle,IDNSHIFTN,Obj,id,"num_shift_n");
  SetToggleFromObjField(Handle,IDNLOGPOW,Obj,id,"num_log_pow");
  SetToggleFromObjField(Handle,IDNNOZERO,Obj,id,"num_no_zero");
  SetTextFromObjField(Handle,IDNAUTONORM,Obj,id,"num_auto_norm");
}

void NumDialog::CloseWindow(int retVal)
{
  int a,j;
  char *format;

  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  if (SetObjFieldFromList(Handle,IDNNUMBERING,Obj,Id,"num")) return;
  if (SetObjFieldFromText(Handle,IDNBEGIN,Obj,Id,"num_begin")) return;
  if (SetObjFieldFromText(Handle,IDNSTEP,Obj,Id,"num_step")) return;
  if (SetObjFieldFromText(Handle,IDNNUM,Obj,Id,"num_num")) return;
  if (SetObjFieldFromText(Handle,IDNHEAD,Obj,Id,"num_head")) return;
  if ((format=(char *)memalloc(10))==NULL) return;
  j=0;
  j+=sprintf(format+j,"%%");
  if (SendDlgItemMessage(IDNFZERO,BM_GETCHECK,0,0))
    j+=sprintf(format+j,"%c",'+');
  a=SendDlgItemMessage(IDNFFRAC,CB_GETCURSEL,0,0);
  if (a==0) j+=sprintf(format+j,"%c",'g');
  else j+=sprintf(format+j,".%df",a-1);
  if (putobj(Obj,"num_format",Id,format)==-1) return;
  if (SetObjFieldFromText(Handle,IDNTAIL,Obj,Id,"num_tail")) return;
  if (SetObjFieldFromList(Handle,IDNALIGN,Obj,Id,"num_align")) return;
  if (SetObjFieldFromList(Handle,IDNDIR,Obj,Id,"num_direction")) return;
  if (SetObjFieldFromText(Handle,IDNSHIFTP,Obj,Id,"num_shift_p")) return;
  if (SetObjFieldFromText(Handle,IDNSHIFTN,Obj,Id,"num_shift_n")) return;
  if (SetObjFieldFromToggle(Handle,IDNLOGPOW,Obj,Id,"num_log_pow")) return;
  if (SetObjFieldFromToggle(Handle,IDNNOZERO,Obj,Id,"num_no_zero")) return;
  if (SetObjFieldFromText(Handle,IDNAUTONORM,Obj,Id,"num_auto_norm")) return;
  TMyDialog::CloseWindow(retVal);
}

class AxisBaseDialog  : public TMyDialog {
  public:
    AxisBaseDialog(TWindow *parent,TResId resID,struct objlist *obj,int id);
    ~AxisBaseDialog();
  protected:
    struct objlist *Obj;
    int Id;
    ColorButton *Col;
    void SetupWindow();
    void SetupItem(int id);
    void CopyClicked() {
      int sel;
      if ((sel=CopyClick(this,Obj,Id,AxisCB))!=-1) SetupItem(sel);
    }
    void CloseWindow(int retVal);
    DECLARE_RESPONSE_TABLE(AxisBaseDialog);
};

DEFINE_RESPONSE_TABLE1(AxisBaseDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
END_RESPONSE_TABLE;

AxisBaseDialog::AxisBaseDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
:TMyDialog(parent,resID)
{
  Obj=obj;
  Id=id;
  Col=new ColorButton(this,IDACOLD);
}

AxisBaseDialog::~AxisBaseDialog()
{
  delete Col;
}

void AxisBaseDialog::SetupWindow()
{
  TMyDialog::SetupWindow();
  SetupItem(Id);
}

void AxisBaseDialog::SetupItem(int id)
{
  int R,G,B;

  SetStyleFromObjField(Handle,IDASTYLE,Obj,id,"style");
  SetComboList(Handle,IDAWIDTH,cblinewidth,CBLINEWIDTH);
  SetTextFromObjField(Handle,IDAWIDTH,Obj,id,"width");
  SetToggleFromObjField(Handle,IDABASELINE,Obj,id,"baseline");
  SetListFromObjField(Handle,IDAARROW,Obj,id,"arrow");
  SetTextFromObjField(Handle,IDAALENGTH,Obj,id,"arrow_length");
  SetTextFromObjField(Handle,IDAAWIDTH,Obj,id,"arrow_width");
  SetListFromObjField(Handle,IDAWAVE,Obj,id,"wave");
  SetTextFromObjField(Handle,IDAWLENGTH,Obj,id,"wave_length");
  SetComboList(Handle,IDAWWIDTH,cblinewidth,CBLINEWIDTH);
  SetTextFromObjField(Handle,IDAWWIDTH,Obj,id,"wave_width");
  getobj(Obj,"R",id,0,NULL,&R);
  getobj(Obj,"G",id,0,NULL,&G);
  getobj(Obj,"B",id,0,NULL,&B);
  Col->SetColor(R,G,B);
}

void AxisBaseDialog::CloseWindow(int retVal)
{
  int R,G,B;

  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  if (SetObjFieldFromStyle(Handle,IDASTYLE,Obj,Id,"style")) return;
  if (SetObjFieldFromText(Handle,IDAWIDTH,Obj,Id,"width")) return;
  if (SetObjFieldFromToggle(Handle,IDABASELINE,Obj,Id,"baseline")) return;
  if (SetObjFieldFromList(Handle,IDAARROW,Obj,Id,"arrow")) return;
  if (SetObjFieldFromText(Handle,IDAALENGTH,Obj,Id,"arrow_length")) return;
  if (SetObjFieldFromText(Handle,IDAAWIDTH,Obj,Id,"arrow_width")) return;
  if (SetObjFieldFromList(Handle,IDAWAVE,Obj,Id,"wave")) return;
  if (SetObjFieldFromText(Handle,IDAWLENGTH,Obj,Id,"wave_length")) return;
  if (SetObjFieldFromText(Handle,IDAWWIDTH,Obj,Id,"wave_width")) return;
  Col->GetColor(&R,&G,&B);
  if (putobj(Obj,"R",Id,&R)==-1) return;
  if (putobj(Obj,"G",Id,&G)==-1) return;
  if (putobj(Obj,"B",Id,&B)==-1) return;
  TMyDialog::CloseWindow(retVal);
}

class AxisPosDialog  : public TMyDialog {
  public:
    AxisPosDialog(TWindow *parent,TResId resID,struct objlist *obj,int id);
    ~AxisPosDialog();
  protected:
    struct objlist *Obj;
    int Id;
    AxisComboBox *Adjust;
    void SetupWindow();
    void SetupItem(int id);
    void CopyClicked() {
      int sel;
      if ((sel=CopyClick(this,Obj,Id,AxisCB))!=-1) SetupItem(sel);
    }
    void CloseWindow(int retVal);
    DECLARE_RESPONSE_TABLE(AxisPosDialog);
};

DEFINE_RESPONSE_TABLE1(AxisPosDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
END_RESPONSE_TABLE;

AxisPosDialog::AxisPosDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
:TMyDialog(parent,resID)
{
  Obj=obj;
  Id=id;
  Adjust=new AxisComboBox(this,IDAADJUST);
}

AxisPosDialog::~AxisPosDialog()
{
  delete Adjust;
}

void AxisPosDialog::SetupWindow()
{
  TMyDialog::SetupWindow();
  SetupItem(Id);
}

void AxisPosDialog::SetupItem(int id)
{
  char *valstr;
  int i,j,oid;
  int lastinst;
  char buf[10];

  SetTextFromObjField(Handle,IDAX,Obj,id,"x");
  SetTextFromObjField(Handle,IDAY,Obj,id,"y");
  SetTextFromObjField(Handle,IDALENGTH,Obj,id,"length");
  SetComboList(Handle,IDADIR,cbdirection,CBDIRECTION);
  SetTextFromObjField(Handle,IDADIR,Obj,id,"direction");
  lastinst=chkobjlastinst(Obj);
  SendDlgItemMessage(IDAADJUST,CB_RESETCONTENT,0,0);
  for (j=0;j<=lastinst;j++) {
    getobj(Obj,"oid",j,0,NULL,&oid);
    sprintf(buf,"^%d",oid);
    SendDlgItemMessage(IDAADJUST,CB_INSERTSTRING,j,(LPARAM)buf);
  }
  sgetobjfield(Obj,id,"adjust_axis",NULL,&valstr,FALSE,FALSE,FALSE);
  for (i=0;(valstr[i]!='\0') && (valstr[i]!=':');i++);
  if (valstr[i]==':') i++;
  SetDlgItemText(IDAADJUST,valstr+i); memfree(valstr);
  SetTextFromObjField(Handle,IDAADJUSTPOS,Obj,id,"adjust_position");
}

void AxisPosDialog::CloseWindow(int retVal)
{
  int len;
  char *buf;

  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  if (SetObjFieldFromText(Handle,IDAX,Obj,Id,"x")) return;
  if (SetObjFieldFromText(Handle,IDAY,Obj,Id,"y")) return;
  if (SetObjFieldFromText(Handle,IDALENGTH,Obj,Id,"length")) return;
  if (SetObjFieldFromText(Handle,IDADIR,Obj,Id,"direction")) return;
  len=SendDlgItemMessage(IDAADJUST,WM_GETTEXTLENGTH,0,0);
  if ((len>0) && ((buf=(char *)memalloc(len+6))!=NULL)) {
    strcpy(buf,"axis:");
    GetDlgItemText(IDAADJUST,buf+5,len+1);
    if (sputobjfield(Obj,Id,"adjust_axis",buf)!=0) {
      memfree(buf);
      return;
    }
    memfree(buf);
  } else putobj(Obj,"adjust_axis",Id,NULL);
  if (SetObjFieldFromText(Handle,IDAADJUSTPOS,Obj,Id,"adjust_position")) return;
  TMyDialog::CloseWindow(retVal);
}

DEFINE_RESPONSE_TABLE1(AxisDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
  EV_CHILD_NOTIFY(IDDELETE,BN_CLICKED,DeleteClicked),
  EV_CHILD_NOTIFY(IDABASE,BN_CLICKED,BaseClicked),
  EV_CHILD_NOTIFY(IDAGAUGE,BN_CLICKED,GaugeClicked),
  EV_CHILD_NOTIFY(IDANUM,BN_CLICKED,NumberClicked),
  EV_CHILD_NOTIFY(IDAFONT,BN_CLICKED,FontClicked),
  EV_CHILD_NOTIFY(IDAPOS,BN_CLICKED,PosClicked),
  EV_CHILD_NOTIFY(IDACLEAR,BN_CLICKED,ClearClicked),
  EV_CHILD_NOTIFY(IDAFILE,BN_CLICKED,FileClicked),
  EV_CHILD_NOTIFY(IDATYPE,CBN_SELCHANGE,ClearClicked),
END_RESPONSE_TABLE;

AxisDialog::AxisDialog(TWindow *parent,TResId resID,struct objlist *obj,int id,int candel)
:TMyDialog(parent,resID)
{
  Obj=obj;
  Id=id;
  CanDel=candel;
  MinE=new SelEdit(this,IDAMIN);
  MaxE=new SelEdit(this,IDAMAX);
  IncE=new SelEdit(this,IDAINC);
  Ref=new AxisComboBox(this,IDAREF);
}

AxisDialog::~AxisDialog()
{
  delete Ref;
  delete MinE;
  delete MaxE;
  delete IncE;
}

void AxisDialog::SetupWindow()
{
  char *title;
  int len,leng;
  char *group;

  TMyDialog::SetupWindow();
  if (!CanDel) ::EnableWindow(GetDlgItem(IDDELETE),FALSE);
  len=GetWindowTextLength();
  getobj(Obj,"group",Id,0,NULL,&group);
  if (group==NULL) group="";
  leng=strlen(group);
  if ((title=(char *)memalloc(len+leng+11))!=NULL) {
    GetWindowText(title,len+1);
    sprintf(title+len," %d %s",Id,group);
    SetCaption(title);
    memfree(title);
  }
  SetupItem(Id);
}

void AxisDialog::GaugeClicked()
{
  GaugeDialog *dlg;

  dlg=new GaugeDialog(this,DIALOG_GAUGE,Obj,Id);
  dlg->Execute();
  delete dlg;
}

void AxisDialog::BaseClicked()
{
  AxisBaseDialog *dlg;

  dlg=new AxisBaseDialog(this,DIALOG_AXISBASE,Obj,Id);
  dlg->Execute();
  delete dlg;
}

void AxisDialog::NumberClicked()
{
  NumDialog *dlg;

  dlg=new NumDialog(this,DIALOG_NUM,Obj,Id);
  dlg->Execute();
  delete dlg;
}

void AxisDialog::FontClicked()
{
  AxisFontDialog *dlg;

  dlg=new AxisFontDialog(this,DIALOG_AXISFONT,Obj,Id);
  dlg->Execute();
  delete dlg;
}

void AxisDialog::PosClicked()
{
  AxisPosDialog *dlg;

  dlg=new AxisPosDialog(this,DIALOG_AXISPOS,Obj,Id);
  dlg->Execute();
  delete dlg;
}

void AxisDialog::SetupItem(int id)
{
  char *valstr;
  int i,j;
  int lastinst;
  int oid;
  char buf[30];
  double min,max,inc;
  int div;
  struct narray *array;
  int num;
  double *data;

  SendDlgItemMessage(IDAMIN,CB_RESETCONTENT,0,0);
  SendDlgItemMessage(IDAMAX,CB_RESETCONTENT,0,0);
  SendDlgItemMessage(IDAINC,CB_RESETCONTENT,0,0);
  getobj(Obj,"scale_history",Id,0,NULL,&array);
  if (array!=NULL) {
    num=arraynum(array)/3;
    data=(double *)arraydata(array);
    for (j=0;j<num;j++) {
      sprintf(buf,"%.15g",data[0+j*3]);
      SendDlgItemMessage(IDAMIN,CB_INSERTSTRING,j,(LPARAM)buf);
      sprintf(buf,"%.15g",data[1+j*3]);
      SendDlgItemMessage(IDAMAX,CB_INSERTSTRING,j,(LPARAM)buf);
      sprintf(buf,"%.15g",data[2+j*3]);
      SendDlgItemMessage(IDAINC,CB_INSERTSTRING,j,(LPARAM)buf);
    }
  }
  getobj(Obj,"min",id,0,NULL,&min);
  getobj(Obj,"max",id,0,NULL,&max);
  getobj(Obj,"inc",id,0,NULL,&inc);
  if ((min==0) && (max==0) && (inc==0)) {
    ClearClicked();
  } else {
    sprintf(buf,"%.15g",min);
    SetDlgItemText(IDAMIN,buf);
    sprintf(buf,"%.15g",max);
    SetDlgItemText(IDAMAX,buf);
    sprintf(buf,"%.15g",inc);
    SetDlgItemText(IDAINC,buf);
  }
  getobj(Obj,"div",id,0,NULL,&div);
  SetTextFromObjField(Handle,IDADIV,Obj,id,"div");
  SetListFromObjField(Handle,IDATYPE,Obj,id,"type");
  lastinst=chkobjlastinst(Obj);
  SendDlgItemMessage(IDAREF,CB_RESETCONTENT,0,0);
  for (j=0;j<=lastinst;j++) {
    getobj(Obj,"oid",j,0,NULL,&oid);
    sprintf(buf,"^%d",oid);
    SendDlgItemMessage(IDAREF,CB_INSERTSTRING,j,(LPARAM)buf);
  }
  sgetobjfield(Obj,id,"reference",NULL,&valstr,FALSE,FALSE,FALSE);
  for (i=0;(valstr[i]!='\0') && (valstr[i]!=':');i++);
  if (valstr[i]==':') i++;
  SetDlgItemText(IDAREF,valstr+i); memfree(valstr);
}

void AxisDialog::CloseWindow(int retVal)
{
  int len;
  char *buf;

  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  exeobj(Obj,"scale_push",Id,0,NULL);
  if (SetObjFieldFromText(Handle,IDAMIN,Obj,Id,"min")) return;
  if (SetObjFieldFromText(Handle,IDAMAX,Obj,Id,"max")) return;
  if (SetObjFieldFromText(Handle,IDAINC,Obj,Id,"inc")) return;
  if (SetObjFieldFromText(Handle,IDADIV,Obj,Id,"div")) return;
  if (SetObjFieldFromList(Handle,IDATYPE,Obj,Id,"type")) return;
  len=SendDlgItemMessage(IDAREF,WM_GETTEXTLENGTH,0,0);
  if (len==0) {
    if (sputobjfield(Obj,Id,"reference",NULL)!=0) return;
  } else if ((buf=(char *)memalloc(len+6))!=NULL) {
    strcpy(buf,"axis:");
    GetDlgItemText(IDAREF,buf+5,len+1);
    if (sputobjfield(Obj,Id,"reference",buf)!=0) {
      memfree(buf);
      return;
    }
    memfree(buf);
  }
  TMyDialog::CloseWindow(retVal);
}

void AxisDialog::ClearClicked()
{
  SetDlgItemText(IDAMIN,"0");
  SetDlgItemText(IDAMAX,"0");
  SetDlgItemText(IDAINC,"0");
}

void AxisDialog::FileClicked()
{
  int anum,room,type;
  char *buf,s[30];
  char *argv2[3];
  struct objlist *fobj;
  struct narray farray;
  int a,i,j,num,*array;
  SelectDialog *dlg;
  struct narray *result;

  if ((fobj=chkobject("file"))==NULL) return;
  if (chkobjlastinst(fobj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,fobj,FileCB,
                       (struct narray *)&farray,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&farray);
    array=(int *)arraydata(&farray);
    anum=chkobjlastinst(Obj);
    if ((num>0) && (anum!=0)) {
      if ((buf=(char *)memalloc(6*num+6))!=NULL) {
        j=0;
        j+=sprintf(buf+j,"file:");
        for (i=0;i<num;i++)
          if (i==num-1) j+=sprintf(buf+j,"%d",array[i]);
          else j+=sprintf(buf+j,"%d,",array[i]);
        room=0;
        argv2[0]=(char *)buf;
        argv2[1]=(char *)&room;
        argv2[2]=NULL;
        a=SendDlgItemMessage(IDATYPE,CB_GETCURSEL,0,0);
        if (a!=CB_ERR) {
          if ((getobj(Obj,"type",Id,0,NULL,&type)==-1)
          || (putobj(Obj,"type",Id,&a)==-1)) {
            arraydel(&farray);
            delete dlg;
            return;
          }
        }
        getobj(Obj,"get_auto_scale",Id,2,argv2,&result);
        memfree(buf);
        if (arraynum(result)==3) {
          sprintf(s,"%.15g",*(double *)arraynget(result,0));
          SetDlgItemText(IDAMIN,s);
          sprintf(s,"%.15g",*(double *)arraynget(result,1));
          SetDlgItemText(IDAMAX,s);
          sprintf(s,"%.15g",*(double *)arraynget(result,2));
          SetDlgItemText(IDAINC,s);
        }
        putobj(Obj,"type",Id,&type);
      }
    }
  }
  arraydel(&farray);
  delete dlg;
}

DEFINE_RESPONSE_TABLE1(TAxisWindow, TSubWindow)
  EV_WM_SIZE,
  EV_WM_LBUTTONDOWN,
  EV_WM_RBUTTONDOWN,
  EV_WM_LBUTTONDBLCLK,
  EV_WM_KEYDOWN,
  EV_WM_PAINT,
  EV_COMMAND(CM_SELECTK,AxisSelect),
  EV_COMMAND(CM_DELETEK,AxisDelete),
  EV_COMMAND(CM_COPYK,AxisCopy),
  EV_COMMAND(CM_TOPK,AxisTop),
  EV_COMMAND(CM_LASTK,AxisLast),
  EV_COMMAND(CM_UPDATEK,AxisUpdate),
  EV_COMMAND(CM_HIDDENK,AxisHidden),
  EV_COMMAND(CM_UPK,AxisUp),
  EV_COMMAND(CM_DOWNK,AxisDown),
END_RESPONSE_TABLE;

TAxisWindow::TAxisWindow(TWindow *parent):TSubWindow(parent)
{
  menulocal.axisopen=TRUE;
  Attr.Style|=WS_VSCROLL|WS_HSCROLL;
  col1=GetSysColor(COLOR_WINDOWTEXT);
  col1h=GetSysColor(COLOR_GRAYTEXT);
  col2=(GetSysColor(COLOR_WINDOW) ^ GetSysColor(COLOR_HIGHLIGHT)) & RGB(255,255,255);
  brush1=CreateSolidBrush(col2);
  brush2=CreateSolidBrush(GetSysColor(COLOR_WINDOW));
  pen1=(HPEN)GetStockObject(NULL_PEN);
  pen2=(HPEN)GetStockObject(NULL_PEN);
  PopupMenu.AppendMenu(MF_STRING,CM_SELECTK,"Focus (SPC)");
  PopupMenu.AppendMenu(MF_STRING,CM_UPDATEK,"Update (CR)");
  PopupMenu.AppendMenu(MF_STRING,CM_DELETEK,"Delete (DEL)");
  PopupMenu.AppendMenu(MF_STRING,CM_TOPK,"Top (HOME)");
  PopupMenu.AppendMenu(MF_STRING,CM_UPK,"Up (SHIFT+UP)");
  PopupMenu.AppendMenu(MF_STRING,CM_DOWNK,"Down (SHIFT+DOWN)");
  PopupMenu.AppendMenu(MF_STRING,CM_LASTK,"Last (END)");
  PopupMenu.AppendMenu(MF_STRING,CM_COPYK,"Duplicate (INS)");
  PopupMenu.AppendMenu(MF_STRING,CM_HIDDENK,"Hide (TAB)");
}

void TAxisWindow::SetupWindow()
{
  TSubWindow::SetupWindow();
  select=-1;
  Scroller = new TScroller(this,1,1,0,0);
  Scroller->XLine=NgraphApp->FWidth;
  Scroller->YLine=NgraphApp->FHeight;
  Scroller->TrackMode=FALSE;
  obj=chkobject("axis");
  axisnum=chkobjlastinst(obj);
  ChangeAxisNum();
}

TAxisWindow::~TAxisWindow()
{
  menulocal.axisopen=FALSE;
  ::DeleteObject(brush1);
  ::DeleteObject(brush2);
  ::DeleteObject(pen1);
  ::DeleteObject(pen2);
  NgraphApp->DeleteAxisWindow();
}

void TAxisWindow::EvSize(UINT sizeType, TSize& size)
{
  TSubWindow::EvSize(sizeType,size);
  ChangeAxisNum();
}

void TAxisWindow::EvLButtonDown(UINT,TPoint& point)
{
  int sel;
  TClientDC dc(Handle);
  TPoint po;

  if (menulock || globallock) return;
  sel=(Scroller->YPos+point.y)/NgraphApp->FHeight-1;
  if (sel!=select) {
    po.x=Scroller->XPos;
    po.y=Scroller->YPos;
    dc.SetWindowOrg(po);
    SelectAxis(dc,sel);
  }
}

void TAxisWindow::EvRButtonDown(UINT modKeys,TPoint& point)
{
  int sel;
  TClientDC dc(Handle);
  TPoint po;
  TRect rect;
  TPoint point2;

  if (menulock || globallock) return;
  sel=(Scroller->YPos+point.y)/NgraphApp->FHeight-1;
  if (sel!=select) {
    po.x=Scroller->XPos;
    po.y=Scroller->YPos;
    dc.SetWindowOrg(po);
    SelectAxis(dc,sel);
  }
  sel=select;
  if ((sel>=0) && (sel<=axisnum)) {
    PopupMenu.EnableMenuItem(CM_SELECTK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_DELETEK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_UPDATEK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_TOPK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_UPK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_DOWNK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_LASTK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_COPYK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_HIDDENK,MF_ENABLED);
  } else {
    PopupMenu.EnableMenuItem(CM_SELECTK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_DELETEK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_UPDATEK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_TOPK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_UPK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_DOWNK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_LASTK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_COPYK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_HIDDENK,MF_GRAYED);
  }
  GetWindowRect(rect);
  point2.x=point.x+rect.left;
  point2.y=point.y+rect.top;
  PopupMenu.TrackPopupMenu(TPM_LEFTBUTTON | TPM_RIGHTBUTTON,point2,0,Handle);
}

void TAxisWindow::EvLButtonDblClk(UINT,TPoint& point)
{
  if (menulock || globallock) return;
  AxisUpdate();
}

void TAxisWindow::EvKeyDown(UINT key,UINT repeatCount,UINT flags)
{
  TClientDC dc(Handle);
  TPoint po;
  int sel;
  TRect rect;
  int h,h1,h2;
  TPoint point;

  if (menulock || globallock) return;
  sel=select;
  switch (key) {
  case VK_DELETE:
    AxisDelete();
    break;
  case VK_INSERT:
    AxisCopy();
    break;
  case VK_HOME:
    AxisTop();
    break;
  case VK_END:
    AxisLast();
    break;
  case VK_RETURN:
    AxisUpdate();
    break;
  case VK_SPACE:
    AxisSelect();
    break;
  case VK_TAB:
    AxisHidden();
    break;
  case VK_DOWN:
    if (GetAsyncKeyState(VK_SHIFT) & 0x8000) AxisDown();
    else {
      if ((sel==-1) && (axisnum>=0)) sel=0;
      else if (sel<axisnum) sel++;
      GetClientRect(rect);
      h=(rect.bottom-rect.top)/NgraphApp->FHeight;
      h1=Scroller->YPos/NgraphApp->FHeight-1;
      h2=Scroller->YPos/NgraphApp->FHeight-1+h;
      if (sel<h1) {
        Scroller->YPos=(sel+1)*NgraphApp->FHeight;
        Invalidate();
      } else if (sel>=h2) {
        Scroller->YPos=(sel+2-h)*NgraphApp->FHeight;
        Invalidate();
      }
      if (sel!=select) {
        po.x=Scroller->XPos;
        po.y=Scroller->YPos;
        dc.SetWindowOrg(po);
        SelectAxis(dc,sel);
      }
    }
    break;
  case VK_UP:
    if (GetAsyncKeyState(VK_SHIFT) & 0x8000) AxisUp();
    else {
      if ((sel==-1) && (axisnum>=0)) sel=axisnum;
      else if (sel>0) sel--;
      GetClientRect(rect);
      h=(rect.bottom-rect.top)/NgraphApp->FHeight;
      h1=Scroller->YPos/NgraphApp->FHeight-1;
      h2=Scroller->YPos/NgraphApp->FHeight-1+h;
      if (sel<h1) {
        Scroller->YPos=(sel+1)*NgraphApp->FHeight;
        Invalidate();
      } else if (sel>=h2) {
        Scroller->YPos=(sel+2-h)*NgraphApp->FHeight;
        Invalidate();
      }
      if (sel<h1) {
        Scroller->YPos=(sel+1)*NgraphApp->FHeight;
        Invalidate();
      } else if (sel>=h2) {
        Scroller->YPos=(sel+2)*NgraphApp->FHeight-(rect.bottom-rect.top);
        Invalidate();
      }
      if (sel!=select) {
        po.x=Scroller->XPos;
        po.y=Scroller->YPos;
        dc.SetWindowOrg(po);
        SelectAxis(dc,sel);
      }
    }
    break;
  case 'P':
    if (GetAsyncKeyState(VK_CONTROL) & 0x8000) {
      sel=select;
      if ((sel>=0) && (sel<=axisnum)) {
        PopupMenu.EnableMenuItem(CM_SELECTK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_DELETEK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_UPDATEK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_TOPK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_LASTK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_COPYK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_HIDDENK,MF_ENABLED);
      } else {
        PopupMenu.EnableMenuItem(CM_SELECTK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_DELETEK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_UPDATEK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_TOPK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_LASTK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_COPYK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_HIDDENK,MF_GRAYED);
      }
      GetWindowRect(rect);
      point.x=rect.left;
      point.y=rect.top;
      PopupMenu.TrackPopupMenu(TPM_LEFTBUTTON | TPM_RIGHTBUTTON,point,0,Handle);
    }
    break;
  default:
    NgraphApp->EvKeyDown2(key,repeatCount,flags);
    break;
  }
  TSubWindow::EvKeyDown(key,repeatCount,flags);
}

void TAxisWindow::AxisDelete()
{
  int sel;

  if (menulock || globallock) return;
  NgraphApp->UnFocus();
  sel=select;
  if ((sel>=0) && (sel<=axisnum)) {
    AxisDel(sel);
    Update(TRUE);
    NgraphApp->FileUpdate();
    NgraphApp->Changed=TRUE;
  }
}

void TAxisWindow::AxisCopy()
{
  int sel;
  int j,id,perm,type;
  char *field;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=0) && (sel<=axisnum)) {
    if ((id=newobj(obj))>=0) {
      for (j=0;j<chkobjfieldnum(obj);j++) {
        field=chkobjfieldname(obj,j);
        perm=chkobjperm(obj,field);
        type=chkobjfieldtype(obj,field);
        if (((perm&NREAD)!=0) && ((perm&NWRITE)!=0) && (type<NVFUNC))
          copyobj(obj,field,id,sel);
      }
      axisnum++;
      NgraphApp->Changed=TRUE;
    }
    select=sel;
    Update(FALSE);
  }
}

void TAxisWindow::AxisTop()
{
  int sel;

  if (menulock || globallock) return;
  NgraphApp->UnFocus();
  sel=select;
  if ((sel>=0) && (sel<=axisnum)) {
    movetopobj(obj,sel);
    select=0;
    AxisMove(sel,0);
    Update(FALSE);
    NgraphApp->FileUpdate();
    NgraphApp->Changed=TRUE;
  }
}

void TAxisWindow::AxisUp()
{
  int sel;

  if (menulock || globallock) return;
  NgraphApp->UnFocus();
  sel=select;
  if ((sel>=1) && (sel<=axisnum)) {
    moveupobj(obj,sel);
    select=sel-1;
    AxisMove(sel,sel-1);
    Update(FALSE);
    NgraphApp->FileUpdate();
    NgraphApp->Changed=TRUE;
  }
}

void TAxisWindow::AxisDown()
{
  int sel;

  if (menulock || globallock) return;
  NgraphApp->UnFocus();
  sel=select;
  if ((sel>=0) && (sel<=axisnum-1)) {
    movedownobj(obj,sel);
    select=sel+1;
    AxisMove(sel,sel+1);
    Update(FALSE);
    NgraphApp->FileUpdate();
    NgraphApp->Changed=TRUE;
  }
}

void TAxisWindow::AxisLast()
{
  int sel;

  if (menulock || globallock) return;
  NgraphApp->UnFocus();
  sel=select;
  if ((sel>=0) && (sel<=axisnum)) {
    movelastobj(obj,sel);
    select=axisnum;
    AxisMove(sel,axisnum);
    Update(FALSE);
    NgraphApp->FileUpdate();
    NgraphApp->Changed=TRUE;
  }
}

void TAxisWindow::AxisUpdate()
{
  int sel,ret;
  AxisDialog *dlg;
  int update;

  if (menulock || globallock) return;
  NgraphApp->UnFocus();
  sel=select;
  update=FALSE;
  if ((sel>=0) && (sel<=axisnum)) {
    dlg=new AxisDialog(this,DIALOG_AXIS,obj,sel,TRUE);
    if ((ret=dlg->Execute())==IDDELETE) {
      AxisDel(sel);
      update=TRUE;
      select=-1;
    } else if (ret==IDOK) update=TRUE;
    if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
    delete dlg;
  }
  if (update) Update(FALSE);
}

void TAxisWindow::AxisSelect()
{
  int sel;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=0) && (sel<=axisnum)) NgraphApp->Focus(obj,sel);
}

void TAxisWindow::AxisHidden()
{
  int sel;
  int hiden;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=0) && (sel<=axisnum)) {
    getobj(obj,"hidden",sel,0,NULL,&hiden);
    hiden=hiden?FALSE:TRUE;
    putobj(obj,"hidden",sel,&hiden);
    Update(FALSE);
    NgraphApp->Changed=TRUE;
  }
}

void TAxisWindow::Update(int clear)
{
  if (clear) select=-1;
  axisnum=chkobjlastinst(obj);
  ChangeAxisNum();
  Invalidate();
}

void TAxisWindow::ChangeAxisNum()
{
  int width,height;
  TRect rect;

  width=76*NgraphApp->FWidth;
  height=(axisnum+2)*NgraphApp->FHeight;
  Scroller->SetRange(width,height);
  Scroller->SetPageSize();
}

void TAxisWindow::SelectAxis(HDC dc,int sel)
{
  int x,y,smode;

  if ((select!=-1) && (sel!=select)) {
    x=NgraphApp->FWidth;
    y=(select+2)*NgraphApp->FHeight;
    smode=::SetROP2(dc,R2_XORPEN);
    ::SelectObject(dc,brush1);
    ::SelectObject(dc,pen1);
    ::Rectangle(dc,x-NgraphApp->FWidth,y-NgraphApp->FHeight,x+76*NgraphApp->FWidth,y);
    ::SetROP2(dc,smode);
  }
  if ((sel>=0) && (sel<=axisnum)) select=sel;
  else select=-1;
  if (select!=-1) {
    x=NgraphApp->FWidth;
    y=(select+2)*NgraphApp->FHeight;
    smode=::SetROP2(dc,R2_XORPEN);
    ::SelectObject(dc,brush1);
    ::SelectObject(dc,pen1);
    ::Rectangle(dc,x-NgraphApp->FWidth,y-NgraphApp->FHeight,x+76*NgraphApp->FWidth,y);
    ::SetROP2(dc,smode);
  }
}

void TAxisWindow::EvPaint()
{
  PAINTSTRUCT ps;
  HDC dc;
  RECT rc;
  int i;
  char buf[256];
  RECT rect;
  int h,w,x,y,len,cx;
  double min,max,inc;
  char *valstr;
  HFONT orgfont;
  int hiden;
  HBRUSH brush;
  char *name;

  if (::GetUpdateRect(Handle,&rc,FALSE)) {
    dc=::BeginPaint(Handle,&ps);
    if (globallock) {
      ::EndPaint(Handle,&ps);
      return;
    }
    Scroller->BeginView(TDC(dc),TRect(rc));
    axisnum=chkobjlastinst(obj);
    orgfont=(HFONT)SelectObject(dc,menulocal.menufont);
    SetTextAlign(dc,TA_BOTTOM | TA_LEFT);
    h=NgraphApp->FHeight;
    w=NgraphApp->FWidth;

    brush=CreateSolidBrush(GetSysColor(COLOR_MENU));
    rect.top=0;
    rect.bottom=h;
    rect.left=0;
    rect.right=76*w+w;
	FillRect(dc,&rect,brush);
    DeleteObject(brush);
    SetTextColor(dc,GetSysColor(COLOR_MENUTEXT));
    SetBkMode(dc,TRANSPARENT);
    x=w;
    y=h;
    rect.top=y-NgraphApp->FHeight;
    rect.bottom=y;
    rect.left=x;
    rect.right=x+6*w;
    len=sprintf(buf,"#");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=6*w;
    rect.left=x;
    rect.right=x+5*w;
    len=sprintf(buf,"name");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=5*w;
    rect.left=x;
    rect.right=x+10*w;
    len=sprintf(buf,"    min");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=10*w;
    rect.left=x;
    rect.right=x+10*w;
    len=sprintf(buf,"    max");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=10*w;
    rect.left=x;
    rect.right=x+10*w;
    len=sprintf(buf,"    inc");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=10*w;
    rect.left=x;
    rect.right=x+7*w;
    len=sprintf(buf," type  ");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=7*w;
    rect.left=x;
    rect.right=x+6*w;
    len=sprintf(buf,"     x");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=6*w;
    rect.left=x;
    rect.right=x+6*w;
    len=sprintf(buf,"     y");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=6*w;
    rect.left=x;
    rect.right=x+6*w;
    len=sprintf(buf,"   dir");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=6*w;
    rect.left=x+w;
    rect.right=x+8*w;
    len=sprintf(buf,"^#");
    ::ExtTextOut(dc,x+w,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);

    for (i=0;i<=axisnum;i++)
    if (((i+2)*h>=(rc.top+Scroller->YPos))
    && ((i-1)*h<=(rc.bottom+Scroller->YPos))) {
      x=w;
      y=(i+2)*h;
      ::SelectObject(dc,pen2);
      ::SelectObject(dc,brush2);
	  ::Rectangle(dc,x-w,y-h,x+76*w,y);
      rect.top=y-NgraphApp->FHeight;
      rect.bottom=y;

      getobj(obj,"hidden",i,0,NULL,&hiden);
      if (hiden) SetTextColor(dc,col1h);
      else SetTextColor(dc,col1);

      getobj(obj,"id",i,0,NULL,&cx);
      rect.left=x;
	  rect.right=x+6*w;
	  len=sprintf(buf,"%d",cx);
	  ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
	  x+=6*w;

  	  getobj(obj,"group",i,0,NULL,&name);
      rect.left=x;
	  rect.right=x+5*w;
      if (name!=NULL) {
        ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,name,strlen(name),NULL);
      } else {
        ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,"....................",20,NULL);
      }
	  x+=5*w;

      getobj(obj,"min",i,0,NULL,&min);
      getobj(obj,"max",i,0,NULL,&max);
      getobj(obj,"inc",i,0,NULL,&inc);

      rect.left=x;
      rect.right=x+10*w;
      if ((min==0) && (max==0)) len=sprintf(buf,"---------");
      else len=sprintf(buf,"%+.2e",min);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
      x+=10*w;

      rect.left=x;
      rect.right=x+10*w;
      if ((min==0) && (max==0)) len=sprintf(buf,"---------");
      else len=sprintf(buf,"%+.2e",max);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
      x+=10*w;

      rect.left=x;
      rect.right=x+10*w;
      if (inc==0) len=sprintf(buf,"---------");
      else len=sprintf(buf,"%+.2e",inc);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
      x+=10*w;

      sgetobjfield(obj,i,"type",NULL,&valstr,FALSE,FALSE,FALSE);
      rect.left=x+w;
      rect.right=x+7*w;
      ::ExtTextOut(dc,x+w,y,ETO_CLIPPED,&rect,valstr,strlen(valstr),NULL);
      memfree(valstr);
      x+=7*w;

      getobj(obj,"x",i,0,NULL,&cx);
      rect.left=x;
      rect.right=x+6*w;
      len=sprintf(buf,"%6d",cx);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
      x+=6*w;

      getobj(obj,"y",i,0,NULL,&cx);
      rect.left=x;
      rect.right=x+6*w;
      len=sprintf(buf,"%6d",cx);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
      x+=6*w;

      getobj(obj,"direction",i,0,NULL,&cx);
      rect.left=x;
      rect.right=x+6*w;
      len=sprintf(buf,"%6d",cx);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
      x+=6*w;

	  getobj(obj,"oid",i,0,NULL,&cx);
      rect.left=x+w;
	  rect.right=x+8*w;
	  len=sprintf(buf,"^%d",cx);
	  ::ExtTextOut(dc,x+w,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
	  x+=8*w;

    }
    SelectAxis(dc,select);
    SetScrollPos(SB_HORZ,Scroller->XPos,TRUE);
    SetScrollPos(SB_VERT,Scroller->YPos,TRUE);
    SelectObject(dc,orgfont);
    ::EndPaint(Handle,&ps);
  }
}

void TAxisWindow::ScaleClear()
{
  struct objlist *obj;
  int i;

  if (menulock || globallock) return;
  if ((obj=chkobject("axis"))==NULL) return;
  for (i=0;i<=chkobjlastinst(obj);i++) {
    exeobj(obj,"clear",i,0,NULL);
    NgraphApp->Changed=TRUE;
  }
  Update(FALSE);
}

void TMyWindow::CmAxisNewFrame()
{
  TAxisWindow *win;
  SectionDialog *dlg;
  struct objlist *obj,*obj2;
  int idx,idy,idu,idr,idg,ret;
  int type,x,y,lenx,leny;
  struct narray group;
  char *argv[2];

  if (menulock || globallock) return;
  if ((obj=chkobject("axis"))==NULL) return;
  if ((obj2=getobject("axisgrid"))==NULL) return;
  idx=newobj(obj);
  idy=newobj(obj);
  idu=newobj(obj);
  idr=newobj(obj);
  idg=-1;
  arrayinit(&group,sizeof(int));
  type=1;
  x=3500;
  y=22000;
  lenx=14000;
  leny=14000;
  arrayadd(&group,&type);
  arrayadd(&group,&idx);
  arrayadd(&group,&idy);
  arrayadd(&group,&idu);
  arrayadd(&group,&idr);
  arrayadd(&group,&x);
  arrayadd(&group,&y);
  arrayadd(&group,&lenx);
  arrayadd(&group,&leny);
  argv[0]=(char *)&group;
  argv[1]=NULL;
  exeobj(obj,"default_grouping",idr,1,argv);
  arraydel(&group);
  dlg=new SectionDialog(this,DIALOG_SECTION,x,y,lenx,leny,obj,idx,idy,idu,idr,obj2,&idg,FALSE);
  ret=dlg->Execute();
  if ((ret==IDDELETE) || (ret==IDCANCEL)) {
    if (idg!=-1) delobj(obj2,idg);
    delobj(obj,idr);
    delobj(obj,idu);
    delobj(obj,idy);
    delobj(obj,idx);
  } else  NgraphApp->Changed=TRUE;
  delete dlg;
  win=(TAxisWindow *)pTAxisWindow;
  if (win!=NULL) win->Update(TRUE);
}

void TMyWindow::CmAxisNewSection()
{
  TAxisWindow *win;
  SectionDialog *dlg;
  struct objlist *obj,*obj2;
  int idx,idy,idu,idr,idg,ret,oidx,oidy;
  int type,x,y,lenx,leny;
  struct narray group;
  char *argv[2];
  char *ref;

  if (menulock || globallock) return;
  if ((obj=getobject("axis"))==NULL) return;
  if ((obj2=getobject("axisgrid"))==NULL) return;
  idx=newobj(obj);
  idy=newobj(obj);
  idu=newobj(obj);
  idr=newobj(obj);
  idg=newobj(obj2);
  arrayinit(&group,sizeof(int));
  type=2;
  x=3500;
  y=22000;
  lenx=14000;
  leny=14000;
  arrayadd(&group,&type);
  arrayadd(&group,&idx);
  arrayadd(&group,&idy);
  arrayadd(&group,&idu);
  arrayadd(&group,&idr);
  arrayadd(&group,&x);
  arrayadd(&group,&y);
  arrayadd(&group,&lenx);
  arrayadd(&group,&leny);
  argv[0]=(char *)&group;
  argv[1]=NULL;
  exeobj(obj,"default_grouping",idr,1,argv);
  arraydel(&group);
  if (idg>=0) {
    getobj(obj,"oid",idx,0,NULL,&oidx);
    if ((ref=(char *)memalloc(15))!=NULL) {
      sprintf(ref,"axis:^%d",oidx);
      putobj(obj2,"axis_x",idg,ref);
    }
    getobj(obj,"oid",idy,0,NULL,&oidy);
    if ((ref=(char *)memalloc(15))!=NULL) {
      sprintf(ref,"axis:^%d",oidy);
      putobj(obj2,"axis_y",idg,ref);
    }
  }
  dlg=new SectionDialog(this,DIALOG_SECTION,x,y,lenx,leny,obj,idx,idy,idu,idr,obj2,&idg,TRUE);
  ret=dlg->Execute();
  if ((ret==IDDELETE) || (ret==IDCANCEL)) {
    delobj(obj2,idg);
    delobj(obj,idr);
    delobj(obj,idu);
    delobj(obj,idy);
    delobj(obj,idx);
  } else NgraphApp->Changed=TRUE;
  delete dlg;
  win=(TAxisWindow *)pTAxisWindow;
  if (win!=NULL) win->Update(TRUE);
}

void TMyWindow::CmAxisNewCross()
{
  TAxisWindow *win;
  CrossDialog *dlg;
  struct objlist *obj;
  int idx,idy,ret;
  int type,x,y,lenx,leny;
  struct narray group;
  char *argv[2];

  if (menulock || globallock) return;
  if ((obj=chkobject("axis"))==NULL) return;
  idx=newobj(obj);
  idy=newobj(obj);
  arrayinit(&group,sizeof(int));
  type=3;
  x=3500;
  y=22000;
  lenx=14000;
  leny=14000;
  arrayadd(&group,&type);
  arrayadd(&group,&idx);
  arrayadd(&group,&idy);
  arrayadd(&group,&x);
  arrayadd(&group,&y);
  arrayadd(&group,&lenx);
  arrayadd(&group,&leny);
  argv[0]=(char *)&group;
  argv[1]=NULL;
  exeobj(obj,"default_grouping",idy,1,argv);
  arraydel(&group);

  dlg=new CrossDialog(this,DIALOG_CROSS,x,y,lenx,leny,obj,idx,idy);
  ret=dlg->Execute();
  if ((ret==IDDELETE) || (ret==IDCANCEL)) {
    delobj(obj,idy);
    delobj(obj,idx);
  } else NgraphApp->Changed=TRUE;
  delete dlg;
  win=(TAxisWindow *)pTAxisWindow;
  if (win!=NULL) win->Update(TRUE);
}

void TMyWindow::CmAxisNewSingle()
{
  struct objlist *obj;
  int id,ret;
  TAxisWindow *win;
  AxisDialog *dlg;

  if (menulock || globallock) return;
  if ((obj=chkobject("axis"))==NULL) return;
  if ((id=newobj(obj))>=0) {
    dlg=new AxisDialog(this,DIALOG_AXIS,obj,id,TRUE);
    ret=dlg->Execute();
    if ((ret==IDDELETE) || (ret==IDCANCEL)) {
      delobj(obj,id);
    } else NgraphApp->Changed=TRUE;
    delete dlg;
    win=(TAxisWindow *)pTAxisWindow;
    if (win!=NULL) win->Update(TRUE);
  }
}

void TMyWindow::CmAxisDel()
{
  TAxisWindow *win;
  CopyDialog *dlg;
  struct objlist *obj;
  int i;

  if (menulock || globallock) return;
  if ((obj=chkobject("axis"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new CopyDialog((TWindow *)this,DIALOG_COPY,obj,-1,AxisCB);
  if (dlg->Execute()==IDOK) {
    i=dlg->sel;
    delete dlg;
  } else {
    delete dlg;
    return;
  }
  AxisDel(i);
  NgraphApp->Changed=TRUE;
  win=(TAxisWindow *)pTAxisWindow;
  if (win!=NULL) win->Update(TRUE);
  NgraphApp->FileUpdate();
}

void TMyWindow::CmAxisUpdate()
{
  TAxisWindow *win;
  CopyDialog *dlg;
  AxisDialog *fdlg;
  struct objlist *obj;
  int i,ret;

  if (menulock || globallock) return;
  if ((obj=chkobject("axis"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new CopyDialog((TWindow *)this,DIALOG_COPY,obj,-1,AxisCB);
  if (dlg->Execute()==IDOK) {
    i=dlg->sel;
    delete dlg;
    if (i<0) return;
  } else {
    delete dlg;
    return;
  }
  fdlg=new AxisDialog(this,DIALOG_AXIS,obj,i,TRUE);
  if ((ret=fdlg->Execute())==IDDELETE) {
    AxisDel(i);
  }
  if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
  win=(TAxisWindow *)pTAxisWindow;
  if (win!=NULL) win->Update(TRUE);
  NgraphApp->FileUpdate();
}

void TMyWindow::CmAxisZoom()
{
  TAxisWindow *win;
  SelectDialog *dlg;
  ZoomDialog *zm;
  struct narray farray;
  struct objlist *obj;
  int i;
  int *array,num,room;
  double zoom,min,max,mid,wd;
  char *argv[4];

  if (menulock || globallock) return;
  if ((obj=chkobject("axis"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  zm=new ZoomDialog((TWindow *)this,DIALOG_ZOOM);
  if ((zm->Execute()==IDOK) && (zm->zoom>0)) {
    zoom=zm->zoom/10000.0;
    dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,AxisCB,
                         (struct narray *)&farray,NULL);
    if (dlg->Execute()==IDOK) {
      num=arraynum(&farray);
      array=(int *)arraydata(&farray);
      for (i=0;i<num;i++) {
        getobj(obj,"min",array[i],0,NULL,&min);
        getobj(obj,"max",array[i],0,NULL,&max);
        wd=(max-min)/2;
        if (wd!=0) {
          mid=(min+max)/2;
          min=mid-wd*zoom;
          max=mid+wd*zoom;
          room=1;
          argv[0]=(char *)&min;
          argv[1]=(char *)&max;
          argv[2]=(char *)&room;
          argv[3]=NULL;
          exeobj(obj,"scale",array[i],3,argv);
          NgraphApp->Changed=TRUE;
        }
      }
      win=(TAxisWindow *)pTAxisWindow;
      if (win!=NULL) win->Update(TRUE);
    }
    arraydel(&farray);
    delete dlg;
  }
  delete zm;
}

void TMyWindow::CmAxisClear()
{
  TAxisWindow *win;
  SelectDialog *dlg;
  struct narray farray;
  struct objlist *obj;
  int i;
  int *array,num;

  if (menulock || globallock) return;
  if ((obj=chkobject("axis"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,AxisCB,
                       (struct narray *)&farray,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&farray);
    array=(int *)arraydata(&farray);
    for (i=0;i<num;i++) {
      exeobj(obj,"scale_push",array[i],0,NULL);
      exeobj(obj,"clear",array[i],0,NULL);
      NgraphApp->Changed=TRUE;
    }
    win=(TAxisWindow *)pTAxisWindow;
    if (win!=NULL) win->Update(TRUE);
  }
  arraydel(&farray);
  delete dlg;
}

void TMyWindow::CmAxisGridNew()
{
  struct objlist *obj;
  int id,ret;
  GridDialog *dlg;

  if (menulock || globallock) return;
  if ((obj=chkobject("axisgrid"))==NULL) return;
  if ((id=newobj(obj))>=0) {
    dlg=new GridDialog(this,DIALOG_GRID,obj,id,TRUE);
    ret=dlg->Execute();
    if ((ret==IDDELETE) || (ret==IDCANCEL)) {
      delobj(obj,id);
    } else NgraphApp->Changed=TRUE;
    delete dlg;
  }
}

void TMyWindow::CmAxisGridDel()
{
  SelectDialog *dlg;
  struct narray farray;
  struct objlist *obj;
  int i;
  int num,*array;

  if (menulock || globallock) return;
  if ((obj=chkobject("axisgrid"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,GridCB,
                       (struct narray *)&farray,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&farray);
    array=(int *)arraydata(&farray);
    for (i=num-1;i>=0;i--) {
      delobj(obj,array[i]);
      NgraphApp->Changed=TRUE;
    }
  }
  arraydel(&farray);
  delete dlg;
}

void TMyWindow::CmAxisGridUpdate()
{
  SelectDialog *dlg;
  GridDialog *fdlg;
  struct narray farray;
  struct objlist *obj;
  int i,j,ret;
  int *array,num;

  if (menulock || globallock) return;
  if ((obj=chkobject("axisgrid"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,GridCB,
                       (struct narray *)&farray,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&farray);
    array=(int *)arraydata(&farray);
    for (i=0;i<num;i++) {
      fdlg=new GridDialog(this,DIALOG_GRID,obj,array[i],TRUE);
      if ((ret=fdlg->Execute())==IDDELETE) {
        delobj(obj,array[i]);
        for (j=i+1;j<num;j++) array[j]--;
      }
      if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
      delete fdlg;
    }
  }
  arraydel(&farray);
  delete dlg;
}

void TMyWindow::CmScaleUndo()
{
  char *argv[1];
  struct objlist *obj;
  struct narray farray;
  int i,num,*array;
  SelectDialog *dlg;
  TAxisWindow *win;

  if (menulock || globallock) return;
  if ((obj=chkobject("axis"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,AxisCB,
                       (struct narray *)&farray,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&farray);
    array=(int *)arraydata(&farray);
    for (i=num-1;i>=0;i--) {
      argv[0]=NULL;
      exeobj(obj,"scale_pop",array[i],0,argv);
      NgraphApp->Changed=TRUE;
    }
    win=(TAxisWindow *)pTAxisWindow;
    if (win!=NULL) win->Update(TRUE);
  }
  arraydel(&farray);
}

void TMyWindow::CmWindowAxis()
{
  TAxisWindow *win;
  TRect rect;
  int x,y;

  if (menulock || globallock) return;
  if (pTAxisWindow==NULL) {
    win=(TAxisWindow *)pTAxisWindow=new TAxisWindow();
    win->Frame=new TSubFrameWindow(this,"Axis Window",win,
    &menulocal.axisx,&menulocal.axisy,&menulocal.axiswidth,&menulocal.axisheight);
    win->Frame->SetIcon(GetApplication(),ICON_7);
    NgraphApp->Frame->GetWindowRect(rect);
    x=rect.left;
    y=rect.top;
    if (menulocal.axiswidth==CW_USEDEFAULT) win->Frame->Attr.W=FWidth*76+2*FWidth;
    else win->Frame->Attr.W=menulocal.axiswidth;
    if (menulocal.axisheight==CW_USEDEFAULT) win->Frame->Attr.H=FHeight*10;
    else win->Frame->Attr.H=menulocal.axisheight;
    if (menulocal.axisx==CW_USEDEFAULT) win->Frame->Attr.X=CW_USEDEFAULT;
    else {
      win->Frame->Attr.X=menulocal.axisx+x;
      if (win->Frame->Attr.X<0) win->Frame->Attr.X=0;
    }
    if (menulocal.axisy==CW_USEDEFAULT) win->Frame->Attr.Y=CW_USEDEFAULT;
    else {
      win->Frame->Attr.Y=menulocal.axisy+y;
      if (win->Frame->Attr.Y<0) win->Frame->Attr.Y=0;
    }
    win->Frame->Create();
  } else {
    win=(TAxisWindow *)pTAxisWindow;
    win->CloseWindow();
  }
}

void TMyWindow::CeWindowAxis(TCommandEnabler& ce)
{
  if (pTAxisWindow==NULL) ce.SetCheck(TCommandEnabler::Unchecked);
  else ce.SetCheck(TCommandEnabler::Checked);
}


