/* 
 * 
 * This file is part of "Ngraph for Windows".
 * 
 * Copyright (C) 2015, Satoshi ISHIZAKA. isizaka@msa.biglobe.ne.jp
 * 
 * "Ngraph for Windows" is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * "Ngraph for Windows" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

/*
 *
 * winfile.cpp
 *
 */

#include <owl\pch.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <fcntl.h>
#include <io.h>
#include <dir.h>
#include <process.h>

extern "C" {
#include "ngraph.h"
#include "object.h"
#include "ioutil.h"
#include "nstring.h"
#include "mathcode.h"
#include "mathfn.h"
#include "gra.h"
#include "spline.h"
#include "config.h"
#include "ioutil.h"
#include "winfopen.h"
#include "owinmenu.h"
}

#include "bitmap.rh"
#include "winmenu.rh"
#include "winmenu.h"
#include "windialg.h"
#include "wincommn.h"
#include "winview.h"
#include "winfile.h"

#define FITSAVE "fit.ngp"

class DirectoryDialog  : public TMyDialog {
  public:
    DirectoryDialog(TWindow *parent,TResId resID);
  protected:
    void SetupWindow();
    void CloseWindow(int retVal);
};

DirectoryDialog::DirectoryDialog(TWindow *parent,TResId resID):TMyDialog(parent,resID)
{
}

void DirectoryDialog::SetupWindow()
{
  int len;
  char *cwd;

  TMyDialog::SetupWindow();
  len=GetCurrentDirectory(0,NULL);
  if ((cwd=(char *)memalloc(len+1))!=NULL) {
    GetCurrentDirectory(len,cwd);
  }
  SetDlgItemText(IDDIRDIR,cwd);
}

void DirectoryDialog::CloseWindow(int retVal)
{
  int len;
  char *cwd;

  if (retVal==IDOK) {
    len=SendDlgItemMessage(IDDIRDIR,WM_GETTEXTLENGTH,0,0);
    if ((cwd=(char *)memalloc(len+1))!=NULL) {
      GetDlgItemText(IDDIRDIR,cwd,len+1);
      SetCurrentDirectory(cwd);
    }
  }
  TMyDialog::CloseWindow(retVal);
}

class MoveDialog  : public TMyDialog {
  public:
    MoveDialog(TWindow *parent,TResId resID,struct objlist *obj,int id);
  protected:
    struct objlist *Obj;
    int Id;
    void SetupWindow();
    void CloseWindow(int retValue=IDOK);
    void AddClicked();
    void RemoveClicked();
    DECLARE_RESPONSE_TABLE(MoveDialog);
};

DEFINE_RESPONSE_TABLE1(MoveDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDMVADDB,BN_CLICKED,AddClicked),
  EV_CHILD_NOTIFY(IDMVREMOVEB,BN_CLICKED,RemoveClicked),
END_RESPONSE_TABLE;

MoveDialog::MoveDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
: TMyDialog(parent,resID)
{
  Obj=obj;
  Id=id;
}

void MoveDialog::AddClicked()
{
  int len,a;
  double x,y;
  char *buf,buf2[128];
  char *endptr;

  len=SendDlgItemMessage(IDMVADD,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)memalloc(len+1))!=NULL) {
    GetDlgItemText(IDMVADD,buf,len+1);
    a=strtol(buf,&endptr,10);
    if (endptr[0]=='\0') {
      memfree(buf);
      len=SendDlgItemMessage(IDMVX,WM_GETTEXTLENGTH,0,0);
      if ((buf=(char *)memalloc(len+1))!=NULL) {
        GetDlgItemText(IDMVX,buf,len+1);
        x=strtod(buf,&endptr);
        if (endptr[0]=='\0') {
          memfree(buf);
          len=SendDlgItemMessage(IDMVY,WM_GETTEXTLENGTH,0,0);
          if ((buf=(char *)memalloc(len+1))!=NULL) {
            GetDlgItemText(IDMVY,buf,len+1);
            y=strtod(buf,&endptr);
            if (endptr[0]=='\0') {
              memfree(buf);
              sprintf(buf2,"%8d  %+.15e %+.15e",a,x,y);
              SendDlgItemMessage(IDMVLIST,LB_INSERTSTRING,-1,(LPARAM)buf2);
            } else memfree(buf);
          }
        } else memfree(buf);
      }
    } else memfree(buf);
  }
}

void MoveDialog::RemoveClicked()
{
  int i,count;

  count=SendDlgItemMessage(IDMVLIST,LB_GETCOUNT,0,0);
  for (i=count-1;i>=0;i--)
    if (SendDlgItemMessage(IDMVLIST,LB_GETSEL,i,0)>0)
      SendDlgItemMessage(IDMVLIST,LB_DELETESTRING,i,0);
}

void MoveDialog::SetupWindow()
{
  int j,movenum,line;
  double x,y;
  struct narray *move,*movex,*movey;
  char buf[128];

  TMyDialog::SetupWindow();
  SendDlgItemMessage(IDMVLIST,LB_RESETCONTENT,0,0);
  getobj(Obj,"move_data",Id,0,NULL,&move);
  getobj(Obj,"move_data_x",Id,0,NULL,&movex);
  getobj(Obj,"move_data_y",Id,0,NULL,&movey);
  movenum=arraynum(move);
  if (arraynum(movex)<movenum) movenum=arraynum(movex);
  if (arraynum(movey)<movenum) movenum=arraynum(movey);
  if (movenum>0) {
    for (j=0;j<movenum;j++) {
      line=*(int *)arraynget(move,j);
      x=*(double *)arraynget(movex,j);
      y=*(double *)arraynget(movey,j);
      sprintf(buf,"%8d  %+.15e %+.15e",line,x,y);
      SendDlgItemMessage(IDMVLIST,LB_INSERTSTRING,-1,(LPARAM)buf);
    }
  }
}

void MoveDialog::CloseWindow(int retValue)
{
  int i,j,count,movenum,line,len,a;
  double x,y;
  struct narray *move,*movex,*movey;
  char *buf,*buf2;
  char *endptr;

  if (retValue==IDOK) {
    getobj(Obj,"move_data",Id,0,NULL,&move);
    getobj(Obj,"move_data_x",Id,0,NULL,&movex);
    getobj(Obj,"move_data_y",Id,0,NULL,&movey);
    if (move!=NULL) {
      putobj(Obj,"move_data",Id,NULL);
      move=NULL;
    }
    if (movex!=NULL) {
      putobj(Obj,"move_data_x",Id,NULL);
      movex=NULL;
    }
    if (movey!=NULL) {
      putobj(Obj,"move_data_y",Id,NULL);
      movey=NULL;
    }
    count=SendDlgItemMessage(IDMVLIST,LB_GETCOUNT,0,0);
    for (i=0;i<count;i++) {
      len=SendDlgItemMessage(IDMVLIST,LB_GETTEXTLEN,i,0);
      if ((buf=(char *)memalloc(len+1))!=NULL) {
        SendDlgItemMessage(IDMVLIST,LB_GETTEXT,i,(LPARAM)buf);
        buf2=buf;
        a=strtol(buf2,&endptr,10);
        buf2=endptr;
        x=strtod(buf2,&endptr);
        buf2=endptr;
        y=strtod(buf2,&endptr);
        memfree(buf);
        if (move==NULL) move=arraynew(sizeof(int));
        if (movex==NULL) movex=arraynew(sizeof(double));
        if (movey==NULL) movey=arraynew(sizeof(double));
        movenum=arraynum(move);
        if (arraynum(movex)<movenum) movenum=arraynum(movex);
        if (arraynum(movey)<movenum) movenum=arraynum(movey);
        for (j=0;j<movenum;j++) {
          line=*(int *)arraynget(move,j);
          if (line==a) break;
        }
        if (j==movenum) {
          arrayadd(move,&a);
          arrayadd(movex,&x);
          arrayadd(movey,&y);
        }
      }
    }
    putobj(Obj,"move_data",Id,move);
    putobj(Obj,"move_data_x",Id,movex);
    putobj(Obj,"move_data_y",Id,movey);
  }
  TMyDialog::CloseWindow();
}

class MaskDialog  : public TMyDialog {
  public:
    MaskDialog(TWindow *parent,TResId resID,struct objlist *obj,int id);
  protected:
    struct objlist *Obj;
    int Id;
    void SetupWindow();
    void CloseWindow(int retValue=IDOK);
    void AddClicked();
    void RemoveClicked();
    DECLARE_RESPONSE_TABLE(MaskDialog);
};

DEFINE_RESPONSE_TABLE1(MaskDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDMKADDB,BN_CLICKED,AddClicked),
  EV_CHILD_NOTIFY(IDMKREMOVEB,BN_CLICKED,RemoveClicked),
END_RESPONSE_TABLE;

MaskDialog::MaskDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
: TMyDialog(parent,resID)
{
  Obj=obj;
  Id=id;
}

void MaskDialog::AddClicked()
{
  int len,a;
  char *buf,buf2[20];
  char *endptr;

  len=SendDlgItemMessage(IDMKADD,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)memalloc(len+1))!=NULL) {
    GetDlgItemText(IDMKADD,buf,len+1);
    a=strtol(buf,&endptr,10);
    if (endptr[0]=='\0') {
      sprintf(buf2,"%8d",a);
      SendDlgItemMessage(IDMKLIST,LB_INSERTSTRING,-1,(LPARAM)buf2);
    }
    memfree(buf);
  }
}

void MaskDialog::RemoveClicked()
{
  int i,count;

  count=SendDlgItemMessage(IDMKLIST,LB_GETCOUNT,0,0);
  for (i=count-1;i>=0;i--)
    if (SendDlgItemMessage(IDMKLIST,LB_GETSEL,i,0)>0)
      SendDlgItemMessage(IDMKLIST,LB_DELETESTRING,i,0);
}

void MaskDialog::SetupWindow()
{
  int j,masknum,line;
  struct narray *mask;
  char buf[20];

  TMyDialog::SetupWindow();
  SendDlgItemMessage(IDMKLIST,LB_RESETCONTENT,0,0);
  getobj(Obj,"mask",Id,0,NULL,&mask);
  if ((masknum=arraynum(mask))>0) {
    for (j=0;j<masknum;j++) {
      line=*(int *)arraynget(mask,j);
      sprintf(buf,"%8d",line);
      SendDlgItemMessage(IDMKLIST,LB_INSERTSTRING,-1,(LPARAM)buf);
    }
  }
}

void MaskDialog::CloseWindow(int retValue)
{
  int i,j,count,masknum,line,len,a;
  struct narray *mask;
  char *buf;
  char *endptr;

  if (retValue==IDOK) {
    getobj(Obj,"mask",Id,0,NULL,&mask);
    if (mask!=NULL) {
      putobj(Obj,"mask",Id,NULL);
      mask=NULL;
    }
    count=SendDlgItemMessage(IDMKLIST,LB_GETCOUNT,0,0);
    for (i=0;i<count;i++) {
      len=SendDlgItemMessage(IDMKLIST,LB_GETTEXTLEN,i,0);
      if ((buf=(char *)memalloc(len+1))!=NULL) {
        SendDlgItemMessage(IDMKLIST,LB_GETTEXT,i,(LPARAM)buf);
        a=strtol(buf,&endptr,10);
        memfree(buf);
        if (mask==NULL) mask=arraynew(sizeof(int));
        masknum=arraynum(mask);
        for (j=0;j<masknum;j++) {
          line=*(int *)arraynget(mask,j);
          if (line==a) break;
        }
        if (j==masknum) arrayadd(mask,&a);
      }
    }
    putobj(Obj,"mask",Id,mask);
  }
  TMyDialog::CloseWindow();
}

char *FitCB(struct objlist *obj,int id) {
  char *valstr,*s;

  if ((s=(char *)memalloc(128))==NULL) return NULL;
  sgetobjfield(obj,id,"type",NULL,&valstr,FALSE,FALSE,FALSE);
  sprintf(s,"%-5d %.100s",id,valstr);
  memfree(valstr);
  return s;
}

class FitDialog  : public TMyDialog {
  public:
    FitDialog(TWindow *parent,TResId resID,struct objlist *obj,int id);
    ~FitDialog();
  protected:
    struct objlist *Obj;
    int Id,Lastid;
    void SetupWindow();
    void SetupItem(int result,int id);
    int Apply();
    void Draw();
    void DeleteClicked() {CloseWindow(IDDELETE);}
    void CopyClicked() {
      int sel;
      if ((sel=CopyClick(this,Obj,Id,FitCB))!=-1) SetupItem(FALSE,sel);
    }
    void ResultClicked();
    void ClipClicked();
    void CloseWindow(int retVal);
    int LoadConfig(int errmes);
    void LoadClicked();
    void SaveClicked();
    DECLARE_RESPONSE_TABLE(FitDialog);
};

DEFINE_RESPONSE_TABLE1(FitDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
  EV_CHILD_NOTIFY(IDDELETE,BN_CLICKED,DeleteClicked),
  EV_CHILD_NOTIFY(IDFTRESULTB,BN_CLICKED,ResultClicked),
  EV_CHILD_NOTIFY(IDFTCLIPB,BN_CLICKED,ClipClicked),
  EV_CHILD_NOTIFY(IDFTDRAWB,BN_CLICKED,Draw),
  EV_CHILD_NOTIFY(IDFTLOAD,BN_CLICKED,LoadClicked),
  EV_CHILD_NOTIFY(IDFTSAVE,BN_CLICKED,SaveClicked),
END_RESPONSE_TABLE;


FitDialog::FitDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
:TMyDialog(parent,resID)
{
  Obj=obj;
  Id=id;
  Lastid=chkobjlastinst(Obj);
}

FitDialog::~FitDialog()
{
  int i,lastid;

  lastid=chkobjlastinst(Obj);
  for (i=lastid;i>Lastid;i--) delobj(Obj,i);
}

void FitDialog::SetupWindow()
{
  char *title;
  int len;

  TMyDialog::SetupWindow();
  len=GetWindowTextLength();
  if ((title=(char *)memalloc(len+10))!=NULL) {
    GetWindowText(title,len+1);
    sprintf(title+len," %d",Id);
    SetCaption(title);
    memfree(title);
  }
  SetupItem(TRUE,Id);
}

void FitDialog::SetupItem(int result,int id)
{
  int a;

  SetListFromObjField(Handle,IDFTTYPE,Obj,id,"type");
  SendDlgItemMessage(IDFTPOLY,CB_RESETCONTENT,0,0);
  SendDlgItemMessage(IDFTPOLY,CB_INSERTSTRING,-1,(LPARAM)"1");
  SendDlgItemMessage(IDFTPOLY,CB_INSERTSTRING,-1,(LPARAM)"2");
  SendDlgItemMessage(IDFTPOLY,CB_INSERTSTRING,-1,(LPARAM)"3");
  SendDlgItemMessage(IDFTPOLY,CB_INSERTSTRING,-1,(LPARAM)"4");
  SendDlgItemMessage(IDFTPOLY,CB_INSERTSTRING,-1,(LPARAM)"5");
  SendDlgItemMessage(IDFTPOLY,CB_INSERTSTRING,-1,(LPARAM)"6");
  SendDlgItemMessage(IDFTPOLY,CB_INSERTSTRING,-1,(LPARAM)"7");
  SendDlgItemMessage(IDFTPOLY,CB_INSERTSTRING,-1,(LPARAM)"8");
  SendDlgItemMessage(IDFTPOLY,CB_INSERTSTRING,-1,(LPARAM)"9");
  getobj(Obj,"poly_dimension",id,0,NULL,&a);
  a--;
  SendDlgItemMessage(IDFTPOLY,CB_SETCURSEL,a,0);
  SetTextFromObjField(Handle,IDFTWEIGHT,Obj,id,"weight_func");
  SetToggleFromObjField(Handle,IDFTTHROUGH,Obj,id,"through_point");
  SetTextFromObjField(Handle,IDFTPX,Obj,id,"point_x");
  SetTextFromObjField(Handle,IDFTPY,Obj,id,"point_y");
  SetTextFromObjField(Handle,IDFTMIN,Obj,id,"min");
  SetTextFromObjField(Handle,IDFTMAX,Obj,id,"max");
  SetTextFromObjField(Handle,IDFTDIV,Obj,id,"div");
  SetToggleFromObjField(Handle,IDFTINTERPOLATION,Obj,id,"interpolation");
  SetTextFromObjField(Handle,IDFTFN,Obj,id,"user_func");
  SetTextFromObjField(Handle,IDFTCONV,Obj,id,"converge");
  SetToggleFromObjField(Handle,IDFTDERIV,Obj,id,"derivative");
  SetTextFromObjField(Handle,IDFTP0,Obj,id,"parameter0");
  SetTextFromObjField(Handle,IDFTP1,Obj,id,"parameter1");
  SetTextFromObjField(Handle,IDFTP2,Obj,id,"parameter2");
  SetTextFromObjField(Handle,IDFTP3,Obj,id,"parameter3");
  SetTextFromObjField(Handle,IDFTP4,Obj,id,"parameter4");
  SetTextFromObjField(Handle,IDFTP5,Obj,id,"parameter5");
  SetTextFromObjField(Handle,IDFTP6,Obj,id,"parameter6");
  SetTextFromObjField(Handle,IDFTP7,Obj,id,"parameter7");
  SetTextFromObjField(Handle,IDFTP8,Obj,id,"parameter8");
  SetTextFromObjField(Handle,IDFTP9,Obj,id,"parameter9");
  SetTextFromObjField(Handle,IDFTD0,Obj,id,"derivative0");
  SetTextFromObjField(Handle,IDFTD1,Obj,id,"derivative1");
  SetTextFromObjField(Handle,IDFTD2,Obj,id,"derivative2");
  SetTextFromObjField(Handle,IDFTD3,Obj,id,"derivative3");
  SetTextFromObjField(Handle,IDFTD4,Obj,id,"derivative4");
  SetTextFromObjField(Handle,IDFTD5,Obj,id,"derivative5");
  SetTextFromObjField(Handle,IDFTD6,Obj,id,"derivative6");
  SetTextFromObjField(Handle,IDFTD7,Obj,id,"derivative7");
  SetTextFromObjField(Handle,IDFTD8,Obj,id,"derivative8");
  SetTextFromObjField(Handle,IDFTD9,Obj,id,"derivative9");
}

class FitLoadDialog  : public TMyDialog {
  public:

    FitLoadDialog(TWindow *parent,TResId resID,struct objlist *obj,int sid);

    int sel;

  protected:

    struct objlist *Obj;

    int Sid;

    void SetupWindow();

    void CloseWindow(int retValue=IDCANCEL);

    DECLARE_RESPONSE_TABLE(FitLoadDialog);

};


DEFINE_RESPONSE_TABLE1(FitLoadDialog,TMyDialog)
  EV_LBN_DBLCLK(IDFLLIST,CmOk),
END_RESPONSE_TABLE;

FitLoadDialog::FitLoadDialog(TWindow *parent,TResId resID,struct objlist *obj,
                              int sid)
:TMyDialog(parent,resID)
{
  Obj=obj;
  Sid=sid;
  sel=-1;
}

void FitLoadDialog::SetupWindow()
{
  int i;
  char *s;

  TMyDialog::SetupWindow();
  SendDlgItemMessage(IDFLLIST,LB_RESETCONTENT,0,0);
  for (i=Sid;i<=chkobjlastinst(Obj);i++) {
    getobj(Obj,"profile",i,0,NULL,&s);
    if (s!=NULL) SendDlgItemMessage(IDFLLIST,LB_INSERTSTRING,-1,(LPARAM)s);
    else SendDlgItemMessage(IDFLLIST,LB_INSERTSTRING,-1,(LPARAM)"");
  }
}

void FitLoadDialog::CloseWindow(int retValue)
{
  sel=SendDlgItemMessage(IDCLIST,LB_GETCURSEL,0,0);
  TMyDialog::CloseWindow(retValue);
}

class FitSaveDialog  : public TMyDialog {
  public:
    FitSaveDialog(TWindow *parent,TResId resID,struct objlist *obj,int sid);
    char *Profile;
  protected:
    struct objlist *Obj;
    int Sid;
    void SetupWindow();
    void CloseWindow(int retVal);
};

FitSaveDialog::FitSaveDialog(TWindow *parent,TResId resID,struct objlist *obj,int sid)
:TMyDialog(parent,resID)
{
  Obj=obj;
  Sid=sid;
  Profile=NULL;
}

void FitSaveDialog::SetupWindow()
{
  int i;
  char *s;

  TMyDialog::SetupWindow();
  SendDlgItemMessage(IDFSPROFILE,CB_RESETCONTENT,0,0);
  for (i=Sid;i<=chkobjlastinst(Obj);i++) {
    getobj(Obj,"profile",i,0,NULL,&s);
    if (s!=NULL) SendDlgItemMessage(IDFSPROFILE,CB_INSERTSTRING,-1,(LPARAM)s);
    else SendDlgItemMessage(IDFSPROFILE,CB_INSERTSTRING,-1,(LPARAM)"");
  }
  SetDlgItemText(IDFSPROFILE,"");
}

void FitSaveDialog::CloseWindow(int retVal)
{
  int len;

  if (retVal==IDOK) {
    len=SendDlgItemMessage(IDFSPROFILE,WM_GETTEXTLENGTH,0,0);
    if ((Profile=(char *)memalloc(len+1))!=NULL) {
      GetDlgItemText(IDFSPROFILE,Profile,len+1);
    }
  }
  TMyDialog::CloseWindow(retVal);
}

int FitDialog::LoadConfig(int errmes)
{
  int lastid;
  int newid;
  struct objlist *shell;
  struct narray sarray;
  char *argv[2];
  char *file;

  lastid=chkobjlastinst(Obj);
  if (lastid==Lastid) {
    if ((file=searchscript(FITSAVE))==NULL) {
      if (errmes) {
#ifdef ENGLISH
        MessageBox("Setting file not found.",FITSAVE,MB_OK|MB_ICONEXCLAMATION);
#else
        MessageBox("ݒt@C܂",FITSAVE,MB_OK|MB_ICONEXCLAMATION);
#endif
      }
      return FALSE;
    }
    if ((shell=chkobject("shell"))==NULL) return FALSE;
    newid=newobj(shell);
    if (newid<0) {
      memfree(file);
      return FALSE;
    }
    arrayinit(&sarray,sizeof(char *));
    changefilename(file);
    if (arrayadd(&sarray,&file)==NULL) {
      memfree(file);
      arraydel2(&sarray);
      return FALSE;
    }
    argv[0]=(char *)&sarray;
    argv[1]=NULL;
    exeobj(shell,"shell",newid,1,argv);
    arraydel2(&sarray);
    delobj(shell,newid);
  }
  return TRUE;
}

void FitDialog::LoadClicked()
{
  int lastid,id;
  FitLoadDialog *dlg;

  if (!LoadConfig(TRUE)) return;
  lastid=chkobjlastinst(Obj);
  if ((Lastid<0) || (lastid==Lastid)) {
#ifdef ENGLISH
    MessageBox("No settings.",FITSAVE,MB_OK|MB_ICONEXCLAMATION);
#else
    MessageBox("ݒ肪L܂",FITSAVE,MB_OK|MB_ICONEXCLAMATION);
#endif
    return;
  }
  dlg=new FitLoadDialog((TWindow *)this,DIALOG_FITLOAD,Obj,Lastid+1);
  if ((dlg->Execute()==IDOK) && (dlg->sel>=0)) {
    id=dlg->sel+Lastid+1;
    SetupItem(FALSE,id);
  }
  delete dlg;
  return;
}


void FitDialog::SaveClicked()
{
  FitSaveDialog *dlg;
  int i,id,a,len;
  char *s;
  char *ngpfile;
  int error;
  HANDLE hFile;

  if (!LoadConfig(FALSE)) return;
  dlg=new FitSaveDialog((TWindow *)this,DIALOG_FITSAVE,Obj,Lastid+1);
  if (dlg->Execute()==IDOK) {
    for (i=Lastid+1;i<=chkobjlastinst(Obj);i++) {
      getobj(Obj,"profile",i,0,NULL,&s);
      if (strcmp(s,dlg->Profile)==0) {
#ifdef ENGLISH
        if (MessageBox("Overwrite existing setting?","Confirm",MB_ICONQUESTION|MB_YESNO)
           !=IDYES) {
          delete dlg;
          return;
        }
#else
        if (MessageBox("vt@C̐ݒ肪݂܂B㏑܂H","mF",MB_ICONQUESTION|MB_YESNO)
           !=IDYES) {
          delete dlg;
          return;
        }
#endif
        id=i;
        break;
      }
    }
    if (i>chkobjlastinst(Obj)) id=newobj(Obj);
    if (putobj(Obj,"profile",id,dlg->Profile)==-1) {
      delete dlg;
      return;
    }
    delete dlg;
    if (SetObjFieldFromList(Handle,IDFTTYPE,Obj,id,"type")) return;
    a=SendDlgItemMessage(IDFTPOLY,CB_GETCURSEL,0,0);
    if (a!=CB_ERR) {
      a++;
      if (putobj(Obj,"poly_dimension",id,&a)==-1) return;
    }
    if (SetObjFieldFromText(Handle,IDFTWEIGHT,Obj,id,"weight_func")) return;
    if (SetObjFieldFromToggle(Handle,IDFTTHROUGH,Obj,id,"through_point")) return;
    if (SetObjFieldFromText(Handle,IDFTPX,Obj,id,"point_x")) return;
    if (SetObjFieldFromText(Handle,IDFTPY,Obj,id,"point_y")) return;
    if (SetObjFieldFromText(Handle,IDFTMIN,Obj,id,"min")) return;
    if (SetObjFieldFromText(Handle,IDFTMAX,Obj,id,"max")) return;
    if (SetObjFieldFromText(Handle,IDFTDIV,Obj,id,"div")) return;
    if (SetObjFieldFromToggle(Handle,IDFTINTERPOLATION,Obj,id,"interpolation")) return;
    if (SetObjFieldFromText(Handle,IDFTFN,Obj,id,"user_func")) return;
    if (SetObjFieldFromToggle(Handle,IDFTDERIV,Obj,id,"derivative")) return;
    if (SetObjFieldFromText(Handle,IDFTCONV,Obj,id,"converge")) return;
    if (SetObjFieldFromText(Handle,IDFTP0,Obj,id,"parameter0")) return;
    if (SetObjFieldFromText(Handle,IDFTP1,Obj,id,"parameter1")) return;
    if (SetObjFieldFromText(Handle,IDFTP2,Obj,id,"parameter2")) return;
    if (SetObjFieldFromText(Handle,IDFTP3,Obj,id,"parameter3")) return;
    if (SetObjFieldFromText(Handle,IDFTP4,Obj,id,"parameter4")) return;
    if (SetObjFieldFromText(Handle,IDFTP5,Obj,id,"parameter5")) return;
    if (SetObjFieldFromText(Handle,IDFTP6,Obj,id,"parameter6")) return;
    if (SetObjFieldFromText(Handle,IDFTP7,Obj,id,"parameter7")) return;
    if (SetObjFieldFromText(Handle,IDFTP8,Obj,id,"parameter8")) return;
    if (SetObjFieldFromText(Handle,IDFTP9,Obj,id,"parameter9")) return;
    if (SetObjFieldFromText(Handle,IDFTD0,Obj,id,"derivative0")) return;
    if (SetObjFieldFromText(Handle,IDFTD1,Obj,id,"derivative1")) return;
    if (SetObjFieldFromText(Handle,IDFTD2,Obj,id,"derivative2")) return;
    if (SetObjFieldFromText(Handle,IDFTD3,Obj,id,"derivative3")) return;
    if (SetObjFieldFromText(Handle,IDFTD4,Obj,id,"derivative4")) return;
    if (SetObjFieldFromText(Handle,IDFTD5,Obj,id,"derivative5")) return;
    if (SetObjFieldFromText(Handle,IDFTD6,Obj,id,"derivative6")) return;
    if (SetObjFieldFromText(Handle,IDFTD7,Obj,id,"derivative7")) return;
    if (SetObjFieldFromText(Handle,IDFTD8,Obj,id,"derivative8")) return;
    if (SetObjFieldFromText(Handle,IDFTD9,Obj,id,"derivative9")) return;

    if ((ngpfile=getscriptname(FITSAVE))==NULL) return;
    unchangefilename(ngpfile);
    error=FALSE;
    hFile=nopen(ngpfile,O_CREAT|O_TRUNC|O_RDWR,NFMODE);
    for (i=Lastid+1;i<=chkobjlastinst(Obj);i++) {
      getobj(Obj,"save",i,0,NULL,&s);
      len=strlen(s);
      if (len!=nwrite(hFile,s,len)) error=TRUE;
      if (nwrite(hFile,"\n",1)!=1) error=TRUE;
    }
    nclose(hFile);
    if (error) MessageBox("I/O error: Write","Error:",MB_ICONEXCLAMATION|MB_OK);
    memfree(ngpfile);
  } else delete dlg;
}

void FitDialog::Draw()
{

  if (!Apply()) return;
  SetupItem(TRUE,Id);
  NgraphApp->pTViewWindow->Draw(FALSE);
}


void FitDialog::ResultClicked()
{
  char buf[1024];
  int type,num;
  double derror,correlation;
  double coe[10];
  char *equation;
  int i,j,dim,dimension;
  int maxdim,need2pass;
  struct narray needarray;
  char *math,*code;
  int *needdata,tbl[10];
  char *inst;

  if ((inst=chkobjinst(Obj,Id))==NULL) return;
  if (_getobj(Obj,"type",inst,&type)) return;
  if (_getobj(Obj,"poly_dimension",inst,&dimension)) return;
  if (_getobj(Obj,"number",inst,&num)) return;
  if (_getobj(Obj,"error",inst,&derror)) return;
  if (_getobj(Obj,"correlation",inst,&correlation)) return;
  if (_getobj(Obj,"%00",inst,&(coe[0]))) return;
  if (_getobj(Obj,"%01",inst,&(coe[1]))) return;
  if (_getobj(Obj,"%02",inst,&(coe[2]))) return;
  if (_getobj(Obj,"%03",inst,&(coe[3]))) return;
  if (_getobj(Obj,"%04",inst,&(coe[4]))) return;
  if (_getobj(Obj,"%05",inst,&(coe[5]))) return;
  if (_getobj(Obj,"%06",inst,&(coe[6]))) return;
  if (_getobj(Obj,"%07",inst,&(coe[7]))) return;
  if (_getobj(Obj,"%08",inst,&(coe[8]))) return;
  if (_getobj(Obj,"%09",inst,&(coe[9]))) return;
  if (_getobj(Obj,"equation",inst,&equation)) return;
  if (_getobj(Obj,"user_func",inst,&math)) return;
  if (equation==NULL) {
    sprintf(buf,"Undefined");
  } else if (type!=4) {
    i=0;
    if (type==0) dim=dimension+1;
    else dim=2;
    if (type==0) i+=sprintf(buf+i,"Eq: %%0i*X^i (i=0-%d)\n\n",dim-1);
    else if (type==1) i+=sprintf(buf+i,"Eq: exp(%%00)*X^%%01\n\n");
    else if (type==2) i+=sprintf(buf+i,"Eq: exp(%%01*X+%%00)\n\n");
    else if (type==3) i+=sprintf(buf+i,"Eq: %%01*Ln(X)+%%00\n\n");
    for (j=0;j<dim;j++)
      i+=sprintf(buf+i,"       %%0%d = %.7e\n",j,coe[j]);
    i+=sprintf(buf+i,"\n");
    i+=sprintf(buf+i,"    points = %d\n",num);
    i+=sprintf(buf+i,"    <DY^2> = %.7e\n",derror);
    if (correlation>=0)
      i+=sprintf(buf+i,"|r| or |R| = %.7e\n",correlation);
    else
      i+=sprintf(buf+i,"|r| or |R| = -------------\n");
   } else {
    arrayinit(&needarray,sizeof(int));
    mathcode(math,&code,&needarray,NULL,&maxdim,&need2pass,
             TRUE,FALSE,FALSE,TRUE,FALSE,FALSE,FALSE,FALSE,FALSE,FALSE,FALSE);
    memfree(code);
    dim=arraynum(&needarray);
    needdata=(int *)arraydata(&needarray);
    for (i=0;i<dim;i++) tbl[i]=needdata[i];
    arraydel(&needarray);
    i=0;
    i+=sprintf(buf+i,"Eq: User defined\n\n");
    for (j=0;j<dim;j++)
      i+=sprintf(buf+i,"       %%0%d = %.7e\n",tbl[j],coe[tbl[j]]);
    i+=sprintf(buf+i,"\n");
    i+=sprintf(buf+i,"    points = %d\n",num);
    i+=sprintf(buf+i,"    <DY^2> = %.7e\n",derror);
    if (correlation>=0)
      i+=sprintf(buf+i,"|r| or |R| = %.7e\n",correlation);
    else
      i+=sprintf(buf+i,"|r| or |R| = -------------\n");
  }
  MessageBox(buf,"Fitting Results",
      MB_OK|MB_TASKMODAL|MB_ICONINFORMATION|MB_SETFOREGROUND);
}

void FitDialog::ClipClicked()
{
  char buf[1024];
  int type,num;
  double derror,correlation;
  double coe[10];
  char *equation;
  int i,j,dim,dimension;
  int maxdim,need2pass;
  struct narray needarray;
  char *math,*code;
  int *needdata,tbl[10];
  char *inst;
  HANDLE HD;
  char far *Clip;

  if ((inst=chkobjinst(Obj,Id))==NULL) return;
  if (_getobj(Obj,"type",inst,&type)) return;
  if (_getobj(Obj,"poly_dimension",inst,&dimension)) return;
  if (_getobj(Obj,"number",inst,&num)) return;
  if (_getobj(Obj,"error",inst,&derror)) return;
  if (_getobj(Obj,"correlation",inst,&correlation)) return;
  if (_getobj(Obj,"%00",inst,&(coe[0]))) return;
  if (_getobj(Obj,"%01",inst,&(coe[1]))) return;
  if (_getobj(Obj,"%02",inst,&(coe[2]))) return;
  if (_getobj(Obj,"%03",inst,&(coe[3]))) return;
  if (_getobj(Obj,"%04",inst,&(coe[4]))) return;
  if (_getobj(Obj,"%05",inst,&(coe[5]))) return;
  if (_getobj(Obj,"%06",inst,&(coe[6]))) return;
  if (_getobj(Obj,"%07",inst,&(coe[7]))) return;
  if (_getobj(Obj,"%08",inst,&(coe[8]))) return;
  if (_getobj(Obj,"%09",inst,&(coe[9]))) return;
  if (_getobj(Obj,"equation",inst,&equation)) return;
  if (_getobj(Obj,"user_func",inst,&math)) return;
  if (equation==NULL) {
    sprintf(buf,"Undefined");
  } else if (type!=4) {
    i=0;
    if (type==0) dim=dimension+1;
    else dim=2;
    if (type==0) i+=sprintf(buf+i,"Eq: %%0i*X^i (i=0-%d)\r\n\r\n",dim-1);
    else if (type==1) i+=sprintf(buf+i,"Eq: exp(%%00)*X^%%01\r\n\r\n");
    else if (type==2) i+=sprintf(buf+i,"Eq: exp(%%01*X+%%00)\r\n\r\n");
    else if (type==3) i+=sprintf(buf+i,"Eq: %%01*Ln(X)+%%00\r\n\r\n");
    for (j=0;j<dim;j++)
      i+=sprintf(buf+i,"       %%0%d = %.7e\r\n",j,coe[j]);
    i+=sprintf(buf+i,"\r\n");
    i+=sprintf(buf+i,"    points = %d\r\n",num);
    i+=sprintf(buf+i,"    <DY^2> = %.7e\r\n",derror);
    if (correlation>=0)
      i+=sprintf(buf+i,"|r| or |R| = %.7e\r\n",correlation);
    else
      i+=sprintf(buf+i,"|r| or |R| = -------------\r\n");
   } else {
    arrayinit(&needarray,sizeof(int));
    mathcode(math,&code,&needarray,NULL,&maxdim,&need2pass,
             TRUE,FALSE,FALSE,TRUE,FALSE,FALSE,FALSE,FALSE,FALSE,FALSE,FALSE);
    memfree(code);
    dim=arraynum(&needarray);
    needdata=(int *)arraydata(&needarray);
    for (i=0;i<dim;i++) tbl[i]=needdata[i];
    arraydel(&needarray);
    i=0;
    i+=sprintf(buf+i,"Eq: User defined\r\n\r\n");
    for (j=0;j<dim;j++)
      i+=sprintf(buf+i,"       %%0%d = %.7e\r\n",tbl[j],coe[tbl[j]]);
    i+=sprintf(buf+i,"\r\n");
    i+=sprintf(buf+i,"    points = %d\r\n",num);
    i+=sprintf(buf+i,"    <DY^2> = %.7e\r\n",derror);
    if (correlation>=0)
      i+=sprintf(buf+i,"|r| or |R| = %.7e\r\n",correlation);
    else
      i+=sprintf(buf+i,"|r| or |R| = -------------\r\n");
  }
  if ((HD=GlobalAlloc(GMEM_MOVEABLE,strlen(buf)+1))!=NULL) {
    Clip=(char *)GlobalLock(HD);
    strcpy(Clip,buf);
    GlobalUnlock(HD);
    ::OpenClipboard(Handle);
    ::EmptyClipboard();
    ::SetClipboardData(CF_TEXT,HD);
    ::CloseClipboard();
  }
}

int FitDialog::Apply()
{
  int a;

  if (SetObjFieldFromList(Handle,IDFTTYPE,Obj,Id,"type")) return FALSE;
  a=SendDlgItemMessage(IDFTPOLY,CB_GETCURSEL,0,0);
  if (a!=CB_ERR) {
    a++;
    if (putobj(Obj,"poly_dimension",Id,&a)==-1) return FALSE;
  }
  if (SetObjFieldFromText(Handle,IDFTWEIGHT,Obj,Id,"weight_func")) return FALSE;
  if (SetObjFieldFromToggle(Handle,IDFTTHROUGH,Obj,Id,"through_point")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTPX,Obj,Id,"point_x")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTPY,Obj,Id,"point_y")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTMIN,Obj,Id,"min")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTMAX,Obj,Id,"max")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTDIV,Obj,Id,"div")) return FALSE;
  if (SetObjFieldFromToggle(Handle,IDFTINTERPOLATION,Obj,Id,"interpolation")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTFN,Obj,Id,"user_func")) return FALSE;
  if (SetObjFieldFromToggle(Handle,IDFTDERIV,Obj,Id,"derivative")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTCONV,Obj,Id,"converge")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTP0,Obj,Id,"parameter0")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTP1,Obj,Id,"parameter1")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTP2,Obj,Id,"parameter2")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTP3,Obj,Id,"parameter3")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTP4,Obj,Id,"parameter4")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTP5,Obj,Id,"parameter5")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTP6,Obj,Id,"parameter6")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTP7,Obj,Id,"parameter7")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTP8,Obj,Id,"parameter8")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTP9,Obj,Id,"parameter9")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTD0,Obj,Id,"derivative0")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTD1,Obj,Id,"derivative1")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTD2,Obj,Id,"derivative2")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTD3,Obj,Id,"derivative3")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTD4,Obj,Id,"derivative4")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTD5,Obj,Id,"derivative5")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTD6,Obj,Id,"derivative6")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTD7,Obj,Id,"derivative7")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTD8,Obj,Id,"derivative8")) return FALSE;
  if (SetObjFieldFromText(Handle,IDFTD9,Obj,Id,"derivative9")) return FALSE;
  NgraphApp->Changed=TRUE;
  return TRUE;
}


void FitDialog::CloseWindow(int retVal)
{
  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  if (!Apply()) return;
  TMyDialog::CloseWindow(retVal);
}


DEFINE_RESPONSE_TABLE1(MarkComboBox,TComboBox)
  EV_WM_DRAWITEM,
END_RESPONSE_TABLE;

char fwbuf[100][10];

void MarkComboBox::EvDrawItem(UINT ctrlId,DRAWITEMSTRUCT far& draw)
{
  int x,y,h,dpi,GC;
  struct mwlocal mwsave;
  TEXTMETRIC tm;
  COLORREF clrBackground;
  COLORREF clrForeground;
  HDC DC;
  int R,G,B,R2,G2,B2;
  COLORREF col1,col2;

  TComboBox::EvDrawItem(ctrlId,draw);
  if (draw.itemState & ODS_DISABLED) {
    col1=GetSysColor(COLOR_GRAYTEXT);
    col2=GetSysColor(COLOR_MENU);
    clrForeground=::SetTextColor(draw.hDC,col1);
    clrBackground=::SetBkColor(draw.hDC,::GetSysColor(COLOR_MENU));
  } else {
    col1=::GetSysColor(draw.itemState & ODS_SELECTED ?
                       COLOR_HIGHLIGHTTEXT:COLOR_WINDOWTEXT);
    col2=::GetSysColor(draw.itemState & ODS_SELECTED ?
                       COLOR_HIGHLIGHT:COLOR_WINDOW);
    clrForeground=::SetTextColor(draw.hDC,col1);
    clrBackground=::SetBkColor(draw.hDC,col2);
    col1=::GetSysColor(COLOR_WINDOWTEXT);
    col2=::GetSysColor(COLOR_WINDOW);
  }
  R=(BYTE)(col1);
  G=(BYTE)(col1 >> 8);
  B=(BYTE)(col1 >> 16);
  R2=(BYTE)(~col2);
  G2=(BYTE)(col2 >> 8);
  B2=(BYTE)(col2 >> 16);
  DC=draw.hDC;
  GetTextMetrics(DC,&tm);
  y=(draw.rcItem.bottom+draw.rcItem.top-tm.tmHeight) / 2;
  x=LOWORD(GetDialogBaseUnits())/4;
  h=draw.rcItem.bottom-draw.rcItem.top;
  ExtTextOut(DC,2*x,y,
             ETO_CLIPPED | ETO_OPAQUE,&(draw.rcItem),
             fwbuf[draw.itemID], lstrlen(fwbuf[draw.itemID]), NULL);
  SetTextColor(DC,clrForeground);
  SetBkColor(DC,clrBackground);
  dpi=nround((draw.rcItem.bottom-draw.rcItem.top)*25.4);
  mwsaveDC(DC,Handle,0,0,&mwsave,dpi,FALSE,NULL);
  GC=_GRAopen(chkobjectname(menulocal.obj),"_output",menulocal.outputobj,
              menulocal.inst,menulocal.output,-1,-1,-1,NULL,mwlocal);
  if (GC>=0) {
    GRAview(GC,draw.rcItem.left/h*100+150,draw.rcItem.top/h*100,
              draw.rcItem.right/h*100+150,draw.rcItem.bottom/h*100,0);
    GRAlinestyle(GC,0,NULL,1,0,0,1000);
    GRAmark(GC,draw.itemID-1,60,55,80,R,G,B,R2,G2,B2);
  }
   _GRAclose(GC);
  mwrestoreDC(&mwsave);
  if (draw.itemState & ODS_FOCUS) DrawFocusRect(DC,&(draw.rcItem));
}

class FileMathDialog  : public TMyDialog {
  public:
    FileMathDialog(TWindow *parent,TResId resID,struct objlist *obj,int id);
    ~FileMathDialog();
  protected:
    struct objlist *Obj;
    int Id;
    void SetupWindow();
    void SetupItem(int id);
    void CopyClicked() {
      int sel;
      if ((sel=CopyClick(this,Obj,Id,FileCB))!=-1) SetupItem(sel);
    }
    void CloseWindow(int retVal);
    DECLARE_RESPONSE_TABLE(FileMathDialog);
};

DEFINE_RESPONSE_TABLE1(FileMathDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
END_RESPONSE_TABLE;


FileMathDialog::FileMathDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
 : TMyDialog(parent,resID)
{
  Obj=obj;
  Id=id;
}

FileMathDialog::~FileMathDialog()
{
}

void FileMathDialog::SetupWindow()
{
  TMyDialog::SetupWindow();
  SetupItem(Id);
}

void FileMathDialog::SetupItem(int id)
{
  int num;
  char **data;

  num=arraynum(menulocal.mathlist);
  data=(char **)arraydata(menulocal.mathlist);
  SetComboList(Handle,IDFMATHX,data,num);
  SetComboList(Handle,IDFMATHY,data,num);
  SetComboList(Handle,IDFFNF,data,num);
  SetComboList(Handle,IDFFNG,data,num);
  SetComboList(Handle,IDFFNH,data,num);
  SetTextFromObjField(Handle,IDFSMX,Obj,id,"smooth_x");
  SetTextFromObjField(Handle,IDFMATHX,Obj,id,"math_x");
  SetTextFromObjField(Handle,IDFSMY,Obj,id,"smooth_y");
  SetTextFromObjField(Handle,IDFMATHY,Obj,id,"math_y");
  SetTextFromObjField(Handle,IDFFNF,Obj,id,"func_f");
  SetTextFromObjField(Handle,IDFFNG,Obj,id,"func_g");
  SetTextFromObjField(Handle,IDFFNH,Obj,id,"func_h");
}

void FileMathDialog::CloseWindow(int retVal)
{
  int len;
  char *buf;

  if ((retVal!=IDOK) && (retVal!=IDFAPPLY)) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  len=SendDlgItemMessage(IDFMATHY,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)memalloc(len+1))!=NULL) {
    GetDlgItemText(IDFMATHY,buf,len+1);
    AddMathList(buf);
    memfree(buf);
  }
  len=SendDlgItemMessage(IDFMATHX,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)memalloc(len+1))!=NULL) {
    GetDlgItemText(IDFMATHX,buf,len+1);
    AddMathList(buf);
    memfree(buf);
  }
  if (SetObjFieldFromText(Handle,IDFSMX,Obj,Id,"smooth_x")) return;
  if (SetObjFieldFromText(Handle,IDFMATHX,Obj,Id,"math_x")) return;
  if (SetObjFieldFromText(Handle,IDFSMY,Obj,Id,"smooth_y")) return;
  if (SetObjFieldFromText(Handle,IDFMATHY,Obj,Id,"math_y")) return;
  if (SetObjFieldFromText(Handle,IDFFNF,Obj,Id,"func_f")) return;
  if (SetObjFieldFromText(Handle,IDFFNG,Obj,Id,"func_g")) return;
  if (SetObjFieldFromText(Handle,IDFFNH,Obj,Id,"func_h")) return;
  TMyDialog::CloseWindow(retVal);
}

class FileLoadDialog  : public TMyDialog {
  public:
    FileLoadDialog(TWindow *parent,TResId resID,struct objlist *obj,int id);
    ~FileLoadDialog();
  protected:
    struct objlist *Obj;
    int Id;
    void SetupWindow();
    void SetupItem(int id);
    void CopyClicked() {
      int sel;
      if ((sel=CopyClick(this,Obj,Id,FileCB))!=-1) SetupItem(sel);
    }
    void CloseWindow(int retVal);
    DECLARE_RESPONSE_TABLE(FileLoadDialog);
};

DEFINE_RESPONSE_TABLE1(FileLoadDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
END_RESPONSE_TABLE;


FileLoadDialog::FileLoadDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
 : TMyDialog(parent,resID)
{
  Obj=obj;
  Id=id;
}

FileLoadDialog::~FileLoadDialog()
{
}

void FileLoadDialog::SetupWindow()
{
  TMyDialog::SetupWindow();
  SetupItem(Id);
}

void FileLoadDialog::SetupItem(int id)
{
  char *ifs,*s;
  int i;

  SetTextFromObjField(Handle,IDFSKIP,Obj,id,"head_skip");
  SetTextFromObjField(Handle,IDFSTEP,Obj,id,"read_step");
  SetTextFromObjField(Handle,IDFFINAL,Obj,id,"final_line");
  SetTextFromObjField(Handle,IDFREMARK,Obj,id,"remark");
  sgetobjfield(Obj,id,"ifs",NULL,&ifs,FALSE,FALSE,FALSE);
  s=nstrnew();

  for (i=0;i<strlen(ifs);i++) {

    if (ifs[i]=='\t') {

      s=nstrccat(s,'\\');

      s=nstrccat(s,'t');

    } else if (ifs[i]=='\\') {

      s=nstrccat(s,'\\');

      s=nstrccat(s,'\\');

    } else s=nstrccat(s,ifs[i]);

  }

  SetDlgItemText(IDFIFS,s);

  memfree(s);

  memfree(ifs);

  SetToggleFromObjField(Handle,IDFCSV,Obj,id,"csv");
}

void FileLoadDialog::CloseWindow(int retVal)
{
  char *ifs,*s;
  int i,len;

  if ((retVal!=IDOK) && (retVal!=IDFAPPLY)) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  if (SetObjFieldFromText(Handle,IDFSKIP,Obj,Id,"head_skip")) return;
  if (SetObjFieldFromText(Handle,IDFSTEP,Obj,Id,"read_step")) return;
  if (SetObjFieldFromText(Handle,IDFFINAL,Obj,Id,"final_line")) return;
  if (SetObjFieldFromText(Handle,IDFREMARK,Obj,Id,"remark")) return;
  len=SendDlgItemMessage(IDFIFS,WM_GETTEXTLENGTH,0,0);
  if ((ifs=(char *)memalloc(len+1))!=NULL) {
    GetDlgItemText(IDFIFS,ifs,len+1);
    s=nstrnew();
    for (i=0;i<strlen(ifs);i++) {

      if ((ifs[i]=='\\') && (ifs[i+1]=='t')) {

        s=nstrccat(s,0x09);

        i++;

      } else if (ifs[i]=='\\') {

        s=nstrccat(s,'\\');

        i++;

      } else s=nstrccat(s,ifs[i]);

    }

    if (sputobjfield(Obj,Id,"ifs",s)!=0) {
      memfree(ifs);
      memfree(s);
      return;
    }
    memfree(ifs);
    memfree(s);
  }
  if (SetObjFieldFromToggle(Handle,IDFCSV,Obj,Id,"csv")) return;
  TMyDialog::CloseWindow(retVal);
}

DEFINE_RESPONSE_TABLE1(FileDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
  EV_CHILD_NOTIFY(IDDELETE,BN_CLICKED,DeleteClicked),
  EV_CHILD_NOTIFY(IDCOPYALL,BN_CLICKED,CopyAllClicked),
  EV_CHILD_NOTIFY(IDFMATHB,BN_CLICKED,MathClicked),
  EV_CHILD_NOTIFY(IDFLOADB,BN_CLICKED,LoadClicked),
  EV_CHILD_NOTIFY(IDFEDIT,BN_CLICKED,EditClicked),
  EV_CHILD_NOTIFY(IDFAPPLY,BN_CLICKED,ApplyClicked),
  EV_CHILD_NOTIFY(IDFFITB,BN_CLICKED,FitClicked),
  EV_CHILD_NOTIFY(IDFMASKB,BN_CLICKED,MaskClicked),
  EV_CHILD_NOTIFY(IDFMOVEB,BN_CLICKED,MoveClicked),
  EV_CHILD_NOTIFY(IDFOPTION,BN_CLICKED,OptionClicked),
  EV_CHILD_NOTIFY(IDFTYPE,LBN_SELCHANGE,TypeChanged),
END_RESPONSE_TABLE;


FileDialog::FileDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
 : TMyDialog(parent,resID)
{
  Mark=new MarkComboBox(this,IDFMARKD);
  Col1=new ColorButton(this,IDFCOL1D);
  Col2=new ColorButton(this,IDFCOL2D);
  Fit=new TStatic(this,IDFFIT,256);
  AxisX=new AxisComboBox(this,IDFAXISX);
  AxisY=new AxisComboBox(this,IDFAXISY);
  Obj=obj;
  Id=id;
}

FileDialog::~FileDialog()
{
  delete Fit;
  delete Col2;
  delete Col1;
  delete Mark;
  delete AxisX;
  delete AxisY;
}

void FileDialog::SetupWindow()
{
  char *title,ch;
  int i,j,len,line,rcode;
  char *argv[2];
  char *s;

  TMyDialog::SetupWindow();
  len=GetWindowTextLength();
  if ((title=(char *)memalloc(len+10))!=NULL) {
    GetWindowText(title,len+1);
    sprintf(title+len," %d",Id);
    SetCaption(title);
    memfree(title);
  }
  SetupItem(TRUE,Id);
  SendDlgItemMessage(IDFCOMMENT,LB_RESETCONTENT,0,0);
  line=10;
  argv[0]=(char *)&line;
  argv[1]=NULL;
  rcode=getobj(Obj,"head_lines",Id,1,argv,&s);
  if ((rcode>=0) && (s!=NULL)) {
    i=0;
    ch='\n';
    while (ch!='\0') {
      for (j=i;(s[j]!='\n') && (s[j]!='\0');j++);
      ch=s[j];
      s[j]='\0';
      SendDlgItemMessage(IDFCOMMENT,LB_INSERTSTRING,-1,(LPARAM)(s+i));
      i=j+1;
    }
  }
  len=GetDialogBaseUnits()*128;
  SendDlgItemMessage(IDFCOMMENT,LB_SETHORIZONTALEXTENT,(WPARAM)len,0);
}

void FileDialog::CopyAllClicked()
{
  int sel;
  int j,perm,type;
  char *field;

  if ((sel=CopyClick(this,Obj,Id,FileCB))!=-1) {
    for (j=0;j<chkobjfieldnum(Obj);j++) {
      field=chkobjfieldname(Obj,j);
      perm=chkobjperm(Obj,field);
      type=chkobjfieldtype(Obj,field);
      if ((strcmp2(field,"name")!=0) && (strcmp2(field,"file")!=0)
      && (strcmp2(field,"fit")!=0)
      && ((perm&NREAD)!=0) && ((perm&NWRITE)!=0) && (type<NVFUNC))
        copyobj(Obj,field,Id,sel);
    }
    FitCopy(Obj,Id,sel);
    SetupItem(FALSE,Id);
    NgraphApp->Changed=TRUE;
  }
}

void FileDialog::DeleteClicked()
{
  CloseWindow(IDDELETE);
}

void FileDialog::ApplyClicked()
{
  CloseWindow(IDFAPPLY);
}

void FileDialog::SetupItem(int file,int id)
{
  char *valstr;
  int i,j,a,R,G,B;
  int lastinst;
  char buf[10];
  struct objlist *aobj;

  if (file) {
	SetTextFromObjField(Handle,IDFNAME,Obj,id,"file");
  }
  aobj=getobject("axis");
  lastinst=chkobjlastinst(aobj);
  SendDlgItemMessage(IDFAXISX,CB_RESETCONTENT,0,0);
  SendDlgItemMessage(IDFAXISY,CB_RESETCONTENT,0,0);
  for (j=0;j<=lastinst;j++) {
    sprintf(buf,"%d",j);
    SendDlgItemMessage(IDFAXISX,CB_INSERTSTRING,j,(LPARAM)buf);
    SendDlgItemMessage(IDFAXISY,CB_INSERTSTRING,j,(LPARAM)buf);
  }
  SetTextFromObjField(Handle,IDFCOLX,Obj,id,"x");
  sgetobjfield(Obj,id,"axis_x",NULL,&valstr,FALSE,FALSE,FALSE);
  for (i=0;(valstr[i]!='\0') && (valstr[i]!=':');i++);
  if (valstr[i]==':') i++;
  SetDlgItemText(IDFAXISX,valstr+i); memfree(valstr);
  SetTextFromObjField(Handle,IDFCOLY,Obj,id,"y");
  sgetobjfield(Obj,id,"axis_y",NULL,&valstr,FALSE,FALSE,FALSE);
  for (i=0;(valstr[i]!='\0') && (valstr[i]!=':');i++);
  if (valstr[i]==':') i++;
  SetDlgItemText(IDFAXISY,valstr+i); memfree(valstr);
  SetListFromObjField(Handle,IDFTYPE,Obj,id,"type");
  SetListFromObjField(Handle,IDFINTP,Obj,id,"interpolation");
  if (id==Id) {
	sgetobjfield(Obj,id,"fit",NULL,&valstr,FALSE,FALSE,FALSE);
	for (i=0;(valstr[i]!='\0') && (valstr[i]!=':');i++);
	if (valstr[i]==':') i++;
	Fit->SetText(valstr+i); memfree(valstr);
  }
  SendDlgItemMessage(IDFMARKD,CB_RESETCONTENT,0,0);
  for (j=-1;j<89;j++) {
    if (j==-1) {
      fwbuf[j+1][0]='-';
      fwbuf[j+1][1]='1';
    } else {
      fwbuf[j+1][0]=j/10+'0';
      fwbuf[j+1][1]=j%10+'0';
    }
    fwbuf[j+1][2]='\0';
    SendDlgItemMessage(IDFMARKD,CB_INSERTSTRING,j+1,(LPARAM)(fwbuf[j+1]));
  }
  getobj(Obj,"mark_type",id,0,NULL,&a);
  SendDlgItemMessage(IDFMARKD,CB_SETCURSEL,a+1,0);
  SetComboList(Handle,IDFSIZE,cbmarksize,CBMARKSIZE);
  SetTextFromObjField(Handle,IDFSIZE,Obj,id,"mark_size");
  SetComboList(Handle,IDFWIDTH,cblinewidth,CBLINEWIDTH);
  SetTextFromObjField(Handle,IDFWIDTH,Obj,id,"line_width");
  SetStyleFromObjField(Handle,IDFSTYLE,Obj,id,"line_style");
  SetListFromObjField(Handle,IDFJOIN,Obj,id,"line_join");
  SetTextFromObjField(Handle,IDFMITER,Obj,id,"line_miter_limit");
  SetToggleFromObjField(Handle,IDFCLIP,Obj,id,"data_clip");
  getobj(Obj,"R",id,0,NULL,&R);
  getobj(Obj,"G",id,0,NULL,&G);
  getobj(Obj,"B",id,0,NULL,&B);
  Col1->SetColor(R,G,B);
  SendDlgItemMessage(IDFCOL1D,BM_SETSTYLE,BS_OWNERDRAW,1);
  getobj(Obj,"R2",id,0,NULL,&R);
  getobj(Obj,"G2",id,0,NULL,&G);
  getobj(Obj,"B2",id,0,NULL,&B);
  Col2->SetColor(R,G,B);
  SendDlgItemMessage(IDFCOL2D,BM_SETSTYLE,BS_OWNERDRAW,1);
  TypeChanged();
}

void FileDialog::EditClicked()
{
  int len,elen;
  char *buf;
  STARTUPINFO si;
  PROCESS_INFORMATION pi;
  HCURSOR orgcursor;

  if (menulocal.editor==NULL) return;
  elen=strlen(menulocal.editor);
  len=SendDlgItemMessage(IDFNAME,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)memalloc(len+elen+4))==NULL) return;
  strcpy(buf,menulocal.editor);
  unchangefilename(buf);
  buf[elen]=' ';
  buf[elen+1]='"';
  if (!GetDlgItemText(IDFNAME,buf+elen+2,len+1)) {
    memfree(buf);
    return;
  }
  unchangefilename(buf+elen+2);
  buf[elen+len+2]='"';
  buf[elen+len+3]='\0';
  si.cb=sizeof(STARTUPINFO);
  si.lpReserved=NULL;
  si.lpDesktop=NULL;
  si.lpTitle=NULL;
  si.dwFlags=0;
  si.wShowWindow=SW_SHOWNORMAL;
  si.cbReserved2=0;
  si.lpReserved2=NULL;
  orgcursor=::SetCursor(LoadCursor(NULL,IDC_APPSTARTING));
  if (CreateProcess(NULL,buf,NULL,NULL,FALSE,NULL,NULL,NULL,&si,&pi)) {
    CloseHandle(pi.hProcess);
    CloseHandle(pi.hThread);
  }
  ::SetCursor(orgcursor);
  memfree(buf);
}

void FileDialog::MathClicked()
{
  FileMathDialog *dlg;

  dlg=new FileMathDialog(this,DIALOG_FILEMATH,Obj,Id);
  dlg->Execute();
  delete dlg;
}

void FileDialog::LoadClicked()
{
  FileLoadDialog *dlg;

  dlg=new FileLoadDialog(this,DIALOG_FILELOAD,Obj,Id);
  dlg->Execute();
  delete dlg;
}

void FileDialog::FitClicked()
{
  FitDialog *dlg;
  struct objlist *fitobj,*obj;
  char *fit,*inst;
  int i,idnum,fitid,fitoid,ret;
  struct narray iarray;
  char *valstr;

  if ((fitobj=chkobject("fit"))==NULL) return;
  if (getobj(Obj,"fit",Id,0,NULL,&fit)==-1) return;
  if (fit!=NULL) {
    arrayinit(&iarray,sizeof(int));
    if (getobjilist(fit,&obj,&iarray,FALSE,NULL)) return;
    idnum=arraynum(&iarray);
    if ((obj!=fitobj) || (idnum<1)) {
      if (putobj(Obj,"fit",Id,NULL)==-1) {
        arraydel(&iarray);
        return;
      }
    } else fitid=*(int *)arraylast(&iarray);
    arraydel(&iarray);
  }
  if (fit==NULL) {
    fitid=newobj(fitobj);
	inst=getobjinst(fitobj,fitid);
	_getobj(fitobj,"oid",inst,&fitoid);
	if ((fit=mkobjlist(fitobj,NULL,fitoid,NULL,TRUE))==NULL) return;
	if (putobj(Obj,"fit",Id,fit)==-1) {
	  memfree(fit);
	  return;
	}
  }
  dlg=new FitDialog(this,DIALOG_FIT,fitobj,fitid);
  ret=dlg->Execute();
  if (ret==IDDELETE) {
    delobj(fitobj,fitid);
    putobj(Obj,"fit",Id,NULL);
  }
  sgetobjfield(Obj,Id,"fit",NULL,&valstr,FALSE,FALSE,FALSE);
  for (i=0;(valstr[i]!='\0') && (valstr[i]!=':');i++);
  if (valstr[i]==':') i++;
  Fit->SetText(valstr+i); memfree(valstr);
  delete dlg;
  SendDlgItemMessage(IDFTYPE,CB_SETCURSEL,19,0);
}

void FileDialog::OptionClicked()
{
  exeobj(Obj,"load_settings",Id,0,NULL);
  SetupItem(FALSE,Id);
}


void FileDialog::MaskClicked()
{
  MaskDialog *dlg;

  dlg=new MaskDialog(this,DIALOG_MASK,Obj,Id);
  dlg->Execute();
  delete dlg;
}

void FileDialog::MoveClicked()
{
  MoveDialog *dlg;

  dlg=new MoveDialog(this,DIALOG_MOVE,Obj,Id);
  dlg->Execute();
  delete dlg;
}


void FileDialog::CloseWindow(int retVal)
{
  int a,len;
  char *buf;
  int R,G,B;

  if ((retVal!=IDOK) && (retVal!=IDFAPPLY)) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  if (SetObjFieldFromText(Handle,IDFNAME,Obj,Id,"file")) return;
  if (SetObjFieldFromText(Handle,IDFCOLX,Obj,Id,"x")) return;
  len=SendDlgItemMessage(IDFAXISX,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)memalloc(len+6))!=NULL) {
    strcpy(buf,"axis:");
    GetDlgItemText(IDFAXISX,buf+5,len+1);
    if (sputobjfield(Obj,Id,"axis_x",buf)!=0) {
      memfree(buf);
      return;
    }
    memfree(buf);
  }
  if (SetObjFieldFromText(Handle,IDFCOLY,Obj,Id,"y")) return;
  len=SendDlgItemMessage(IDFAXISY,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)memalloc(len+6))!=NULL) {
    strcpy(buf,"axis:");
    GetDlgItemText(IDFAXISY,buf+5,len+1);
    if (sputobjfield(Obj,Id,"axis_y",buf)!=0) {
      memfree(buf);
      return;
    }
    memfree(buf);
  }
  if (SetObjFieldFromList(Handle,IDFTYPE,Obj,Id,"type")) return;
  if (SetObjFieldFromList(Handle,IDFINTP,Obj,Id,"interpolation")) return;
  a=SendDlgItemMessage(IDFMARKD,CB_GETCURSEL,0,0);
  if (a!=CB_ERR) {
    a--;
    if (putobj(Obj,"mark_type",Id,&a)==-1) return;
  }
  if (SetObjFieldFromText(Handle,IDFSIZE,Obj,Id,"mark_size")) return;
  if (SetObjFieldFromText(Handle,IDFWIDTH,Obj,Id,"line_width")) return;
  if (SetObjFieldFromStyle(Handle,IDFSTYLE,Obj,Id,"line_style")) return;
  if (SetObjFieldFromList(Handle,IDFJOIN,Obj,Id,"line_join")) return;
  if (SetObjFieldFromText(Handle,IDFMITER,Obj,Id,"line_miter_limit")) return;
  if (SetObjFieldFromToggle(Handle,IDFCLIP,Obj,Id,"data_clip")) return;
  Col1->GetColor(&R,&G,&B);
  if (putobj(Obj,"R",Id,&R)==-1) return;
  if (putobj(Obj,"G",Id,&G)==-1) return;
  if (putobj(Obj,"B",Id,&B)==-1) return;
  Col2->GetColor(&R,&G,&B);
  if (putobj(Obj,"R2",Id,&R)==-1) return;
  if (putobj(Obj,"G2",Id,&G)==-1) return;
  if (putobj(Obj,"B2",Id,&B)==-1) return;
  TMyDialog::CloseWindow(retVal);
}

void FileDialog::TypeChanged()
{
  int type;

  ::EnableWindow(GetDlgItem(IDFMARKD),TRUE);
  ::EnableWindow(GetDlgItem(IDFINTP),TRUE);
  ::EnableWindow(GetDlgItem(IDFCOL1D),TRUE);
  ::EnableWindow(GetDlgItem(IDFCOL2D),TRUE);
  ::EnableWindow(GetDlgItem(IDFSTYLE),TRUE);
  ::EnableWindow(GetDlgItem(IDFWIDTH),TRUE);
  ::EnableWindow(GetDlgItem(IDFSIZE),TRUE);
  ::EnableWindow(GetDlgItem(IDFMITER),TRUE);
  ::EnableWindow(GetDlgItem(IDFJOIN),TRUE);
  type=SendDlgItemMessage(IDFTYPE,CB_GETCURSEL,0,0);
  switch (type) {
  case 0:
    ::EnableWindow(GetDlgItem(IDFINTP),FALSE);
    ::EnableWindow(GetDlgItem(IDFMITER),FALSE);
    ::EnableWindow(GetDlgItem(IDFJOIN),FALSE);
    break;
  case 1:
  case 2:
    ::EnableWindow(GetDlgItem(IDFMARKD),FALSE);
    ::EnableWindow(GetDlgItem(IDFINTP),FALSE);
    ::EnableWindow(GetDlgItem(IDFCOL2D),FALSE);
    ::EnableWindow(GetDlgItem(IDFSIZE),FALSE);
    break;
  case 3:
    ::EnableWindow(GetDlgItem(IDFMARKD),FALSE);
    ::EnableWindow(GetDlgItem(IDFCOL2D),FALSE);
    ::EnableWindow(GetDlgItem(IDFSIZE),FALSE);
    break;
  case 4:
  case 6:
  case 8:
    ::EnableWindow(GetDlgItem(IDFMARKD),FALSE);
    ::EnableWindow(GetDlgItem(IDFINTP),FALSE);
    ::EnableWindow(GetDlgItem(IDFCOL2D),FALSE);
    ::EnableWindow(GetDlgItem(IDFSIZE),FALSE);
    ::EnableWindow(GetDlgItem(IDFMITER),FALSE);
    ::EnableWindow(GetDlgItem(IDFJOIN),FALSE);
    break;
  case 5:
    ::EnableWindow(GetDlgItem(IDFMARKD),FALSE);
    ::EnableWindow(GetDlgItem(IDFINTP),FALSE);
    ::EnableWindow(GetDlgItem(IDFCOL2D),FALSE);
    ::EnableWindow(GetDlgItem(IDFMITER),FALSE);
    ::EnableWindow(GetDlgItem(IDFJOIN),FALSE);
    break;
  case 7:
    ::EnableWindow(GetDlgItem(IDFMARKD),FALSE);
    ::EnableWindow(GetDlgItem(IDFINTP),FALSE);
    ::EnableWindow(GetDlgItem(IDFSIZE),FALSE);
    ::EnableWindow(GetDlgItem(IDFMITER),FALSE);
    ::EnableWindow(GetDlgItem(IDFJOIN),FALSE);
    break;
  case 9:
  case 10:
    ::EnableWindow(GetDlgItem(IDFMARKD),FALSE);
    ::EnableWindow(GetDlgItem(IDFINTP),FALSE);
    ::EnableWindow(GetDlgItem(IDFCOL2D),FALSE);
    ::EnableWindow(GetDlgItem(IDFMITER),FALSE);
    ::EnableWindow(GetDlgItem(IDFJOIN),FALSE);
    break;
  case 11:
  case 12:
    ::EnableWindow(GetDlgItem(IDFMARKD),FALSE);
    ::EnableWindow(GetDlgItem(IDFINTP),FALSE);
    ::EnableWindow(GetDlgItem(IDFCOL2D),FALSE);
    ::EnableWindow(GetDlgItem(IDFSIZE),FALSE);
    break;
  case 13:
  case 14:
  case 17:
  case 18:
    ::EnableWindow(GetDlgItem(IDFMARKD),FALSE);
    ::EnableWindow(GetDlgItem(IDFINTP),FALSE);
    ::EnableWindow(GetDlgItem(IDFCOL2D),FALSE);
    ::EnableWindow(GetDlgItem(IDFMITER),FALSE);
    ::EnableWindow(GetDlgItem(IDFJOIN),FALSE);
    break;
  case 15:
  case 16:
    ::EnableWindow(GetDlgItem(IDFMARKD),FALSE);
    ::EnableWindow(GetDlgItem(IDFINTP),FALSE);
    ::EnableWindow(GetDlgItem(IDFMITER),FALSE);
    ::EnableWindow(GetDlgItem(IDFJOIN),FALSE);
    break;
  case 19:
    ::EnableWindow(GetDlgItem(IDFMARKD),FALSE);
    ::EnableWindow(GetDlgItem(IDFINTP),FALSE);
    ::EnableWindow(GetDlgItem(IDFCOL2D),FALSE);
    ::EnableWindow(GetDlgItem(IDFSIZE),FALSE);
    break;
  }
}

DEFINE_RESPONSE_TABLE1(FileDefDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDFMATHB,BN_CLICKED,MathClicked),
  EV_CHILD_NOTIFY(IDFLOADB,BN_CLICKED,LoadClicked),
END_RESPONSE_TABLE;

FileDefDialog::FileDefDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
 : TMyDialog(parent,resID)
{
  Mark=new MarkComboBox(this,IDFMARKD);
  Col1=new ColorButton(this,IDFCOL1D);
  Col2=new ColorButton(this,IDFCOL2D);
  AxisX=new AxisComboBox(this,IDFAXISX);
  AxisY=new AxisComboBox(this,IDFAXISY);
  Obj=obj;
  Id=id;
}

FileDefDialog::~FileDefDialog()
{
  delete Col2;
  delete Col1;
  delete Mark;
  delete AxisX;
  delete AxisY;
}

void FileDefDialog::SetupWindow()
{
  TMyDialog::SetupWindow();
  SetupItem(Id);
}

void FileDefDialog::MathClicked()
{
  FileMathDialog *dlg;

  dlg=new FileMathDialog(this,DIALOG_FILEMATH,Obj,Id);
  dlg->Execute();
  delete dlg;
}

void FileDefDialog::LoadClicked()
{
  FileLoadDialog *dlg;

  dlg=new FileLoadDialog(this,DIALOG_FILELOAD,Obj,Id);
  dlg->Execute();
  delete dlg;
}

void FileDefDialog::SetupItem(int id)
{
  char *valstr;
  int i,j,a,R,G,B;
  int lastinst;
  char buf[10];
  struct objlist *aobj;

  aobj=getobject("axis");
  lastinst=chkobjlastinst(aobj);
  SendDlgItemMessage(IDFAXISX,CB_RESETCONTENT,0,0);
  SendDlgItemMessage(IDFAXISY,CB_RESETCONTENT,0,0);
  for (j=0;j<=lastinst;j++) {
    sprintf(buf,"%d",j);
    SendDlgItemMessage(IDFAXISX,CB_INSERTSTRING,j,(LPARAM)buf);
    SendDlgItemMessage(IDFAXISY,CB_INSERTSTRING,j,(LPARAM)buf);
  }
  SetTextFromObjField(Handle,IDFCOLX,Obj,id,"x");
  sgetobjfield(Obj,id,"axis_x",NULL,&valstr,FALSE,FALSE,FALSE);
  for (i=0;(valstr[i]!='\0') && (valstr[i]!=':');i++);
  if (valstr[i]==':') i++;
  SetDlgItemText(IDFAXISX,valstr+i); memfree(valstr);
  SetTextFromObjField(Handle,IDFCOLY,Obj,id,"y");
  sgetobjfield(Obj,id,"axis_y",NULL,&valstr,FALSE,FALSE,FALSE);
  for (i=0;(valstr[i]!='\0') && (valstr[i]!=':');i++);
  if (valstr[i]==':') i++;
  SetDlgItemText(IDFAXISY,valstr+i); memfree(valstr);
  SetListFromObjField(Handle,IDFTYPE,Obj,id,"type");
  SetListFromObjField(Handle,IDFINTP,Obj,id,"interpolation");
  SendDlgItemMessage(IDFMARKD,CB_RESETCONTENT,0,0);
  for (j=-1;j<89;j++) {
    if (j==-1) {
      fwbuf[j+1][0]='-';
      fwbuf[j+1][1]='1';
    } else {
      fwbuf[j+1][0]=j/10+'0';
      fwbuf[j+1][1]=j%10+'0';
    }
    fwbuf[j+1][2]='\0';
    SendDlgItemMessage(IDFMARKD,CB_INSERTSTRING,j+1,(LPARAM)(fwbuf[j+1]));
  }
  getobj(Obj,"mark_type",id,0,NULL,&a);
  SendDlgItemMessage(IDFMARKD,CB_SETCURSEL,a+1,0);
  SetComboList(Handle,IDFSIZE,cbmarksize,CBMARKSIZE);
  SetTextFromObjField(Handle,IDFSIZE,Obj,id,"mark_size");
  SetComboList(Handle,IDFWIDTH,cblinewidth,CBLINEWIDTH);
  SetTextFromObjField(Handle,IDFWIDTH,Obj,id,"line_width");
  SetStyleFromObjField(Handle,IDFSTYLE,Obj,id,"line_style");
  SetListFromObjField(Handle,IDFJOIN,Obj,id,"line_join");
  SetTextFromObjField(Handle,IDFMITER,Obj,id,"line_miter_limit");
  SetToggleFromObjField(Handle,IDFCLIP,Obj,id,"data_clip");
  getobj(Obj,"R",id,0,NULL,&R);
  getobj(Obj,"G",id,0,NULL,&G);
  getobj(Obj,"B",id,0,NULL,&B);
  Col1->SetColor(R,G,B);
  SendDlgItemMessage(IDFCOL1D,BM_SETSTYLE,BS_OWNERDRAW,1);
  getobj(Obj,"R2",id,0,NULL,&R);
  getobj(Obj,"G2",id,0,NULL,&G);
  getobj(Obj,"B2",id,0,NULL,&B);
  Col2->SetColor(R,G,B);
  SendDlgItemMessage(IDFCOL2D,BM_SETSTYLE,BS_OWNERDRAW,1);
}

void FileDefDialog::CloseWindow(int retVal)
{
  int len,a;
  char *buf;
  int R,G,B;

  if ((retVal!=IDOK) && (retVal!=IDFAPPLY)) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  if (SetObjFieldFromText(Handle,IDFCOLX,Obj,Id,"x")) return;
  len=SendDlgItemMessage(IDFAXISX,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)memalloc(len+6))!=NULL) {
    strcpy(buf,"axis:");
    GetDlgItemText(IDFAXISX,buf+5,len+1);
    if (sputobjfield(Obj,Id,"axis_x",buf)!=0) {
      memfree(buf);
      return;
    }
    memfree(buf);
  }
  if (SetObjFieldFromText(Handle,IDFCOLY,Obj,Id,"y")) return;
  len=SendDlgItemMessage(IDFAXISY,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)memalloc(len+6))!=NULL) {
    strcpy(buf,"axis:");
    GetDlgItemText(IDFAXISY,buf+5,len+1);
    if (sputobjfield(Obj,Id,"axis_y",buf)!=0) {
      memfree(buf);
      return;
    }
    memfree(buf);
  }
  if (SetObjFieldFromList(Handle,IDFTYPE,Obj,Id,"type")) return;
  if (SetObjFieldFromList(Handle,IDFINTP,Obj,Id,"interpolation")) return;
  a=SendDlgItemMessage(IDFMARKD,CB_GETCURSEL,0,0);
  if (a!=CB_ERR) {
    a--;
    if (putobj(Obj,"mark_type",Id,&a)==-1) return;
  }
  if (SetObjFieldFromText(Handle,IDFSIZE,Obj,Id,"mark_size")) return;
  if (SetObjFieldFromText(Handle,IDFWIDTH,Obj,Id,"line_width")) return;
  if (SetObjFieldFromStyle(Handle,IDFSTYLE,Obj,Id,"line_style")) return;
  if (SetObjFieldFromList(Handle,IDFJOIN,Obj,Id,"line_join")) return;
  if (SetObjFieldFromText(Handle,IDFMITER,Obj,Id,"line_miter_limit")) return;
  if (SetObjFieldFromToggle(Handle,IDFCLIP,Obj,Id,"data_clip")) return;
  Col1->GetColor(&R,&G,&B);
  if (putobj(Obj,"R",Id,&R)==-1) return;
  if (putobj(Obj,"G",Id,&G)==-1) return;
  if (putobj(Obj,"B",Id,&B)==-1) return;
  Col2->GetColor(&R,&G,&B);
  if (putobj(Obj,"R2",Id,&R)==-1) return;
  if (putobj(Obj,"G2",Id,&G)==-1) return;
  if (putobj(Obj,"B2",Id,&B)==-1) return;
  TMyDialog::CloseWindow(retVal);
}

class MathTextDialog  : public TMyDialog {
  public:
    MathTextDialog(TWindow *parent,TResId resID,char *text);
    ~MathTextDialog();
    char *math;
  protected:
    char *Text;
    void SetupWindow();
    void CloseWindow(int retVal);
    DECLARE_RESPONSE_TABLE(MathTextDialog);
};

DEFINE_RESPONSE_TABLE1(MathTextDialog,TMyDialog)
END_RESPONSE_TABLE;

MathTextDialog::MathTextDialog(TWindow *parent,TResId resID,char *text)
 : TMyDialog(parent,resID)
{
  Text=text;
  math=NULL;
}

MathTextDialog::~MathTextDialog()
{
  memfree(math);
}

void MathTextDialog::SetupWindow()
{
  char **data;
  int num;

  TMyDialog::SetupWindow();
  num=arraynum(menulocal.mathlist);
  data=(char **)arraydata(menulocal.mathlist);
  SetComboList(Handle,IDFMT,data,num);
  SetDlgItemText(IDFMT,Text);
}

void MathTextDialog::CloseWindow(int retVal)
{
  int len;

  if (retVal==IDOK) {
    len=SendDlgItemMessage(IDFMT,WM_GETTEXTLENGTH,0,0);
    if ((math=(char *)memalloc(len+1))!=NULL) {

      GetDlgItemText(IDFMT,math,len+1);

    }

  }

  TMyDialog::CloseWindow(retVal);
}


class MathDialog  : public TMyDialog {
  public:
    MathDialog(TWindow *parent,TResId resID,struct objlist *obj);
    ~MathDialog();
  protected:
    struct objlist *Obj;
    int Mode;
    void SetupWindow() {
      TMyDialog::SetupWindow();
      SetupItem();
    }
    void SetupItem();
    void AllClicked();
    void DblClicked();
    void XMathClicked();
    DECLARE_RESPONSE_TABLE(MathDialog);
};


DEFINE_RESPONSE_TABLE1(MathDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDFMALL,BN_CLICKED,AllClicked),
  EV_CHILD_NOTIFY(IDFMLIST,LBN_DBLCLK,DblClicked),
  EV_CHILD_NOTIFY(IDFMXMATH,BN_CLICKED,XMathClicked),
  EV_CHILD_NOTIFY(IDFMYMATH,BN_CLICKED,XMathClicked),
  EV_CHILD_NOTIFY(IDFMF,BN_CLICKED,XMathClicked),
  EV_CHILD_NOTIFY(IDFMG,BN_CLICKED,XMathClicked),
  EV_CHILD_NOTIFY(IDFMH,BN_CLICKED,XMathClicked),
END_RESPONSE_TABLE;


MathDialog::MathDialog(TWindow *parent,TResId resID,struct objlist *obj)
 : TMyDialog(parent,resID)
{
  Obj=obj;
  Mode=0;
}

MathDialog::~MathDialog()
{
}

void MathDialog::SetupItem()
{
  int i,len;
  char *s,*math;
  char *field;

  SendDlgItemMessage(IDFMLIST,LB_RESETCONTENT,0,0);
  if (Mode==0) field="math_x";
  else if (Mode==1) field="math_y";
  else if (Mode==2) field="func_f";
  else if (Mode==3) field="func_g";
  else if (Mode==4) field="func_h";
  for (i=0;i<=chkobjlastinst(Obj);i++) {
    math=NULL;
    getobj(Obj,field,i,0,NULL,&math);
    if (math==NULL) len=0;
    else len=strlen(math);
    if ((s=(char *)memalloc(len+10))!=NULL) {
      if (math==NULL) sprintf(s,"%d:",i);
      else sprintf(s,"%d: %s",i,math);
      SendDlgItemMessage(IDFMLIST,LB_INSERTSTRING,-1,(LPARAM)s);
      memfree(s);
    } else SendDlgItemMessage(IDFMLIST,LB_INSERTSTRING,-1,(LPARAM)"");
  }
  SendDlgItemMessage(IDFMXMATH,BM_SETCHECK,0,0);
  SendDlgItemMessage(IDFMYMATH,BM_SETCHECK,0,0);
  SendDlgItemMessage(IDFMF,BM_SETCHECK,0,0);
  SendDlgItemMessage(IDFMG,BM_SETCHECK,0,0);
  SendDlgItemMessage(IDFMH,BM_SETCHECK,0,0);
  if (Mode==1) SendDlgItemMessage(IDFMYMATH,BM_SETCHECK,1,0);
  else if (Mode==2) SendDlgItemMessage(IDFMF,BM_SETCHECK,1,0);
  else if (Mode==3) SendDlgItemMessage(IDFMG,BM_SETCHECK,1,0);
  else if (Mode==4) SendDlgItemMessage(IDFMH,BM_SETCHECK,1,0);
  else SendDlgItemMessage(IDFMXMATH,BM_SETCHECK,1,0);
}

void MathDialog::DblClicked()
{
  int a,i,num;
  char *buf;
  char *field;
  MathTextDialog *dlg;
  unsigned int *seldata;
  unsigned int caret;

  if (Mode==0) field="math_x";
  else if (Mode==1) field="math_y";
  else if (Mode==2) field="func_f";
  else if (Mode==3) field="func_g";
  else if (Mode==4) field="func_h";
  if ((a=SendDlgItemMessage(IDFMLIST,LB_GETCURSEL,0,0))!=LB_ERR) {
    sgetobjfield(Obj,a,field,NULL,&buf,FALSE,FALSE,FALSE);
    dlg=new MathTextDialog(this,DIALOG_MATHTEXT,buf);
    if (dlg->Execute()==IDOK) {
      for (i=0;i<=chkobjlastinst(Obj);i++)
        if (SendDlgItemMessage(IDFMLIST,LB_GETSEL,i,0)>0) {
          sputobjfield(Obj,i,field,dlg->math);
          AddMathList(buf);
          NgraphApp->Changed=TRUE;
        }
    }
    memfree(buf);
    delete dlg;
  }
  num=SendDlgItemMessage(IDFMLIST,LB_GETSELCOUNT,0,0);
  caret=SendDlgItemMessage(IDFMLIST,LB_GETCARETINDEX,0,0);
  if (num<0) num=0;
  if ((seldata=(unsigned int *)memalloc(sizeof(int)*num))!=NULL) {
    SendDlgItemMessage(IDFMLIST,LB_GETSELITEMS,num,(LPARAM)seldata);
  }
  SetupItem();
  if (seldata!=NULL) {
    for (i=0;i<num;i++) SendDlgItemMessage(IDFMLIST,LB_SETSEL,TRUE,seldata[i]);
    memfree(seldata);
  }
  SendDlgItemMessage(IDFMLIST,LB_SETCARETINDEX,caret,FALSE);
}

void MathDialog::AllClicked()
{
  int i;
  char *field;

  if (Mode==0) field="math_x";
  else if (Mode==1) field="math_y";
  else if (Mode==2) field="func_f";
  else if (Mode==3) field="func_g";
  else if (Mode==4) field="func_h";
  for (i=0;i<=chkobjlastinst(Obj);i++) {
    if (i==0) SendDlgItemMessage(IDFMLIST,LB_SETCARETINDEX,0,TRUE);
    SendDlgItemMessage(IDFMLIST,LB_SETSEL,TRUE,i);
  }
  SendDlgItemMessage(IDFMLIST,LB_SETCARETINDEX,0,FALSE);
  DblClicked();
}

void MathDialog::XMathClicked()
{
  if (SendDlgItemMessage(IDFMYMATH,BM_GETCHECK,0,0)==1) Mode=1;
  else if (SendDlgItemMessage(IDFMF,BM_GETCHECK,0,0)==1) Mode=2;
  else if (SendDlgItemMessage(IDFMG,BM_GETCHECK,0,0)==1) Mode=3;
  else if (SendDlgItemMessage(IDFMH,BM_GETCHECK,0,0)==1) Mode=4;
  else Mode=0;
  SetupItem();
}


DEFINE_RESPONSE_TABLE1(TFileWindow, TSubWindow)
  EV_WM_SIZE,
  EV_WM_LBUTTONDOWN,
  EV_WM_RBUTTONDOWN,
  EV_WM_LBUTTONDBLCLK,
  EV_WM_KEYDOWN,
  EV_WM_PAINT,
  EV_COMMAND(CM_FILEOPENN,FileOpen),
  EV_COMMAND(CM_EDITK,FileEdit),
  EV_COMMAND(CM_DELETEK,FileDelete),
  EV_COMMAND(CM_COPYK,FileCopy),
  EV_COMMAND(CM_COPY2K,FileCopy2),
  EV_COMMAND(CM_TOPK,FileTop),
  EV_COMMAND(CM_LASTK,FileLast),
  EV_COMMAND(CM_UPDATEK,FileUpdate),
  EV_COMMAND(CM_DRAWK,FileDraw),
  EV_COMMAND(CM_HIDDENK,FileHidden),
  EV_COMMAND(CM_UPK,FileUp),
  EV_COMMAND(CM_DOWNK,FileDown),
  EV_COMMAND(CM_FITSETTINGS,FileFit),
END_RESPONSE_TABLE;

TFileWindow::TFileWindow(TWindow *parent):TSubWindow(parent)
{
  menulocal.fileopen=TRUE;
  Attr.Style|=WS_VSCROLL|WS_HSCROLL;
  col1=GetSysColor(COLOR_WINDOWTEXT);
  col1h=GetSysColor(COLOR_GRAYTEXT);
  col1d=RGB(0,0,255);
  col2=(GetSysColor(COLOR_WINDOW) ^ GetSysColor(COLOR_HIGHLIGHT)) & RGB(255,255,255);
  brush1=CreateSolidBrush(col2);
  brush2=CreateSolidBrush(GetSysColor(COLOR_WINDOW));
  pen1=(HPEN)GetStockObject(NULL_PEN);
  pen2=(HPEN)GetStockObject(NULL_PEN);
  PopupMenu.AppendMenu(MF_STRING,CM_FILEOPENN,"Open");
  PopupMenu.AppendMenu(MF_STRING,CM_UPDATEK,"Update (CR)");
  PopupMenu.AppendMenu(MF_STRING,CM_DELETEK,"Close (DEL)");
  PopupMenu.AppendMenu(MF_STRING,CM_TOPK,"Top (HOME)");
  PopupMenu.AppendMenu(MF_STRING,CM_UPK,"Up (SHIFT+UP)");
  PopupMenu.AppendMenu(MF_STRING,CM_DOWNK,"Down (SHIFT+DOWN)");
  PopupMenu.AppendMenu(MF_STRING,CM_LASTK,"Last (END)");
  PopupMenu.AppendMenu(MF_STRING,CM_EDITK,"Edit");
  PopupMenu.AppendMenu(MF_STRING,CM_COPYK,"Duplicate (INS)");
  PopupMenu.AppendMenu(MF_STRING,CM_COPY2K,"Duplicate behind (SHIT+INS)");
  PopupMenu.AppendMenu(MF_STRING,CM_DRAWK,"Draw (SPC)");
  PopupMenu.AppendMenu(MF_STRING,CM_HIDDENK,"Hide (TAB)");
  PopupMenu.AppendMenu(MF_STRING,CM_FITSETTINGS,"Fit settings (Ctrl+F)");
}

void TFileWindow::SetupWindow()
{
  TSubWindow::SetupWindow();
  select=-1;
  Scroller = new TScroller(this,1,1,0,0);
  Scroller->XLine=NgraphApp->FWidth;
  Scroller->YLine=NgraphApp->FHeight;
  Scroller->TrackMode=FALSE;
  obj=chkobject("file");
  filenum=chkobjlastinst(obj);
  ChangeFileNum();
}

TFileWindow::~TFileWindow()
{
  menulocal.fileopen=FALSE;
  ::DeleteObject(brush1);
  ::DeleteObject(brush2);
  ::DeleteObject(pen1);
  ::DeleteObject(pen2);
  NgraphApp->DeleteFileWindow();
}

void TFileWindow::EvSize(UINT sizeType, TSize& size)
{
  TSubWindow::EvSize(sizeType,size);
  ChangeFileNum();
}

void TFileWindow::EvLButtonDown(UINT,TPoint& point)
{
  int sel;
  TClientDC dc(Handle);
  TPoint po;

  if (menulock || globallock) return;
  sel=(Scroller->YPos+point.y)/NgraphApp->FHeight-1;
  if (sel!=select) {
    po.x=Scroller->XPos;
    po.y=Scroller->YPos;
    dc.SetWindowOrg(po);
    SelectFile(dc,sel);
  }
}

void TFileWindow::EvRButtonDown(UINT modKeys,TPoint& point)
{
  TRect rect;
  TPoint point2;
  int sel;
  TClientDC dc(Handle);
  TPoint po;
  struct objlist *fitobj,*obj2;
  char *fit;
  int idnum;
  struct narray iarray;

  if (menulock || globallock) return;
  sel=(Scroller->YPos+point.y)/NgraphApp->FHeight-1;
  if (sel!=select) {
    po.x=Scroller->XPos;
    po.y=Scroller->YPos;
    dc.SetWindowOrg(po);
    SelectFile(dc,sel);
  }
  sel=select;
  if ((sel>=0) && (sel<=filenum)) {
    PopupMenu.EnableMenuItem(CM_FILEOPENN,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_DELETEK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_UPDATEK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_TOPK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_UPK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_DOWNK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_LASTK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_EDITK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_COPYK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_COPY2K,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_DRAWK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_HIDDENK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_FITSETTINGS,MF_GRAYED);
    if ((fitobj=chkobject("fit"))!=NULL) {
      if (getobj(obj,"fit",sel,0,NULL,&fit)!=-1) {
        if (fit!=NULL) {
          arrayinit(&iarray,sizeof(int));
          if (!getobjilist(fit,&obj2,&iarray,FALSE,NULL)) {
            idnum=arraynum(&iarray);
            if ((obj2==fitobj) && (idnum>0))
              PopupMenu.EnableMenuItem(CM_FITSETTINGS,MF_ENABLED);
          }
          arraydel(&iarray);
        }
      }
    }
  } else {
    PopupMenu.EnableMenuItem(CM_FILEOPENN,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_DELETEK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_UPDATEK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_TOPK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_UPK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_DOWNK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_LASTK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_EDITK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_COPYK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_COPY2K,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_DRAWK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_HIDDENK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_FITSETTINGS,MF_GRAYED);
  }
  GetWindowRect(rect);
  point2.x=point.x+rect.left;
  point2.y=point.y+rect.top;
  PopupMenu.TrackPopupMenu(TPM_LEFTBUTTON | TPM_RIGHTBUTTON,point2,0,Handle);
}

void TFileWindow::EvLButtonDblClk(UINT,TPoint& point)
{
  if (menulock || globallock) return;
  FileUpdate();
}

void TFileWindow::EvKeyDown(UINT key,UINT repeatCount,UINT flags)
{
  int sel;
  TClientDC dc(Handle);
  TPoint po;
  TRect rect;
  int h,h1,h2;
  TPoint point;
  struct objlist *fitobj,*obj2;
  char *fit;
  int idnum;
  struct narray iarray;

  if (menulock || globallock) return;
  sel=select;
  switch (key) {
  case VK_DOWN:
    if (GetAsyncKeyState(VK_SHIFT) & 0x8000) FileDown();
    else {
      if ((sel==-1) && (filenum>=0)) sel=0;
      else if (sel<filenum) sel++;
      GetClientRect(rect);
      h=(rect.bottom-rect.top)/NgraphApp->FHeight;
      h1=Scroller->YPos/NgraphApp->FHeight-1;
      h2=Scroller->YPos/NgraphApp->FHeight-1+h;
      if (sel<h1) {
        Scroller->YPos=(sel+1)*NgraphApp->FHeight;
        Invalidate();
      } else if (sel>=h2) {
        Scroller->YPos=(sel+2-h)*NgraphApp->FHeight;
        Invalidate();
      }
      if (sel!=select) {
        po.x=Scroller->XPos;
        po.y=Scroller->YPos;
        dc.SetWindowOrg(po);
        SelectFile(dc,sel);
      }
    }
    break;
  case VK_UP:
    if (GetAsyncKeyState(VK_SHIFT) & 0x8000) FileUp();
    else {
      if ((sel==-1) && (filenum>=0)) sel=filenum;
      else if (sel>0) sel--;
      GetClientRect(rect);
      h=(rect.bottom-rect.top)/NgraphApp->FHeight;
      h1=Scroller->YPos/NgraphApp->FHeight-1;
      h2=Scroller->YPos/NgraphApp->FHeight-1+h;
      if (sel<h1) {
        Scroller->YPos=(sel+1)*NgraphApp->FHeight;
        Invalidate();
      } else if (sel>=h2) {
        Scroller->YPos=(sel+2-h)*NgraphApp->FHeight;
        Invalidate();
      }
      if (sel!=select) {
        po.x=Scroller->XPos;
        po.y=Scroller->YPos;
        dc.SetWindowOrg(po);
        SelectFile(dc,sel);
      }
    }
    break;
  case VK_DELETE:
    FileDelete();
    break;
  case VK_INSERT:
    if (GetAsyncKeyState(VK_SHIFT) & 0x8000) FileCopy2();
    else FileCopy();
    break;
  case VK_HOME:
    FileTop();
    break;
  case VK_END:
    FileLast();
    break;
  case VK_RETURN:
    FileUpdate();
    break;
  case VK_SPACE:
    FileDraw();
    break;
  case VK_TAB:
    FileHidden();
    break;
  case 'F':
    if (GetAsyncKeyState(VK_CONTROL) & 0x8000) FileFit();
    break;
  case 'P':
    if (GetAsyncKeyState(VK_CONTROL) & 0x8000) {
      sel=select;
      if ((sel>=0) && (sel<=filenum)) {
        PopupMenu.EnableMenuItem(CM_FILEOPENN,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_DELETEK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_UPDATEK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_TOPK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_LASTK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_EDITK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_COPYK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_COPY2K,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_DRAWK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_HIDDENK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_FITSETTINGS,MF_GRAYED);
        if ((fitobj=chkobject("fit"))!=NULL) {
          if (getobj(obj,"fit",sel,0,NULL,&fit)!=-1) {
            if (fit!=NULL) {
              arrayinit(&iarray,sizeof(int));
              if (!getobjilist(fit,&obj2,&iarray,FALSE,NULL)) {
                idnum=arraynum(&iarray);
                if ((obj2==fitobj) && (idnum>0))
                  PopupMenu.EnableMenuItem(CM_FITSETTINGS,MF_ENABLED);
              }
              arraydel(&iarray);
            }
          }
        }
      } else {
        PopupMenu.EnableMenuItem(CM_FILEOPENN,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_DELETEK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_UPDATEK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_TOPK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_LASTK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_EDITK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_COPYK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_COPY2K,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_DRAWK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_HIDDENK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_FITSETTINGS,MF_GRAYED);
      }
      GetWindowRect(rect);
      point.x=rect.left;
      point.y=rect.top;
      PopupMenu.TrackPopupMenu(TPM_LEFTBUTTON | TPM_RIGHTBUTTON,point,0,Handle);
    }
    break;
  default:
    NgraphApp->EvKeyDown2(key,repeatCount,flags);
    break;
  }
  TSubWindow::EvKeyDown(key,repeatCount,flags);
}

void TFileWindow::FileOpen()
{
  if (menulock || globallock) return;
  NgraphApp->CmFileOpen();
}

void TFileWindow::FileEdit()
{
  int sel;
  int len,elen;
  char *buf,*name;
  STARTUPINFO si;
  PROCESS_INFORMATION pi;

  if (menulock || globallock) return;
  if (menulocal.editor==NULL) return;
  sel=select;
  if ((sel>=0) && (sel<=filenum)) {
    elen=strlen(menulocal.editor);
    if (getobj(obj,"file",sel,0,NULL,&name)==-1) return;
    if (name!=NULL) {
      len=strlen(name);
      if ((buf=(char *)memalloc(len+elen+4))==NULL) return;
      strcpy(buf,menulocal.editor);
      unchangefilename(buf);
      buf[elen]=' ';
      buf[elen+1]='"';
      strcpy(buf+elen+2,name);
      unchangefilename(buf+elen+2);
      buf[elen+len+2]='"';
      buf[elen+len+3]='\0';
      si.cb=sizeof(STARTUPINFO);
      si.lpReserved=NULL;
      si.lpDesktop=NULL;
      si.lpTitle=NULL;
      si.dwFlags=0;
      si.wShowWindow=SW_SHOWNORMAL;
      si.cbReserved2=0;
      si.lpReserved2=NULL;
      if (CreateProcess(NULL,buf,NULL,NULL,FALSE,NULL,NULL,NULL,&si,&pi)) {
        CloseHandle(pi.hProcess);
        CloseHandle(pi.hThread);
      }
      memfree(buf);
    }
  }
}

void TFileWindow::FileDelete()
{
  int sel;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=0) && (sel<=filenum)) {
    FitDel(obj,sel);
    delobj(obj,sel);
    Update(FALSE);
    NgraphApp->Changed=TRUE;
  }
}

void TFileWindow::FileCopy()
{
  int sel;
  TClientDC dc(Handle);
  int j,id,perm,type;
  char *field;
  TPoint po;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=0) && (sel<=filenum)) {
    if ((id=newobj(obj))>=0) {
      for (j=0;j<chkobjfieldnum(obj);j++) {
        field=chkobjfieldname(obj,j);
        perm=chkobjperm(obj,field);
        type=chkobjfieldtype(obj,field);
        if ((strcmp2(field,"name")!=0) && (strcmp2(field,"fit")!=0)
          && ((perm&NREAD)!=0) && ((perm&NWRITE)!=0) && (type<NVFUNC))
          copyobj(obj,field,id,sel);
      }
      FitCopy(obj,id,sel);
      filenum++;
      NgraphApp->Changed=TRUE;
    }
    select=sel;
    Update(FALSE);
  }
}

void TFileWindow::FileCopy2()
{
  int sel;
  TClientDC dc(Handle);
  int j,id,perm,type;
  char *field;
  TPoint po;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=0) && (sel<=filenum)) {
    if ((id=newobj(obj))>=0) {
      for (j=0;j<chkobjfieldnum(obj);j++) {
        field=chkobjfieldname(obj,j);
        perm=chkobjperm(obj,field);
        type=chkobjfieldtype(obj,field);
        if ((strcmp2(field,"name")!=0) && (strcmp2(field,"fit")!=0)
          && ((perm&NREAD)!=0) && ((perm&NWRITE)!=0) && (type<NVFUNC))
          copyobj(obj,field,id,sel);
      }
      FitCopy(obj,id,sel);
      filenum++;
      for (j=filenum;j>sel+1;j--) moveupobj(obj,j);
      NgraphApp->Changed=TRUE;
    }
    select=sel;
    Update(FALSE);
  }
}

void TFileWindow::FileTop()
{
  int sel;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=0) && (sel<=filenum)) {
    movetopobj(obj,sel);
    select=0;
    Update(FALSE);
    NgraphApp->Changed=TRUE;
  }
}

void TFileWindow::FileUp()
{
  int sel;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=1) && (sel<=filenum)) {
    moveupobj(obj,sel);
    select=sel-1;
    Update(FALSE);
    NgraphApp->Changed=TRUE;
  }
}

void TFileWindow::FileDown()
{
  int sel;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=0) && (sel<=filenum-1)) {
    movedownobj(obj,sel);
    select=sel+1;
    Update(FALSE);
    NgraphApp->Changed=TRUE;
  }
}

void TFileWindow::FileLast()
{
  int sel;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=0) && (sel<=filenum)) {
    movelastobj(obj,sel);
    select=filenum;
    Update(FALSE);
    NgraphApp->Changed=TRUE;
  }
}

void TFileWindow::FileUpdate()
{
  int sel,ret;
  FileDialog *dlg;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=0) && (sel<=filenum)) {
    dlg=new FileDialog(this,DIALOG_FILE,obj,sel);
    if ((ret=dlg->Execute())==IDDELETE) {
      FitDel(obj,sel);
      delobj(obj,sel);
      select=-1;
    }
    if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
    delete dlg;
    Update(FALSE);
  }
}

void TFileWindow::FileDraw()
{
  int i,sel;
  int hiden;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=0) && (sel<=filenum)) {
    for (i=0;i<=filenum;i++) {
      if (i==sel) hiden=FALSE;
      else hiden=TRUE;
      putobj(obj,"hidden",i,&hiden);
    }
  } else {
    hiden=FALSE;
    for (i=0;i<=filenum;i++) putobj(obj,"hidden",i,&hiden);
  }
  NgraphApp->Draw();
  Update(FALSE);
}

void TFileWindow::FileHidden()
{
  int sel;
  int hiden;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=0) && (sel<=filenum)) {
    getobj(obj,"hidden",sel,0,NULL,&hiden);
    hiden=hiden?FALSE:TRUE;
    putobj(obj,"hidden",sel,&hiden);
    Update(FALSE);
    NgraphApp->Changed=TRUE;
  }
}

void TFileWindow::Update(int clear)
{
  if (clear) select=-1;
  filenum=chkobjlastinst(obj);
  ChangeFileNum();
  Invalidate();
}

void TFileWindow::ChangeFileNum()
{
  int width,height;

  width=80*NgraphApp->FWidth+NgraphApp->FHeight;
  height=(filenum+2)*NgraphApp->FHeight;
  Scroller->SetRange(width,height);
  Scroller->SetPageSize();
}

void TFileWindow::FileFit()
{
  FitDialog *dlg;
  struct objlist *fitobj,*obj2;
  char *fit;
  int sel,idnum,fitid,ret;
  struct narray iarray;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=0) && (sel<=filenum)) {
    if ((fitobj=chkobject("fit"))==NULL) return;
    if (getobj(obj,"fit",sel,0,NULL,&fit)==-1) return;
    if (fit!=NULL) {
      arrayinit(&iarray,sizeof(int));
      if (getobjilist(fit,&obj2,&iarray,FALSE,NULL)) {
        arraydel(&iarray);
        return;
      }
      idnum=arraynum(&iarray);
      if ((obj2!=fitobj) || (idnum<1)) {
        putobj(obj,"fit",sel,NULL);
        arraydel(&iarray);
        return;
      } else fitid=*(int *)arraylast(&iarray);
      arraydel(&iarray);
    }
    if (fit==NULL) return;
    dlg=new FitDialog(this,DIALOG_FIT,fitobj,fitid);
    ret=dlg->Execute();
    if (ret==IDDELETE) {
      delobj(fitobj,fitid);
      putobj(obj,"fit",sel,NULL);
    }
    delete dlg;
  }
}

void TFileWindow::SelectFile(HDC dc,int sel)
{
  int x,y,smode;

  if ((select!=-1) && (sel!=select)) {
    x=NgraphApp->FWidth;
    y=(select+2)*NgraphApp->FHeight;
    smode=::SetROP2(dc,R2_XORPEN);
    ::SelectObject(dc,brush1);
    ::SelectObject(dc,pen1);
    ::Rectangle(dc,x-NgraphApp->FWidth,y-NgraphApp->FHeight,
                   x+80*NgraphApp->FWidth+NgraphApp->FHeight,y);
    ::SetROP2(dc,smode);
  }
  if ((sel>=0) && (sel<=filenum)) select=sel;
  else select=-1;
  if (select!=-1) {
    x=NgraphApp->FWidth;
    y=(select+2)*NgraphApp->FHeight;
    smode=::SetROP2(dc,R2_XORPEN);
    ::SelectObject(dc,brush1);
    ::SelectObject(dc,pen1);
    ::Rectangle(dc,x-NgraphApp->FWidth,y-NgraphApp->FHeight,
                   x+80*NgraphApp->FWidth+NgraphApp->FHeight,y);
    ::SetROP2(dc,smode);
  }
}

void TFileWindow::EvPaint()
{
  PAINTSTRUCT ps;
  HDC dc;
  RECT rc;
  int i,j,k,len,x,y,w,h,dpi;
  char buf[256],*valstr;
  char *file,*bfile;
  RECT rect;
  int GC;
  int cx,cy,type,marktype,fr,fg,fb,fr2,fg2,fb2,intp,spcond,spnum;
  int poly[14];
  double spx[7],spy[7],spz[7];
  double spc[6][7],spc2[6];
  struct mwlocal mwsave;
  HFONT orgfont;
  int hiden;
  HBRUSH brush;
  struct narray *mask,*move;

  if (::GetUpdateRect(Handle,&rc,FALSE)) {
    dc=::BeginPaint(Handle,&ps);
    if (globallock) {
      ::EndPaint(Handle,&ps);
      return;
    }
    Scroller->BeginView(TDC(dc),TRect(rc));
    filenum=chkobjlastinst(obj);
    orgfont=(HFONT)SelectObject(dc,menulocal.menufont);
    SetTextAlign(dc,TA_BOTTOM | TA_LEFT);
    h=NgraphApp->FHeight;
    w=NgraphApp->FWidth;

    brush=CreateSolidBrush(GetSysColor(COLOR_MENU));
    rect.top=0;
    rect.bottom=h;
    rect.left=0;
    rect.right=80*w+h+w;
    FillRect(dc,&rect,brush);
    DeleteObject(brush);
    SetTextColor(dc,GetSysColor(COLOR_MENUTEXT));
    SetBkMode(dc,TRANSPARENT);
    x=w;
    y=h;
    rect.top=y-NgraphApp->FHeight;
    rect.bottom=y;
    rect.left=x;
    rect.right=x+6*w;
    len=sprintf(buf,"#");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=6*w;
    rect.left=x;
    rect.right=x+12*w;
    len=sprintf(buf,"file        ");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=13*w;
    rect.left=x;
    rect.right=x+3*w;
    len=sprintf(buf,"  x");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=3*w;
    rect.left=x;
    rect.right=x+3*w;
    len=sprintf(buf,"  y");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=3*w;
    rect.left=x;
    rect.right=x+3*w;
    len=sprintf(buf," ax");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=3*w;
    rect.left=x;
    rect.right=x+3*w;
    len=sprintf(buf," ay");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=3*w;
    rect.left=x;
    rect.right=x+3*w+h;
    len=sprintf(buf," tp");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=3*w+h;
    rect.left=x;
    rect.right=x+6*w;
    len=sprintf(buf,"  size");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=6*w;
    rect.left=x;
    rect.right=x+6*w;
    len=sprintf(buf," width");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=6*w;
    rect.left=x;
    rect.right=x+6*w;
    len=sprintf(buf,"  skip");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=6*w;
    rect.left=x;
    rect.right=x+6*w;
    len=sprintf(buf,"  step");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=6*w;
    rect.left=x;
    rect.right=x+6*w;
    len=sprintf(buf," final");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=6*w;
    rect.left=x;
    rect.right=x+6*w;
    len=sprintf(buf,"   num");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=6*w;
    rect.left=x+w;
    rect.right=x+8*w;
    len=sprintf(buf,"^#");
    ::ExtTextOut(dc,x+w,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);

    for (i=0;i<=filenum;i++)
    if (((i+2)*h>=(rc.top+Scroller->YPos))
    && ((i-1)*h<=(rc.bottom+Scroller->YPos))) {
      x=w;
      y=(i+2)*h;
      ::SelectObject(dc,pen2);
      ::SelectObject(dc,brush2);
      ::Rectangle(dc,x-w,y-h,x+80*w+h,y);
      rect.top=y-NgraphApp->FHeight;
      rect.bottom=y;

      getobj(obj,"hidden",i,0,NULL,&hiden);
      if (hiden) SetTextColor(dc,col1h);
      else {
        getobj(obj,"mask",i,0,NULL,&mask);
        getobj(obj,"move_data",i,0,NULL,&move);
        if ((arraynum(mask)!=0) || (arraynum(move)!=0)) SetTextColor(dc,col1d);
        else SetTextColor(dc,col1);
      }
      getobj(obj,"id",i,0,NULL,&cx);
      rect.left=x;
	  rect.right=x+6*w;
	  len=sprintf(buf,"%d",cx);
	  ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
	  x+=6*w;

	  getobj(obj,"file",i,0,NULL,&file);
      bfile=getbasename(file);
      rect.left=x;
      rect.right=x+12*w;
      if (bfile!=NULL) {
        ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,bfile,strlen(bfile),NULL);
        memfree(bfile);
      } else
        ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,"....................",20,NULL);
      x+=13*w;

      getobj(obj,"x",i,0,NULL,&cx);
      rect.left=x;
      rect.right=x+3*w;
      len=sprintf(buf,"%3d",cx);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
      x+=3*w;

      getobj(obj,"y",i,0,NULL,&cy);
      rect.left=x;
      rect.right=x+3*w;
      len=sprintf(buf,"%3d",cy);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
      x+=3*w;

      sgetobjfield(obj,i,"axis_x",NULL,&valstr,FALSE,FALSE,FALSE);
      for (j=0;(valstr[j]!='\0') && (valstr[j]!=':');j++);
      if (valstr[j]==':') j++;
      rect.left=x;
      rect.right=x+3*w;
      len=sprintf(buf,"%3s",valstr+j);
      memfree(valstr);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
      x+=3*w;

      sgetobjfield(obj,i,"axis_y",NULL,&valstr,FALSE,FALSE,FALSE);
      for (j=0;(valstr[j]!='\0') && (valstr[j]!=':');j++);
      if (valstr[j]==':') j++;
      rect.left=x;
      rect.right=x+3*w;
      len=sprintf(buf,"%3s",valstr+j);
      memfree(valstr);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
      x+=3*w;

      x+=w;
      getobj(obj,"type",i,0,NULL,&type);
      getobj(obj,"R",i,0,NULL,&fr);
      getobj(obj,"G",i,0,NULL,&fg);
      getobj(obj,"B",i,0,NULL,&fb);
      getobj(obj,"R2",i,0,NULL,&fr2);
      getobj(obj,"G2",i,0,NULL,&fg2);
      getobj(obj,"B2",i,0,NULL,&fb2);
      dpi=nround(h*254.0);
      mwsaveDC(dc,Handle,0,0,&mwsave,dpi,FALSE,NULL);
      GC=_GRAopen(chkobjectname(menulocal.obj),"_output",menulocal.outputobj,
                  menulocal.inst,menulocal.output,-1,-1,-1,NULL,mwlocal);
      if (GC>=0) {
        GRAview(GC,x*10/h,y*10/h-10,x*10/h+10,y*10/h,0);
        switch (type) {
        case 0:
          getobj(obj,"mark_type",i,0,NULL,&marktype);
          GRAlinestyle(GC,0,NULL,1,0,0,1000);
          GRAmark(GC,marktype,5,5,8,fr,fg,fb,fr2,fg2,fb2);
          break;
        case 1:
          GRAcolor(GC,fr,fg,fb);
          GRAlinestyle(GC,0,NULL,1,0,0,1000);
          GRAline(GC,0,5,10,5);
          break;
        case 2:
          poly[0]=0;  poly[1]=5; poly[2]=3; poly[3]=2; poly[4]=7; poly[5]=8;
          poly[6]=10;poly[7]=5; poly[8]=7; poly[9]=2; poly[10]=3;poly[11]=8;
          poly[12]=0; poly[13]=5;
          GRAcolor(GC,fr,fg,fb);
          GRAlinestyle(GC,0,NULL,1,0,0,1000);
          GRAdrawpoly(GC,7,poly,0);
          break;
        case 3:
          spx[0]=0; spx[1]=3; spx[2]=7; spx[3]=10;spx[4]=7; spx[5]=3; spx[6]=0;
          spy[0]=5;spy[1]=2; spy[2]=8; spy[3]=5; spy[4]=2; spy[5]=8; spy[6]=5;
          for (j=0;j<7;j++) spz[j]=j;
          getobj(obj,"interpolation",i,0,NULL,&intp);
          if ((intp==0) || (intp==2)) {
            spcond=SPLCND2NDDIF;
            spnum=4;
          } else {
            spcond=SPLCNDPERIODIC;
            spnum=7;
          }
          spline(spz,spx,spc[0],spc[1],spc[2],spnum,spcond,spcond,0,0);
          spline(spz,spy,spc[3],spc[4],spc[5],spnum,spcond,spcond,0,0);
          GRAcolor(GC,fr,fg,fb);
          GRAcurvefirst(GC,0,NULL,NULL,NULL,
                        splinedif,splineint,NULL,spx[0],spy[0]);
          for (j=0;j<spnum-1;j++) {
            for (k=0;k<6;k++) spc2[k]=spc[k][j];
            if (!GRAcurve(GC,spc2,spx[j],spy[j])) break;
          }
          break;
        case 4: case 5: case 6: case 7: case 8:
          GRAcolor(GC,fr,fg,fb);
          if ((type==4) || (type==5)) GRAlinestyle(GC,0,NULL,1,0,0,1000);
          else GRAlinestyle(GC,0,NULL,1,2,0,1000);
          spx[0]=0;
          spy[0]=7;
          spx[1]=10;
          spy[1]=3;
          if ((type==4) || (type==5)) GRAline(GC,spx[0],spy[0],spx[1],spy[1]);
          if (type==5) {
            poly[0]=9;  poly[1]=3;
            poly[2]=spx[1]; poly[3]=spy[1];
            poly[4]=10; poly[5]=4;
            GRAdrawpoly(GC,3,poly,0);
          }
          if ((type==7) || (type==8)) {
            if (type==7) GRAcolor(GC,fr2,fg2,fb2);
            GRArectangle(GC,spx[0],spy[0],spx[1],spy[1],1);
            if (type==7) GRAcolor(GC,fr,fg,fb);
          }
          if ((type==6) || (type==7)) {
            GRAline(GC,spx[0],spy[0],spx[0],spy[1]);
            GRAline(GC,spx[0],spy[1],spx[1],spy[1]);
            GRAline(GC,spx[1],spy[1],spx[1],spy[0]);
            GRAline(GC,spx[1],spy[0],spx[0],spy[0]);
          }
          break;
        case 9: case 10:
          GRAcolor(GC,fr,fg,fb);
          GRAlinestyle(GC,0,NULL,1,0,0,1000);
          if (type==9) {
            GRAline(GC,0,5,10,5);
            GRAline(GC,0,3,0,7);
            GRAline(GC,10,3,10,7);
          } else {
            GRAline(GC,5,1,5,9);
            GRAline(GC,3,1,7,1);
            GRAline(GC,3,9,7,9);
          }
          break;
        case 11: case 12:
          GRAcolor(GC,fr,fg,fb);
          GRAlinestyle(GC,0,NULL,1,0,0,1000);
          if (type==11) {
            GRAmoveto(GC,0,9);
            GRAlineto(GC,3,9);
            GRAlineto(GC,3,5);
            GRAlineto(GC,7,5);
            GRAlineto(GC,7,1);
            GRAlineto(GC,10,1);
          } else {
            GRAmoveto(GC,0,9);
            GRAlineto(GC,0,6);
            GRAlineto(GC,5,6);
            GRAlineto(GC,5,3);
            GRAlineto(GC,10,3);
            GRAlineto(GC,10,1);
          }
          break;
        case 13: case 14: case 15: case 16: case 17: case 18:
          GRAcolor(GC,fr,fg,fb);
          GRAlinestyle(GC,0,NULL,1,2,0,1000);
          if ((type==15) || (type==17)) {
            if (type==15) GRAcolor(GC,fr2,fg2,fb2);
            GRArectangle(GC,0,4,10,6,1);
            if (type==15) GRAcolor(GC,fr,fg,fb);
          }
          if ((type==16) || (type==18)) {
            if (type==16) GRAcolor(GC,fr2,fg2,fb2);
            GRArectangle(GC,4,1,6,9,1);
            if (type==16) GRAcolor(GC,fr,fg,fb);
          }
          if ((type==13) || (type==15)) {
            GRAline(GC,0,4,10,4);
            GRAline(GC,10,4,10,6);
            GRAline(GC,10,6,0,6);
            GRAline(GC,0,6,0,4);
          }
          if ((type==14) || (type==16)) {
            GRAline(GC,4,1,6,1);
            GRAline(GC,6,1,6,9);
            GRAline(GC,6,9,4,9);
            GRAline(GC,4,9,4,1);
          }
          break;
        }
      }
      _GRAclose(GC);
      mwrestoreDC(&mwsave);
      if ((type==3) && (intp>=2)) {
        rect.left=x+h;
        rect.right=x+h+2*w;
        len=sprintf(buf,"B");
        ::ExtTextOut(dc,x+h,y,ETO_CLIPPED,&rect,buf,len,NULL);
      } else if (type==19) {
        rect.left=x;
        rect.right=x+h+2*w;
        len=sprintf(buf,"FIT");
        ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
      }
      x+=h+2*w;

      getobj(obj,"mark_size",i,0,NULL,&cx);
      rect.left=x;
      rect.right=x+6*w;
      len=sprintf(buf,"%6d",cx);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
      x+=6*w;

      getobj(obj,"line_width",i,0,NULL,&cx);
      rect.left=x;
      rect.right=x+6*w;
      len=sprintf(buf,"%6d",cx);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
      x+=6*w;

      getobj(obj,"head_skip",i,0,NULL,&cx);
      rect.left=x;
      rect.right=x+6*w;
      len=sprintf(buf,"%6d",cx);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
      x+=6*w;

      getobj(obj,"read_step",i,0,NULL,&cx);
      rect.left=x;
      rect.right=x+6*w;
      len=sprintf(buf,"%6d",cx);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
      x+=6*w;

      getobj(obj,"final_line",i,0,NULL,&cx);
      rect.left=x;
      rect.right=x+6*w;
      len=sprintf(buf,"%6d",cx);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
      x+=6*w;

      getobj(obj,"data_num",i,0,NULL,&cx);
      rect.left=x;
      rect.right=x+6*w;
      len=sprintf(buf,"%6d",cx);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
      x+=6*w;

      getobj(obj,"oid",i,0,NULL,&cx);
      rect.left=x+w;
      rect.right=x+8*w;
	    len=sprintf(buf,"^%d",cx);
	    ::ExtTextOut(dc,x+w,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
	    x+=8*w;
    }
    SelectFile(dc,select);
    SetScrollPos(SB_HORZ,Scroller->XPos,TRUE);
    SetScrollPos(SB_VERT,Scroller->YPos,TRUE);
    SelectObject(dc,orgfont);
    ::EndPaint(Handle,&ps);
  }
}

char FileOpenBuf[1024]={0};

void TMyWindow::CmFileNew()
{
  char *name;
  TFileWindow *win;
  FileDialog *dlg;
  int id,ret;
  struct objlist *obj;

  if (menulock || globallock) return;
  if ((obj=chkobject("file"))==NULL) return;
  FileOpenBuf[0]='\0';
  EnableFrameWindow(this,false);
  if (nGetOpenFileName(Handle,"Data new",NULL,NULL,
      FileOpenBuf,512,"All Files (*.*)\0*.*\0Data Files (*.dat)\0*.dat\0",FALSE,menulocal.changedirectory)==IDOK) {
    if ((id=newobj(obj))>=0) {
      if ((name=(char *)memalloc(strlen(FileOpenBuf)+1))==NULL) {
        EnableFrameWindow(this,true);
        return;
      }
      strcpy(name,FileOpenBuf);
      changefilename(name);
      AddDataFileList(name);
      putobj(obj,"file",id,name);
      dlg=new FileDialog(this,DIALOG_FILE,obj,id);
      ret=dlg->Execute();
      if ((ret==IDDELETE) || (ret==IDCANCEL)) {
        FitDel(obj,id);
        delobj(obj,id);
      } else NgraphApp->Changed=TRUE;
      delete dlg;
      win=(TFileWindow *)pTFileWindow;
      if (win!=NULL) win->Update(TRUE);
    }
  }
  EnableFrameWindow(this,true);
}

void TMyWindow::CmFileOpen()
{
  char *s,*s2,*dir,*name;
  int dirlen,len,id;
  TFileWindow *win;
  FileDialog *dlg;
  int id0,ids,ide,j,perm,type,ret;
  char *field;
  struct objlist *obj;
  OSVERSIONINFO VersionInformation;

  if (menulock || globallock) return;
  VersionInformation.dwOSVersionInfoSize=sizeof(OSVERSIONINFO);
  GetVersionEx(&VersionInformation);
  if ((obj=chkobject("file"))==NULL) return;
  FileOpenBuf[0]='\0';
  ids=-1;
  EnableFrameWindow(this,false);
  do {
    ret=nGetOpenFileNameMulti(Handle,"Data open",NULL,&(menulocal.fileopendir),
         FileOpenBuf,1024,"All Files (*.*)\0*.*\0Data Files (*.dat)\0*.dat\0",menulocal.changedirectory);
    if ((ret==IDOK) || (ret==3)) {
      if (VersionInformation.dwMajorVersion>=4) {
        s2=FileOpenBuf;
        dir=s2;
        dirlen=strlen(dir);
        s2=s2+strlen(s2)+1;
        if (s2[0]=='\0') {
          if ((id=newobj(obj))>=0) {
            if (ids==-1) ids=id;
            if ((name=(char *)memalloc(dirlen+1))!=NULL) {
              strcpy(name,dir);
              changefilename(name);
              AddDataFileList(name);
              putobj(obj,"file",id,name);
            }
          }
        } else {
          while (s2[0]!='\0') {
            len=strlen(s2);
            if ((id=newobj(obj))>=0) {
              if (ids==-1) ids=id;
              if ((name=(char *)memalloc(dirlen+len+2))!=NULL) {
                strcpy(name,dir);
                if ((dirlen!=0) && (name[dirlen-1]!='\\')) {
                  name[dirlen]='\\';
                  name[dirlen+1]='\0';
                }
                strcat(name,s2);
                changefilename(name);
                AddDataFileList(name);
                putobj(obj,"file",id,name);
              }
            }
            s2+=strlen(s2)+1;
          }
          FileOpenBuf[0]='\0';
        }
      } else {
        s2=FileOpenBuf;
        dir=getitok(&s2,&dirlen," \x09");
        if (s2[0]=='\0') {
          if ((id=newobj(obj))>=0) {
            if (ids==-1) ids=id;
            if ((name=(char *)memalloc(dirlen+1))!=NULL) {
              strcpy(name,dir);
              changefilename(name);
              AddDataFileList(name);
              putobj(obj,"file",id,name);
            }
          }
        } else {
          while (s2[0]!='\0') {
            s=getitok(&s2,&len," \x09");
            if ((id=newobj(obj))>=0) {
              if (ids==-1) ids=id;
              if ((name=(char *)memalloc(dirlen+len+2))!=NULL) {
                strncpy(name,dir,dirlen);
                if ((dirlen!=0) && (name[dirlen-1]!='\\')) {
                  name[dirlen]='\\';
                  name[dirlen+1]='\0';
                  strncat(name,s,len);
                  name[dirlen+1+len]='\0';
                } else {
                  strncat(name,s,len);
                  name[dirlen+len]='\0';
                }
                changefilename(name);
                AddDataFileList(name);
                putobj(obj,"file",id,name);
              }
            }
          }
          FileOpenBuf[0]='\0';
        }
      }
      win=(TFileWindow *)pTFileWindow;
      if (win!=NULL) win->Update(TRUE);
    }
  } while (ret==3);
  EnableFrameWindow(this,true);
  if (ids!=-1) {
    ide=chkobjlastinst(obj);
    id0=-1;
    id=ids;
    while (id<=ide) {
      if (id0!=-1) {
        for (j=0;j<chkobjfieldnum(obj);j++) {
          field=chkobjfieldname(obj,j);
          perm=chkobjperm(obj,field);
          type=chkobjfieldtype(obj,field);
          if ((strcmp2(field,"name")!=0) && (strcmp2(field,"file")!=0)
          && (strcmp2(field,"fit")!=0)
          && ((perm&NREAD)!=0) && ((perm&NWRITE)!=0) && (type<NVFUNC))
            copyobj(obj,field,id,id0);
        }
        FitCopy(obj,id,id0);
        id++;
      } else {
        dlg=new FileDialog(this,DIALOG_FILE,obj,id);
        ret=dlg->Execute();
        if ((ret==IDDELETE) || (ret==IDCANCEL)) {
          FitDel(obj,id);
          delobj(obj,id);
          ide--;
        } else if (ret==IDFAPPLY) {
          id0=id;
          id++;
          NgraphApp->Changed=TRUE;
        } else {
          id++;
          NgraphApp->Changed=TRUE;
        }
        delete dlg;
      }
      win=(TFileWindow *)pTFileWindow;
      if (win!=NULL) win->Update(TRUE);
    }
  }
}

void TMyWindow::CmFileClose()
{
  TFileWindow *win;
  SelectDialog *dlg;
  struct narray farray;
  struct objlist *obj;
  int i;
  int *array,num;

  if (menulock || globallock) return;
  if ((obj=chkobject("file"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,FileCB,
                       (struct narray *)&farray,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&farray);
    array=(int *)arraydata(&farray);
    for (i=num-1;i>=0;i--) {
      FitDel(obj,array[i]);
      delobj(obj,array[i]);
      NgraphApp->Changed=TRUE;
    }
    win=(TFileWindow *)pTFileWindow;
    if (win!=NULL) win->Update(TRUE);
  }
  arraydel(&farray);
  delete dlg;
}

void TMyWindow::CmFileUpdate()
{
  TFileWindow *win;
  SelectDialog *dlg;
  FileDialog *fdlg;
  struct objlist *obj;
  int i,j;
  int *array,num;
  int id0,perm,type,ret;
  char *field;
  struct narray farray;
  int last;

  if (menulock || globallock) return;
  if ((obj=chkobject("file"))==NULL) return;
  last=chkobjlastinst(obj);
  if (last==-1) return;
  else if (last==0) {
    arrayinit(&farray,sizeof(int));
    arrayadd(&farray,&last);
    ret=IDOK;
  } else {
    dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,FileCB,
                         (struct narray *)&farray,NULL);
    ret=dlg->Execute();
    delete dlg;
  }
  if (ret==IDOK) {
    num=arraynum(&farray);
    array=(int *)arraydata(&farray);
    id0=-1;
    for (i=0;i<num;i++) {
      if (id0!=-1) {
        for (j=0;j<chkobjfieldnum(obj);j++) {
          field=chkobjfieldname(obj,j);
          perm=chkobjperm(obj,field);
          type=chkobjfieldtype(obj,field);
          if ((strcmp2(field,"name")!=0) && (strcmp2(field,"file")!=0)
          && (strcmp2(field,"fit")!=0)
		  && ((perm&NREAD)!=0) && ((perm&NWRITE)!=0) && (type<NVFUNC))
			copyobj(obj,field,array[i],array[id0]);
		}
        FitCopy(obj,array[i],array[id0]);
      } else {
        fdlg=new FileDialog(this,DIALOG_FILE,obj,array[i]);
        ret=fdlg->Execute();
        if (ret==IDDELETE) {
          FitDel(obj,array[i]);
          delobj(obj,array[i]);
          for (j=i+1;j<num;j++) array[j]--;
        } else if (ret==IDFAPPLY) id0=i;
        if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
        delete fdlg;
      }
    }
    win=(TFileWindow *)pTFileWindow;
    if (win!=NULL) win->Update(TRUE);
  }
  arraydel(&farray);
}

void TMyWindow::CmFileEdit()
{
  CopyDialog *dlg;
  struct objlist *obj;
  int i;
  int len,elen,last;
  char *buf,*name;
  STARTUPINFO si;
  PROCESS_INFORMATION pi;

  if (menulock || globallock) return;
  if (menulocal.editor==NULL) return;
  if ((obj=chkobject("file"))==NULL) return;
  last=chkobjlastinst(obj);
  if (last==-1) return;
  else if (last==0) i=0;
  else {
    dlg=new CopyDialog((TWindow *)this,DIALOG_COPY,obj,-1,FileCB);
    if (dlg->Execute()==IDOK) {
      i=dlg->sel;
      delete dlg;
    } else {
      delete dlg;
      return;
    }
  }
  if (i<0) return;
  elen=strlen(menulocal.editor);
  if (getobj(obj,"file",i,0,NULL,&name)==-1) return;
  if (name!=NULL) {
    len=strlen(name);
    if ((buf=(char *)memalloc(len+elen+2))==NULL) return;
    strcpy(buf,menulocal.editor);
    unchangefilename(buf);
    buf[elen]=' ';
    strcpy(buf+elen+1,name);
    unchangefilename(buf+elen+1);
    si.cb=sizeof(STARTUPINFO);
    si.lpReserved=NULL;
    si.lpDesktop=NULL;
    si.lpTitle=NULL;
    si.dwFlags=0;
    si.wShowWindow=SW_SHOWNORMAL;
    si.cbReserved2=0;
    si.lpReserved2=NULL;
    if (CreateProcess(NULL,buf,NULL,NULL,FALSE,NULL,NULL,NULL,&si,&pi)) {
      CloseHandle(pi.hProcess);
      CloseHandle(pi.hThread);
    }
    memfree(buf);
  }
}

void TMyWindow::CmFileDirectory()
{
  DirectoryDialog *dlg;

  if (menulock || globallock) return;
  dlg=new DirectoryDialog((TWindow *)this,DIALOG_DIRECTORY);
  dlg->Execute();
}

void TMyWindow::CmSetMathAll()
{
  struct objlist *obj;

  if (menulock || globallock) return;
  if ((obj=chkobject("file"))==NULL) return;
  MathDialog(this,DIALOG_MATH,obj).Execute();
}

void TMyWindow::CmWindowFile()
{
  TFileWindow *win;
  TRect rect;
  int x,y;

  if (menulock || globallock) return;
  if  (pTFileWindow==NULL) {
    win=(TFileWindow *)pTFileWindow=new TFileWindow();
    win->Frame=new TSubFrameWindow(this,"Data Window",win,
    &menulocal.filex,&menulocal.filey,&menulocal.filewidth,&menulocal.fileheight);
    win->Frame->SetIcon(GetApplication(),ICON_3);
    NgraphApp->Frame->GetWindowRect(rect);
    x=rect.left;
    y=rect.top;
    if (menulocal.filewidth==CW_USEDEFAULT) win->Frame->Attr.W=FWidth*80+FHeight+2*FWidth;
    else win->Frame->Attr.W=menulocal.filewidth;
    if (menulocal.fileheight==CW_USEDEFAULT) win->Frame->Attr.H=FHeight*10;
    else win->Frame->Attr.H=menulocal.fileheight;
    if (menulocal.filex==CW_USEDEFAULT) win->Frame->Attr.X=CW_USEDEFAULT;
    else {
      win->Frame->Attr.X=menulocal.filex+x;
      if (win->Frame->Attr.X<0) win->Frame->Attr.X=0;
    }
    if (menulocal.filey==CW_USEDEFAULT) win->Frame->Attr.X=CW_USEDEFAULT;
    else {
      win->Frame->Attr.Y=menulocal.filey+y;
      if (win->Frame->Attr.Y<0) win->Frame->Attr.Y=0;
    }
    win->Frame->Create();
  } else {
    win=(TFileWindow *)pTFileWindow;
    win->CloseWindow();
  }
}

void TMyWindow::CmFileHistory(WPARAM wparam)
{
  int fil,num;
  char **data;
  int ret;
  char *name;
  int len,id;
  TFileWindow *win;
  FileDialog *dlg;
  struct objlist *obj;

  if (menulock || globallock) return;
  num=arraynum(menulocal.datafilelist);
  data=(char **)arraydata(menulocal.datafilelist);
  fil=LOWORD(wparam)-CM_FILE00-1;
  if ((fil<0) || (fil>=num) || (data[fil]==NULL)) return;
  if ((obj=chkobject("file"))==NULL) return;
  if ((id=newobj(obj))>=0) {
    len=strlen(data[fil]);
    if ((name=(char *)memalloc(len+1))!=NULL) {
      strcpy(name,data[fil]);
      putobj(obj,"file",id,name);
      dlg=new FileDialog(this,DIALOG_FILE,obj,id);
      ret=dlg->Execute();
      if ((ret==IDDELETE) || (ret==IDCANCEL)) {
        FitDel(obj,id);
        delobj(obj,id);
      } else {
        NgraphApp->Changed=TRUE;
      }
      delete dlg;
    }
  }
  AddDataFileList(data[fil]);
  win=(TFileWindow *)pTFileWindow;
  if (win!=NULL) win->Update(TRUE);
}

void TMyWindow::CmOptionFileDef()
{
  struct objlist *obj;
  int id;

  if (menulock || globallock) return;
  if ((obj=chkobject("file"))==NULL) return;
  if ((id=newobj(obj))>=0) {
    if (FileDefDialog(this,DIALOG_FILEDEF,obj,id).Execute()==IDOK) {
      if (CheckIniFile()) {
        exeobj(obj,"save_config",id,0,NULL);
      }
    }
    delobj(obj,id);
    NgraphApp->Update2();
  }
}

void TMyWindow::CeWindowFile(TCommandEnabler& ce)
{
  if (pTFileWindow==NULL) ce.SetCheck(TCommandEnabler::Unchecked);
  else ce.SetCheck(TCommandEnabler::Checked);
}

