/* 
 * 
 * This file is part of "Ngraph for Windows".
 * 
 * Copyright (C) 2015, Satoshi ISHIZAKA. isizaka@msa.biglobe.ne.jp
 * 
 * "Ngraph for Windows" is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * "Ngraph for Windows" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

/*
 *
 * winlgnd.cpp
 *
 */

#include <owl\pch.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

extern "C" {
#include "ngraph.h"
#include "object.h"
#include "nstring.h"
#include "jnstring.h"
#include "ioutil.h"
#include "mathfn.h"
#include "gra.h"
#include "spline.h"
#include "owinmenu.h"
}

#include "bitmap.rh"
#include "winmenu.rh"
#include "winmenu.h"
#include "windialg.h"
#include "wincommn.h"
#include "winlgnd.h"

char *LegendLineCB(struct objlist *obj,int id)
{
  struct narray *array;
  int num,*data;
  char *s;

  if ((s=(char *)memalloc(128))==NULL) return NULL;
  getobj(obj,"points",id,0,NULL,&array);
  num=arraynum(array);
  data=(int *)arraydata(array);
  if (num<2) sprintf(s,"%-5d",id);
  else sprintf(s,"%-5d (%d,%d)-",id,data[0],data[1]);
  return s;
}

char *LegendRectCB(struct objlist *obj,int id)
{
  int x1,y1;
  char *s;

  if ((s=(char *)memalloc(128))==NULL) return NULL;
  getobj(obj,"x1",id,0,NULL,&x1);
  getobj(obj,"y1",id,0,NULL,&y1);
  sprintf(s,"%-5d x1:%d y1:%d",id,x1,y1);
  return s;
}

char *LegendArcCB(struct objlist *obj,int id)
{
  int x1,y1;
  char *s;

  if ((s=(char *)memalloc(128))==NULL) return NULL;
  getobj(obj,"x",id,0,NULL,&x1);
  getobj(obj,"y",id,0,NULL,&y1);
  sprintf(s,"%-5d x:%d y:%d",id,x1,y1);
  return s;
}

char *LegendTextCB(struct objlist *obj,int id)
{
  char *text;
  char *s;

  if ((s=(char *)memalloc(128))==NULL) return NULL;
  getobj(obj,"text",id,0,NULL,&text);
  if (text==NULL) sprintf(s,"%-5d",id);
  else sprintf(s,"%-5d %.64s",id,text);
  return s;
}



DEFINE_RESPONSE_TABLE1(LegendCurveDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
  EV_CHILD_NOTIFY(IDDELETE,BN_CLICKED,DeleteClicked),
END_RESPONSE_TABLE;

LegendCurveDialog::LegendCurveDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
:TMyDialog(parent,resID) {
  Obj=obj;
  Id=id;
  Col=new ColorButton(this,IDLCCOLB);
}

void LegendCurveDialog::SetupWindow()
{
  char *title;
  int len;

  TMyDialog::SetupWindow();
  len=GetWindowTextLength();
  if ((title=(char *)memalloc(len+10))!=NULL) {
    GetWindowText(title,len+1);
    sprintf(title+len," %d",Id);
    SetCaption(title);
    memfree(title);
  }
  SetupItem(Id);
}

void LegendCurveDialog::SetupItem(int id)
{
  int R,G,B;

  SetTextFromObjField(Handle,IDLCPOINTS,Obj,id,"points");
  SetStyleFromObjField(Handle,IDLCSTYLE,Obj,id,"style");
  SetComboList(Handle,IDLCWIDTH,cblinewidth,CBLINEWIDTH);
  SetTextFromObjField(Handle,IDLCWIDTH,Obj,id,"width");
  SetListFromObjField(Handle,IDLCJOIN,Obj,id,"join");
  SetTextFromObjField(Handle,IDLCMITER,Obj,id,"miter_limit");
  getobj(Obj,"R",id,0,NULL,&R);
  getobj(Obj,"G",id,0,NULL,&G);
  getobj(Obj,"B",id,0,NULL,&B);
  Col->SetColor(R,G,B);
  SetListFromObjField(Handle,IDLCINTP,Obj,id,"interpolation");
}


void LegendCurveDialog::CloseWindow(int retVal)
{
  int R,G,B;

  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  if (SetObjFieldFromText(Handle,IDLCPOINTS,Obj,Id,"points")) return;
  if (SetObjFieldFromStyle(Handle,IDLCSTYLE,Obj,Id,"style")) return;
  if (SetObjFieldFromText(Handle,IDLCWIDTH,Obj,Id,"width")) return;
  if (SetObjFieldFromList(Handle,IDLCJOIN,Obj,Id,"join")) return;
  if (SetObjFieldFromText(Handle,IDLCMITER,Obj,Id,"miter_limit")) return;
  Col->GetColor(&R,&G,&B);
  if (putobj(Obj,"R",Id,&R)==-1) return;
  if (putobj(Obj,"G",Id,&G)==-1) return;
  if (putobj(Obj,"B",Id,&B)==-1) return;
  if (SetObjFieldFromList(Handle,IDLCINTP,Obj,Id,"interpolation")) return;
  TMyDialog::CloseWindow(retVal);
}


DEFINE_RESPONSE_TABLE1(LegendPolyDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
  EV_CHILD_NOTIFY(IDDELETE,BN_CLICKED,DeleteClicked),
END_RESPONSE_TABLE;

LegendPolyDialog::LegendPolyDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
:TMyDialog(parent,resID) {
  Obj=obj;
  Id=id;
  Col=new ColorButton(this,IDLPCOLB);
}

void LegendPolyDialog::SetupWindow()
{
  char *title;
  int len;

  TMyDialog::SetupWindow();
  len=GetWindowTextLength();
  if ((title=(char *)memalloc(len+10))!=NULL) {
    GetWindowText(title,len+1);
    sprintf(title+len," %d",Id);
    SetCaption(title);
    memfree(title);
  }
  SetupItem(Id);
}

void LegendPolyDialog::SetupItem(int id)
{
  int R,G,B;

  SetTextFromObjField(Handle,IDLPPOINTS,Obj,id,"points");
  SetStyleFromObjField(Handle,IDLPSTYLE,Obj,id,"style");
  SetComboList(Handle,IDLPWIDTH,cblinewidth,CBLINEWIDTH);
  SetTextFromObjField(Handle,IDLPWIDTH,Obj,id,"width");
  SetListFromObjField(Handle,IDLPJOIN,Obj,id,"join");
  SetTextFromObjField(Handle,IDLPMITER,Obj,id,"miter_limit");
  getobj(Obj,"R",id,0,NULL,&R);
  getobj(Obj,"G",id,0,NULL,&G);
  getobj(Obj,"B",id,0,NULL,&B);
  Col->SetColor(R,G,B);
  SetListFromObjField(Handle,IDLPFILL,Obj,id,"fill");
}


void LegendPolyDialog::CloseWindow(int retVal)
{
  int R,G,B;

  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  if (SetObjFieldFromText(Handle,IDLPPOINTS,Obj,Id,"points")) return;
  if (SetObjFieldFromStyle(Handle,IDLPSTYLE,Obj,Id,"style")) return;
  if (SetObjFieldFromText(Handle,IDLPWIDTH,Obj,Id,"width")) return;
  if (SetObjFieldFromList(Handle,IDLPJOIN,Obj,Id,"join")) return;
  if (SetObjFieldFromText(Handle,IDLPMITER,Obj,Id,"miter_limit")) return;
  Col->GetColor(&R,&G,&B);
  if (putobj(Obj,"R",Id,&R)==-1) return;
  if (putobj(Obj,"G",Id,&G)==-1) return;
  if (putobj(Obj,"B",Id,&B)==-1) return;
  if (SetObjFieldFromList(Handle,IDLPFILL,Obj,Id,"fill")) return;
  TMyDialog::CloseWindow(retVal);
}


DEFINE_RESPONSE_TABLE1(ArrowHeadStatic,TStatic)
  EV_WM_PAINT,
END_RESPONSE_TABLE;

ArrowHeadStatic::ArrowHeadStatic(TWindow *parent,int resId)
:TStatic(parent,resId)
{
  ang=45;
  wid=20000;
}

void ArrowHeadStatic::SetLenWid(int angle,int width)
{
  ang=angle;
  wid=width;
}

void ArrowHeadStatic::EvPaint()
{
  PAINTSTRUCT ps;
  RECT rect;
  HRGN hClip;
  HDC dc;
  HPEN pen,spen;
  HBRUSH brush,sbrush;
  int x,y,w,h,lw,len;
  POINT points[3];

  dc=::BeginPaint(Handle,&ps);
  ::GetClientRect(Handle,&rect);
  hClip=CreateRectRgn(rect.left,rect.top,rect.right,rect.bottom);
  SelectClipRgn(dc,hClip);
  DeleteObject(hClip);
  brush=CreateSolidBrush(RGB(255,255,255));
  FillRect(dc,&rect,brush);
  DeleteObject(brush);
  pen=(HPEN)GetStockObject(BLACK_PEN);
  brush=CreateSolidBrush(RGB(0,0,0));
  spen=(HPEN)SelectObject(dc,pen);
  sbrush=(HBRUSH)SelectObject(dc,brush);
  len=wid*0.5/tan(ang*0.5*MPI/180);
  x=rect.left;
  y=rect.top;
  w=rect.right-rect.left;
  h=rect.bottom-rect.top;
  if (w/20<h/20) lw=w/20;
  else lw=h/20;
  Rectangle(dc,x+nround(lw*(len/10000.0)),y+(h-lw)/2,x+w,y+(h+lw)/2);
  points[0].x=x;
  points[0].y=y+h/2;
  points[1].x=x+nround(lw*(len/10000.0));
  points[1].y=y+h/2-nround(lw*(wid/20000.0));
  points[2].x=x+nround(lw*(len/10000.0));
  points[2].y=y+h/2+nround(lw*(wid/20000.0));
  BeginPath(dc);
  Polygon(dc,points,3);
  CloseFigure(dc);
  EndPath(dc);
  FillPath(dc);
  SelectObject(dc,spen);
  SelectObject(dc,sbrush);
  DeleteObject(pen);
  DeleteObject(brush);
  ::EndPaint(Handle,&ps);
}


DEFINE_RESPONSE_TABLE1(LegendArrowDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
  EV_CHILD_NOTIFY(IDDELETE,BN_CLICKED,DeleteClicked),
  EV_WM_HSCROLL,
  EV_WM_VSCROLL,
END_RESPONSE_TABLE;

LegendArrowDialog::LegendArrowDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
:TMyDialog(parent,resID) {
  Obj=obj;
  Id=id;
  Col=new ColorButton(this,IDLACOLB);
  LEN=new TStatic(this,IDLAALENGTH,16);
  WID=new TStatic(this,IDLAAWIDTH,16);
  LENSC=new TScrollBar(this,IDLASCH);
  WIDSC=new TScrollBar(this,IDLASCV);
  HeadWin=new ArrowHeadStatic(this,IDLAWIN);
}

void LegendArrowDialog::SetupWindow()
{
  char *title;
  int len;

  TMyDialog::SetupWindow();
  len=GetWindowTextLength();
  if ((title=(char *)memalloc(len+10))!=NULL) {
    GetWindowText(title,len+1);
    sprintf(title+len," %d",Id);
    SetCaption(title);
    memfree(title);
  }
  SetupItem(Id);
}

void LegendArrowDialog::SetupItem(int id)
{
  int len,R,G,B;
  char buf[16];

  SetTextFromObjField(Handle,IDLAPOINTS,Obj,id,"points");
  SetStyleFromObjField(Handle,IDLASTYLE,Obj,id,"style");
  SetComboList(Handle,IDLAWIDTH,cblinewidth,CBLINEWIDTH);
  SetTextFromObjField(Handle,IDLAWIDTH,Obj,id,"width");
  SetListFromObjField(Handle,IDLAJOIN,Obj,id,"join");
  SetTextFromObjField(Handle,IDLAMITER,Obj,id,"miter_limit");
  SetListFromObjField(Handle,IDLAHEAD,Obj,id,"arrow");
  getobj(Obj,"arrow_width",id,0,NULL,&wid);
  wid=(wid/1000)*1000;
  sprintf(buf,"%d%c",wid/100,'%');
  WID->SetText(buf);
  WIDSC->SetRange(10,200);
  WIDSC->SetPosition(10+(200-wid/1000));
  getobj(Obj,"arrow_length",id,0,NULL,&len);
  ang=atan(0.5*wid/len)*2*180/MPI;
  if (ang<10) ang=10;
  else if (ang>170) ang=170;
  sprintf(buf,"%d deg",ang);
  LEN->SetText(buf);
  LENSC->SetRange(10,170);
  LENSC->SetPosition(ang);
  HeadWin->SetLenWid(ang,wid);
  getobj(Obj,"R",id,0,NULL,&R);
  getobj(Obj,"G",id,0,NULL,&G);
  getobj(Obj,"B",id,0,NULL,&B);
  Col->SetColor(R,G,B);
}


void LegendArrowDialog::CloseWindow(int retVal)
{
  int len,R,G,B;

  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  if (SetObjFieldFromText(Handle,IDLAPOINTS,Obj,Id,"points")) return;
  if (SetObjFieldFromStyle(Handle,IDLASTYLE,Obj,Id,"style")) return;
  if (SetObjFieldFromText(Handle,IDLAWIDTH,Obj,Id,"width")) return;
  if (SetObjFieldFromList(Handle,IDLAJOIN,Obj,Id,"join")) return;
  if (SetObjFieldFromText(Handle,IDLAMITER,Obj,Id,"miter_limit")) return;
  if (SetObjFieldFromList(Handle,IDLAHEAD,Obj,Id,"arrow")) return;
  len=wid*0.5/tan(ang*0.5*MPI/180);
  if (putobj(Obj,"arrow_length",Id,&len)==-1) return;
  if (putobj(Obj,"arrow_width",Id,&wid)==-1) return;
  Col->GetColor(&R,&G,&B);
  if (putobj(Obj,"R",Id,&R)==-1) return;
  if (putobj(Obj,"G",Id,&G)==-1) return;
  if (putobj(Obj,"B",Id,&B)==-1) return;
  TMyDialog::CloseWindow(retVal);
}

void LegendArrowDialog::EvHScroll(UINT scrollCode, UINT thumbPos, HWND hWndCtl)
{
  int Min,Max,Sc,X,ScPos;
  char buf[16];

  LENSC->GetRange(Min,Max);
  X=ang;
  switch (scrollCode) {
  case SB_PAGEUP:
    Sc=-10;
    break;
  case SB_PAGEDOWN:
    Sc=10;
    break;
  case SB_LINEUP:
    Sc=-1;
    break;
  case SB_LINEDOWN:
    Sc=1;
    break;
  case SB_THUMBPOSITION:
    Sc=thumbPos-X;
    break;
  case SB_THUMBTRACK:
    Sc=thumbPos-X;
    break;
  default:
    Sc=0;
    break;
  }
  ScPos=X+Sc;
  ScPos=max(Min,ScPos);
  ScPos=min(Max,ScPos);
  Sc=ScPos-X;
  X=ScPos;
  if (Sc==0) return;
  LENSC->SetPosition(X);
  ang=X;
  sprintf(buf,"%d deg",ang);
  LEN->SetText(buf);
  HeadWin->SetLenWid(ang,wid);
  HeadWin->Invalidate();
}

void LegendArrowDialog::EvVScroll(UINT scrollCode, UINT thumbPos, HWND hWndCtl)
{
  int Min,Max,Sc,X,ScPos;
  char buf[5];

  WIDSC->GetRange(Min,Max);
  X=10+(200-wid/1000);
  switch (scrollCode) {
  case SB_PAGEUP:
    Sc=-10;
    break;
  case SB_PAGEDOWN:
    Sc=10;
    break;
  case SB_LINEUP:
    Sc=-1;
    break;
  case SB_LINEDOWN:
    Sc=1;
    break;
  case SB_THUMBPOSITION:
    Sc=thumbPos-X;
    break;
  case SB_THUMBTRACK:
    Sc=thumbPos-X;
    break;
  default:
    Sc=0;
    break;
  }
  ScPos=X+Sc;
  ScPos=max(Min,ScPos);
  ScPos=min(Max,ScPos);
  Sc=ScPos-X;
  X=ScPos;
  if (Sc==0) return;
  WIDSC->SetPosition(X);
  wid=(10+(200-X))*1000;
  sprintf(buf,"%d%c",wid/100,'%');
  WID->SetText(buf);
  HeadWin->SetLenWid(ang,wid);
  HeadWin->Invalidate();
}




DEFINE_RESPONSE_TABLE1(LegendRectDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
  EV_CHILD_NOTIFY(IDDELETE,BN_CLICKED,DeleteClicked),
END_RESPONSE_TABLE;

LegendRectDialog::LegendRectDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
:TMyDialog(parent,resID) {
  Obj=obj;
  Id=id;
  Col=new ColorButton(this,IDLRCOLB);
  Col2=new ColorButton(this,IDLRCOLB2);
}

void LegendRectDialog::SetupWindow()
{
  char *title;
  int len;

  TMyDialog::SetupWindow();
  len=GetWindowTextLength();
  if ((title=(char *)memalloc(len+10))!=NULL) {
    GetWindowText(title,len+1);
    sprintf(title+len," %d",Id);
    SetCaption(title);
    memfree(title);
  }
  SetupItem(Id);
}

void LegendRectDialog::SetupItem(int id)
{
  int R,G,B;

  SetTextFromObjField(Handle,IDLRX1,Obj,id,"x1");
  SetTextFromObjField(Handle,IDLRY1,Obj,id,"y1");
  SetTextFromObjField(Handle,IDLRX2,Obj,id,"x2");
  SetTextFromObjField(Handle,IDLRY2,Obj,id,"y2");
  SetStyleFromObjField(Handle,IDLRSTYLE,Obj,id,"style");
  SetComboList(Handle,IDLRWIDTH,cblinewidth,CBLINEWIDTH);
  SetTextFromObjField(Handle,IDLRWIDTH,Obj,id,"width");
  getobj(Obj,"R",id,0,NULL,&R);
  getobj(Obj,"G",id,0,NULL,&G);
  getobj(Obj,"B",id,0,NULL,&B);
  Col->SetColor(R,G,B);
  SetToggleFromObjField(Handle,IDLRFILL,Obj,id,"fill");
  getobj(Obj,"R2",id,0,NULL,&R);
  getobj(Obj,"G2",id,0,NULL,&G);
  getobj(Obj,"B2",id,0,NULL,&B);
  Col2->SetColor(R,G,B);
  SetToggleFromObjField(Handle,IDLRFRAME,Obj,id,"frame");
}


void LegendRectDialog::CloseWindow(int retVal)
{
  int R,G,B;

  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  if (SetObjFieldFromText(Handle,IDLRX1,Obj,Id,"x1")) return;
  if (SetObjFieldFromText(Handle,IDLRY1,Obj,Id,"y1")) return;
  if (SetObjFieldFromText(Handle,IDLRX2,Obj,Id,"x2")) return;
  if (SetObjFieldFromText(Handle,IDLRY2,Obj,Id,"y2")) return;
  if (SetObjFieldFromStyle(Handle,IDLRSTYLE,Obj,Id,"style")) return;
  if (SetObjFieldFromText(Handle,IDLRWIDTH,Obj,Id,"width")) return;
  Col->GetColor(&R,&G,&B);
  if (putobj(Obj,"R",Id,&R)==-1) return;
  if (putobj(Obj,"G",Id,&G)==-1) return;
  if (putobj(Obj,"B",Id,&B)==-1) return;
  if (SetObjFieldFromToggle(Handle,IDLRFILL,Obj,Id,"fill")) return;
  Col2->GetColor(&R,&G,&B);
  if (putobj(Obj,"R2",Id,&R)==-1) return;
  if (putobj(Obj,"G2",Id,&G)==-1) return;
  if (putobj(Obj,"B2",Id,&B)==-1) return;
  if (SetObjFieldFromToggle(Handle,IDLRFRAME,Obj,Id,"frame")) return;
  TMyDialog::CloseWindow(retVal);
}


DEFINE_RESPONSE_TABLE1(LegendArcDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
  EV_CHILD_NOTIFY(IDDELETE,BN_CLICKED,DeleteClicked),
END_RESPONSE_TABLE;

LegendArcDialog::LegendArcDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
:TMyDialog(parent,resID) {
  Obj=obj;
  Id=id;
  Col=new ColorButton(this,IDLARCOLB);
}

void LegendArcDialog::SetupWindow()
{
  char *title;
  int len;

  TMyDialog::SetupWindow();
  len=GetWindowTextLength();
  if ((title=(char *)memalloc(len+10))!=NULL) {
    GetWindowText(title,len+1);
    sprintf(title+len," %d",Id);
    SetCaption(title);
    memfree(title);
  }
  SetupItem(Id);
}

void LegendArcDialog::SetupItem(int id)
{
  int R,G,B;

  SetTextFromObjField(Handle,IDLARX,Obj,id,"x");
  SetTextFromObjField(Handle,IDLARY,Obj,id,"y");
  SetTextFromObjField(Handle,IDLARX2,Obj,id,"rx");
  SetTextFromObjField(Handle,IDLARY2,Obj,id,"ry");
  SetStyleFromObjField(Handle,IDLARSTYLE,Obj,id,"style");
  SetComboList(Handle,IDLARWIDTH,cblinewidth,CBLINEWIDTH);
  SetTextFromObjField(Handle,IDLARWIDTH,Obj,id,"width");
  SetTextFromObjField(Handle,IDLARANGLE1,Obj,id,"angle1");
  SetTextFromObjField(Handle,IDLARANGLE2,Obj,id,"angle2");
  SetToggleFromObjField(Handle,IDLARPIE,Obj,id,"pieslice");
  getobj(Obj,"R",id,0,NULL,&R);
  getobj(Obj,"G",id,0,NULL,&G);
  getobj(Obj,"B",id,0,NULL,&B);
  Col->SetColor(R,G,B);
  SetToggleFromObjField(Handle,IDLARFILL,Obj,id,"fill");
}


void LegendArcDialog::CloseWindow(int retVal)
{
  int R,G,B;

  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  if (SetObjFieldFromText(Handle,IDLARX,Obj,Id,"x")) return;
  if (SetObjFieldFromText(Handle,IDLARY,Obj,Id,"y")) return;
  if (SetObjFieldFromText(Handle,IDLARX2,Obj,Id,"rx")) return;
  if (SetObjFieldFromText(Handle,IDLARY2,Obj,Id,"ry")) return;
  if (SetObjFieldFromStyle(Handle,IDLARSTYLE,Obj,Id,"style")) return;
  if (SetObjFieldFromText(Handle,IDLARWIDTH,Obj,Id,"width")) return;
  if (SetObjFieldFromText(Handle,IDLARANGLE1,Obj,Id,"angle1")) return;
  if (SetObjFieldFromText(Handle,IDLARANGLE2,Obj,Id,"angle2")) return;
  if (SetObjFieldFromToggle(Handle,IDLARPIE,Obj,Id,"pieslice")) return;
  Col->GetColor(&R,&G,&B);
  if (putobj(Obj,"R",Id,&R)==-1) return;
  if (putobj(Obj,"G",Id,&G)==-1) return;
  if (putobj(Obj,"B",Id,&B)==-1) return;
  if (SetObjFieldFromToggle(Handle,IDLARFILL,Obj,Id,"fill")) return;
  TMyDialog::CloseWindow(retVal);
}


DEFINE_RESPONSE_TABLE1(LegendMarkDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
  EV_CHILD_NOTIFY(IDDELETE,BN_CLICKED,DeleteClicked),
END_RESPONSE_TABLE;

LegendMarkDialog::LegendMarkDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
:TMyDialog(parent,resID) {
  Obj=obj;
  Id=id;
  Col1=new ColorButton(this,IDLMCOL1B);
  Col2=new ColorButton(this,IDLMCOL2B);
  Mark=new MarkComboBox(this,IDLMTYPE);
}

void LegendMarkDialog::SetupWindow()
{
  char *title;
  int len;

  TMyDialog::SetupWindow();
  len=GetWindowTextLength();
  if ((title=(char *)memalloc(len+10))!=NULL) {
    GetWindowText(title,len+1);
    sprintf(title+len," %d",Id);
    SetCaption(title);
    memfree(title);
  }
  SetupItem(Id);
}

void LegendMarkDialog::SetupItem(int id)
{
  int j,a;
  int R,G,B;

  SetTextFromObjField(Handle,IDLMX,Obj,id,"x");
  SetTextFromObjField(Handle,IDLMY,Obj,id,"y");
  SendDlgItemMessage(IDLMTYPE,CB_RESETCONTENT,0,0);
  for (j=-1;j<89;j++) {
    if (j==-1) {
      fwbuf[j+1][0]='-';
      fwbuf[j+1][1]='1';
    } else {
      fwbuf[j+1][0]=j/10+'0';
      fwbuf[j+1][1]=j%10+'0';
    }
    fwbuf[j+1][2]='\0';
    SendDlgItemMessage(IDLMTYPE,CB_INSERTSTRING,j+1,(LPARAM)(fwbuf[j+1]));
  }
  getobj(Obj,"type",id,0,NULL,&a);
  SendDlgItemMessage(IDLMTYPE,CB_SETCURSEL,a+1,0);
  SetComboList(Handle,IDLMSIZE,cbmarksize,CBMARKSIZE);
  SetTextFromObjField(Handle,IDLMSIZE,Obj,id,"size");
  SetStyleFromObjField(Handle,IDLMSTYLE,Obj,id,"style");
  SetComboList(Handle,IDLMWIDTH,cblinewidth,CBLINEWIDTH);
  SetTextFromObjField(Handle,IDLMWIDTH,Obj,id,"width");
  getobj(Obj,"R",id,0,NULL,&R);
  getobj(Obj,"G",id,0,NULL,&G);
  getobj(Obj,"B",id,0,NULL,&B);
  Col1->SetColor(R,G,B);
  getobj(Obj,"R2",id,0,NULL,&R);
  getobj(Obj,"G2",id,0,NULL,&G);
  getobj(Obj,"B2",id,0,NULL,&B);
  Col2->SetColor(R,G,B);
}


void LegendMarkDialog::CloseWindow(int retVal)
{
  int a;
  int R,G,B;

  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  if (SetObjFieldFromText(Handle,IDLMX,Obj,Id,"x")) return;
  if (SetObjFieldFromText(Handle,IDLMY,Obj,Id,"y")) return;
  a=SendDlgItemMessage(IDLMTYPE,CB_GETCURSEL,0,0);
  if (a!=CB_ERR) {
    a--;
    if (putobj(Obj,"type",Id,&a)==-1) return;
  }
  if (SetObjFieldFromText(Handle,IDLMSIZE,Obj,Id,"size")) return;
  if (SetObjFieldFromStyle(Handle,IDLMSTYLE,Obj,Id,"style")) return;
  if (SetObjFieldFromText(Handle,IDLMWIDTH,Obj,Id,"width")) return;
  Col1->GetColor(&R,&G,&B);
  if (putobj(Obj,"R",Id,&R)==-1) return;
  if (putobj(Obj,"G",Id,&G)==-1) return;
  if (putobj(Obj,"B",Id,&B)==-1) return;
  Col2->GetColor(&R,&G,&B);
  if (putobj(Obj,"R2",Id,&R)==-1) return;
  if (putobj(Obj,"G2",Id,&G)==-1) return;
  if (putobj(Obj,"B2",Id,&B)==-1) return;
  TMyDialog::CloseWindow(retVal);
}


DEFINE_RESPONSE_TABLE1(LegendTextDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
  EV_CHILD_NOTIFY(IDDELETE,BN_CLICKED,DeleteClicked),
END_RESPONSE_TABLE;

LegendTextDialog::LegendTextDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
:TMyDialog(parent,resID) {
  Obj=obj;
  Id=id;
  Col=new ColorButton(this,IDLTCOLB);
}

void LegendTextDialog::SetupWindow()
{
  char *title;
  int len;

  TMyDialog::SetupWindow();
  len=GetWindowTextLength();
  if ((title=(char *)memalloc(len+10))!=NULL) {
    GetWindowText(title,len+1);
    sprintf(title+len," %d",Id);
    SetCaption(title);
    memfree(title);
  }
  SetupItem(Id);
}

bool LegendTextDialog::EvInitDialog(HWND hWndFocus)
{
  char *text;

  TMyDialog::EvInitDialog(hWndFocus);
  getobj(Obj,"text",Id,0,NULL,&text);
  if (text==NULL) {
    ::SetFocus(GetDlgItem(IDLTTEXT));
    return FALSE;
  } else return TRUE;
}

void LegendTextDialog::SetupItem(int id)
{
  int j,selfont,R,G,B;
  struct fontmap *fcur;
  char *font;

#ifndef ENGLISH
  SendDlgItemMessage(IDLTGREEK,BM_SETCHECK,0,0);
#endif
  SetTextFromObjField(Handle,IDLTX,Obj,id,"x");
  SetTextFromObjField(Handle,IDLTY,Obj,id,"y");
  SetTextFromObjField(Handle,IDLTTEXT,Obj,id,"text");
  SetComboList(Handle,IDLTDIRECTION,cbdirection,CBDIRECTION);
  SetTextFromObjField(Handle,IDLTDIRECTION,Obj,id,"direction");
  SetTextFromObjField(Handle,IDLTSPACE,Obj,id,"space");
  SetComboList(Handle,IDLTPT,cbtextpt,CBTEXTPT);
  SetTextFromObjField(Handle,IDLTPT,Obj,id,"pt");
  SetTextFromObjField(Handle,IDLTSCRIPT,Obj,id,"script_size");
  getobj(Obj,"font",id,0,NULL,&font);
  SendDlgItemMessage(IDLTFONT,CB_RESETCONTENT,0,0);
  fcur=mwlocal->fontmaproot;
  j=0;
  selfont=-1;
  while (fcur!=NULL) {
    if (fcur->charset!=SHIFTJIS_CHARSET) {
      SendDlgItemMessage(IDLTFONT,CB_INSERTSTRING,-1,(LPARAM)fcur->fontalias);
      if ((font!=NULL) && (strcmp(font,fcur->fontalias)==0)) selfont=j;
      j++;
    }
    fcur=fcur->next;
  }
  if (selfont!=-1) SendDlgItemMessage(IDLTFONT,CB_SETCURSEL,selfont,0);
#ifndef ENGLISH
  getobj(Obj,"jfont",id,0,NULL,&font);
  SendDlgItemMessage(IDLTJFONT,CB_RESETCONTENT,0,0);
  fcur=mwlocal->fontmaproot;
  j=0;
  selfont=-1;
  while (fcur!=NULL) {
    if (fcur->charset==SHIFTJIS_CHARSET) {
      SendDlgItemMessage(IDLTJFONT,CB_INSERTSTRING,-1,(LPARAM)fcur->fontalias);
      if ((font!=NULL) && (strcmp(font,fcur->fontalias)==0)) selfont=j;
      j++;
    }
    fcur=fcur->next;
  }
  if (selfont!=-1) SendDlgItemMessage(IDLTJFONT,CB_SETCURSEL,selfont,0);
#endif
  getobj(Obj,"R",id,0,NULL,&R);
  getobj(Obj,"G",id,0,NULL,&G);
  getobj(Obj,"B",id,0,NULL,&B);
  Col->SetColor(R,G,B);
  SetToggleFromObjField(Handle,IDLTRAW,Obj,id,"raw");
}


void LegendTextDialog::CloseWindow(int retVal)
{
  int R,G,B;
  int len;
  char *buf;
  int i,j,k,a;
  unsigned int code;

  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  if (SetObjFieldFromText(Handle,IDLTX,Obj,Id,"x")) return;
  if (SetObjFieldFromText(Handle,IDLTY,Obj,Id,"y")) return;
#ifndef ENGLISH
  a=SendDlgItemMessage(IDLTGREEK,BM_GETCHECK,0,0);
#else
  a=FALSE;
#endif
  if (!a) {
    if (SetObjFieldFromText(Handle,IDLTTEXT,Obj,Id,"text")) return;
  } else {
    len=SendDlgItemMessage(IDLTTEXT,WM_GETTEXTLENGTH,0,0);
    if ((buf=(char *)memalloc(len+1))==NULL) return;
    GetDlgItemText(IDLTTEXT,buf,len+1);
    j=0;
    for (i=0;buf[i]!='\0';i++) {
      if (niskanji((unsigned char)buf[i])) {
        code=njms2jis(((unsigned char)buf[i] << 8)+(unsigned char)buf[i+1]);
        if ((code>>8)==0x26) {
          for (k=0;k<48;k++) if (greektable[k].jis==(code & 0xff)) break;
          if (k!=48) {
            buf[j]=greektable[k].symbol;
            j++;
          } else {
            buf[j]=buf[i];
            buf[j+1]=buf[i+1];
            j+=2;
          }
        } else {
          buf[j]=buf[i];
          buf[j+1]=buf[i+1];
          j+=2;
        }
        i++;
      } else {
        buf[j]=buf[i];
        j++;
      }
    }
    buf[j]='\0';
    if (sputobjfield(Obj,Id,"text",buf)!=0) {
      memfree(buf);
      return;
    }
    memfree(buf);
  }
  if (SetObjFieldFromText(Handle,IDLTDIRECTION,Obj,Id,"direction")) return;
  if (SetObjFieldFromText(Handle,IDLTSPACE,Obj,Id,"space")) return;
  if (SetObjFieldFromText(Handle,IDLTPT,Obj,Id,"pt")) return;
  if (SetObjFieldFromText(Handle,IDLTSCRIPT,Obj,Id,"script_size")) return;
  if (SetObjFieldFromText(Handle,IDLTFONT,Obj,Id,"font")) return;
#ifndef ENGLISH
  if (SetObjFieldFromText(Handle,IDLTJFONT,Obj,Id,"jfont")) return;
#endif
  Col->GetColor(&R,&G,&B);
  if (putobj(Obj,"R",Id,&R)==-1) return;
  if (putobj(Obj,"G",Id,&G)==-1) return;
  if (putobj(Obj,"B",Id,&B)==-1) return;
  if (SetObjFieldFromToggle(Handle,IDLTRAW,Obj,Id,"raw")) return;
  TMyDialog::CloseWindow(retVal);
}


LegendTextDefDialog::LegendTextDefDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
:TMyDialog(parent,resID) {
  Obj=obj;
  Id=id;
  Col=new ColorButton(this,IDLTCOLB);
}

void LegendTextDefDialog::SetupWindow()
{
  TMyDialog::SetupWindow();
  SetupItem(Id);
}

void LegendTextDefDialog::SetupItem(int id)
{
  int j,selfont,R,G,B;
  struct fontmap *fcur;
  char *font;

  SetComboList(Handle,IDLTDIRECTION,cbdirection,CBDIRECTION);
  SetTextFromObjField(Handle,IDLTDIRECTION,Obj,id,"direction");
  SetTextFromObjField(Handle,IDLTSPACE,Obj,id,"space");
  SetComboList(Handle,IDLTPT,cbtextpt,CBTEXTPT);
  SetTextFromObjField(Handle,IDLTPT,Obj,id,"pt");
  SetTextFromObjField(Handle,IDLTSCRIPT,Obj,id,"script_size");
  getobj(Obj,"font",id,0,NULL,&font);
  SendDlgItemMessage(IDLTFONT,CB_RESETCONTENT,0,0);
  fcur=mwlocal->fontmaproot;
  j=0;
  selfont=-1;
  while (fcur!=NULL) {
    if (fcur->charset!=SHIFTJIS_CHARSET) {
      SendDlgItemMessage(IDLTFONT,CB_INSERTSTRING,-1,(LPARAM)fcur->fontalias);
      if (strcmp(font,fcur->fontalias)==0) selfont=j;
      j++;
    }
    fcur=fcur->next;
  }
  if (selfont!=-1) SendDlgItemMessage(IDLTFONT,CB_SETCURSEL,selfont,0);
#ifndef ENGLISH
  getobj(Obj,"jfont",id,0,NULL,&font);
  SendDlgItemMessage(IDLTJFONT,CB_RESETCONTENT,0,0);
  fcur=mwlocal->fontmaproot;
  j=0;
  selfont=-1;
  while (fcur!=NULL) {
    if (fcur->charset==SHIFTJIS_CHARSET) {
      SendDlgItemMessage(IDLTJFONT,CB_INSERTSTRING,-1,(LPARAM)fcur->fontalias);
      if (strcmp(font,fcur->fontalias)==0) selfont=j;
      j++;
    }
    fcur=fcur->next;
  }
  if (selfont!=-1) SendDlgItemMessage(IDLTJFONT,CB_SETCURSEL,selfont,0);
#endif
  getobj(Obj,"R",id,0,NULL,&R);
  getobj(Obj,"G",id,0,NULL,&G);
  getobj(Obj,"B",id,0,NULL,&B);
  Col->SetColor(R,G,B);
  SetToggleFromObjField(Handle,IDLTRAW,Obj,id,"raw");
}


void LegendTextDefDialog::CloseWindow(int retVal)
{
  int R,G,B;

  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  if (SetObjFieldFromText(Handle,IDLTDIRECTION,Obj,Id,"direction")) return;
  if (SetObjFieldFromText(Handle,IDLTSPACE,Obj,Id,"space")) return;
  if (SetObjFieldFromText(Handle,IDLTPT,Obj,Id,"pt")) return;
  if (SetObjFieldFromText(Handle,IDLTSCRIPT,Obj,Id,"script_size")) return;
  if (SetObjFieldFromText(Handle,IDLTFONT,Obj,Id,"font")) return;
#ifndef ENGLISH
  if (SetObjFieldFromText(Handle,IDLTJFONT,Obj,Id,"jfont")) return;
#endif
  Col->GetColor(&R,&G,&B);
  if (putobj(Obj,"R",Id,&R)==-1) return;
  if (putobj(Obj,"G",Id,&G)==-1) return;
  if (putobj(Obj,"B",Id,&B)==-1) return;
  if (SetObjFieldFromToggle(Handle,IDLTRAW,Obj,Id,"raw")) return;
  TMyDialog::CloseWindow(retVal);
}




char *legendlist[LEGENDNUM]=
{"line","curve","polygon","rectangle","arc","mark","text"};

DEFINE_RESPONSE_TABLE1(TLegendWindow, TSubWindow)
  EV_WM_SIZE,
  EV_WM_LBUTTONDOWN,
  EV_WM_RBUTTONDOWN,
  EV_WM_LBUTTONDBLCLK,
  EV_WM_KEYDOWN,
  EV_WM_PAINT,
  EV_COMMAND(CM_SELECTK,LegendSelect),
  EV_COMMAND(CM_DELETEK,LegendDelete),
  EV_COMMAND(CM_COPYK,LegendCopy),
  EV_COMMAND(CM_COPY2K,LegendCopy2),
  EV_COMMAND(CM_TOPK,LegendTop),
  EV_COMMAND(CM_LASTK,LegendLast),
  EV_COMMAND(CM_UPDATEK,LegendUpdate),
  EV_COMMAND(CM_HIDDENK,LegendHidden),
  EV_COMMAND(CM_UPK,LegendUp),
  EV_COMMAND(CM_DOWNK,LegendDown),
END_RESPONSE_TABLE;

TLegendWindow::TLegendWindow(TWindow *parent):TSubWindow(parent)
{
  menulocal.legendopen=TRUE;
  Attr.Style|=WS_VSCROLL|WS_HSCROLL;
  col1=GetSysColor(COLOR_WINDOWTEXT);
  col1h=GetSysColor(COLOR_GRAYTEXT);
  col2=(GetSysColor(COLOR_WINDOW) ^ GetSysColor(COLOR_HIGHLIGHT)) & RGB(255,255,255);
  brush1=CreateSolidBrush(col2);
  brush2=CreateSolidBrush(GetSysColor(COLOR_WINDOW));
  pen1=(HPEN)GetStockObject(NULL_PEN);
  pen2=(HPEN)GetStockObject(NULL_PEN);
  PopupMenu.AppendMenu(MF_STRING,CM_SELECTK,"Focus (SPC)");
  PopupMenu.AppendMenu(MF_STRING,CM_UPDATEK,"Update (CR)");
  PopupMenu.AppendMenu(MF_STRING,CM_DELETEK,"Delete (DEL)");
  PopupMenu.AppendMenu(MF_STRING,CM_TOPK,"Top (HOME)");
  PopupMenu.AppendMenu(MF_STRING,CM_UPK,"Up (SHIFT+UP)");
  PopupMenu.AppendMenu(MF_STRING,CM_DOWNK,"Down (SHIFT+DOWN)");
  PopupMenu.AppendMenu(MF_STRING,CM_LASTK,"Last (END)");
  PopupMenu.AppendMenu(MF_STRING,CM_COPYK,"Duplicate (INS)");
  PopupMenu.AppendMenu(MF_STRING,CM_COPY2K,"Duplicate behind (SHIFT+INS)");
  PopupMenu.AppendMenu(MF_STRING,CM_HIDDENK,"Hide (TAB)");
}

void TLegendWindow::SetupWindow()
{
  int i;

  TSubWindow::SetupWindow();
  select=-1;
  Scroller = new TScroller(this,1,1,0,0);
  Scroller->XLine=NgraphApp->FWidth;
  Scroller->YLine=NgraphApp->FHeight;
  Scroller->TrackMode=FALSE;
  for (i=0;i<LEGENDNUM;i++) {
    obj[i]=chkobject(legendlist[i]);
    legend[i]=chkobjlastinst(obj[i]);
  }
  ChangeLegendNum();
}


TLegendWindow::~TLegendWindow()
{
  menulocal.legendopen=FALSE;
  ::DeleteObject(brush1);
  ::DeleteObject(brush2);
  ::DeleteObject(pen1);
  ::DeleteObject(pen2);
  NgraphApp->DeleteLegendWindow();
}

void TLegendWindow::EvSize(UINT sizeType, TSize& size)
{
  TSubWindow::EvSize(sizeType,size);
  ChangeLegendNum();
}

void TLegendWindow::EvLButtonDown(UINT,TPoint& point)
{
  int sel;
  TClientDC dc(Handle);
  TPoint po;

  if (menulock || globallock) return;
  sel=(Scroller->YPos+point.y)/NgraphApp->FHeight-1;
  if (sel!=select) {
	po.x=Scroller->XPos;
	po.y=Scroller->YPos;
	dc.SetWindowOrg(po);
	SelectLegend(dc,sel);
  }
}

void TLegendWindow::EvRButtonDown(UINT modKeys,TPoint& point)
{
  int sel;
  TClientDC dc(Handle);
  TPoint po;
  TRect rect;
  TPoint point2;
  int i,num;

  if (menulock || globallock) return;
  sel=(Scroller->YPos+point.y)/NgraphApp->FHeight-1;
  if (sel!=select) {
	po.x=Scroller->XPos;
	po.y=Scroller->YPos;
	dc.SetWindowOrg(po);
	SelectLegend(dc,sel);
  }
  sel=select;
  num=0;
  for (i=0;i<LEGENDNUM;i++) num+=legend[i]+1;
  if ((sel>=0) && (sel<=num)) {
    PopupMenu.EnableMenuItem(CM_SELECTK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_DELETEK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_UPDATEK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_TOPK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_UPK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_DOWNK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_LASTK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_COPYK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_COPY2K,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_HIDDENK,MF_ENABLED);
  } else {
    PopupMenu.EnableMenuItem(CM_SELECTK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_DELETEK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_UPDATEK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_TOPK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_UPK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_DOWNK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_LASTK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_COPYK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_COPY2K,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_HIDDENK,MF_GRAYED);
  }
  GetWindowRect(rect);
  point2.x=point.x+rect.left;
  point2.y=point.y+rect.top;
  PopupMenu.TrackPopupMenu(TPM_LEFTBUTTON | TPM_RIGHTBUTTON,point2,0,Handle);
}

void TLegendWindow::EvLButtonDblClk(UINT,TPoint& point)
{
  if (menulock || globallock) return;
  LegendUpdate();
}

void TLegendWindow::EvKeyDown(UINT key,UINT repeatCount,UINT flags)
{
  int sel,num,i;
  TClientDC dc(Handle);
  TPoint po;
  TRect rect;
  int h,h1,h2;
  TPoint point;

  if (menulock || globallock) return;
  sel=select;
  switch (key) {
  case VK_DELETE:
    LegendDelete();
	break;
  case VK_INSERT:
    if (GetAsyncKeyState(VK_SHIFT) & 0x8000) LegendCopy2();
    else LegendCopy();
	break;
  case VK_HOME:
    LegendTop();
	break;
  case VK_END:
    LegendLast();
	break;
  case VK_RETURN:
    LegendUpdate();
    break;
  case VK_SPACE:
    LegendSelect();
    break;
  case VK_TAB:
    LegendHidden();
    break;
  case VK_DOWN:
    if (GetAsyncKeyState(VK_SHIFT) & 0x8000) LegendDown();
    else {
      num=0;
      for (i=0;i<LEGENDNUM;i++) num+=legend[i]+1;
      num--;
      if ((sel==-1) && (num>=0)) sel=0;
      else if (sel<num) sel++;
      GetClientRect(rect);
      h=(rect.bottom-rect.top)/NgraphApp->FHeight;
      h1=Scroller->YPos/NgraphApp->FHeight-1;
      h2=Scroller->YPos/NgraphApp->FHeight-1+h;
      if (sel<h1) {
        Scroller->YPos=(sel+1)*NgraphApp->FHeight;
        Invalidate();
      } else if (sel>=h2) {
        Scroller->YPos=(sel+2-h)*NgraphApp->FHeight;
        Invalidate();
      }
      if (sel!=select) {
        po.x=Scroller->XPos;
        po.y=Scroller->YPos;
        dc.SetWindowOrg(po);
        SelectLegend(dc,sel);
      }
    }
	break;
  case VK_UP:
    if (GetAsyncKeyState(VK_SHIFT) & 0x8000) LegendUp();
    else {
      num=0;
      for (i=0;i<LEGENDNUM;i++) num+=legend[i]+1;
      num--;
      if ((sel==-1) && (num>=0)) sel=num;
      else if (sel>0) sel--;
      GetClientRect(rect);
      h=(rect.bottom-rect.top)/NgraphApp->FHeight;
      h1=Scroller->YPos/NgraphApp->FHeight-1;
      h2=Scroller->YPos/NgraphApp->FHeight-1+h;
      if (sel<h1) {
        Scroller->YPos=(sel+1)*NgraphApp->FHeight;
        Invalidate();
      } else if (sel>=h2) {
        Scroller->YPos=(sel+2-h)*NgraphApp->FHeight;
        Invalidate();
      }
      if (sel!=select) {
        po.x=Scroller->XPos;
        po.y=Scroller->YPos;
        dc.SetWindowOrg(po);
        SelectLegend(dc,sel);
      }
    }
	break;
  case 'P':
    if (GetAsyncKeyState(VK_CONTROL) & 0x8000) {
      sel=select;
      num=0;
      for (i=0;i<LEGENDNUM;i++) num+=legend[i]+1;
      if ((sel>=0) && (sel<=num)) {
        PopupMenu.EnableMenuItem(CM_SELECTK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_DELETEK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_UPDATEK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_TOPK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_LASTK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_COPYK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_COPY2K,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_HIDDENK,MF_ENABLED);
      } else {
        PopupMenu.EnableMenuItem(CM_SELECTK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_DELETEK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_UPDATEK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_TOPK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_LASTK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_COPYK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_COPY2K,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_HIDDENK,MF_GRAYED);
      }
      GetWindowRect(rect);
      point.x=rect.left;
      point.y=rect.top;
      PopupMenu.TrackPopupMenu(TPM_LEFTBUTTON | TPM_RIGHTBUTTON,point,0,Handle);
    }
    break;
  default:
    NgraphApp->EvKeyDown2(key,repeatCount,flags);
    break;
  }
  TSubWindow::EvKeyDown(key,repeatCount,flags);
}

void TLegendWindow::LegendDelete()
{
  int sel,i,num;

  if (menulock || globallock) return;
  NgraphApp->UnFocus();
  sel=select;
  num=0;
  for (i=0;i<LEGENDNUM;i++) {
    if ((sel>=num) && (sel<=num+legend[i])) {
      delobj(obj[i],sel-num);
      Update(TRUE);
      NgraphApp->Changed=TRUE;
      break;
    }
    num+=legend[i]+1;
  }
}

void TLegendWindow::LegendCopy()
{
  int sel,i,num;
  int j,id,perm,type;
  char *field;

  if (menulock || globallock) return;
  sel=select;
  num=0;
  for (i=0;i<LEGENDNUM;i++) {
    if ((sel>=num) && (sel<=num+legend[i])) {
      if ((id=newobj(obj[i]))>=0) {
        for (j=0;j<chkobjfieldnum(obj[i]);j++) {
          field=chkobjfieldname(obj[i],j);
          perm=chkobjperm(obj[i],field);
          type=chkobjfieldtype(obj[i],field);
          if ((strcmp2(field,"name")!=0)
            && ((perm&NREAD)!=0) && ((perm&NWRITE)!=0) && (type<NVFUNC))
            copyobj(obj[i],field,id,sel-num);
        }
        legend[i]++;
        NgraphApp->Changed=TRUE;
      }
      select=sel;
      Update(FALSE);
      break;
    }
    num+=legend[i]+1;
  }
}

void TLegendWindow::LegendCopy2()
{
  int sel,i,num;
  int j,id,perm,type;
  char *field;

  if (menulock || globallock) return;
  sel=select;
  num=0;
  for (i=0;i<LEGENDNUM;i++) {
    if ((sel>=num) && (sel<=num+legend[i])) {
      if ((id=newobj(obj[i]))>=0) {
        for (j=0;j<chkobjfieldnum(obj[i]);j++) {
          field=chkobjfieldname(obj[i],j);
          perm=chkobjperm(obj[i],field);
          type=chkobjfieldtype(obj[i],field);
          if ((strcmp2(field,"name")!=0)
            && ((perm&NREAD)!=0) && ((perm&NWRITE)!=0) && (type<NVFUNC))
            copyobj(obj[i],field,id,sel-num);
        }
        legend[i]++;
        for (j=legend[i];j>sel-num+1;j--) moveupobj(obj[i],j);
        NgraphApp->Changed=TRUE;
      }
      select=sel;
      Update(FALSE);
      break;
    }
    num+=legend[i]+1;
  }
}

void TLegendWindow::LegendTop()
{
  int sel,i,num;

  if (menulock || globallock) return;
  sel=select;
  num=0;
  for (i=0;i<LEGENDNUM;i++) {
    if ((sel>=num) && (sel<=num+legend[i])) {
      movetopobj(obj[i],sel-num);
      select=num;
      Update(FALSE);
      NgraphApp->Changed=TRUE;
      break;
    }
    num+=legend[i]+1;
  }
}

void TLegendWindow::LegendUp()
{
  int sel,i,num;

  if (menulock || globallock) return;
  sel=select;
  num=0;
  for (i=0;i<LEGENDNUM;i++) {
    if ((sel>=num+1) && (sel<=num+legend[i])) {
      moveupobj(obj[i],sel-num);
      select=sel-1;
      Update(FALSE);
      NgraphApp->Changed=TRUE;
      break;
    }
    num+=legend[i]+1;
  }
}

void TLegendWindow::LegendDown()
{
  int sel,i,num;

  if (menulock || globallock) return;
  sel=select;
  num=0;
  for (i=0;i<LEGENDNUM;i++) {
    if ((sel>=num) && (sel<=num+legend[i]-1)) {
      movedownobj(obj[i],sel-num);
      select=sel+1;
      Update(FALSE);
      NgraphApp->Changed=TRUE;
      break;
    }
    num+=legend[i]+1;
  }
}

void TLegendWindow::LegendLast()
{
  int sel,i,num;

  if (menulock || globallock) return;
  sel=select;
  num=0;
  for (i=0;i<LEGENDNUM;i++) {
    if ((sel>=num) && (sel<=num+legend[i])) {
      movelastobj(obj[i],sel-num);
      select=num+legend[i];
      Update(FALSE);
      NgraphApp->Changed=TRUE;
      break;
    }
    num+=legend[i]+1;
  }
}

void TLegendWindow::LegendUpdate()
{
  int sel,i,num;
  LegendArrowDialog *dlgll;
  LegendCurveDialog *dlglc;
  LegendPolyDialog *dlglp;
  LegendRectDialog *dlglr;
  LegendArcDialog *dlglar;
  LegendMarkDialog *dlglm;
  LegendTextDialog *dlglt;
  int update,ret;

  if (menulock || globallock) return;
  NgraphApp->UnFocus();
  sel=select;
  num=0;
  update=FALSE;
  for (i=0;i<LEGENDNUM;i++) {
    if ((sel>=num) && (sel<=num+legend[i])) {
      ret=IDCANCEL;
      if (i==0) {
        dlgll=new LegendArrowDialog(this,DIALOG_LEGENDARROW,obj[i],sel-num);
        ret=dlgll->Execute();
        delete dlgll;
      } else if (i==1) {
        dlglc=new LegendCurveDialog(this,DIALOG_LEGENDCURVE,obj[i],sel-num);
        ret=dlglc->Execute();
        delete dlglc;
      } else if (i==2) {
        dlglp=new LegendPolyDialog(this,DIALOG_LEGENDPOLY,obj[i],sel-num);
        ret=dlglp->Execute();
        delete dlglp;
      } else if (i==3) {
        dlglr=new LegendRectDialog(this,DIALOG_LEGENDRECT,obj[i],sel-num);
        ret=dlglr->Execute();
        delete dlglr;
      } else if (i==4) {
        dlglar=new LegendArcDialog(this,DIALOG_LEGENDARC,obj[i],sel-num);
        ret=dlglar->Execute();
        delete dlglar;
      } else if (i==5) {
        dlglm=new LegendMarkDialog(this,DIALOG_LEGENDMARK,obj[i],sel-num);
        ret=dlglm->Execute();
        delete dlglm;
      } else if (i==6) {
        dlglt=new LegendTextDialog(this,DIALOG_LEGENDTEXT,obj[i],sel-num);
        ret=dlglt->Execute();
        delete dlglt;
      }
      if (ret==IDDELETE) {
        delobj(obj[i],sel-num);
        update=TRUE;
        select=-1;
      } else if (ret==IDOK) update=TRUE;
      if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
    }
    num+=legend[i]+1;
  }
  if (update) Update(FALSE);
}

void TLegendWindow::LegendSelect()
{
  int sel,i,num;

  if (menulock || globallock) return;
  sel=select;
  num=0;
  for (i=0;i<LEGENDNUM;i++) {
    if ((sel>=num) && (sel<=num+legend[i])) {
      NgraphApp->Focus(obj[i],sel-num);
      break;
    }
    num+=legend[i]+1;
  }
}

void TLegendWindow::LegendHidden()
{
  int sel,i,num;
  int hiden;

  if (menulock || globallock) return;
  NgraphApp->UnFocus();
  sel=select;
  num=0;
  for (i=0;i<LEGENDNUM;i++) {
    if ((sel>=num) && (sel<=num+legend[i])) {
      getobj(obj[i],"hidden",sel-num,0,NULL,&hiden);
      hiden=hiden?FALSE:TRUE;
      putobj(obj[i],"hidden",sel-num,&hiden);
      Update(FALSE);
      NgraphApp->Changed=TRUE;
      break;
    }
    num+=legend[i]+1;
  }
}

void TLegendWindow::Update(int clear)
{
  int i;

  if (clear) select=-1;
  for (i=0;i<LEGENDNUM;i++) legend[i]=chkobjlastinst(obj[i]);
  ChangeLegendNum();
  Invalidate();
}

void TLegendWindow::ChangeLegendNum()
{
  int i,num,width,height;

  num=1;
  for (i=0;i<LEGENDNUM;i++) num+=legend[i]+1;
  width=63*NgraphApp->FWidth;
  height=num*NgraphApp->FHeight;
  Scroller->SetRange(width,height);
  Scroller->SetPageSize();
}

void TLegendWindow::SelectLegend(HDC dc,int sel)
{
  int i,num,x,y,smode;

  if ((select!=-1) && (sel!=select)) {
    x=NgraphApp->FWidth;
    y=(select+2)*NgraphApp->FHeight;
    smode=::SetROP2(dc,R2_XORPEN);
    ::SelectObject(dc,brush1);
    ::SelectObject(dc,pen1);
    ::Rectangle(dc,x-NgraphApp->FWidth,y-NgraphApp->FHeight,x+63*NgraphApp->FWidth,y);
    ::SetROP2(dc,smode);
  }
  num=-1;
  for (i=0;i<LEGENDNUM;i++) num+=legend[i]+1;
  if ((sel>=0) && (sel<=num)) select=sel;
  else select=-1;
  if (select!=-1) {
    x=NgraphApp->FWidth;
    y=(select+2)*NgraphApp->FHeight;
    smode=::SetROP2(dc,R2_XORPEN);
    ::SelectObject(dc,brush1);
    ::SelectObject(dc,pen1);
    ::Rectangle(dc,x-NgraphApp->FWidth,y-NgraphApp->FHeight,x+63*NgraphApp->FWidth,y);
    ::SetROP2(dc,smode);
  }
}

void TLegendWindow::EvPaint()
{
  PAINTSTRUCT ps;
  HDC dc;
  RECT rc;
  int i,k,tot;
  char buf[256];
  RECT rect;
  int h,w,x,y,len,x1,y1,x2,y2,type,cx;
  struct narray *array;
  int *points;
  char *text;
  HFONT orgfont;
  int hiden;
  HBRUSH brush;
  int x0,y0;
  char *valstr;

  if (::GetUpdateRect(Handle,&rc,FALSE)) {
    dc=::BeginPaint(Handle,&ps);
    if (globallock) {
      ::EndPaint(Handle,&ps);
      return;
    }
    Scroller->BeginView(TDC(dc),TRect(rc));
    orgfont=(HFONT)SelectObject(dc,menulocal.menufont);
    SetTextAlign(dc,TA_BOTTOM | TA_LEFT);
    h=NgraphApp->FHeight;
    w=NgraphApp->FWidth;

    brush=CreateSolidBrush(GetSysColor(COLOR_MENU));
    rect.top=0;
    rect.bottom=h;
    rect.left=0;
    rect.right=63*w+w;
	FillRect(dc,&rect,brush);
    DeleteObject(brush);
    SetTextColor(dc,GetSysColor(COLOR_MENUTEXT));
    SetBkMode(dc,TRANSPARENT);
    x=w;
    y=h;
    rect.top=y-NgraphApp->FHeight;
    rect.bottom=y;
    rect.left=x;
    rect.right=x+6*w;
    len=sprintf(buf,"#");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=6*w;
    rect.left=x;
    rect.right=x+10*w;
    len=sprintf(buf,"object");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=10*w;
    rect.left=x;
    rect.right=x+25*w;
    len=sprintf(buf,"property");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=25*w;
    rect.left=x;
    rect.right=x+6*w;
    len=sprintf(buf,"     x");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=6*w;
    rect.left=x;
    rect.right=x+6*w;
    len=sprintf(buf,"     y");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=6*w;
    rect.left=x+w;
    rect.right=x+8*w;
    len=sprintf(buf,"^#");
    ::ExtTextOut(dc,x+w,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);

    tot=0;
    for (k=0;k<LEGENDNUM;k++) {
      legend[k]=chkobjlastinst(obj[k]);
      for (i=0;i<=legend[k];i++)
      if (((tot+i+2)*h>=(rc.top+Scroller->YPos))
      && ((tot+i-1)*h<=(rc.bottom+Scroller->YPos))) {
        x=w;
        y=(tot+i+2)*h;
        ::SelectObject(dc,pen2);
        ::SelectObject(dc,brush2);
		::Rectangle(dc,x-w,y-h,x+63*w,y);
        rect.top=y-NgraphApp->FHeight;
        rect.bottom=y;

        getobj(obj[k],"hidden",i,0,NULL,&hiden);
        if (hiden) SetTextColor(dc,col1h);
        else SetTextColor(dc,col1);

		rect.left=x;
		rect.right=x+6*w;
		getobj(obj[k],"id",i,0,NULL,&cx);
		len=sprintf(buf,"%d",cx);
		::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
		x+=6*w;

        rect.left=x;
		rect.right=x+10*w;
		len=sprintf(buf,"%s",legendlist[k]);
		::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
		x+=10*w;

		switch (k) {
        case 0:
          getobj(obj[k],"points",i,0,NULL,&array);
          points=(int *)arraydata(array);
          if (arraynum(array)<2) {
            x0=0;
            y0=0;
          } else {
            x0=points[0];
            y0=points[1];
          }
          sgetobjfield(obj[k],i,"arrow",NULL,&valstr,FALSE,FALSE,FALSE);
          rect.left=x;
          rect.right=x+25*w;
          len=sprintf(buf,"points:%-6d arrow:%s",arraynum(array)/2,valstr);
          memfree(valstr);
          ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
          x+=25*w;
          break;
        case 1:
        case 2:
          getobj(obj[k],"points",i,0,NULL,&array);
          points=(int *)arraydata(array);
          if (arraynum(array)<2) {
            x0=0;
            y0=0;
          } else {
            x0=points[0];
            y0=points[1];
          }
          rect.left=x;
          rect.right=x+25*w;
          len=sprintf(buf,"points:%-6d",arraynum(array)/2);
          ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
          x+=25*w;
          break;
        case 3:
          getobj(obj[k],"x1",i,0,NULL,&x1);
          getobj(obj[k],"y1",i,0,NULL,&y1);
          x0=x1;
          y0=y1;
          getobj(obj[k],"x2",i,0,NULL,&x2);
          getobj(obj[k],"y2",i,0,NULL,&y2);
          rect.left=x;
          rect.right=x+25*w;
          len=sprintf(buf,"w:%-6d h:%-6d",abs(x1-x2),abs(y1-y2));
          ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
          x+=25*w;
          break;
        case 4:
          getobj(obj[k],"x",i,0,NULL,&x1);
          getobj(obj[k],"y",i,0,NULL,&y1);
          x0=x1;
          y0=y1;
          getobj(obj[k],"rx",i,0,NULL,&x2);
          getobj(obj[k],"ry",i,0,NULL,&y2);
          rect.left=x;
          rect.right=x+25*w;
          len=sprintf(buf,"rx:%-6d ry:%-6d",x2,y2);
          ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
          x+=25*w;
          break;
        case 5:
          getobj(obj[k],"x",i,0,NULL,&x1);
          getobj(obj[k],"y",i,0,NULL,&y1);
          x0=x1;
          y0=y1;
          getobj(obj[k],"type",i,0,NULL,&type);
          rect.left=x;
          rect.right=x+25*w;
          len=sprintf(buf,"type:%-6d",type);
          ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
          x+=25*w;
          break;
        case 6:
          getobj(obj[k],"x",i,0,NULL,&x1);
          getobj(obj[k],"y",i,0,NULL,&y1);
          x0=x1;
          y0=y1;
          getobj(obj[k],"text",i,0,NULL,&text);
          rect.left=x;
          rect.right=x+25*w;
          len=sprintf(buf,"%.25s",text);
          ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
          x+=25*w;
          break;
        }
		rect.left=x;
		rect.right=x+6*w;
		len=sprintf(buf,"%6d",x0);
		::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
		x+=6*w;
		rect.left=x;
		rect.right=x+6*w;
		len=sprintf(buf,"%6d",y0);
		::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
		x+=6*w;
		rect.left=x+w;
		rect.right=x+8*w;
        getobj(obj[k],"oid",i,0,NULL,&cx);
		len=sprintf(buf,"^%d",cx);
		::ExtTextOut(dc,x+w,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
		x+=8*w;

      }
      tot+=legend[k]+1;
    }
    SelectLegend(dc,select);
    SetScrollPos(SB_HORZ,Scroller->XPos,TRUE);
    SetScrollPos(SB_VERT,Scroller->YPos,TRUE);
    SelectObject(dc,orgfont);
    ::EndPaint(Handle,&ps);
  }
}


void TMyWindow::CmLineDel()
{
  SelectDialog *dlg;
  struct narray array;
  struct objlist *obj;
  int i;
  int num,*data;

  if (menulock || globallock) return;
  if ((obj=chkobject("line"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,LegendLineCB,
                       (struct narray *)&array,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&array);
    data=(int *)arraydata(&array);
    for (i=num-1;i>=0;i--) {
      delobj(obj,data[i]);
      NgraphApp->Changed=TRUE;
    }
    if (pTLegendWindow!=NULL) ((TLegendWindow *)pTLegendWindow)->Update(TRUE);
  }
  arraydel(&array);
  delete dlg;
}

void TMyWindow::CmLineUpdate()
{
  SelectDialog *dlg;
  LegendArrowDialog *dlg2;
  struct narray array;
  struct objlist *obj;
  int i,j,ret;
  int *data,num;

  if (menulock || globallock) return;
  if ((obj=chkobject("line"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,LegendLineCB,
                       (struct narray *)&array,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&array);
    data=(int *)arraydata(&array);
    for (i=0;i<num;i++) {
      dlg2=new LegendArrowDialog(this,DIALOG_LEGENDARROW,obj,data[i]);
      if ((ret=dlg2->Execute())==IDDELETE) {
        delobj(obj,data[i]);
        for (j=i+1;j<num;j++) data[j]--;
      }
      if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
      delete dlg2;
    }
    if (pTLegendWindow!=NULL) ((TLegendWindow *)pTLegendWindow)->Update(TRUE);
  }
  arraydel(&array);
  delete dlg;
}


void TMyWindow::CmCurveDel()
{
  SelectDialog *dlg;
  struct narray array;
  struct objlist *obj;
  int i;
  int num,*data;

  if (menulock || globallock) return;
  if ((obj=chkobject("curve"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,LegendLineCB,
                       (struct narray *)&array,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&array);
    data=(int *)arraydata(&array);
    for (i=num-1;i>=0;i--) {
      delobj(obj,data[i]);
      NgraphApp->Changed=TRUE;
    }
    if (pTLegendWindow!=NULL) ((TLegendWindow *)pTLegendWindow)->Update(TRUE);
  }
  arraydel(&array);
  delete dlg;
}

void TMyWindow::CmCurveUpdate()
{
  SelectDialog *dlg;
  LegendCurveDialog *dlg2;
  struct narray array;
  struct objlist *obj;
  int i,j,ret;
  int *data,num;

  if (menulock || globallock) return;
  if ((obj=chkobject("curve"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,LegendLineCB,
                       (struct narray *)&array,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&array);
    data=(int *)arraydata(&array);
    for (i=0;i<num;i++) {
      dlg2=new LegendCurveDialog(this,DIALOG_LEGENDCURVE,obj,data[i]);
      if ((ret=dlg2->Execute())==IDDELETE) {
        delobj(obj,data[i]);
        for (j=i+1;j<num;j++) data[j]--;
      }
      if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
      delete dlg2;
    }
    if (pTLegendWindow!=NULL) ((TLegendWindow *)pTLegendWindow)->Update(TRUE);
  }
  arraydel(&array);
  delete dlg;
}

void TMyWindow::CmPolyDel()
{
  SelectDialog *dlg;
  struct narray array;
  struct objlist *obj;
  int i;
  int num,*data;

  if (menulock || globallock) return;
  if ((obj=chkobject("polygon"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,LegendLineCB,
                       (struct narray *)&array,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&array);
    data=(int *)arraydata(&array);
    for (i=num-1;i>=0;i--) {
      delobj(obj,data[i]);
      NgraphApp->Changed=TRUE;
    }
    if (pTLegendWindow!=NULL) ((TLegendWindow *)pTLegendWindow)->Update(TRUE);
  }
  arraydel(&array);
  delete dlg;
}

void TMyWindow::CmPolyUpdate()
{
  SelectDialog *dlg;
  LegendPolyDialog *dlg2;
  struct narray array;
  struct objlist *obj;
  int i,j,ret;
  int *data,num;

  if (menulock || globallock) return;
  if ((obj=chkobject("polygon"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,LegendLineCB,
                       (struct narray *)&array,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&array);
    data=(int *)arraydata(&array);
    for (i=0;i<num;i++) {
      dlg2=new LegendPolyDialog(this,DIALOG_LEGENDPOLY,obj,data[i]);
      if ((ret=dlg2->Execute())==IDDELETE) {
        delobj(obj,data[i]);
        for (j=i+1;j<num;j++) data[j]--;
      }
      if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
      delete dlg2;
    }
    if (pTLegendWindow!=NULL) ((TLegendWindow *)pTLegendWindow)->Update(TRUE);
  }
  arraydel(&array);
  delete dlg;
}

void TMyWindow::CmRectDel()
{
  SelectDialog *dlg;
  struct narray array;
  struct objlist *obj;
  int i;
  int num,*data;

  if (menulock || globallock) return;
  if ((obj=chkobject("rectangle"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,LegendRectCB,
                       (struct narray *)&array,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&array);
    data=(int *)arraydata(&array);
    for (i=num-1;i>=0;i--) {
      delobj(obj,data[i]);
      NgraphApp->Changed=TRUE;
    }
    if (pTLegendWindow!=NULL) ((TLegendWindow *)pTLegendWindow)->Update(TRUE);
  }
  arraydel(&array);
  delete dlg;
}

void TMyWindow::CmRectUpdate()
{
  SelectDialog *dlg;
  LegendRectDialog *dlg2;
  struct narray array;
  struct objlist *obj;
  int i,j,ret;
  int *data,num;

  if (menulock || globallock) return;
  if ((obj=chkobject("rectangle"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,LegendRectCB,
                       (struct narray *)&array,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&array);
    data=(int *)arraydata(&array);
    for (i=0;i<num;i++) {
      dlg2=new LegendRectDialog(this,DIALOG_LEGENDRECT,obj,data[i]);
      if ((ret=dlg2->Execute())==IDDELETE) {
        delobj(obj,data[i]);
        for (j=i+1;j<num;j++) data[j]--;
      }
      if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
      delete dlg2;
    }
    if (pTLegendWindow!=NULL) ((TLegendWindow *)pTLegendWindow)->Update(TRUE);
  }
  arraydel(&array);
  delete dlg;
}

void TMyWindow::CmArcDel()
{
  SelectDialog *dlg;
  struct narray array;
  struct objlist *obj;
  int i;
  int num,*data;

  if (menulock || globallock) return;
  if ((obj=chkobject("arc"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,LegendArcCB,
                       (struct narray *)&array,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&array);
    data=(int *)arraydata(&array);
    for (i=num-1;i>=0;i--) {
      delobj(obj,data[i]);
      NgraphApp->Changed=TRUE;
    }
    if (pTLegendWindow!=NULL) ((TLegendWindow *)pTLegendWindow)->Update(TRUE);
  }
  arraydel(&array);
  delete dlg;
}

void TMyWindow::CmArcUpdate()
{
  SelectDialog *dlg;
  LegendArcDialog *dlg2;
  struct narray array;
  struct objlist *obj;
  int i,j,ret;
  int *data,num;

  if (menulock || globallock) return;
  if ((obj=chkobject("arc"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,LegendArcCB,
                       (struct narray *)&array,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&array);
    data=(int *)arraydata(&array);
    for (i=0;i<num;i++) {
      dlg2=new LegendArcDialog(this,DIALOG_LEGENDARC,obj,data[i]);
      if ((ret=dlg2->Execute())==IDDELETE) {
        delobj(obj,data[i]);
        for (j=i+1;j<num;j++) data[j]--;
      }
      if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
      delete dlg2;
    }
    if (pTLegendWindow!=NULL) ((TLegendWindow *)pTLegendWindow)->Update(TRUE);
  }
  arraydel(&array);
  delete dlg;
}

void TMyWindow::CmMarkDel()
{
  SelectDialog *dlg;
  struct narray array;
  struct objlist *obj;
  int i;
  int num,*data;

  if (menulock || globallock) return;
  if ((obj=chkobject("mark"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,LegendArcCB,
                       (struct narray *)&array,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&array);
    data=(int *)arraydata(&array);
    for (i=num-1;i>=0;i--) {
      delobj(obj,data[i]);
      NgraphApp->Changed=TRUE;
    }
    if (pTLegendWindow!=NULL) ((TLegendWindow *)pTLegendWindow)->Update(TRUE);
  }
  arraydel(&array);
  delete dlg;
}

void TMyWindow::CmMarkUpdate()
{
  SelectDialog *dlg;
  LegendMarkDialog *dlg2;
  struct narray array;
  struct objlist *obj;
  int i,j,ret;
  int *data,num;

  if (menulock || globallock) return;
  if ((obj=chkobject("mark"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,LegendArcCB,
                       (struct narray *)&array,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&array);
    data=(int *)arraydata(&array);
    for (i=0;i<num;i++) {
      dlg2=new LegendMarkDialog(this,DIALOG_LEGENDMARK,obj,data[i]);
      if ((ret=dlg2->Execute())==IDDELETE) {
        delobj(obj,data[i]);
        for (j=i+1;j<num;j++) data[j]--;
      }
      if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
      delete dlg2;
    }
    if (pTLegendWindow!=NULL) ((TLegendWindow *)pTLegendWindow)->Update(TRUE);
  }
  arraydel(&array);
  delete dlg;
}

void TMyWindow::CmTextDel()
{
  SelectDialog *dlg;
  struct narray array;
  struct objlist *obj;
  int i;
  int num,*data;

  if (menulock || globallock) return;
  if ((obj=chkobject("text"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,LegendTextCB,
                       (struct narray *)&array,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&array);
    data=(int *)arraydata(&array);
    for (i=num-1;i>=0;i--) {
      delobj(obj,data[i]);
      NgraphApp->Changed=TRUE;
    }
    if (pTLegendWindow!=NULL) ((TLegendWindow *)pTLegendWindow)->Update(TRUE);
  }
  arraydel(&array);
  delete dlg;
}

void TMyWindow::CmTextUpdate()
{
  SelectDialog *dlg;
  LegendTextDialog *dlg2;
  struct narray array;
  struct objlist *obj;
  int i,j,ret;
  int *data,num;

  if (menulock || globallock) return;
  if ((obj=chkobject("text"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,LegendTextCB,
                       (struct narray *)&array,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&array);
    data=(int *)arraydata(&array);
    for (i=0;i<num;i++) {
      dlg2=new LegendTextDialog(this,DIALOG_LEGENDTEXT,obj,data[i]);
      if ((ret=dlg2->Execute())==IDDELETE) {
        delobj(obj,data[i]);
        for (j=i+1;j<num;j++) data[j]--;
      }
      if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
      delete dlg2;
    }
    if (pTLegendWindow!=NULL) ((TLegendWindow *)pTLegendWindow)->Update(TRUE);
  }
  arraydel(&array);
  delete dlg;
}

void TMyWindow::CmWindowLegend()
{
  TLegendWindow *win;
  TRect rect;
  int x,y;

  if (menulock || globallock) return;
  if (pTLegendWindow==NULL) {
    win=(TLegendWindow *)pTLegendWindow=new TLegendWindow();
    win->Frame=new TSubFrameWindow(this,"Legend Window",win,
    &menulocal.legendx,&menulocal.legendy,&menulocal.legendwidth,&menulocal.legendheight);
    win->Frame->SetIcon(GetApplication(),ICON_6);
    NgraphApp->Frame->GetWindowRect(rect);
    x=rect.left;
    y=rect.top;
    if (menulocal.legendwidth==CW_USEDEFAULT) win->Frame->Attr.W=FWidth*63+2*FWidth;
    else win->Frame->Attr.W=menulocal.legendwidth;
    if (menulocal.legendheight==CW_USEDEFAULT) win->Frame->Attr.H=FHeight*10;
    else win->Frame->Attr.H=menulocal.legendheight;
    if (menulocal.legendx==CW_USEDEFAULT) win->Frame->Attr.X=CW_USEDEFAULT;
    else {
      win->Frame->Attr.X=menulocal.legendx+x;
      if (win->Frame->Attr.X<0) win->Frame->Attr.X=0;
    }
    if (menulocal.legendy==CW_USEDEFAULT) win->Frame->Attr.Y=CW_USEDEFAULT;
    else {
      win->Frame->Attr.Y=menulocal.legendy+y;
      if (win->Frame->Attr.Y<0) win->Frame->Attr.Y=0;
    }
    win->Frame->Create();
  } else {
    win=(TLegendWindow *)pTLegendWindow;
    win->CloseWindow();
  }
}

void TMyWindow::CmOptionTextDef()
{
  struct objlist *obj;
  int id;

  if (menulock || globallock) return;
  if ((obj=chkobject("text"))==NULL) return;
  if ((id=newobj(obj))>=0) {
    if (LegendTextDefDialog(this,DIALOG_LEGENDTEXTDEF,obj,id).Execute()==IDOK) {
      if (CheckIniFile()) {
        exeobj(obj,"save_config",id,0,NULL);
      }
    }
    delobj(obj,id);
    NgraphApp->Update2();
  }
}

void TMyWindow::CeWindowLegend(TCommandEnabler& ce)
{
  if (pTLegendWindow==NULL) ce.SetCheck(TCommandEnabler::Unchecked);
  else ce.SetCheck(TCommandEnabler::Checked);
}

