/* 
 * 
 * This file is part of "Ngraph for Windows".
 * 
 * Copyright (C) 2015, Satoshi ISHIZAKA. isizaka@msa.biglobe.ne.jp
 * 
 * "Ngraph for Windows" is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * "Ngraph for Windows" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

/*
 *
 * winmenu.cpp
 *
 */

#include <owl\pch.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

extern "C" {
#include "ngraph.h"
#include "object.h"
#include "ioutil.h"
#include "nstring.h"
#include "config.h"
#include "winprt.h"
#include "owinmenu.h"
}

#include "bitmap.rh"
#include "winmenu.rh"
#include "winmenu.h"
#include "windialg.h"
#include "wincommn.h"
#include "winview.h"
#include "winfile.h"
#include "winaxis.h"
#include "winlgnd.h"
#include "winmerge.h"

#define DOCDIR "/doc"
#define HELP "ngraph.htm"

int menulock=FALSE;
HICON AboutIcon;
#define ICONSIZE 48

void TSubWindow::Activate()
{
  NgraphApp->ActiveWindow=this;
}

DEFINE_RESPONSE_TABLE1(TSubFrameWindow, TDecoratedFrame)
  EV_WM_MOVE,
  EV_WM_SIZE,
  EV_WM_ACTIVATE,
  EV_WM_DESTROY,
END_RESPONSE_TABLE;

void TSubFrameWindow::WindowShift(int dx,int dy)
{
  TRect rect;

  GetWindowRect(rect);
  MoveWindow(rect.left+dx,rect.top+dy,rect.right-rect.left,rect.bottom-rect.top,TRUE);
}

int moveflag=FALSE;

void TSubFrameWindow::EvMove(TPoint &clientOrigin)
{
  TRect rect,frect;
  int dx,dy;
  int shift;

  TDecoratedFrame::EvMove(clientOrigin);
  shift=(GetAsyncKeyState(VK_CONTROL) & 0x8000)
     || (GetAsyncKeyState(VK_SHIFT) & 0x8000);
  if (menulocal.movechild) shift=!shift;

  if (!moveflag && shift && (framex!=0) && (framey!=0) && (this==NgraphApp->Frame)) {
    moveflag=TRUE;
    GetWindowRect(rect);
    dx=rect.left-framex;
    dy=rect.top-framey;
    if ((NgraphApp->pTFileWindow!=NULL) && (NgraphApp->pTFileWindow->Frame!=this))
      NgraphApp->pTFileWindow->Frame->WindowShift(dx,dy);
    if ((NgraphApp->pTAxisWindow!=NULL) && (NgraphApp->pTAxisWindow->Frame!=this))
      NgraphApp->pTAxisWindow->Frame->WindowShift(dx,dy);
    if ((NgraphApp->pTLegendWindow!=NULL) && (NgraphApp->pTLegendWindow->Frame!=this))
      NgraphApp->pTLegendWindow->Frame->WindowShift(dx,dy);
    if ((NgraphApp->pTMergeWindow!=NULL) && (NgraphApp->pTMergeWindow->Frame!=this))
      NgraphApp->pTMergeWindow->Frame->WindowShift(dx,dy);
    if ((NgraphApp->pTCoordWindow!=NULL) && (NgraphApp->pTCoordWindow->Frame!=this))
      NgraphApp->pTCoordWindow->Frame->WindowShift(dx,dy);
    if ((NgraphApp->pTDialogWindow!=NULL) && (NgraphApp->pTDialogWindow->Frame!=this))
      NgraphApp->pTDialogWindow->Frame->WindowShift(dx,dy);
    moveflag=FALSE;
  }
  if ((X!=NULL) && (Y!=NULL)) {
    GetWindowRect(rect);
    NgraphApp->Frame->GetWindowRect(frect);
    *X=rect.left-frect.left;
    *Y=rect.top-frect.top;
  }
  GetWindowRect(frect);
  framex=frect.left;
  framey=frect.top;
}

void TSubFrameWindow::EvSize(UINT sizeType, TSize& size)
{
  TRect rect;

  TDecoratedFrame::EvSize(sizeType,size);
  if ((Width!=NULL) && (Height!=NULL)) {
    GetWindowRect(rect);
    *Width=rect.right-rect.left;
    *Height=rect.bottom-rect.top;
  }
}

void TSubFrameWindow::EvDestroy()
{
  TRect rect,frect;

  if ((X!=NULL) && (Y!=NULL)) {
    GetWindowRect(rect);
    NgraphApp->Frame->GetWindowRect(frect);
    *X=rect.left-frect.left;
    *Y=rect.top-frect.top;
  }
  TDecoratedFrame::EvDestroy();
}

void TSubFrameWindow::EvActivate(uint active,bool minimized,THandle hWndOther)
{
  TFrameWindow::EvActivate(active,minimized,hWndOther);
  client->Activate();
}

class AbWinStatic : public TStatic {
  public:
    AbWinStatic(TWindow *parent,int resId) : TStatic(parent,resId) {};
    ~AbWinStatic() {};
  protected:
    void EvPaint();
    DECLARE_RESPONSE_TABLE(AbWinStatic);
};

DEFINE_RESPONSE_TABLE1(AbWinStatic,TStatic)
  EV_WM_PAINT,
END_RESPONSE_TABLE;

void AbWinStatic::EvPaint()
{
  PAINTSTRUCT ps;
  HDC dc;
  HFONT sfont;
  HBRUSH brush;
  RECT rect,rect2;
  HFONT font,font2;
  LOGFONT IDFont;
  int i,x,y,x2,y2,delta,spc,spc2;

  dc=::BeginPaint(Handle,&ps);
  ::GetClientRect(Handle,&rect);
  IDFont.lfHeight=(rect.bottom-rect.top)*6/10;
  spc=(rect.bottom-rect.top)*6/10/20;
  IDFont.lfWidth=0;
  IDFont.lfEscapement=0;
  IDFont.lfOrientation=0;
  IDFont.lfUnderline=0;
  IDFont.lfStrikeOut=0;
  IDFont.lfWeight=0;
  IDFont.lfItalic=0;
  IDFont.lfPitchAndFamily=0;
  IDFont.lfCharSet=ANSI_CHARSET;
  IDFont.lfOutPrecision=0;
  IDFont.lfClipPrecision=0;
  IDFont.lfQuality=PROOF_QUALITY;
  strcpy(IDFont.lfFaceName,"Arial Bold Italic");
  font=(HFONT)CreateFontIndirect(&IDFont);
  IDFont.lfHeight=(rect.bottom-rect.top)*3/10;
  spc2=(rect.bottom-rect.top)*3/10/20;
  strcpy(IDFont.lfFaceName,"Times New Roman Italic");
  font2=(HFONT)CreateFontIndirect(&IDFont);
  x=rect.left+(rect.right-rect.left)*3/20;
  y=rect.top+(rect.bottom-rect.top)*6/10;
  x2=rect.left+(rect.right-rect.left)*5/10;
  y2=rect.bottom-(rect.bottom-rect.top)*1/20;
  delta=(rect.bottom-rect.top)*2/30;

  for (i=0;i<256;i++) {
    rect2.left=rect.left;
    rect2.right=rect.right;
    rect2.top=rect.top+(rect.bottom-rect.top)*i/256;
    rect2.bottom=rect.top+(rect.bottom-rect.top)*(i+1)/256;
    brush=CreateSolidBrush(RGB(31+i*96/255,31+i*96/255,95+i*160/255));
    FillRect(dc,&rect2,brush);
    DeleteObject(brush);
  }
  brush=CreateSolidBrush(RGB(255,255,255));
  rect2.left=rect.left;
  rect2.top=rect.bottom-1;
  rect2.right=rect.right;
  rect2.bottom=rect.bottom;
  FillRect(dc,&rect2,brush);
  rect2.left=rect.right-1;
  rect2.top=rect.top;
  rect2.right=rect.right;
  rect2.bottom=rect.bottom;
  FillRect(dc,&rect2,brush);
  DeleteObject(brush);
  brush=CreateSolidBrush(RGB(0,0,0));
  rect2.left=rect.left;
  rect2.top=rect.top;
  rect2.right=rect.left+1;
  rect2.bottom=rect.bottom-1;
  FillRect(dc,&rect2,brush);
  rect2.left=rect.left;
  rect2.top=rect.top;
  rect2.right=rect.right-1;
  rect2.bottom=rect.top+1;
  FillRect(dc,&rect2,brush);
  DeleteObject(brush);
  SetBkMode(dc,TRANSPARENT);
  sfont=(HFONT)SelectObject(dc,font);
  SetTextAlign(dc,TA_BOTTOM | TA_LEFT);
  SetTextCharacterExtra(dc,-spc);
  SetTextColor(dc,RGB(0,0,0));
  TextOut(dc,x+delta,y+delta,"Ngraph",6);
  SetTextColor(dc,RGB(0,255,255));
  TextOut(dc,x,y,"Ngraph",6);
  SelectObject(dc,font2);
  SetTextCharacterExtra(dc,-spc2);
  SetTextColor(dc,RGB(255,255,255));
  TextOut(dc,x2,y2,"for Windows",11);
  SelectObject(dc,sfont);

  DeleteObject(font);
  DeleteObject(font2);
  ::EndPaint(Handle,&ps);
}

class AbIconStatic : public TStatic {
  public:
    AbIconStatic::AbIconStatic(TWindow *parent,int resId) : TStatic(parent,resId) {};
    ~AbIconStatic() {};
  protected:
    void EvPaint();
    DECLARE_RESPONSE_TABLE(AbIconStatic);
};

DEFINE_RESPONSE_TABLE1(AbIconStatic,TStatic)
  EV_WM_PAINT,
END_RESPONSE_TABLE;

void AbIconStatic::EvPaint()
{
  PAINTSTRUCT ps;
  HDC dc;
  RECT rect;
  HBRUSH brush;

  dc=::BeginPaint(Handle,&ps);
  ::GetClientRect(Handle,&rect);
  brush=CreateSolidBrush(RGB(0,128,128));
  rect.left=0;
  rect.top=0;
  rect.right=50;
  rect.bottom=50;
  FillRect(dc,&rect,brush);
  DeleteObject(brush);
  ::DrawIconEx(dc,1,1,AboutIcon,ICONSIZE,ICONSIZE,0,NULL,DI_NORMAL);
  ::EndPaint(Handle,&ps);
}

DEFINE_RESPONSE_TABLE1(TMyDialog,TDialog)
END_RESPONSE_TABLE;

class AboutDialog  : public TMyDialog {
  public:
    AboutDialog(TWindow *parent,TResId resID) :TMyDialog(parent,resID) {
      Icon=new AbIconStatic(this,IDABICON);
      Win=new AbWinStatic(this,IDABWIN);
      Version=new TStatic(this,IDABVERSION,20);
      CopyRight=new TStatic(this,IDABCOPYRIGHT,100);
      Mail=new TStatic(this,IDABMAIL,40);
      Web=new TStatic(this,IDABWEB,40);
    }
    ~AboutDialog() {
      delete Win;
      delete Version;
      delete CopyRight;
      delete Mail;
    }
  protected:
    TStatic *Version,*CopyRight,*Mail,*Icon,*Web;
    AbWinStatic *Win;
    void SetupWindow() {
      struct objlist *obj;
      char *version,ver[20];
      char *mail,*web;
      char *copyright;

      TMyDialog::SetupWindow();
      if ((obj=chkobject("system"))==NULL) return;
      getobj(obj,"version",0,0,NULL,&version);
      sprintf(ver,"%s"GUIVERSION,version);
      Version->SetText(ver);
      getobj(obj,"copyright",0,0,NULL,&copyright);
      CopyRight->SetText(copyright);
      getobj(obj,"e-mail",0,0,NULL,&mail);
      Mail->SetText(mail);
      getobj(obj,"web",0,0,NULL,&web);
      Web->SetText(web);
    }
};

TMyApp::TMyApp(char *file):
TApplication()
{
  File=file;
}

TMyApp::~TMyApp()
{
}

void defaultwindowconfig()
{
  int w,h,x,y;
  RECT rect;
  TRect trect;

  NgraphApp->Frame->GetWindowRect(trect);
  x=trect.left;
  y=trect.top;
  ::GetWindowRect(GetDesktopWindow(),&rect);
  w=rect.right-rect.left;
  h=w/2*1.2;
  if (menulocal.fileopen) {
    if (menulocal.filewidth==CW_USEDEFAULT) menulocal.filewidth=w/4;
    if (menulocal.fileheight==CW_USEDEFAULT) menulocal.fileheight=h/4;
    if (menulocal.filex==CW_USEDEFAULT) menulocal.filex=-menulocal.filewidth-1;
    if (menulocal.filey==CW_USEDEFAULT) menulocal.filey=0;
    if (menulocal.filex+x+menulocal.filewidth>rect.right) menulocal.filex=rect.right-x-menulocal.filewidth;
    if (menulocal.filex+x<0) menulocal.filex=-x;
    if (menulocal.filey+y+menulocal.fileheight>rect.bottom) menulocal.filey=rect.bottom-y-menulocal.fileheight;
    if (menulocal.filey+y<0) menulocal.filey=-y;
  }
  if (menulocal.axisopen) {
    if (menulocal.axiswidth==CW_USEDEFAULT) menulocal.axiswidth=w/4;
    if (menulocal.axisheight==CW_USEDEFAULT) menulocal.axisheight=h/4;
    if (menulocal.axisx==CW_USEDEFAULT) menulocal.axisx=-menulocal.axiswidth-1;
    if (menulocal.axisy==CW_USEDEFAULT) menulocal.axisy=menulocal.fileheight+1;
    if (menulocal.axisx+x+menulocal.axiswidth>rect.right) menulocal.axisx=rect.right-x-menulocal.axiswidth;
    if (menulocal.axisx+x<0) menulocal.axisx=-x;
    if (menulocal.axisy+y+menulocal.axisheight>rect.bottom) menulocal.axisy=rect.bottom-y-menulocal.axisheight;
    if (menulocal.axisy+y<0) menulocal.axisy=-y;
  }
  if (menulocal.coordopen) {
    if (menulocal.coordwidth==CW_USEDEFAULT) menulocal.coordwidth=w/4;
    if (menulocal.coordheight==CW_USEDEFAULT) menulocal.coordheight=h/4;
    if (menulocal.coordx==CW_USEDEFAULT) menulocal.coordx=-menulocal.coordwidth-1;
    if (menulocal.coordy==CW_USEDEFAULT) menulocal.coordy=menulocal.fileheight+menulocal.axisheight+2;
    if (menulocal.coordx+x+menulocal.coordwidth>rect.right) menulocal.coordx=rect.right-x-menulocal.coordwidth;
    if (menulocal.coordx+x<0) menulocal.coordx=-x;
    if (menulocal.coordy+y+menulocal.coordheight>rect.bottom) menulocal.coordy=rect.bottom-y-menulocal.coordheight;
    if (menulocal.coordy+y<0) menulocal.coordy=-y;
  }
}

void TMyApp::InitInstance()
{
  int i;
  int num;
  TRect rect;
  char *ext;
  char **data;
  HMENU Menu;
  struct objlist *aobj;

  TApplication::InitInstance();

  num=arraynum(menulocal.ngpfilelist);
  data=(char **)arraydata(menulocal.ngpfilelist);
  Menu=NgraphApp->Frame->GetMenu();
  for (i=0;i<num;i++)
    InsertMenu(Menu,CM_GRAPH00,MF_BYCOMMAND|MF_STRING,CM_GRAPH00+i+1,data[i]);
  if (num>0) {
    DeleteMenu(Menu,CM_GRAPH00,MF_BYCOMMAND);
    InsertMenu(Menu,CM_GRAPH00+1,MF_BYCOMMAND|MF_SEPARATOR,CM_GRAPH00,NULL);
  }
  num=arraynum(menulocal.datafilelist);
  data=(char **)arraydata(menulocal.datafilelist);
  Menu=NgraphApp->Frame->GetMenu();
  for (i=0;i<num;i++)
    InsertMenu(Menu,CM_FILE00,MF_BYCOMMAND|MF_STRING,CM_FILE00+i+1,data[i]);
  if (num>0) {
    DeleteMenu(Menu,CM_FILE00,MF_BYCOMMAND);
    InsertMenu(Menu,CM_FILE00+1,MF_BYCOMMAND|MF_SEPARATOR,CM_FILE00,NULL);
  }
  NgraphApp->Frame->DrawMenuBar();

  if (!menulocal.statusb) GetMainWindow()->SendMessage(WM_COMMAND,IDW_STATUSBAR);
  defaultwindowconfig();
  if (menulocal.coordopen) NgraphApp->CmWindowCoord();
  if (menulocal.dialogopen) NgraphApp->CmWindowDialog();
  if (menulocal.mergeopen) NgraphApp->CmWindowMerge();
  if (menulocal.legendopen) NgraphApp->CmWindowLegend();
  if (menulocal.axisopen) NgraphApp->CmWindowAxis();
  if (menulocal.fileopen) NgraphApp->CmWindowFile();
  menulocal.hWnd=GetMainWindow()->Handle;

  if ((aobj=getobject("axis"))!=NULL) {
    for (i=0;i<=chkobjlastinst(aobj);i++) exeobj(aobj,"tight",i,0,NULL);
  }
  if ((aobj=getobject("axisgrid"))!=NULL) {
    for (i=0;i<=chkobjlastinst(aobj);i++) exeobj(aobj,"tight",i,0,NULL);
  }

  if (File!=NULL) {
    if ((ext=getextention(File))!=NULL) {
      if ((strcmp0(ext,"PRM")==0) || (strcmp0(ext,"prm")==0))
        NgraphApp->LoadPrmFile(File);
      else if ((strcmp0(ext,"NGP")==0) || (strcmp0(ext,"ngp")==0))
        LoadNgpFile(File,menulocal.ignorepath,menulocal.expand,menulocal.expanddir,FALSE,NULL);
    }
  }
  SetForegroundWindow(NgraphApp->Frame->Handle);
  GetMainWindow()->PostMessage(WM_COMMAND,CM_VIEWERDRAWB);
}

void TMyApp::InitMainWindow()
{
  TViewWindow *win;
  RECT rect;
  int x,y,w,h;

  NgraphApp=(TMyWindow *)win=new TViewWindow;
  win->pTViewWindow=win;
  win->Frame=new TSubFrameWindow(0,systemname,win,NULL,NULL,NULL,NULL);
  ::GetWindowRect(GetDesktopWindow(),&rect);
  x=rect.left;
  y=rect.top;
  w=rect.right-rect.left;
  h=rect.bottom-rect.top;
  if (menulocal.menux==CW_USEDEFAULT) menulocal.menux=x+w*3/8;
  if (menulocal.menuy==CW_USEDEFAULT) menulocal.menuy=y+h/8;
  if (menulocal.menuwidth==CW_USEDEFAULT) menulocal.menuwidth=w/2;
  if (menulocal.menuheight==CW_USEDEFAULT) menulocal.menuheight=w*1.2/2;
  win->Frame->Attr.X=menulocal.menux;
  win->Frame->Attr.Y=menulocal.menuy;
  win->Frame->Attr.W=menulocal.menuwidth;
  win->Frame->Attr.H=menulocal.menuheight;

  win->Frame->Attr.AccelTable=WINMENU;

  win->SB= new TStatusBar(win->Frame,TGadget::Recessed,TStatusBar::SizeGrip);
  win->Zoom=new TTextGadget(0,TTextGadget::Recessed,TTextGadget::Left,4);
  win->PointX=new TTextGadget(0,TTextGadget::Recessed,TTextGadget::Left,4);
  win->PointY=new TTextGadget(0,TTextGadget::Recessed,TTextGadget::Left,4);
  win->SB->Insert(*win->Zoom);
  win->SB->Insert(*win->PointX);
  win->SB->Insert(*win->PointY);
  win->Frame->Insert(*(win->SB),TDecoratedFrame::Bottom);

  if ((menulocal.controlb1f==3) || (menulocal.controlb1f==4)) 
    win->CB=new TControlBar(win->Frame,TControlBar::Vertical);
  else
    win->CB=new TControlBar(win->Frame,TControlBar::Horizontal);
  win->fileB=new TButtonGadget(BITMAP_FILEWIN,CM_WINDOWFILEB);
  win->fileB->SetNotchCorners(FALSE);
  win->CB->Insert(*win->fileB);
  win->axisB=new TButtonGadget(BITMAP_AXISWIN,CM_WINDOWAXISB);
  win->axisB->SetNotchCorners(FALSE);
  win->CB->Insert(*win->axisB);
  win->legendB=new TButtonGadget(BITMAP_LEGENDWIN,CM_WINDOWLEGENDB);
  win->legendB->SetNotchCorners(FALSE);
  win->CB->Insert(*win->legendB);
  win->mergeB=new TButtonGadget(BITMAP_MERGEWIN,CM_WINDOWMERGEB);
  win->mergeB->SetNotchCorners(FALSE);
  win->CB->Insert(*win->mergeB);
  win->coordB=new TButtonGadget(BITMAP_COORDWIN,CM_WINDOWCOORDINATEB);
  win->coordB->SetNotchCorners(FALSE);
  win->CB->Insert(*win->coordB);
  win->dialogB=new TButtonGadget(BITMAP_DIALOGWIN,CM_WINDOWDIALOGB);
  win->dialogB->SetNotchCorners(FALSE);
  win->CB->Insert(*win->dialogB);
  win->Separator1=new TSeparatorGadget(10);
  win->CB->Insert(*(win->Separator1));

  win->openfileB=new TButtonGadget(BITMAP_FILEOPEN,CM_FILEOPENN);
  win->openfileB->SetNotchCorners(FALSE);
  win->CB->Insert(*(win->openfileB));
  win->loadB=new TButtonGadget(BITMAP_LOAD,CM_GRAPHLOAD);
  win->loadB->SetNotchCorners(FALSE);
  win->CB->Insert(*(win->loadB));
  win->saveB=new TButtonGadget(BITMAP_SAVE,CM_GRAPHSAVE);
  win->saveB->SetNotchCorners(FALSE);
  win->CB->Insert(*(win->saveB));
  win->axisClearB=new TButtonGadget(BITMAP_SCALECLEAR,CM_AXISCLEAR);
  win->axisClearB->SetNotchCorners(FALSE);
  win->CB->Insert(*(win->axisClearB));
  win->drawB=new TButtonGadget(BITMAP_DRAW,CM_VIEWERDRAWB);
  win->drawB->SetNotchCorners(FALSE);
  win->CB->Insert(*(win->drawB));
  win->clearB=new TButtonGadget(BITMAP_CLEAR,CM_VIEWERCLEAR);
  win->clearB->SetNotchCorners(FALSE);
  win->CB->Insert(*(win->clearB));
  win->clipB=new TButtonGadget(BITMAP_CLIPBORD,CM_VIEWERCLIPBOARD);
  win->clipB->SetNotchCorners(FALSE);
  win->CB->Insert(*(win->clipB));
  win->printB=new TButtonGadget(BITMAP_PRINT,CM_PRINTPRINT);
  win->printB->SetNotchCorners(FALSE);
  win->CB->Insert(*(win->printB));
  win->exviewB=new TButtonGadget(BITMAP_EXVIEW,CM_PRINTVIEWER);
  win->exviewB->SetNotchCorners(FALSE);
  win->CB->Insert(*(win->exviewB));

  win->Separator2=new TSeparatorGadget(10);
  win->CB->Insert(*(win->Separator2));

  win->MathB=new TButtonGadget(BITMAP_MATH,CM_FILEMATHB);
  win->MathB->SetNotchCorners(FALSE);
  win->CB->Insert(*(win->MathB));
  win->ScaleUndoB=new TButtonGadget(BITMAP_SCALEUNDO,CM_SCALEUNDOB);
  win->ScaleUndoB->SetNotchCorners(FALSE);
  win->CB->Insert(*(win->ScaleUndoB));

  win->Separator3=new TSeparatorGadget(10);
  win->CB->Insert(*(win->Separator3));
  win->CB->SetHintMode(TGadgetWindow::EnterHints);

  win->interruptB=new TButtonGadget(BITMAP_INTERRUPT,CM_INTERRUPT);
  win->interruptB->SetNotchCorners(FALSE);
  win->CB->Insert(*(win->interruptB));

  if (menulocal.controlb1f==1)
    win->Frame->Insert(*(win->CB),TDecoratedFrame::Top);
  else if (menulocal.controlb1f==2)
    win->Frame->Insert(*(win->CB),TDecoratedFrame::Bottom);
  else if (menulocal.controlb1f==3)
    win->Frame->Insert(*(win->CB),TDecoratedFrame::Left);
  else if (menulocal.controlb1f==4)
    win->Frame->Insert(*(win->CB),TDecoratedFrame::Right);

  if ((menulocal.controlb2f==3) || (menulocal.controlb2f==4)) 
    win->CB2=new TControlBar(win->Frame,TControlBar::Vertical);
  else
    win->CB2=new TControlBar(win->Frame,TControlBar::Horizontal);
  win->Separator0=new TSeparatorGadget(24);
  win->CB2->Insert(*(win->Separator0));
  win->CB2->SetHintMode(TGadgetWindow::EnterHints);

  if (menulocal.controlb2f==1)
    win->Frame->Insert(*(win->CB2),TDecoratedFrame::Top);
  else if (menulocal.controlb2f==2)
    win->Frame->Insert(*(win->CB2),TDecoratedFrame::Bottom);
  else if (menulocal.controlb2f==3)
    win->Frame->Insert(*(win->CB2),TDecoratedFrame::Left);
  else if (menulocal.controlb2f==4)
    win->Frame->Insert(*(win->CB2),TDecoratedFrame::Right);

  win->Frame->AssignMenu(WINMENU);
  SetMainWindow(win->Frame);
  GetMainWindow()->SetIcon(this,ICON_1);
  GetMainWindow()->SetIconSm(this,ICON_1);
  AboutIcon=(HICON)::LoadImage(hInstance,MAKEINTRESOURCE(ICON_1),IMAGE_ICON,
                               ICONSIZE,ICONSIZE,LR_DEFAULTCOLOR);
}

bool TMyApp::IdleAction(long idleCount)
{
  NgraphApp->Interrupt=FALSE;
  if (!menulock && !globallock) eventloop();
  return TApplication::IdleAction(idleCount);
}

class TMyWindow *NgraphApp;

DEFINE_RESPONSE_TABLE1(TMyWindow, TSubWindow)
  EV_WM_DROPFILES,
  EV_COMMAND(CM_SYSTEMSHELL,CmSystemShell),
  EV_COMMAND(CM_FILENEWN,CmFileNew),
  EV_COMMAND(CM_FILEOPENN,CmFileOpen),
  EV_COMMAND(CM_FILECLOSEN,CmFileClose),
  EV_COMMAND(CM_FILEUPDATE,CmFileUpdate),
  EV_COMMAND(CM_FILEEDIT,CmFileEdit),
  EV_COMMAND(CM_FILEDIRECTORY,CmFileDirectory),
  EV_COMMAND_AND_ID(CM_FILE00+1,CmFileHistory),
  EV_COMMAND_AND_ID(CM_FILE00+2,CmFileHistory),
  EV_COMMAND_AND_ID(CM_FILE00+3,CmFileHistory),
  EV_COMMAND_AND_ID(CM_FILE00+4,CmFileHistory),
  EV_COMMAND_AND_ID(CM_FILE00+5,CmFileHistory),
  EV_COMMAND_AND_ID(CM_FILE00+6,CmFileHistory),
  EV_COMMAND_AND_ID(CM_FILE00+7,CmFileHistory),
  EV_COMMAND_AND_ID(CM_FILE00+8,CmFileHistory),
  EV_COMMAND_AND_ID(CM_FILE00+9,CmFileHistory),
  EV_COMMAND_AND_ID(CM_FILE00+10,CmFileHistory),
  EV_COMMAND(CM_AXISNEWFRAME,CmAxisNewFrame),
  EV_COMMAND(CM_AXISNEWSECTION,CmAxisNewSection),
  EV_COMMAND(CM_AXISNEWCROSS,CmAxisNewCross),
  EV_COMMAND(CM_AXISNEWSINGLE,CmAxisNewSingle),
  EV_COMMAND(CM_AXISDEL,CmAxisDel),
  EV_COMMAND(CM_AXISUPDATE,CmAxisUpdate),
  EV_COMMAND(CM_AXISZOOM,CmAxisZoom),
  EV_COMMAND(CM_AXISCLEAR,CmAxisClear),
  EV_COMMAND(CM_AXISGRIDNEW,CmAxisGridNew),
  EV_COMMAND(CM_AXISGRIDDEL,CmAxisGridDel),
  EV_COMMAND(CM_AXISGRIDUPDATE,CmAxisGridUpdate),
  EV_COMMAND(CM_LINEDEL,CmLineDel),
  EV_COMMAND(CM_LINEUPDATE,CmLineUpdate),
  EV_COMMAND(CM_CURVEDEL,CmCurveDel),
  EV_COMMAND(CM_CURVEUPDATE,CmCurveUpdate),
  EV_COMMAND(CM_POLYDEL,CmPolyDel),
  EV_COMMAND(CM_POLYUPDATE,CmPolyUpdate),
  EV_COMMAND(CM_RECTDEL,CmRectDel),
  EV_COMMAND(CM_RECTUPDATE,CmRectUpdate),
  EV_COMMAND(CM_ARCDEL,CmArcDel),
  EV_COMMAND(CM_ARCUPDATE,CmArcUpdate),
  EV_COMMAND(CM_MARKDEL,CmMarkDel),
  EV_COMMAND(CM_MARKUPDATE,CmMarkUpdate),
  EV_COMMAND(CM_TEXTDEL,CmTextDel),
  EV_COMMAND(CM_TEXTUPDATE,CmTextUpdate),
  EV_COMMAND(CM_GRAPHPAGE,CmGraphPage),
  EV_COMMAND(CM_GRAPHLOAD,CmGraphLoad),
  EV_COMMAND(CM_GRAPHSAVE,CmGraphSave),
  EV_COMMAND(CM_GRAPHOVERWRITE,CmGraphOverWrite),
  EV_COMMAND(CM_GRAPHNEWFRAME,CmGraphNewFrame),
  EV_COMMAND(CM_GRAPHNEWSECTION,CmGraphNewSection),
  EV_COMMAND(CM_GRAPHNEWCROSS,CmGraphNewCross),
  EV_COMMAND(CM_GRAPHALLCLEAR,CmGraphAllClear),
  EV_COMMAND(CM_GRAPHSWITCH,CmGraphSwitch),
  EV_COMMAND_AND_ID(CM_GRAPH00+1,CmGraphHistory),
  EV_COMMAND_AND_ID(CM_GRAPH00+2,CmGraphHistory),
  EV_COMMAND_AND_ID(CM_GRAPH00+3,CmGraphHistory),
  EV_COMMAND_AND_ID(CM_GRAPH00+4,CmGraphHistory),
  EV_COMMAND_AND_ID(CM_GRAPH00+5,CmGraphHistory),
  EV_COMMAND_AND_ID(CM_GRAPH00+6,CmGraphHistory),
  EV_COMMAND_AND_ID(CM_GRAPH00+7,CmGraphHistory),
  EV_COMMAND_AND_ID(CM_GRAPH00+8,CmGraphHistory),
  EV_COMMAND_AND_ID(CM_GRAPH00+9,CmGraphHistory),
  EV_COMMAND_AND_ID(CM_GRAPH00+10,CmGraphHistory),
  EV_COMMAND(CM_MERGEOPEN,CmMergeOpen),
  EV_COMMAND(CM_MERGECLOSE,CmMergeClose),
  EV_COMMAND(CM_MERGEUPDATE,CmMergeUpdate),
  EV_COMMAND(CM_VIEWERDRAW,CmViewerDraw),
  EV_COMMAND(CM_VIEWERCLEAR,CmViewerClear),
  EV_COMMAND(CM_VIEWERCLIPBOARD,CmViewerClipboard),
  EV_COMMAND(CM_VIEWERDRAWB,CmViewerDrawB),
  EV_COMMAND(CM_SCRIPTEXEC,CmScriptExec),
  EV_COMMAND(CM_PREFFONT,CmOptionFont),
  EV_COMMAND(CM_PREFVIEWER,CmOptionViewer),
  EV_COMMAND(CM_PREFEXVIEWER,CmOptionExtViewer),
  EV_COMMAND(CM_PREFEXDRIVER,CmOptionExtDriver),
  EV_COMMAND(CM_PREFSCRIPT,CmOptionScript),
  EV_COMMAND(CM_PREFMISC,CmOptionMisc),
  EV_COMMAND(CM_PREFSAVE,CmOptionSaveDefault),
  EV_COMMAND(CM_PREFSAVENGP,CmOptionSaveNgp),
  EV_COMMAND(CM_PREFFILEDEF,CmOptionFileDef),
  EV_COMMAND(CM_PREFTEXTDEF,CmOptionTextDef),
  EV_COMMAND(CM_WINDOWFILE,CmWindowFile),
  EV_COMMAND(CM_WINDOWAXIS,CmWindowAxis),
  EV_COMMAND(CM_WINDOWLEGEND,CmWindowLegend),
  EV_COMMAND(CM_WINDOWMERGE,CmWindowMerge),
  EV_COMMAND(CM_WINDOWDIALOG,CmWindowDialog),
  EV_COMMAND(CM_WINDOWCOORDINATE,CmWindowCoord),
  EV_COMMAND(CM_WINDOWFILEB,CmWindowFile),
  EV_COMMAND(CM_WINDOWAXISB,CmWindowAxis),
  EV_COMMAND(CM_WINDOWLEGENDB,CmWindowLegend),
  EV_COMMAND(CM_WINDOWMERGEB,CmWindowMerge),
  EV_COMMAND(CM_WINDOWDIALOGB,CmWindowDialog),
  EV_COMMAND(CM_WINDOWCONFIG,CmWindowConfig),
  EV_COMMAND(CM_WINDOWCONFIG2,CmWindowConfig2),
  EV_COMMAND(CM_WINDOWMOVE,CmWindowMove),
  EV_COMMAND(CM_WINDOWCOORDINATEB,CmWindowCoord),
  EV_COMMAND(CM_PRINTSETUP,CmPrintSetup),
  EV_COMMAND(CM_PRINTPRINT,CmPrintPrint),
  EV_COMMAND(CM_PRINTVIEWER,CmPrintViewer),
  EV_COMMAND(CM_PRINTDRIVER,CmPrintDriver),
  EV_COMMAND(CM_PRINTGRAFILE,CmPrintGRAFile),
  EV_COMMAND(CM_PRINTDATAFILE,CmPrintDataFile),
  EV_COMMAND(CM_HELPABOUT,CmHelpAbout),
  EV_COMMAND(CM_HELPHELP,CmHelpHelp),
  EV_COMMAND(CM_FILEMATHB,CmSetMathAll),
  EV_COMMAND(CM_SCALEUNDOB,CmScaleUndo),
  EV_COMMAND(CM_INTERRUPT,CmInterrupt),

  EV_COMMAND(CM_POINTPB,PointBPushed),
  EV_COMMAND(CM_LEGENDPB,LegendBPushed),
  EV_COMMAND(CM_LINEPB,LineBPushed),
  EV_COMMAND(CM_CURVEPB,CurveBPushed),
  EV_COMMAND(CM_RECTPB,RectBPushed),
  EV_COMMAND(CM_ARCPB,ArcBPushed),
  EV_COMMAND(CM_POLYPB,PolyBPushed),
  EV_COMMAND(CM_MARKPB,MarkBPushed),
  EV_COMMAND(CM_TEXTPB,TextBPushed),
  EV_COMMAND(CM_GAUSSPB,GaussBPushed),
  EV_COMMAND(CM_AXISPB,AxisBPushed),
  EV_COMMAND(CM_TRIMPB,TrimBPushed),
  EV_COMMAND(CM_FRAMEPB,FrameBPushed),
  EV_COMMAND(CM_SECTIONPB,SectionBPushed),
  EV_COMMAND(CM_CROSSPB,CrossBPushed),
  EV_COMMAND(CM_SINGLEPB,SingleBPushed),
  EV_COMMAND(CM_DATAPB,DataBPushed),
  EV_COMMAND(CM_EVALPB,EvalBPushed),
  EV_COMMAND(CM_ZOOMPB,ZoomBPushed),

  EV_COMMAND_ENABLE(CM_WINDOWFILE,CeWindowFile),
  EV_COMMAND_ENABLE(CM_WINDOWAXIS,CeWindowAxis),
  EV_COMMAND_ENABLE(CM_WINDOWLEGEND,CeWindowLegend),
  EV_COMMAND_ENABLE(CM_WINDOWMERGE,CeWindowMerge),
  EV_COMMAND_ENABLE(CM_WINDOWDIALOG,CeWindowDialog),
  EV_COMMAND_ENABLE(CM_WINDOWCOORDINATE,CeWindowCoord),
  EV_COMMAND_ENABLE(CM_WINDOWMOVE,CeWindowMove),
END_RESPONSE_TABLE;

TMyWindow::TMyWindow(TWindow *parent):TSubWindow(parent)
{
  int i;

  Init(parent,0,0);
  FileName=NULL;
  Changed=FALSE;
  pTMergeWindow=NULL;
  pTFileWindow=NULL;
  pTAxisWindow=NULL;
  pTLegendWindow=NULL;
  pTDialogWindow=NULL;
  pTCoordWindow=NULL;
  ActiveWindow=NULL;
  for (i=0;i<16;i++) custColors[i]=menulocal.customcolor[i];
  Interrupt=FALSE;
  infostr=NULL;
  Printer=NULL;
}

TMyWindow::~TMyWindow()
{
  memfree(infostr);
  memfree(FileName);
  if (Printer!=NULL) delete Printer;
}

void TMyWindow::EnableFrameWindow(TWindow *win,bool enable)
{
  if ((pTViewWindow!=NULL) && (pTViewWindow!=win))
    pTViewWindow->Frame->EnableWindow(enable);
  if ((pTFileWindow!=NULL) && (pTFileWindow!=win))
    pTFileWindow->Frame->EnableWindow(enable);
  if ((pTAxisWindow!=NULL) && (pTAxisWindow!=win))
    pTAxisWindow->Frame->EnableWindow(enable);
  if ((pTMergeWindow!=NULL) && (pTMergeWindow!=win))
    pTMergeWindow->Frame->EnableWindow(enable);
  if ((pTDialogWindow!=NULL) && (pTDialogWindow!=win))
    pTDialogWindow->Frame->EnableWindow(enable);
  if ((pTCoordWindow!=NULL) && (pTCoordWindow!=win))
    pTCoordWindow->Frame->EnableWindow(enable);
  if ((pTLegendWindow!=NULL) && (pTLegendWindow!=win))
    pTLegendWindow->Frame->EnableWindow(enable);
}

void TMyWindow::Focus(struct objlist *legendobj,int id) {
  if (pTViewWindow!=NULL)
     ((TViewWindow *)pTViewWindow)->Focus(legendobj,id);
}

void TMyWindow::UnFocus() {
  if (pTViewWindow!=NULL)
     ((TViewWindow *)pTViewWindow)->UnFocus();
}

void TMyWindow::Draw() {
  if (pTViewWindow!=NULL)
     ((TViewWindow *)pTViewWindow)->Draw(FALSE);
}

void TMyWindow::EvKeyDown2(UINT key,UINT repeatCount,UINT flags)
{
  int win0,win;

  switch (key) {
  case VK_F1:
  case 'Z':
    if ((key==VK_F1) || (key=='Z') && (GetAsyncKeyState(VK_CONTROL) & 0x8000)) {
      if (ActiveWindow==pTViewWindow) win=0;
      else if (ActiveWindow==pTFileWindow) win=1;
      else if (ActiveWindow==pTAxisWindow) win=2;
      else if (ActiveWindow==pTLegendWindow) win=3;
      else if (ActiveWindow==pTMergeWindow) win=4;
      else if (ActiveWindow==pTDialogWindow) win=5;
      else if (ActiveWindow==pTCoordWindow) win=6;
      else win=0;
      win0=win;
      do {
        if (GetAsyncKeyState(VK_SHIFT) & 0x8000) win--;
        else win++;
        if (win==7) win=0;
        if (win==-1) win=6;
        if (win==0) {
          if (pTViewWindow!=NULL) {
            pTViewWindow->Frame->SetActiveWindow();
            break;
          }
        } else if (win==1) {
          if (pTFileWindow!=NULL) {
            pTFileWindow->Frame->SetActiveWindow();
            break;
          }
        } else if (win==2) {
          if (pTAxisWindow!=NULL) {
            pTAxisWindow->Frame->SetActiveWindow();
            break;
          }
        } else if (win==3) {
          if (pTLegendWindow!=NULL) {
            pTLegendWindow->Frame->SetActiveWindow();
            break;
          }
        } else if (win==4) {
          if (pTMergeWindow!=NULL) {
            pTMergeWindow->Frame->SetActiveWindow();
            break;
          }
        } else if (win==5) {
          if (pTDialogWindow!=NULL) {
            pTDialogWindow->Frame->SetActiveWindow();
            break;
          }
        } else if (win==6) {
          if (pTCoordWindow!=NULL) {
            pTCoordWindow->Frame->SetActiveWindow();
            break;
          }
        }
      } while (win!=win0);
    }
    break;
  case VK_F2:
    break;
  case VK_F3:
    CmWindowFile();
    break;
  case VK_F4:
    CmWindowAxis();
    break;
  case VK_F5:
    CmWindowLegend();
    break;
  case VK_F6:
    CmWindowMerge();
    break;
  case VK_F7:
    CmWindowDialog();
    break;
  case VK_F8:
    CmWindowCoord();
    break;
  case 'V':
    break;
  case 'F':
    if (GetAsyncKeyState(VK_CONTROL) & 0x8000) CmWindowFile();
    break;
  case 'A':
    if (GetAsyncKeyState(VK_CONTROL) & 0x8000) CmWindowAxis();
    break;
  case 'L':
    if (GetAsyncKeyState(VK_CONTROL) & 0x8000) CmWindowLegend();
    break;
  case 'M':
    if (GetAsyncKeyState(VK_CONTROL) & 0x8000) CmWindowMerge();
    break;
  case 'I':
    if (GetAsyncKeyState(VK_CONTROL) & 0x8000) CmWindowDialog();
    break;
  case 'C':
    if (GetAsyncKeyState(VK_CONTROL) & 0x8000) CmWindowCoord();
    break;
  }
}

int TMyWindow::ChkInterrupt()
{
  MSG msg;
  TRect Bound;
  int x,y;

  if (!GetInputState()) return FALSE;
  while (TRUE) {
    if (PeekMessage(&msg,Handle,WM_KEYDOWN,WM_KEYDOWN,PM_REMOVE)) {
      if (msg.wParam==VK_ESCAPE) {
        Interrupt=TRUE;
        break;
      }
    }
    if (PeekMessage(&msg,NgraphApp->CB->Handle,
                    WM_LBUTTONDOWN,WM_LBUTTONDOWN,PM_REMOVE)) {
      Bound=NgraphApp->interruptB->GetBounds();
      x=LOWORD(msg.lParam);
      y=HIWORD(msg.lParam);
      if ((Bound.left<=x) && (x<=Bound.right)
      && (Bound.top<=y) && (y<=Bound.bottom)) {
        Interrupt=TRUE;
        break;
      }
    } else break;
  }
  if (Interrupt) {
    Interrupt=FALSE;
    return  TRUE;
  }
  return FALSE;
}

void TMyWindow::CmInterrupt()
{
}

void TMyWindow::SetCoord(int x,int y)
{
  if (pTCoordWindow!=NULL) ((TCoordWindow *)pTCoordWindow)->SetCoord(x,y);
}

void TMyWindow::SetPoint(int x,int y)
{
  char buf[20];

  sprintf(buf,"X:%d",x);
  PointX->SetText(buf);
  sprintf(buf,"Y:%d",y);
  PointY->SetText(buf);
  SB->HandleMessage(WM_PAINT);
}

void TMyWindow::SetZoom(double zm)
{
  char buf[20];

  sprintf(buf,"%.2f%%",zm*100);
  Zoom->SetText(buf);
  SB->HandleMessage(WM_PAINT);
}

void TMyWindow::ResetZoom()
{
  Zoom->SetText("");
  SB->HandleMessage(WM_PAINT);
}

void TMyWindow::SetStatusBar(char *mes)
{
  hcrWait=LoadCursor(NULL,IDC_WAIT);
  hcrOld=::SetCursor(hcrWait);
  SB->SetText(mes);
  SB->HandleMessage(WM_PAINT);
}

void TMyWindow::ResetStatusBar()
{
  SB->SetText(NULL);
  ::SetCursor(hcrOld);
}

void TMyWindow::SetMenuFont()
{
  HDC dc;
  TEXTMETRIC tm;
  LOGFONT IDFont;
  HFONT orgfont;
  int dpi;

  dc=GetDC(Handle);
  dpi=GetDeviceCaps(dc,LOGPIXELSX);
  if (menulocal.fontname!=NULL) {
    IDFont.lfHeight=menulocal.fontsize*dpi/72;
    IDFont.lfWidth=0;
    IDFont.lfEscapement=0;
    IDFont.lfOrientation=0;
    IDFont.lfUnderline=0;
    IDFont.lfStrikeOut=0;
    if (menulocal.fontitalic==FONT_ROMAN) {
      IDFont.lfWeight=400;
      IDFont.lfItalic=FALSE;
    } else if (menulocal.fontitalic==FONT_ITALIC) {
      IDFont.lfWeight=400;
      IDFont.lfItalic=TRUE;
    } else if (menulocal.fontitalic==FONT_BOLD) {
      IDFont.lfWeight=700;
      IDFont.lfItalic=FALSE;
    } else if (menulocal.fontitalic==FONT_BOLDITALIC) {
      IDFont.lfWeight=700;
      IDFont.lfItalic=TRUE;
    } else {
      IDFont.lfWeight=0;
      IDFont.lfItalic=0;
    }
    IDFont.lfPitchAndFamily=FIXED_PITCH;
    IDFont.lfCharSet=menulocal.fontcharset;
    IDFont.lfOutPrecision=0;
    IDFont.lfClipPrecision=0;
    IDFont.lfQuality=PROOF_QUALITY;
    strcpy(IDFont.lfFaceName,menulocal.fontname);
    if (menulocal.menufont!=NULL) DeleteObject(menulocal.menufont);
    menulocal.menufont=(HFONT)CreateFontIndirect(&IDFont);
  } else {
    if (menulocal.menufont!=NULL) DeleteObject(menulocal.menufont);
    menulocal.menufont=(HFONT)GetStockObject(SYSTEM_FIXED_FONT);
  }
  orgfont=(HFONT)SelectObject(dc,menulocal.menufont);
  GetTextMetrics(dc,&tm);
  SelectObject(dc,orgfont);
  ReleaseDC(Handle,dc);
  FHeight=tm.tmHeight+tm.tmDescent+tm.tmExternalLeading;
  FWidth=tm.tmAveCharWidth;
  if (FHeight<1) FHeight=1;
  if (FWidth<1) FWidth=1;
}

void TMyWindow::SetupWindow()
{
  TSubWindow::SetupWindow();
  SetMenuFont();
  DragAcceptFiles(TRUE);
}


bool TMyWindow::CanClose()
{
  int quit;

  if (menulock || globallock) return FALSE;
  quit=CheckSave();
  if (quit) {
    SaveHistory();
  }
  return quit;
}

void TMyWindow::CmSystemShell()
{
  struct objlist *obj,*robj,*shell;
  char *inst;
  int idn,allocnow;

  if (menulock || globallock) return;
  GetApplication()->PumpWaitingMessages();
  menulock=TRUE;
  obj=menulocal.obj;
  inst=menulocal.inst;
  idn=getobjtblpos(obj,"_evloop",&robj);
  registerevloop(chkobjectname(obj),"_evloop",robj,idn,inst,NULL);
  if ((shell=chkobject("shell"))!=NULL) {
    allocnow=nAllocConsole();
    exeobj(shell,"shell",0,0,NULL);
    nFreeConsole(allocnow);
  }
  unregisterevloop(robj,idn,inst);
  menulock=FALSE;
  SetForegroundWindow(menulocal.hWnd);
  NgraphApp->Update();
}

void TMyWindow::CmHelpAbout()
{
  AboutDialog(this,DIALOG_ABOUT).Execute();
}

void TMyWindow::CmHelpHelp()
{
  char *HelpFile;
  char *libdir,*docdir;
  struct objlist *sys;

  if ((sys=getobject("system"))==NULL) return;
  if (getobj(sys,"lib_dir",0,0,NULL,&libdir)==-1) return;
  if ((docdir=(char *)memalloc(strlen(libdir)+5))==NULL) return;
  strcpy(docdir,libdir);
  strcat(docdir,DOCDIR);
  if (!findfilename(docdir,CONFSEP,HELP)) {
    MessageBox("HTML file not found.","Error:",MB_OK|MB_ICONEXCLAMATION);
    memfree(docdir);
    return;
  } else if ((HelpFile=getfilename(docdir,CONFSEP,HELP))==NULL) {
    memfree(docdir);
    return;
  }
  unchangefilename(docdir);
  unchangefilename(HelpFile);
  ShellExecute(Handle,"open",HelpFile,NULL,docdir,0);
  memfree(docdir);
  memfree(HelpFile);
}

void TMyWindow::Update()
{
  if (pTViewWindow!=NULL) ((TViewWindow *)pTViewWindow)->Update(TRUE);
  if (pTFileWindow!=NULL) ((TFileWindow *)pTFileWindow)->Update(TRUE);
  if (pTAxisWindow!=NULL) ((TAxisWindow *)pTAxisWindow)->Update(TRUE);
  if (pTLegendWindow!=NULL) ((TLegendWindow *)pTLegendWindow)->Update(TRUE);
  if (pTMergeWindow!=NULL) ((TMergeWindow *)pTMergeWindow)->Update(TRUE);
  if (pTDialogWindow!=NULL) ((TDialogWindow *)pTDialogWindow)->Update(TRUE);
  if (pTCoordWindow!=NULL) ((TCoordWindow *)pTCoordWindow)->Update(TRUE);
}

void TMyWindow::Update2()
{
  if (pTFileWindow!=NULL) ((TFileWindow *)pTFileWindow)->Update(TRUE);
  if (pTAxisWindow!=NULL) ((TAxisWindow *)pTAxisWindow)->Update(TRUE);
  if (pTLegendWindow!=NULL) ((TLegendWindow *)pTLegendWindow)->Update(TRUE);
  if (pTMergeWindow!=NULL) ((TMergeWindow *)pTMergeWindow)->Update(TRUE);
  if (pTDialogWindow!=NULL) ((TDialogWindow *)pTDialogWindow)->Update(TRUE);
  if (pTCoordWindow!=NULL) ((TCoordWindow *)pTCoordWindow)->Update(TRUE);
}

void TMyWindow::ChangePage()
{
  if (pTViewWindow!=NULL) ((TViewWindow *)pTViewWindow)->CloseGDC();
  CloseGRA();
  OpenGRA();
  if (pTViewWindow!=NULL) ((TViewWindow *)pTViewWindow)->OpenGDC();
  if (pTViewWindow!=NULL) ((TViewWindow *)pTViewWindow)->ChangeScroller();
  CheckPage();
  if (pTViewWindow!=NULL) ((TViewWindow *)pTViewWindow)->ChangeDPI();
}

void TMyWindow::FileUpdate()
{
  if (pTFileWindow!=NULL) ((TFileWindow *)pTFileWindow)->Update(TRUE);
}

int TMyWindow::ViewOpened()
{
  if (pTViewWindow!=NULL) return TRUE;
  else return FALSE;
}


void TMyWindow::CmWindowMove()
{
  if (menulocal.movechild) menulocal.movechild=FALSE;
  else menulocal.movechild=TRUE;
}

void TMyWindow::CeWindowMove(TCommandEnabler& ce)
{
  if (menulocal.movechild) ce.SetCheck(TCommandEnabler::Checked);
  else ce.SetCheck(TCommandEnabler::Unchecked);
}

void TMyWindow::CmWindowConfig()
{
  if (pTCoordWindow!=NULL) CmWindowCoord();
  if (pTDialogWindow!=NULL) CmWindowDialog();
  if (pTMergeWindow!=NULL) CmWindowMerge();
  if (pTLegendWindow!=NULL) CmWindowLegend();
  if (pTAxisWindow!=NULL) CmWindowAxis();
  if (pTFileWindow!=NULL) CmWindowFile();
  PostMessage(WM_COMMAND,CM_WINDOWCONFIG2);
}

void TMyWindow::CmWindowConfig2()
{
  initwindowconfig();
  mwinwindowconfig();
  defaultwindowconfig();
  if (menulocal.coordopen) CmWindowCoord();
  if (menulocal.dialogopen) CmWindowDialog();
  if (menulocal.mergeopen) CmWindowMerge();
  if (menulocal.legendopen) CmWindowLegend();
  if (menulocal.axisopen) CmWindowAxis();
  if (menulocal.fileopen) CmWindowFile();
}

void TMyWindow::DisplayDialog(char *str)
{
  SetInfoText(str);
  if (pTDialogWindow!=NULL) ((TDialogWindow *)pTDialogWindow)->DrawInfoText();
}

void TMyWindow::DisplayStatus(char *str)
{
  SB->SetHintText(NULL);
  SB->SetText(str);
  SB->HandleMessage(WM_PAINT);
}

void TMyWindow::SetFrameCaption(char *file)
{
  char *fullpath;
  int len;
  char *buf;

  if (file==NULL) {
    fullpath=NULL;
    len=0;
  } else {
    fullpath=getfullpath(file);
    len=strlen(fullpath);
  }
  if ((buf=(char *)memalloc(len+10))!=NULL) {
    strcpy(buf,"Ngraph");
    if (fullpath!=NULL) {
      strcat(buf,": ");
      strcat(buf,fullpath);
    }
    NgraphApp->Frame->SetCaption(buf);
    memfree(buf);
  }
  memfree(fullpath);
}

void TMyWindow::SetInfoText(char *text)
{
  memfree(infostr);
  infostr=(char *)memalloc(strlen(text)+1);
  if (infostr!=NULL) strcpy(infostr,text);
}


int TMyWindow::PutStderr(char *s)
{
  int len;

  MessageBox(s,"Error:",MB_OK|MB_ICONEXCLAMATION);
  Update2();
  len=strlen(s);
  return len+1;
}

int TMyWindow::InputYN(char *mes)
{
  int ret;

  ret=MessageBox(mes,"Question",MB_YESNO|MB_ICONQUESTION);
  Update2();
  return (ret==IDYES)?TRUE:FALSE;
}

void TMyWindow::EvDropFiles(TDropInfo drop)
{
  int totalNumberOfFiles;
  char *name,*ext;
  int i,len,id;
  TFileWindow *win;
  FileDialog *dlg;
  struct objlist *obj;
  int id0,j,perm,type,ret;
  char *field;
  int load;

  totalNumberOfFiles=drop.DragQueryFileCount();
  if (menulock || globallock) {
    drop.DragFinish();
    return;
  }
  if (totalNumberOfFiles==1) {
    len=drop.DragQueryFileNameLen(0)+1;
    if ((name=(char *)memalloc(len+1))!=NULL) {
      drop.DragQueryFile(0,name,len);
      load=FALSE;
      if ((ext=getextention(name))!=NULL) {
        if ((strcmp0(ext,"PRM")==0) || (strcmp0(ext,"prm")==0)) {
          LoadPrmFile(name);
          load=TRUE;
        } else if ((strcmp0(ext,"NGP")==0) || (strcmp0(ext,"ngp")==0)) {
          LoadNgpFile(name,menulocal.ignorepath,menulocal.expand,menulocal.expanddir,FALSE,NULL);
          load=TRUE;
        }
      }
      memfree(name);
      if (load) {
        drop.DragFinish();
        PostMessage(WM_COMMAND,CM_VIEWERDRAWB);
        return;
      }
    }
  }
  if ((obj=chkobject("file"))==NULL) {
    drop.DragFinish();
    return;
  }
  id0=-1;
  for (i=0;i<totalNumberOfFiles;i++) {
    if ((id=newobj(obj))>=0) {
      len=drop.DragQueryFileNameLen(i)+1;
      if ((name=(char *)memalloc(len+1))!=NULL) {
        drop.DragQueryFile(i,name,len);
        changefilename(name);
        AddDataFileList(name);
        putobj(obj,"file",id,name);
        if (id0!=-1) {
		  for (j=0;j<chkobjfieldnum(obj);j++) {
			field=chkobjfieldname(obj,j);
			perm=chkobjperm(obj,field);
			type=chkobjfieldtype(obj,field);
			if ((strcmp2(field,"name")!=0) && (strcmp2(field,"file")!=0)
			&& (strcmp2(field,"fit")!=0)
			&& ((perm&NREAD)!=0) && ((perm&NWRITE)!=0) && (type<NVFUNC))
			  copyobj(obj,field,id,id0);
		  }
          FitCopy(obj,id,id0);
		} else {
		  dlg=new FileDialog(this,DIALOG_FILE,obj,id);
          ret=dlg->Execute();
          if ((ret==IDDELETE) || (ret==IDCANCEL)) {
            FitDel(obj,id);
            delobj(obj,id);
          } else {
            if (ret==IDFAPPLY) id0=id;
            NgraphApp->Changed=TRUE;
          }
          delete dlg;
        }
      }
    }
  }
  drop.DragFinish();
  SetForegroundWindow(Handle);
  win=(TFileWindow *)pTFileWindow;
  if (win!=NULL) win->Update(TRUE);
}





