/* 
 * 
 * This file is part of "Ngraph for Windows".
 * 
 * Copyright (C) 2015, Satoshi ISHIZAKA. isizaka@msa.biglobe.ne.jp
 * 
 * "Ngraph for Windows" is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * "Ngraph for Windows" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

/*
 *
 * winmerge.cpp
 *
 */

#include <owl\pch.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

extern "C" {
#include "ngraph.h"
#include "object.h"
#include "nstring.h"
#include "ioutil.h"
#include "winfopen.h"
#include "owinmenu.h"
}

#include "bitmap.rh"
#include "winmenu.rh"
#include "winmenu.h"
#include "windialg.h"
#include "wincommn.h"
#include "winmerge.h"

DEFINE_RESPONSE_TABLE1(MergeDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDCOPY,BN_CLICKED,CopyClicked),
  EV_CHILD_NOTIFY(IDDELETE,BN_CLICKED,DeleteClicked),
END_RESPONSE_TABLE;


MergeDialog::MergeDialog(TWindow *parent,TResId resID,struct objlist *obj,int id)
:TMyDialog(parent,resID)
{
  Obj=obj;
  Id=id;
}

void MergeDialog::SetupWindow()
{
  char *title;
  int len;

  TMyDialog::SetupWindow();
  len=GetWindowTextLength();
  if ((title=(char *)memalloc(len+10))!=NULL) {
    GetWindowText(title,len+1);
    sprintf(title+len," %d",Id);
    SetCaption(title);
    memfree(title);
  }
  SetupItem(TRUE,Id);
}

void MergeDialog::SetupItem(int file,int id)
{
  if (file) {
    SetTextFromObjField(Handle,IDMNAME,Obj,id,"file");
  }
  SetTextFromObjField(Handle,IDMTOP,Obj,id,"top_margin");
  SetTextFromObjField(Handle,IDMLEFT,Obj,id,"left_margin");
  SetTextFromObjField(Handle,IDMZOOM,Obj,id,"zoom");
  SetToggleFromObjField(Handle,IDMGREEK,Obj,id,"symbol_greek");
}

void MergeDialog::CloseWindow(int retVal)
{
  if (retVal!=IDOK) {
    TMyDialog::CloseWindow(retVal);
    return;
  }
  if (SetObjFieldFromText(Handle,IDMNAME,Obj,Id,"file")) return;
  if (SetObjFieldFromText(Handle,IDMTOP,Obj,Id,"top_margin")) return;
  if (SetObjFieldFromText(Handle,IDMLEFT,Obj,Id,"left_margin")) return;
  if (SetObjFieldFromText(Handle,IDMZOOM,Obj,Id,"zoom")) return;
  if (SetObjFieldFromToggle(Handle,IDMGREEK,Obj,Id,"symbol_greek")) return;
  TMyDialog::CloseWindow(retVal);
}




DEFINE_RESPONSE_TABLE1(TMergeWindow, TSubWindow)
  EV_WM_SIZE,
  EV_WM_LBUTTONDOWN,
  EV_WM_RBUTTONDOWN,
  EV_WM_LBUTTONDBLCLK,
  EV_WM_KEYDOWN,
  EV_WM_PAINT,
  EV_COMMAND(CM_MERGEOPEN,MergeOpen),
  EV_COMMAND(CM_SELECTK,MergeSelect),
  EV_COMMAND(CM_DELETEK,MergeDelete),
  EV_COMMAND(CM_COPYK,MergeCopy),
  EV_COMMAND(CM_TOPK,MergeTop),
  EV_COMMAND(CM_LASTK,MergeLast),
  EV_COMMAND(CM_UPDATEK,MergeUpdate),
  EV_COMMAND(CM_HIDDENK,MergeHidden),
  EV_COMMAND(CM_UPK,MergeUp),
  EV_COMMAND(CM_DOWNK,MergeDown),
END_RESPONSE_TABLE;

TMergeWindow::TMergeWindow(TWindow *parent):TSubWindow(parent)
{
  menulocal.mergeopen=TRUE;
  Attr.Style|=WS_VSCROLL|WS_HSCROLL;
  col1=GetSysColor(COLOR_WINDOWTEXT);
  col1h=GetSysColor(COLOR_GRAYTEXT);
  col2=(GetSysColor(COLOR_WINDOW) ^ GetSysColor(COLOR_HIGHLIGHT)) & RGB(255,255,255);
  brush1=CreateSolidBrush(col2);
  brush2=CreateSolidBrush(GetSysColor(COLOR_WINDOW));
  pen1=(HPEN)GetStockObject(NULL_PEN);
  pen2=(HPEN)GetStockObject(NULL_PEN);
  PopupMenu.AppendMenu(MF_STRING,CM_MERGEOPEN,"Open");
  PopupMenu.AppendMenu(MF_STRING,CM_SELECTK,"Focus (SPC)");
  PopupMenu.AppendMenu(MF_STRING,CM_UPDATEK,"Update (CR)");
  PopupMenu.AppendMenu(MF_STRING,CM_DELETEK,"Close (DEL)");
  PopupMenu.AppendMenu(MF_STRING,CM_TOPK,"Top (HOME)");
  PopupMenu.AppendMenu(MF_STRING,CM_UPK,"Up (SHIFT+UP)");
  PopupMenu.AppendMenu(MF_STRING,CM_DOWNK,"Down (SHIFT+DOWN)");
  PopupMenu.AppendMenu(MF_STRING,CM_LASTK,"Last (END)");
  PopupMenu.AppendMenu(MF_STRING,CM_COPYK,"Duplicate (INS)");
  PopupMenu.AppendMenu(MF_STRING,CM_HIDDENK,"Hide (TAB)");
}

void TMergeWindow::SetupWindow()
{
  TSubWindow::SetupWindow();
  select=-1;
  Scroller = new TScroller(this,1,1,0,0);
  Scroller->XLine=NgraphApp->FWidth;
  Scroller->YLine=NgraphApp->FHeight;
  Scroller->TrackMode=FALSE;
  obj=chkobject("merge");
  filenum=chkobjlastinst(obj);
  ChangeFileNum();
}


TMergeWindow::~TMergeWindow()
{
  menulocal.mergeopen=FALSE;
  ::DeleteObject(brush1);
  ::DeleteObject(brush2);
  ::DeleteObject(pen1);
  ::DeleteObject(pen2);
  NgraphApp->DeleteMergeWindow();
}

void TMergeWindow::EvSize(UINT sizeType, TSize& size)
{
  TSubWindow::EvSize(sizeType,size);
  ChangeFileNum();
}

void TMergeWindow::EvLButtonDown(UINT,TPoint& point)
{
  int sel;
  TClientDC dc(Handle);
  TPoint po;

  if (menulock || globallock) return;
  sel=(Scroller->YPos+point.y)/NgraphApp->FHeight-1;
  if (sel!=select) {
    po.x=Scroller->XPos;
    po.y=Scroller->YPos;
	dc.SetWindowOrg(po);
    SelectFile(dc,sel);
  }
}

void TMergeWindow::EvRButtonDown(UINT modKeys,TPoint& point)
{
  int sel;
  TClientDC dc(Handle);
  TPoint po;
  TRect rect;
  TPoint point2;

  if (menulock || globallock) return;
  sel=(Scroller->YPos+point.y)/NgraphApp->FHeight-1;
  if (sel!=select) {
    po.x=Scroller->XPos;
    po.y=Scroller->YPos;
	dc.SetWindowOrg(po);
    SelectFile(dc,sel);
  }
  sel=select;
  if ((sel>=0) && (sel<=filenum)) {
    PopupMenu.EnableMenuItem(CM_MERGEOPEN,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_DELETEK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_UPDATEK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_TOPK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_UPK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_DOWNK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_LASTK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_COPYK,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_HIDDENK,MF_ENABLED);
  } else {
    PopupMenu.EnableMenuItem(CM_MERGEOPEN,MF_ENABLED);
    PopupMenu.EnableMenuItem(CM_DELETEK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_UPDATEK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_TOPK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_UPK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_DOWNK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_LASTK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_COPYK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_HIDDENK,MF_GRAYED);
  }
  GetWindowRect(rect);
  point2.x=point.x+rect.left;
  point2.y=point.y+rect.top;
  PopupMenu.TrackPopupMenu(TPM_LEFTBUTTON | TPM_RIGHTBUTTON,point2,0,Handle);
}

void TMergeWindow::EvLButtonDblClk(UINT,TPoint& point)
{
  if (menulock || globallock) return;
  MergeUpdate();
}

void TMergeWindow::EvKeyDown(UINT key,UINT repeatCount,UINT flags)
{
  int sel;
  TClientDC dc(Handle);
  TPoint po;
  TRect rect;
  int h,h1,h2;
  TPoint point;

  if (menulock || globallock) return;
  sel=select;
  switch (key) {
  case VK_DELETE:
    MergeDelete();
    break;
  case VK_INSERT:
    MergeCopy();
    break;
  case VK_HOME:
    MergeTop();
    break;
  case VK_END:
    MergeLast();
    break;
  case VK_RETURN:
    MergeUpdate();
    break;
 case VK_SPACE:
    MergeSelect();
    break;
  case VK_TAB:
    MergeHidden();
    break;
  case VK_DOWN:
    if (GetAsyncKeyState(VK_SHIFT) & 0x8000) MergeDown();
    else {
      if ((sel==-1) && (filenum>=0)) sel=0;
      else if (sel<filenum) sel++;
      GetClientRect(rect);
      h=(rect.bottom-rect.top)/NgraphApp->FHeight;
      h1=Scroller->YPos/NgraphApp->FHeight-1;
      h2=Scroller->YPos/NgraphApp->FHeight-1+h;
      if (sel<h1) {
        Scroller->YPos=(sel+1)*NgraphApp->FHeight;
        Invalidate();
      } else if (sel>=h2) {
        Scroller->YPos=(sel+2-h)*NgraphApp->FHeight;
        Invalidate();
      }
      if (sel!=select) {
        po.x=Scroller->XPos;
        po.y=Scroller->YPos;
        dc.SetWindowOrg(po);
        SelectFile(dc,sel);
      }
    }
    break;
  case VK_UP:
    if (GetAsyncKeyState(VK_SHIFT) & 0x8000) MergeUp();
    else {
      if ((sel==-1) && (filenum>=0)) sel=filenum;
      else if (sel>0) sel--;
      GetClientRect(rect);
      h=(rect.bottom-rect.top)/NgraphApp->FHeight;
      h1=Scroller->YPos/NgraphApp->FHeight-1;
      h2=Scroller->YPos/NgraphApp->FHeight-1+h;
      if (sel<h1) {
        Scroller->YPos=(sel+1)*NgraphApp->FHeight;
        Invalidate();
      } else if (sel>=h2) {
        Scroller->YPos=(sel+2-h)*NgraphApp->FHeight;
        Invalidate();
      }
      if (sel!=select) {
        po.x=Scroller->XPos;
        po.y=Scroller->YPos;
        dc.SetWindowOrg(po);
        SelectFile(dc,sel);
      }
    }
    break;
  case 'P':
    if (GetAsyncKeyState(VK_CONTROL) & 0x8000) {
      sel=select;
      if ((sel>=0) && (sel<=filenum)) {
        PopupMenu.EnableMenuItem(CM_MERGEOPEN,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_SELECTK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_DELETEK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_UPDATEK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_TOPK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_LASTK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_COPYK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_HIDDENK,MF_ENABLED);
      } else {
        PopupMenu.EnableMenuItem(CM_MERGEOPEN,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_SELECTK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_DELETEK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_UPDATEK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_TOPK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_LASTK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_COPYK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_HIDDENK,MF_GRAYED);
      }
      GetWindowRect(rect);
      point.x=rect.left;
      point.y=rect.top;
      PopupMenu.TrackPopupMenu(TPM_LEFTBUTTON | TPM_RIGHTBUTTON,point,0,Handle);
    }
    break;
  default:
    NgraphApp->EvKeyDown2(key,repeatCount,flags);
    break;
  }
  TSubWindow::EvKeyDown(key,repeatCount,flags);
}

void TMergeWindow::MergeOpen()
{
  NgraphApp->CmMergeOpen();
}

void TMergeWindow::MergeDelete()
{
  int sel;

  if (menulock || globallock) return;
  NgraphApp->UnFocus();
  sel=select;
  if ((sel>=0) && (sel<=filenum)) {
    delobj(obj,sel);
    Update(FALSE);
    NgraphApp->Changed=TRUE;
  }
}

void TMergeWindow::MergeCopy()
{
  int sel;
  int j,id,perm,type;
  char *field;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=0) && (sel<=filenum)) {
    if ((id=newobj(obj))>=0) {
      for (j=0;j<chkobjfieldnum(obj);j++) {
        field=chkobjfieldname(obj,j);
        perm=chkobjperm(obj,field);
        type=chkobjfieldtype(obj,field);
        if ((strcmp2(field,"name")!=0)
          && ((perm&NREAD)!=0) && ((perm&NWRITE)!=0) && (type<NVFUNC))
          copyobj(obj,field,id,sel);
      }
      filenum++;
      NgraphApp->Changed=TRUE;
    }
    select=sel;
    Update(FALSE);
  }
}

void TMergeWindow::MergeTop()
{
  int sel;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=0) && (sel<=filenum)) {
    movetopobj(obj,sel);
    select=0;
    Update(FALSE);
    NgraphApp->Changed=TRUE;
  }
}

void TMergeWindow::MergeUp()
{
  int sel;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=1) && (sel<=filenum)) {
    moveupobj(obj,sel);
    select=sel-1;
    Update(FALSE);
    NgraphApp->Changed=TRUE;
  }
}

void TMergeWindow::MergeDown()
{
  int sel;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=0) && (sel<=filenum-1)) {
    movedownobj(obj,sel);
    select=sel+1;
    Update(FALSE);
    NgraphApp->Changed=TRUE;
  }
}

void TMergeWindow::MergeLast()
{
  int sel;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=0) && (sel<=filenum)) {
    movelastobj(obj,sel);
    select=filenum;
    Update(FALSE);
    NgraphApp->Changed=TRUE;
  }
}

void TMergeWindow::MergeUpdate()
{
  int sel,ret;
  MergeDialog *dlg;

  if (menulock || globallock) return;
  NgraphApp->UnFocus();
  sel=select;
  if ((sel>=0) && (sel<=filenum)) {
    dlg=new MergeDialog(this,DIALOG_MERGE,obj,sel);
    if ((ret=dlg->Execute())==IDDELETE) {
      delobj(obj,sel);
      select=-1;
    }
    if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
    delete dlg;
    Update(FALSE);
  }
}

void TMergeWindow::MergeSelect()
{
  int sel;

  if (menulock || globallock) return;
  sel=select;
  if ((sel>=0) && (sel<=filenum)) NgraphApp->Focus(obj,sel);
}

void TMergeWindow::MergeHidden()
{
  int sel;
  int hiden;

  if (menulock || globallock) return;
  NgraphApp->UnFocus();
  sel=select;
  if ((sel>=0) && (sel<=filenum)) {
    getobj(obj,"hidden",sel,0,NULL,&hiden);
    hiden=hiden?FALSE:TRUE;
    putobj(obj,"hidden",sel,&hiden);
    Update(FALSE);
    NgraphApp->Changed=TRUE;
  }
}

void TMergeWindow::Update(int clear)
{
  if (clear) select=-1;
  filenum=chkobjlastinst(obj);
  ChangeFileNum();
  Invalidate();
}

void TMergeWindow::ChangeFileNum()
{
  int width,height;

  width=47*NgraphApp->FWidth;
  height=(filenum+2)*NgraphApp->FHeight;
  Scroller->SetRange(width,height);
  Scroller->SetPageSize();
}

void TMergeWindow::SelectFile(HDC dc,int sel)
{
  int x,y,smode;

  if ((select!=-1) && (sel!=select)) {
    x=NgraphApp->FWidth;
    y=(select+2)*NgraphApp->FHeight;
    smode=::SetROP2(dc,R2_XORPEN);
    ::SelectObject(dc,brush1);
    ::SelectObject(dc,pen1);
    ::Rectangle(dc,x-NgraphApp->FWidth,y-NgraphApp->FHeight,x+47*NgraphApp->FWidth,y);
    ::SetROP2(dc,smode);
  }
  if ((sel>=0) && (sel<=filenum)) select=sel;
  else select=-1;
  if (select!=-1) {
    x=NgraphApp->FWidth;
    y=(select+2)*NgraphApp->FHeight;
    smode=::SetROP2(dc,R2_XORPEN);
    ::SelectObject(dc,brush1);
    ::SelectObject(dc,pen1);
    ::Rectangle(dc,x-NgraphApp->FWidth,y-NgraphApp->FHeight,x+47*NgraphApp->FWidth,y);
    ::SetROP2(dc,smode);
  }
}

void TMergeWindow::EvPaint()
{
  PAINTSTRUCT ps;
  HDC dc;
  RECT rc;
  int i;
  char buf[256];
  char *file,*bfile;
  RECT rect;
  int h,w,x,y,len,cx;
  HFONT orgfont;
  int hiden;
  HBRUSH brush;

  if (::GetUpdateRect(Handle,&rc,FALSE)) {
    dc=::BeginPaint(Handle,&ps);
    if (globallock) {
      ::EndPaint(Handle,&ps);
      return;
    }
    Scroller->BeginView(TDC(dc),TRect(rc));
    filenum=chkobjlastinst(obj);
    orgfont=(HFONT)SelectObject(dc,menulocal.menufont);
    SetTextAlign(dc,TA_BOTTOM | TA_LEFT);
    h=NgraphApp->FHeight;
    w=NgraphApp->FWidth;

    brush=CreateSolidBrush(GetSysColor(COLOR_MENU));
    rect.top=0;
    rect.bottom=h;
    rect.left=0;
    rect.right=47*w+w;
	FillRect(dc,&rect,brush);
    DeleteObject(brush);
    SetTextColor(dc,GetSysColor(COLOR_MENUTEXT));
    SetBkMode(dc,TRANSPARENT);
    x=w;
    y=h;
    rect.top=y-NgraphApp->FHeight;
    rect.bottom=y;
    rect.left=x;
    rect.right=x+6*w;
    len=sprintf(buf,"#");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=6*w;
    rect.left=x;
    rect.right=x+13*w;
    len=sprintf(buf,"file");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=13*w;
    rect.left=x;
    rect.right=x+6*w;
    len=sprintf(buf,"    tm");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=6*w;
    rect.left=x;
    rect.right=x+6*w;
    len=sprintf(buf,"    lm");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=6*w;
    rect.left=x;
    rect.right=x+6*w;
    len=sprintf(buf,"    zm");
    ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
    x+=6*w;
    rect.left=x+w;
    rect.right=x+8*w;
    len=sprintf(buf,"^#");
    ::ExtTextOut(dc,x+w,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);

    for (i=0;i<=filenum;i++)
    if (((i+2)*h>=(rc.top+Scroller->YPos))
    && ((i-1)*h<=(rc.bottom+Scroller->YPos))) {
      x=w;
      y=(i+2)*h;
      ::SelectObject(dc,pen2);
      ::SelectObject(dc,brush2);
	  ::Rectangle(dc,x-w,y-h,x+47*w,y);
      rect.top=y-NgraphApp->FHeight;
      rect.bottom=y;

      getobj(obj,"hidden",i,0,NULL,&hiden);
      if (hiden) SetTextColor(dc,col1h);
      else SetTextColor(dc,col1);

      getobj(obj,"id",i,0,NULL,&cx);
      rect.left=x;
	  rect.right=x+6*w;
	  len=sprintf(buf,"%d",cx);
	  ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
	  x+=6*w;

      getobj(obj,"file",i,0,NULL,&file);
      bfile=getbasename(file);
      rect.left=x;
      rect.right=x+12*w;
      if (bfile!=NULL) {
        ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,bfile,strlen(bfile),NULL);
        memfree(bfile);
      } else
        ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,"....................",20,NULL);
      x+=13*w;

      getobj(obj,"top_margin",i,0,NULL,&cx);
      rect.left=x;
      rect.right=x+6*w;
      len=sprintf(buf,"%6d",cx);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,buf,len,NULL);
      x+=6*w;

      getobj(obj,"left_margin",i,0,NULL,&cx);
      rect.left=x;
      rect.right=x+6*w;
      len=sprintf(buf,"%6d",cx);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
      x+=6*w;

      getobj(obj,"zoom",i,0,NULL,&cx);
      rect.left=x;
      rect.right=x+6*w;
      len=sprintf(buf,"%6d",cx);
      ::ExtTextOut(dc,x,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
      x+=6*w;

      getobj(obj,"oid",i,0,NULL,&cx);
      rect.left=x+w;
	  rect.right=x+8*w;
	  len=sprintf(buf,"^%d",cx);
	  ::ExtTextOut(dc,x+w,y,ETO_CLIPPED,&rect,(char *)buf,len,NULL);
	  x+=8*w;

    }
    SelectFile(dc,select);
    SetScrollPos(SB_HORZ,Scroller->XPos,TRUE);
    SetScrollPos(SB_VERT,Scroller->YPos,TRUE);
    SelectObject(dc,orgfont);
    ::EndPaint(Handle,&ps);
  }
}

char MergeOpenBuf[512]={0};

void TMyWindow::CmMergeOpen()
{
  struct objlist *obj;
  char *name;
  int id,ret;
  TMergeWindow *win;
  MergeDialog *dlg;

  if (menulock || globallock) return;
  if ((obj=chkobject("merge"))==NULL) return;
  MergeOpenBuf[0]='\0';
  EnableFrameWindow(this,false);
  if (nGetOpenFileName(Handle,"Merge open","gra",NULL,
      MergeOpenBuf,512,"GRA Files (*.gra)\0*.gra\0All Files (*.*)\0*.*\0",TRUE,menulocal.changedirectory)==IDOK) {
    if ((id=newobj(obj))>=0) {
      if ((name=(char *)memalloc(strlen(MergeOpenBuf)+1))==NULL) {
        EnableFrameWindow(this,true);
        return;
      }
      strcpy(name,MergeOpenBuf);
      changefilename(name);
      putobj(obj,"file",id,name);
      dlg=new MergeDialog(this,DIALOG_MERGE,obj,id);
      ret=dlg->Execute();
      if ((ret==IDDELETE) || (ret==IDCANCEL)) {
        delobj(obj,id);
      } else NgraphApp->Changed=TRUE;
      delete dlg;
    }
    win=(TMergeWindow *)pTMergeWindow;
    if (win!=NULL) win->Update(TRUE);
  }
  EnableFrameWindow(this,true);
}

void TMyWindow::CmMergeClose()
{
  TMergeWindow *win;
  SelectDialog *dlg;
  struct narray farray;
  struct objlist *obj;
  int i;
  int num,*array;

  if (menulock || globallock) return;
  if ((obj=chkobject("merge"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,FileCB,
                       (struct narray *)&farray,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&farray);
    array=(int *)arraydata(&farray);
    for (i=num-1;i>=0;i--) {
      delobj(obj,array[i]);
      NgraphApp->Changed=TRUE;
    }
    win=(TMergeWindow *)pTMergeWindow;
    if (win!=NULL) win->Update(TRUE);
  }
  arraydel(&farray);
  delete dlg;
}

void TMyWindow::CmMergeUpdate()
{
  TMergeWindow *win;
  SelectDialog *dlg;
  MergeDialog *fdlg;
  struct narray farray;
  struct objlist *obj;
  int i,j,ret;
  int *array,num;

  if (menulock || globallock) return;
  if ((obj=chkobject("merge"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,FileCB,
                       (struct narray *)&farray,NULL);
  if (dlg->Execute()==IDOK) {
    num=arraynum(&farray);
    array=(int *)arraydata(&farray);
    for (i=0;i<num;i++) {
      fdlg=new MergeDialog(this,DIALOG_MERGE,obj,array[i]);
      if ((ret=fdlg->Execute())==IDDELETE) {
        delobj(obj,array[i]);
        for (j=i+1;j<num;j++) array[j]--;
      }
      if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
      delete fdlg;
    }
    win=(TMergeWindow *)pTMergeWindow;
    if (win!=NULL) win->Update(TRUE);
  }
  arraydel(&farray);
  delete dlg;
}

void TMyWindow::CmWindowMerge()
{
  TMergeWindow *win;
  TRect rect;
  int x,y;

  if (menulock || globallock) return;
  if (pTMergeWindow==NULL) {
    win=(TMergeWindow *)pTMergeWindow=new TMergeWindow();
    win->Frame=new TSubFrameWindow(this,"Merge Window",win,
    &menulocal.mergex,&menulocal.mergey,&menulocal.mergewidth,&menulocal.mergeheight);
    win->Frame->SetIcon(GetApplication(),ICON_4);
    NgraphApp->Frame->GetWindowRect(rect);
    x=rect.left;
    y=rect.top;
    if (menulocal.mergewidth==CW_USEDEFAULT) win->Frame->Attr.W=FWidth*47+2*FWidth;
    else win->Frame->Attr.W=menulocal.mergewidth;
    if (menulocal.mergeheight==CW_USEDEFAULT) win->Frame->Attr.H=FHeight*10;
    else win->Frame->Attr.H=menulocal.mergeheight;
    if (menulocal.mergex==CW_USEDEFAULT) win->Frame->Attr.X=CW_USEDEFAULT;
    else {
      win->Frame->Attr.X=menulocal.mergex+x;
      if (win->Frame->Attr.X<0) win->Frame->Attr.X=0;
    }
    if (menulocal.mergey==CW_USEDEFAULT) win->Frame->Attr.Y=CW_USEDEFAULT;
    else {
      win->Frame->Attr.Y=menulocal.mergey+y;
      if (win->Frame->Attr.Y<0) win->Frame->Attr.Y=0;
    }
    win->Frame->Create();
  } else {
    win=(TMergeWindow *)pTMergeWindow;
    win->CloseWindow();
  }
}

void TMyWindow::CeWindowMerge(TCommandEnabler& ce)
{
  if (pTMergeWindow==NULL) ce.SetCheck(TCommandEnabler::Unchecked);
  else ce.SetCheck(TCommandEnabler::Checked);
}

