/* 
 * 
 * This file is part of "Ngraph for Windows".
 * 
 * Copyright (C) 2015, Satoshi ISHIZAKA. isizaka@msa.biglobe.ne.jp
 * 
 * "Ngraph for Windows" is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * "Ngraph for Windows" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

/*
 *
 * winview.cpp
 *
 */

#include <owl\pch.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <limits.h>
#include <io.h>

extern "C" {
#include "ngraph.h"
#include "object.h"
#include "ioutil.h"
#include "nstring.h"
#include "config.h"
#include "mathfn.h"
#include "gra.h"
#include "spline.h"
#include "owinmenu.h"
#include "winfopen.h"
}

#include "bitmap.rh"
#include "winmenu.rh"
#include "winmenu.h"
#include "windialg.h"
#include "wincommn.h"
#include "winview.h"
#include "winlgnd.h"
#include "winlgndx.h"
#include "winaxis.h"
#include "winmerge.h"
#include "winfile.h"

DEFINE_RESPONSE_TABLE1(EvalDialog,TMyDialog)
  EV_CHILD_NOTIFY(IDEVMASK,BN_CLICKED,MaskClicked),
  EV_CHILD_NOTIFY(IDEVMOVE,BN_CLICKED,MoveClicked),
END_RESPONSE_TABLE;

TMyScroller::TMyScroller(TWindow* window,int xUnit,int yUnit,double xRange,double yRange) : TScroller(window,xUnit,yUnit,xRange,yRange)
{
}

EvalDialog::EvalDialog(TWindow *parent,TResId resID,struct objlist *obj,int num,struct narray *iarray)
: TMyDialog(parent,resID)
{
  Obj=obj;
  Num=num;
  arrayinit(iarray,sizeof(int));
  sel=iarray;
}

void EvalDialog::MaskClicked()
{
  CloseWindow(IDEVMASK);
}

void EvalDialog::MoveClicked()
{
  CloseWindow(IDEVMOVE);
}


void EvalDialog::CloseWindow(int retValue)
{
  int i,count;

  if ((retValue==IDEVMASK) || (retValue==IDEVMOVE)) {
    count=SendDlgItemMessage(IDEVLIST,LB_GETCOUNT,0,0);
    for (i=0;i<count;i++)
      if (SendDlgItemMessage(IDEVLIST,LB_GETSEL,i,0)>0) arrayadd(sel,&i);
  }
  TMyDialog::CloseWindow(retValue);
}

struct evaltype evallist[100];
struct narray sellist;
char evbuf[256];

void EvalDialog::SetupWindow()
{
  int i,j;

  TMyDialog::SetupWindow();
  SendDlgItemMessage(IDEVLIST,LB_RESETCONTENT,0,0);
  for (i=0;i<Num;i++) {
    j=0;
    j+=sprintf(evbuf+j,"%6d ",evallist[i].id);
    j+=sprintf(evbuf+j,"%8d ",evallist[i].line);
    j+=sprintf(evbuf+j,"%+.15e %+.15e",evallist[i].x,evallist[i].y);
    SendDlgItemMessage(IDEVLIST,LB_INSERTSTRING,-1,(LPARAM)evbuf);
  }
}

char MetaFileBuf[512];

class ClipboardDialog  : public TMyDialog {
  public:
    ClipboardDialog(TWindow *parent,TResId resID,int *enhmeta,char **file,int *dpi);
    ~ClipboardDialog();
    char **File;
  protected:
    TStatic *DPI;
    TScrollBar *DPIS;
    int *DpiP;
    int *EnhMeta;
    void SetupWindow();
    void CloseWindow(int retVal);
    void BrowseClicked();
    void EvHScroll(UINT scrollCode, UINT thumbPos, HWND hWndCtl);
    DECLARE_RESPONSE_TABLE(ClipboardDialog);
};

DEFINE_RESPONSE_TABLE1(ClipboardDialog,TMyDialog)
  EV_WM_HSCROLL,
  EV_CHILD_NOTIFY(IDCLBROWSE,BN_CLICKED,BrowseClicked),
END_RESPONSE_TABLE;

ClipboardDialog::ClipboardDialog(TWindow *parent,TResId resID,int *enhmeta,char **file,int *dpi)
:TMyDialog(parent,resID)
{
  EnhMeta=enhmeta;
  File=file;
  DPI=new TStatic(this,IDCLDPI,5);
  DPIS=new TScrollBar(this,IDCLDPIS);
  DpiP=dpi;
}

ClipboardDialog::~ClipboardDialog()
{
  delete DPIS;
  delete DPI;
}

void ClipboardDialog::BrowseClicked()
{
  MetaFileBuf[0]='\0';
  if (SendDlgItemMessage(IDCLENHMETA,BM_GETCHECK,0,0)) {
    if (nGetSaveFileName(Handle,"Windows Enhanced Metafile output","emf",NULL,
        MetaFileBuf,512,"Enhanced Metafile (*.emf)\0*.emf\0",FALSE)==IDOK) {
      SendDlgItemMessage(IDCLFILE,WM_SETTEXT,0,(LPARAM)MetaFileBuf);
    }
  } else {
    if (nGetSaveFileName(Handle,"Windows Metafile output","wmf",NULL,
        MetaFileBuf,512,"Metafile (*.wmf)\0*.wmf\0",FALSE)==IDOK) {
      SendDlgItemMessage(IDCLFILE,WM_SETTEXT,0,(LPARAM)MetaFileBuf);
    }
  }
}

void ClipboardDialog::SetupWindow()
{
  char buf[256];

  TMyDialog::SetupWindow();
  SendDlgItemMessage(IDCLMETA,BM_SETCHECK,0,0);
  if (*EnhMeta) SendDlgItemMessage(IDCLENHMETA,BM_SETCHECK,1,0);
  else SendDlgItemMessage(IDCLMETA,BM_SETCHECK,1,0);
  sprintf(buf,"%d",*DpiP);
  DPI->SetText(buf);
  DPIS->SetRange(96,2540);
  DPIS->SetPosition(*DpiP);
}

void ClipboardDialog::CloseWindow(int retVal)
{
  int len;
  char *buf;

  if (SendDlgItemMessage(IDCLENHMETA,BM_GETCHECK,0,0)) *EnhMeta=TRUE;
  else *EnhMeta=FALSE;
  len=SendDlgItemMessage(IDCLFILE,WM_GETTEXTLENGTH,0,0);
  if (len!=0) {
    if ((*File=(char *)memalloc(len+1))!=NULL) {
      GetDlgItemText(IDCLFILE,*File,len+1);
      if (access(*File,04)==0) {
        buf=(char *)memalloc(strlen(*File)+60);
        if (buf!=NULL) {
          sprintf(buf,
#ifdef ENGLISH
                   "`%s'\n\nOverwrite existing file?",
#else
                   "`%s'\n\nt@C݂܂B㏑Ă낵łH",
#endif
                   *File);
          if (MessageBox(buf,"Metafile",MB_ICONQUESTION|MB_YESNO)!=IDYES) {
            memfree(buf);
            memfree(*File);
            *File=NULL;
            TMyDialog::CloseWindow(IDCANCEL);
            return;
          }
          memfree(buf);
        } else {
          if (MessageBox(
#ifdef ENGLISH
                   "Overwrite existing file?",
#else
                   "`%s'\n\nt@C݂܂B㏑Ă낵łH",
#endif
                   "Driver",MB_ICONQUESTION|MB_YESNO)!=IDNO) {
            memfree(*File);
            *File=NULL;
            TMyDialog::CloseWindow(IDCANCEL);
            return;
          }
        }
      }
    } else {
      TMyDialog::CloseWindow(IDCANCEL);
      return;
    }
  } else *File=NULL;
  TMyDialog::CloseWindow(retVal);
}

void ClipboardDialog::EvHScroll(UINT scrollCode, UINT thumbPos, HWND hWndCtl)
{
  int Min,Max,Sc,X,ScPos;
  char buf[5];

  DPIS->GetRange(Min,Max);
  X=*DpiP;
  switch (scrollCode) {
  case SB_PAGEUP:
    Sc=-10;
    break;
  case SB_PAGEDOWN:
    Sc=10;
    break;
  case SB_LINEUP:
    Sc=-1;
    break;
  case SB_LINEDOWN:
    Sc=1;
    break;
  case SB_THUMBPOSITION:
    Sc=thumbPos-X;
    break;
  case SB_THUMBTRACK:
    Sc=thumbPos-X;
    break;
  default:
    Sc=0;
    break;
  }
  ScPos=X+Sc;
  ScPos=max(Min,ScPos);
  ScPos=min(Max,ScPos);
  Sc=ScPos-X;
  X=ScPos;
  if (Sc==0) return;
  DPIS->SetPosition(X);
  *DpiP=X;
  sprintf(buf,"%d",*DpiP);
  DPI->SetText(buf);
}

struct focuslist {
  struct objlist *obj;
  int oid;
  int ofsx,ofsy;
};

struct pointslist {
  int x,y;
};

DEFINE_RESPONSE_TABLE1(TViewWindow, TMyWindow)
  EV_WM_SIZE,
  EV_WM_LBUTTONDOWN,
  EV_WM_PAINT,
  EV_WM_LBUTTONDOWN,
  EV_WM_RBUTTONDOWN,
  EV_WM_LBUTTONUP,
  EV_WM_LBUTTONDBLCLK,
  EV_WM_MOUSEMOVE,
  EV_WM_KEYDOWN,
  EV_WM_KEYUP,
  EV_COMMAND(CM_DELETEK,ViewDelete),
  EV_COMMAND(CM_UPDATEK,ViewUpdate),
  EV_COMMAND(CM_COPYK,ViewCopy),
  EV_COMMAND(CM_TOPK,ViewTop),
  EV_COMMAND(CM_LASTK,ViewLast),
  EV_COMMAND(CM_CROSSK,ViewCross),
END_RESPONSE_TABLE;

#define MOUSENONE 0
#define MOUSEPOINT 1
#define MOUSEDRAG  2
#define MOUSEZOOM1 3
#define MOUSEZOOM2 4
#define MOUSEZOOM3 5
#define MOUSEZOOM4 6
#define MOUSECHANGE 7

TViewWindow::TViewWindow(TWindow *parent):TMyWindow(parent)
{
}

TViewWindow::~TViewWindow()
{
  if (region!=NULL) DeleteObject(region);
  region=NULL;
  CloseGDC();
  CloseGRA();
  NgraphApp->CB2->Remove(*PointB);
  NgraphApp->CB2->Remove(*LegendB);
  NgraphApp->CB2->Remove(*LineB);
  NgraphApp->CB2->Remove(*CurveB);
  NgraphApp->CB2->Remove(*RectB);
  NgraphApp->CB2->Remove(*ArcB);
  NgraphApp->CB2->Remove(*PolyB);
  NgraphApp->CB2->Remove(*MarkB);
  NgraphApp->CB2->Remove(*TextB);
  NgraphApp->CB2->Remove(*GaussB);
  NgraphApp->CB2->Remove(*AxisB);
  NgraphApp->CB2->Remove(*TrimB);
  NgraphApp->CB2->Remove(*FrameB);
  NgraphApp->CB2->Remove(*SectionB);
  NgraphApp->CB2->Remove(*CrossB);
  NgraphApp->CB2->Remove(*SingleB);
  NgraphApp->CB2->Remove(*DataB);
  NgraphApp->CB2->Remove(*EvalB);
  NgraphApp->CB2->Remove(*ZoomB);
  if (NgraphApp->CB2->Handle!=NULL) NgraphApp->CB2->Invalidate();
  delete PointB;
  delete LegendB;
  delete LineB;
  delete CurveB;
  delete RectB;
  delete ArcB;
  delete PolyB;
  delete MarkB;
  delete TextB;
  delete GaussB;
  delete AxisB;
  delete TrimB;
  delete FrameB;
  delete SectionB;
  delete CrossB;
  delete SingleB;
  delete DataB;
  delete EvalB;
  delete ZoomB;
  delete PointBM;
  delete LegendBM;
  delete LineBM;
  delete CurveBM;
  delete RectBM;
  delete ArcBM;
  delete PolyBM;
  delete MarkBM;
  delete TextBM;
  delete GaussBM;
  delete AxisBM;
  delete TrimBM;
  delete FrameBM;
  delete SectionBM;
  delete CrossBM;
  delete SingleBM;
  delete DataBM;
  delete EvalBM;
  delete ZoomBM;
  arrayfree2(focusobj);
  ShowFrame=FALSE;
  arrayfree2(points);
  ReleaseCapture();
}

void TViewWindow::SetupWindow()
{
  TMyWindow::SetupWindow();
  Attr.Style|=WS_VSCROLL|WS_HSCROLL;
  Scroller = new TMyScroller(this,1,1,0,0);
  Scroller->XLine=NgraphApp->FWidth;
  Scroller->YLine=NgraphApp->FHeight;
  Scroller->AutoMode=TRUE;
  menulocal.GRAoid=-1;
  menulocal.GRAinst=NULL;
  PointB=new TButtonGadget(BITMAP_POINT,CM_POINTPB,TButtonGadget::Exclusive);
  PointB->SetNotchCorners(FALSE);
  LegendB=new TButtonGadget(BITMAP_LEGENDPOINT,CM_LEGENDPB,TButtonGadget::Exclusive);
  LegendB->SetNotchCorners(FALSE);
  LineB=new TButtonGadget(BITMAP_LEGENDLINE,CM_LINEPB,TButtonGadget::Exclusive);
  LineB->SetNotchCorners(FALSE);
  CurveB=new TButtonGadget(BITMAP_LEGENDCURVE,CM_CURVEPB,TButtonGadget::Exclusive);
  CurveB->SetNotchCorners(FALSE);
  RectB=new TButtonGadget(BITMAP_LEGENDRECT,CM_RECTPB,TButtonGadget::Exclusive);
  RectB->SetNotchCorners(FALSE);
  ArcB=new TButtonGadget(BITMAP_LEGENDARC,CM_ARCPB,TButtonGadget::Exclusive);
  ArcB->SetNotchCorners(FALSE);
  PolyB=new TButtonGadget(BITMAP_LEGENDPOLYGON,CM_POLYPB,TButtonGadget::Exclusive);
  PolyB->SetNotchCorners(FALSE);
  MarkB=new TButtonGadget(BITMAP_LEGENDMARK,CM_MARKPB,TButtonGadget::Exclusive);
  MarkB->SetNotchCorners(FALSE);
  TextB=new TButtonGadget(BITMAP_LEGENDTEXT,CM_TEXTPB,TButtonGadget::Exclusive);
  TextB->SetNotchCorners(FALSE);
  GaussB=new TButtonGadget(BITMAP_LEGENDGAUSS,CM_GAUSSPB,TButtonGadget::Exclusive);
  GaussB->SetNotchCorners(FALSE);
  AxisB=new TButtonGadget(BITMAP_AXISPOINT,CM_AXISPB,TButtonGadget::Exclusive);
  AxisB->SetNotchCorners(FALSE);
  TrimB=new TButtonGadget(BITMAP_AXISTRIMMING,CM_TRIMPB,TButtonGadget::Exclusive);
  TrimB->SetNotchCorners(FALSE);
  FrameB=new TButtonGadget(BITMAP_AXISFRAME,CM_FRAMEPB,TButtonGadget::Exclusive);
  FrameB->SetNotchCorners(FALSE);
  SectionB=new TButtonGadget(BITMAP_AXISSECTION,CM_SECTIONPB,TButtonGadget::Exclusive);
  SectionB->SetNotchCorners(FALSE);
  CrossB=new TButtonGadget(BITMAP_AXISCROSS,CM_CROSSPB,TButtonGadget::Exclusive);
  CrossB->SetNotchCorners(FALSE);
  SingleB=new TButtonGadget(BITMAP_AXISSINGLE,CM_SINGLEPB,TButtonGadget::Exclusive);
  SingleB->SetNotchCorners(FALSE);
  DataB=new TButtonGadget(BITMAP_DATAPOINT,CM_DATAPB,TButtonGadget::Exclusive);
  DataB->SetNotchCorners(FALSE);
  EvalB=new TButtonGadget(BITMAP_EVALUATE,CM_EVALPB,TButtonGadget::Exclusive);
  EvalB->SetNotchCorners(FALSE);
  ZoomB=new TButtonGadget(BITMAP_ZOOM,CM_ZOOMPB,TButtonGadget::Exclusive);
  ZoomB->SetNotchCorners(FALSE);
  NgraphApp->CB2->Insert(*PointB);
  NgraphApp->CB2->Insert(*LegendB);
  NgraphApp->CB2->Insert(*LineB);
  NgraphApp->CB2->Insert(*CurveB);
  NgraphApp->CB2->Insert(*PolyB);
  NgraphApp->CB2->Insert(*RectB);
  NgraphApp->CB2->Insert(*ArcB);
  NgraphApp->CB2->Insert(*MarkB);
  NgraphApp->CB2->Insert(*TextB);
  NgraphApp->CB2->Insert(*GaussB);
  NgraphApp->CB2->Insert(*AxisB);
  NgraphApp->CB2->Insert(*TrimB);
  NgraphApp->CB2->Insert(*FrameB);
  NgraphApp->CB2->Insert(*SectionB);
  NgraphApp->CB2->Insert(*CrossB);
  NgraphApp->CB2->Insert(*SingleB);
  NgraphApp->CB2->Insert(*DataB);
  NgraphApp->CB2->Insert(*EvalB);
  NgraphApp->CB2->Insert(*ZoomB);
  NgraphApp->CB2->Remove(*(NgraphApp->Separator0));
  NgraphApp->CB2->Insert(*(NgraphApp->Separator0));
  NgraphApp->CB2->LayoutSession();
  Mode=PointB;
  Mode->SetButtonState(TButtonGadget::Down);
  Capture=FALSE;
  MoveData=FALSE;
  MouseMode=MOUSENONE;
  focusobj=arraynew(sizeof(struct focuslist *));
  points=arraynew(sizeof(struct pointslist *));
  FrameOfsX=FrameOfsY=0;
  LineX=LineY=0;
  CrossX=CrossY=0;
  ShowFrame=FALSE;
  ShowLine=FALSE;
  ShowRect=FALSE;
  ShowCross=FALSE;
  allclear=TRUE;
  PaintLock=FALSE;
  region=NULL;
  PopupMenu.AppendMenu(MF_STRING,CM_DELETEK,"Delete (DEL)");
  PopupMenu.AppendMenu(MF_STRING,CM_UPDATEK,"Update (CR)");
  PopupMenu.AppendMenu(MF_STRING,CM_COPYK,"Duplicate (INS)");
  PopupMenu.AppendMenu(MF_STRING,CM_TOPK,"Top (HOME)");
  PopupMenu.AppendMenu(MF_STRING,CM_LASTK,"Last (END)");
  PopupMenu.AppendMenu(MF_STRING,CM_CROSSK,"Show cross (TAB)");

  PointBM=new TBitmap(hInstance,BITMAP_POINT);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_POINTPB,*PointBM);
  LegendBM=new TBitmap(hInstance,BITMAP_LEGENDPOINT);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_LEGENDPB,*LegendBM);
  LineBM=new TBitmap(hInstance,BITMAP_LEGENDLINE);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_LINEPB,*LineBM);
  CurveBM=new TBitmap(hInstance,BITMAP_LEGENDCURVE);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_CURVEPB,*CurveBM);
  PolyBM=new TBitmap(hInstance,BITMAP_LEGENDPOLYGON);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_POLYPB,*PolyBM);
  RectBM=new TBitmap(hInstance,BITMAP_LEGENDRECT);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_RECTPB,*RectBM);
  ArcBM=new TBitmap(hInstance,BITMAP_LEGENDARC);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_ARCPB,*ArcBM);
  MarkBM=new TBitmap(hInstance,BITMAP_LEGENDMARK);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_MARKPB,*MarkBM);
  TextBM=new TBitmap(hInstance,BITMAP_LEGENDTEXT);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_TEXTPB,*TextBM);
  GaussBM=new TBitmap(hInstance,BITMAP_LEGENDGAUSS);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_GAUSSPB,*GaussBM);
  AxisBM=new TBitmap(hInstance,BITMAP_AXISPOINT);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_AXISPB,*AxisBM);
  TrimBM=new TBitmap(hInstance,BITMAP_AXISTRIMMING);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_TRIMPB,*TrimBM);
  FrameBM=new TBitmap(hInstance,BITMAP_AXISFRAME);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_FRAMEPB,*FrameBM);
  SectionBM=new TBitmap(hInstance,BITMAP_AXISSECTION);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_SECTIONPB,*SectionBM);
  CrossBM=new TBitmap(hInstance,BITMAP_AXISCROSS);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_CROSSPB,*CrossBM);
  SingleBM=new TBitmap(hInstance,BITMAP_AXISSINGLE);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_SINGLEPB,*SingleBM);
  DataBM=new TBitmap(hInstance,BITMAP_DATAPOINT);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_DATAPB,*DataBM);
  EvalBM=new TBitmap(hInstance,BITMAP_EVALUATE);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_EVALPB,*EvalBM);
  ZoomBM=new TBitmap(hInstance,BITMAP_ZOOM);
  PopupMenu2.AppendMenu(MF_BITMAP,CM_ZOOMPB,*ZoomBM);
  SetBkgndColor(RGB(mwlocal->bgR,mwlocal->bgG,mwlocal->bgB));

  OpenGRA();
  OpenGDC();
  ChangeScroller();
  ChangeDPI();
}

void TMyWindow::PointBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
     win=(TViewWindow *)pTViewWindow;
     win->Mode->SetButtonState(TButtonGadget::Up);
     win->Mode=win->PointB;
     win->Mode->SetButtonState(TButtonGadget::Down);
     win->ButtonPushed();
  }
}

void TMyWindow::LegendBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
    win=(TViewWindow *)pTViewWindow;
    win->Mode->SetButtonState(TButtonGadget::Up);
    win->Mode=win->LegendB;
    win->Mode->SetButtonState(TButtonGadget::Down);
    win->ButtonPushed();
  }
}

void TMyWindow::LineBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
    win=(TViewWindow *)pTViewWindow;
    win->Mode->SetButtonState(TButtonGadget::Up);
    win->Mode=win->LineB;
    win->Mode->SetButtonState(TButtonGadget::Down);
    win->ButtonPushed();
  }
}

void TMyWindow::CurveBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
    win=(TViewWindow *)pTViewWindow;
    win->Mode->SetButtonState(TButtonGadget::Up);
    win->Mode=win->CurveB;
    win->Mode->SetButtonState(TButtonGadget::Down);
    win->ButtonPushed();
  }
}

void TMyWindow::RectBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
    win=(TViewWindow *)pTViewWindow;
    win->Mode->SetButtonState(TButtonGadget::Up);
    win->Mode=win->RectB;
    win->Mode->SetButtonState(TButtonGadget::Down);
    win->ButtonPushed();
  }
}

void TMyWindow::ArcBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
    win=(TViewWindow *)pTViewWindow;
    win->Mode->SetButtonState(TButtonGadget::Up);
    win->Mode=win->ArcB;
    win->Mode->SetButtonState(TButtonGadget::Down);
    win->ButtonPushed();
  }
}

void TMyWindow::PolyBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
    win=(TViewWindow *)pTViewWindow;
    win->Mode->SetButtonState(TButtonGadget::Up);
    win->Mode=win->PolyB;
    win->Mode->SetButtonState(TButtonGadget::Down);
    win->ButtonPushed();
  }
}

void TMyWindow::MarkBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
    win=(TViewWindow *)pTViewWindow;
    win->Mode->SetButtonState(TButtonGadget::Up);
    win->Mode=win->MarkB;
    win->Mode->SetButtonState(TButtonGadget::Down);
    win->ButtonPushed();
  }
}

void TMyWindow::TextBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
    win=(TViewWindow *)pTViewWindow;
    win->Mode->SetButtonState(TButtonGadget::Up);
    win->Mode=win->TextB;
    win->Mode->SetButtonState(TButtonGadget::Down);
    win->ButtonPushed();
  }
}

void TMyWindow::GaussBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
    win=(TViewWindow *)pTViewWindow;
    win->Mode->SetButtonState(TButtonGadget::Up);
    win->Mode=win->GaussB;
    win->Mode->SetButtonState(TButtonGadget::Down);
    win->ButtonPushed();
  }
}

void TMyWindow::AxisBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
    win=(TViewWindow *)pTViewWindow;
    win->Mode->SetButtonState(TButtonGadget::Up);
    win->Mode=win->AxisB;
    win->Mode->SetButtonState(TButtonGadget::Down);
    win->ButtonPushed();
  }
}

void TMyWindow::TrimBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
    win=(TViewWindow *)pTViewWindow;
    win->Mode->SetButtonState(TButtonGadget::Up);
    win->Mode=win->TrimB;
    win->Mode->SetButtonState(TButtonGadget::Down);
    win->ButtonPushed();
  }
}

void TMyWindow::FrameBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
    win=(TViewWindow *)pTViewWindow;
    win->Mode->SetButtonState(TButtonGadget::Up);
    win->Mode=win->FrameB;
    win->Mode->SetButtonState(TButtonGadget::Down);
    win->ButtonPushed();
  }
}

void TMyWindow::SectionBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
    win=(TViewWindow *)pTViewWindow;
    win->Mode->SetButtonState(TButtonGadget::Up);
    win->Mode=win->SectionB;
    win->Mode->SetButtonState(TButtonGadget::Down);
    win->ButtonPushed();
  }
}

void TMyWindow::CrossBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
    win=(TViewWindow *)pTViewWindow;
    win->Mode->SetButtonState(TButtonGadget::Up);
    win->Mode=win->CrossB;
    win->Mode->SetButtonState(TButtonGadget::Down);
    win->ButtonPushed();
  }
}

void TMyWindow::SingleBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
    win=(TViewWindow *)pTViewWindow;
    win->Mode->SetButtonState(TButtonGadget::Up);
    win->Mode=win->SingleB;
    win->Mode->SetButtonState(TButtonGadget::Down);
    win->ButtonPushed();
  }
}

void TMyWindow::DataBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
    win=(TViewWindow *)pTViewWindow;
    win->Mode->SetButtonState(TButtonGadget::Up);
    win->Mode=win->DataB;
    win->Mode->SetButtonState(TButtonGadget::Down);
    win->ButtonPushed();
  }
}

void TMyWindow::EvalBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
    win=(TViewWindow *)pTViewWindow;
    win->Mode->SetButtonState(TButtonGadget::Up);
    win->Mode=win->EvalB;
    win->Mode->SetButtonState(TButtonGadget::Down);
    win->ButtonPushed();
  }
}

void TMyWindow::ZoomBPushed()
{
  TViewWindow *win;

  if (pTViewWindow!=NULL) {
    win=(TViewWindow *)pTViewWindow;
    win->Mode->SetButtonState(TButtonGadget::Up);
    win->Mode=win->ZoomB;
    win->Mode->SetButtonState(TButtonGadget::Down);
    win->ButtonPushed();
  }
}

void TViewWindow::ButtonPushed()
{
  UnFocus();
  if ((Mode==PointB) || (Mode==LegendB) || (Mode==AxisB) || (Mode==TrimB)
   || (Mode==DataB) || (Mode==EvalB)) {
    SetCursor(NULL,IDC_ARROW);
  } else if (Mode==TextB) {
    SetCursor(NULL,IDC_IBEAM);
  } else if (Mode==ZoomB) {
    SetCursor(NgraphApp->GetApplication(),CURSOR_ZOOMIN);
  } else {
    SetCursor(NULL,IDC_CROSS);
  }
  ReleaseCapture();
  Capture=FALSE;
  MouseMode=MOUSENONE;
}

void TViewWindow::CloseWindow(int retValue)
{
  CloseGDC();
  CloseGRA();
  TWindow::CloseWindow(retValue);
}

void TViewWindow::ShowPoints(HDC dc)
{
  int i,num,x1,y1,x2,y2;
  struct pointslist **po;
  int smode;
  HPEN pen,spen;
  double zoom;
  COLORREF col1,col2;

  col1=(~ GetSysColor(COLOR_WINDOW)) & RGB(255,255,255);
  col2=GetSysColor(COLOR_WINDOW);
  pen=CreatePen(PS_DOT,0,col1);
  spen=(HPEN)SelectObject(dc,pen);
  smode=SetROP2(dc,R2_XORPEN);
  num=arraynum(points);
  po=(struct pointslist **)arraydata(points);
  zoom=menulocal.PaperZoom/10000.0;
  if ((Mode==RectB) || (Mode==ArcB) || (Mode==GaussB)
  || (Mode==FrameB) || (Mode==SectionB) || (Mode==CrossB)) {
	if (num==2) {
	  pen=CreatePen(PS_DOT,0,col1);
	  DeleteObject(SelectObject(dc,pen));
	  x1=mwd2p(po[0]->x*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
	  y1=mwd2p(po[0]->y*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
	  x2=mwd2p(po[1]->x*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
	  y2=mwd2p(po[1]->y*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
	  MoveToEx(dc,x1,y1,NULL);
	  LineTo(dc,x1,y2);
	  LineTo(dc,x2,y2);
	  LineTo(dc,x2,y1);
	  LineTo(dc,x1,y1);
	}
  } else {
	pen=CreatePen(PS_SOLID,0,col2);
	DeleteObject(SelectObject(dc,pen));
	for (i=0;i<num;i++) {
	  x1=mwd2p(po[i]->x*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
	  y1=mwd2p(po[i]->y*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
	  MoveToEx(dc,x1-4,y1,NULL);
	  LineTo(dc,x1+5,y1);
	  MoveToEx(dc,x1,y1-4,NULL);
	  LineTo(dc,x1,y1+5);
	}
	if (num>=1) {
	  x1=mwd2p(po[0]->x*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
	  y1=mwd2p(po[0]->y*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
	  MoveToEx(dc,x1,y1,NULL);
	}
	pen=CreatePen(PS_SOLID,0,col2);
	DeleteObject(SelectObject(dc,pen));
	for (i=1;i<num;i++) {
	  x1=mwd2p(po[i]->x*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
	  y1=mwd2p(po[i]->y*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
	  LineTo(dc,x1,y1);
	}
  }
  DeleteObject(SelectObject(dc,spen));
  SetROP2(dc,smode);
}

void TViewWindow::Focus(struct objlist *fobj,int id)
{
  int oid;
  char *inst;
  struct narray *sarray;
  char **sdata;
  int snum;
  struct objlist *dobj;
  int did;
  char *dfield;
  int i;
  struct focuslist *focus;
  struct savedstdio save;
  HDC dc;
  int man;

  if (chkobjchild(chkobject("legend"),fobj)) {
    NgraphApp->LegendBPushed();
  } else if (chkobjchild(chkobject("axis"),fobj)) {
    NgraphApp->AxisBPushed();
  } else if (chkobjchild(chkobject("merge"),fobj)) {
    NgraphApp->PointBPushed();
  } else return;
  ButtonPushed();
  UnFocus();
  dc=GetDC(Handle);
  inst=chkobjinst(fobj,id);
  _getobj(fobj,"oid",inst,&oid);
  ignorestdio(&save);
  if (fobj!=NULL) {
    if (_getobj(menulocal.obj,"_list",menulocal.inst,&sarray)) return;
    if ((snum=arraynum(sarray))==0) return;
    sdata=(char **)arraydata(sarray);
    for (i=1;i<snum;i++) {
      if (((dobj=getobjlist(sdata[i],&did,&dfield,NULL))!=NULL)
      && (fobj==dobj) && (did==oid)) {
        if ((focus=(struct focuslist *)memalloc(sizeof(struct focuslist)))!=NULL) {
          if (chkobjchild(chkobject("axis"),dobj)) {
            getobj(fobj,"group_manager",id,0,NULL,&man);
            getobj(fobj,"oid",man,0,NULL,&did);
          }
          focus->obj=dobj;
          focus->oid=did;
          arrayadd(focusobj,&focus);
          ShowFrame=TRUE;
          ShowFocusFrame(dc);
        }
        MouseMode=MOUSENONE;
        break;
      }
    }
  }
  ReleaseDC(Handle,dc);
  restorestdio(&save);
}

void TViewWindow::Match(char *objname,int x1,int y1,int x2,int y2,int err)
{
  struct objlist *fobj;
  char *argv[6];
  struct narray *sarray;
  char **sdata;
  int snum;
  struct objlist *dobj;
  int did;
  char *dfield,*dinst;
  int i,match;
  struct focuslist *focus;
  int minx,miny,maxx,maxy;
  struct savedstdio save;

  ignorestdio(&save);
  minx=min(x1,x2);
  miny=min(y1,y2);
  maxx=max(x1,x2);
  maxy=max(y1,y2);
  argv[0]=(char *)&minx;
  argv[1]=(char *)&miny;
  argv[2]=(char *)&maxx;
  argv[3]=(char *)&maxy;
  argv[4]=(char *)&err;
  argv[5]=NULL;
  if ((fobj=chkobject(objname))!=NULL) {
    if (_getobj(menulocal.obj,"_list",menulocal.inst,&sarray)) return;
    if ((snum=arraynum(sarray))==0) return;
    sdata=(char **)arraydata(sarray);
    for (i=1;i<snum;i++) {
      if (((dobj=getobjlist(sdata[i],&did,&dfield,NULL))!=NULL)
       && chkobjchild(fobj,dobj)) {
        if ((dinst=chkobjinstoid(dobj,did))!=NULL) {
          _exeobj(dobj,"match",dinst,5,argv);
          _getobj(dobj,"match",dinst,&match);
          if (match) {
            if ((focus=(struct focuslist *)memalloc(sizeof(struct focuslist)))!=NULL) {
              focus->obj=dobj;
              focus->oid=did;
              arrayadd(focusobj,&focus);
            }
          }
        }
      }
    }
  }
  restorestdio(&save);
}

void TViewWindow::Trimming(int x1,int y1,int x2,int y2)
{
  SelectDialog *dlg;
  struct narray farray;
  struct objlist *obj;
  int i,id;
  int *array,num;
  int vx1,vy1,vx2,vy2;
  int maxx,maxy,minx,miny;
  int dir,rcode1,rcode2,room;
  double ax,ay,ip1,ip2,min,max;
  char *argv[4];

  if ((x1==x2) && (y1==y2)) return;
  if ((obj=chkobject("axis"))==NULL) return;
  if (chkobjlastinst(obj)==-1) return;
  dlg=new SelectDialog((TWindow *)this,DIALOG_SELECT,obj,AxisCB,
                       (struct narray *)&farray,NULL);
  if (dlg->Execute()==IDOK) {
    vx1=x1-x2;
    vy1=y1-y2;
    vx2=x2-x1;
    vy2=y1-y2;

    num=arraynum(&farray);
    array=(int *)arraydata(&farray);

    for (i=0;i<num;i++) {
      id=array[i];
      getobj(obj,"direction",id,0,NULL,&dir);
      ax=cos(dir/18000.0*MPI);
      ay=-sin(dir/18000.0*MPI);
      ip1=ax*vx1+ay*vy1;
      ip2=ax*vx2+ay*vy2;
      if (fabs(ip1)>fabs(ip2)) {
        if (ip1>0) {
          maxx=x1;
          maxy=y1;
          minx=x2;
          miny=y2;
        } else if (ip1<0) {
          maxx=x2;
          maxy=y2;
          minx=x1;
          miny=y1;
        } else {
          maxx=minx=0;
          maxy=miny=0;
        }
      } else {
        if (ip2>0) {
          maxx=x2;
          maxy=y1;
          minx=x1;
          miny=y2;
        } else if (ip2<0) {
          maxx=x1;
          maxy=y2;
          minx=x2;
          miny=y1;
        } else {
          maxx=minx=0;
          maxy=miny=0;
        }
      }
      if ((minx!=maxx) && (miny!=maxy)) {
        argv[0]=(char *)&minx;
        argv[1]=(char *)&miny;
        argv[2]=NULL;
        rcode1=getobj(obj,"coordinate",id,2,argv,&min);
        argv[0]=(char *)&maxx;
        argv[1]=(char *)&maxy;
        argv[2]=NULL;
        rcode2=getobj(obj,"coordinate",id,2,argv,&max);
        if ((rcode1!=-1) && (rcode2!=-1)) {
          room=1;
          exeobj(obj,"scale_push",id,0,NULL);
          argv[0]=(char *)&min;
          argv[1]=(char *)&max;
          argv[2]=(char *)&room;
          argv[3]=NULL;
          exeobj(obj,"scale",id,3,argv);
          NgraphApp->Changed=TRUE;
        }
      }
    }
    AdjustAxis();
    allclear=TRUE;
    NgraphApp->Update();
  }
  arraydel(&farray);
  delete dlg;
}

void TViewWindow::Evaluate(int x1,int y1,int x2,int y2,int err)
{
  struct objlist *fileobj;
  char *argv[7];
  struct narray *sarray;
  char **sdata;
  int snum;
  struct objlist *dobj;
  int did,id,limit;
  char *dfield,*dinst;
  int i,j;
  struct narray *eval;
  int evalnum,tot;
  int minx,miny,maxx,maxy;
  struct savedstdio save;
  double line,dx,dy;
  char mes[256];
  EvalDialog *dlg;
  int ret;
  int selnum,sel;
  int masknum,iline;
  struct narray *mask;

  ignorestdio(&save);
  minx=min(x1,x2);
  miny=min(y1,y2);
  maxx=max(x1,x2);
  maxy=max(y1,y2);
  limit=100;
  argv[0]=(char *)&minx;
  argv[1]=(char *)&miny;
  argv[2]=(char *)&maxx;
  argv[3]=(char *)&maxy;
  argv[4]=(char *)&err;
  argv[5]=(char *)&limit;
  argv[6]=NULL;
  if ((fileobj=chkobject("file"))!=NULL) {
	sprintf(mes,"Evaluating.");
	NgraphApp->SetStatusBar(mes);
	if (_getobj(menulocal.obj,"_list",menulocal.inst,&sarray)) return;
	if ((snum=arraynum(sarray))==0) return;
	sdata=(char **)arraydata(sarray);
	tot=0;
	for (i=1;i<snum;i++) {
	  if (((dobj=getobjlist(sdata[i],&did,&dfield,NULL))!=NULL)
	   && (fileobj==dobj)) {
		if ((dinst=chkobjinstoid(dobj,did))!=NULL) {
		  _getobj(dobj,"id",dinst,&id);
		  _exeobj(dobj,"evaluate",dinst,6,argv);
		  _getobj(dobj,"evaluate",dinst,&eval);
		  evalnum=arraynum(eval)/3;
		  for (j=0;j<evalnum;j++) {
			if (tot>=limit) break;
			tot++;
			line=*(double *)arraynget(eval,j*3+0);
			dx=*(double *)arraynget(eval,j*3+1);
			dy=*(double *)arraynget(eval,j*3+2);
			evallist[tot-1].id=id;
			evallist[tot-1].line=nround(line);
			evallist[tot-1].x=dx;
			evallist[tot-1].y=dy;
		  }
		  if (tot>=limit) break;
		}
	  }
	}
	NgraphApp->ResetStatusBar();
    if (tot>0) {
      dlg=new EvalDialog(this,DIALOG_EVALUATE,fileobj,tot,&sellist);
      ret=dlg->Execute();
      delete dlg;
      selnum=arraynum(&sellist);
      if (ret==IDEVMASK) {
        for (i=0;i<selnum;i++) {
          sel=*(int *)arraynget(&sellist,i);
          getobj(fileobj,"mask",evallist[sel].id,0,NULL,&mask);
          if (mask==NULL) {
            mask=arraynew(sizeof(int));
            putobj(fileobj,"mask",evallist[sel].id,mask);
          }
          masknum=arraynum(mask);
          for (j=0;j<masknum;j++) {
            iline=*(int *)arraynget(mask,j);
            if (iline==evallist[sel].line) break;
          }
          if (j==masknum) {
            arrayadd(mask,&(evallist[sel].line));
            NgraphApp->Changed=TRUE;
          }
        }
        arraydel(&sellist);
      } else if ((ret==IDEVMOVE) && (selnum>0)) {
        SetCursor(NULL,IDC_CROSS);
        Capture=TRUE;
        MoveData=TRUE;
      }
    }
  }
  restorestdio(&save);
}

int TViewWindow::FileUpdate(int x1,int y1,int x2,int y2,int err)
{
  struct objlist *fileobj;
  char *argv[7];
  struct narray *sarray;
  char **sdata;
  int snum;
  struct objlist *dobj;
  int did,id,limit;
  char *dfield,*dinst;
  int i,j;
  struct narray *eval;
  int evalnum;
  int minx,miny,maxx,maxy;
  struct savedstdio save;
  char mes[256];
  FileDialog *dlg;
  struct narray dfile;
  int *ddata,dnum;
  int ret,ret2;

  ret=FALSE;
  ignorestdio(&save);
  minx=min(x1,x2);
  miny=min(y1,y2);
  maxx=max(x1,x2);
  maxy=max(y1,y2);
  limit=1;
  argv[0]=(char *)&minx;
  argv[1]=(char *)&miny;
  argv[2]=(char *)&maxx;
  argv[3]=(char *)&maxy;
  argv[4]=(char *)&err;
  argv[5]=(char *)&limit;
  argv[6]=NULL;
  if ((fileobj=chkobject("file"))!=NULL) {
    arrayinit(&dfile,sizeof(int));
	sprintf(mes,"Searching for data.");
	NgraphApp->SetStatusBar(mes);
	if (_getobj(menulocal.obj,"_list",menulocal.inst,&sarray)) return ret;
	if ((snum=arraynum(sarray))==0) return ret;
	sdata=(char **)arraydata(sarray);
	for (i=1;i<snum;i++) {
	  if (((dobj=getobjlist(sdata[i],&did,&dfield,NULL))!=NULL)
	   && chkobjchild(fileobj,dobj)) {
		if ((dinst=chkobjinstoid(dobj,did))!=NULL) {
		  _getobj(dobj,"id",dinst,&id);
		  _exeobj(dobj,"evaluate",dinst,6,argv);
		  _getobj(dobj,"evaluate",dinst,&eval);
		  evalnum=arraynum(eval)/3;
          if (evalnum!=0) arrayadd(&dfile,&id);
		}
	  }
	}
	NgraphApp->ResetStatusBar();
    dnum=arraynum(&dfile);
    ddata=(int *)arraydata(&dfile);
    for (i=0;i<dnum;i++) {
      id=ddata[i];
      dlg=new FileDialog(this,DIALOG_FILE,fileobj,id);
	   if ((ret2=dlg->Execute())==IDDELETE) {
        FitDel(fileobj,id);
        delobj(fileobj,id);
        for (j=i+1;j<dnum;j++) if (ddata[j]>id) ddata[j]=ddata[j]-1;
      }
      if (ret2!=IDCANCEL) NgraphApp->Changed=TRUE;
	   delete dlg;
      ret=TRUE;
    }
    arraydel(&dfile);
  }
  restorestdio(&save);
  return ret;
}

void TViewWindow::GetLargeFrame(int *minx,int *miny,int *maxx,int *maxy)
{
  int i,num;
  struct focuslist **focus;
  struct narray *abbox;
  int bboxnum,*bbox;
  char *inst;
  struct savedstdio save;

  ignorestdio(&save);
  *minx=*miny=*maxx=*maxy=0;
  num=arraynum(focusobj);
  focus=(struct focuslist **)arraydata(focusobj);
  if ((inst=chkobjinstoid(focus[0]->obj,focus[0]->oid))!=NULL) {
    _exeobj(focus[0]->obj,"bbox",inst,0,NULL);
    _getobj(focus[0]->obj,"bbox",inst,&abbox);
    bboxnum=arraynum(abbox);
    bbox=(int *)arraydata(abbox);
    if (bboxnum>=4) {
      *minx=bbox[0];
      *miny=bbox[1];
      *maxx=bbox[2];
      *maxy=bbox[3];
    }
  }
  for (i=1;i<num;i++)
  if ((inst=chkobjinstoid(focus[i]->obj,focus[i]->oid))!=NULL) {
    _exeobj(focus[i]->obj,"bbox",inst,0,NULL);
    _getobj(focus[i]->obj,"bbox",inst,&abbox);
    bboxnum=arraynum(abbox);
    bbox=(int *)arraydata(abbox);
    if (bboxnum>=4) {
      if (bbox[0]<*minx) *minx=bbox[0];
      if (bbox[1]<*miny) *miny=bbox[1];
      if (bbox[2]>*maxx) *maxx=bbox[2];
      if (bbox[3]>*maxy) *maxy=bbox[3];
    }
  }
  restorestdio(&save);
}

void TViewWindow::ShowFocusLine(HDC dc,int change)
{
  int j,num;
  struct focuslist **focus;
  struct narray *abbox;
  int bboxnum;
  int *bbox;
  HPEN pen,spen;
  int smode;
  int x1,y1,x2,y2,ofsx,ofsy;
  char *inst;
  struct savedstdio save;
  COLORREF col;
  double zoom;
  int frame;
  char *group;

  ignorestdio(&save);
  col=(~GetSysColor(COLOR_WINDOW)) & RGB(255,255,255);
  pen=CreatePen(PS_DOT,0,col);
  spen=(HPEN)SelectObject(dc,pen);
  smode=SetROP2(dc,R2_XORPEN);
  num=arraynum(focusobj);
  focus=(struct focuslist **)arraydata(focusobj);
  zoom=menulocal.PaperZoom/10000.0;
  if (num==1) {
	if ((inst=chkobjinstoid(focus[0]->obj,focus[0]->oid))!=NULL) {
	  _exeobj(focus[0]->obj,"bbox",inst,0,NULL);
	  _getobj(focus[0]->obj,"bbox",inst,&abbox);
	  bboxnum=arraynum(abbox);
	  bbox=(int *)arraydata(abbox);
      frame=FALSE;
      if (focus[0]->obj==chkobject("rectangle")) frame=TRUE;
      if (focus[0]->obj==chkobject("axis")) {
        _getobj(focus[0]->obj,"group",inst,&group);
        if ((group!=NULL) && (group[0]!='a')) frame=TRUE;
      }
      if (!frame) {
        for (j=4;j<bboxnum;j+=2) {
          if (change==(j-4)/2) {
            ofsx=LineX;
            ofsy=LineY;
          } else {
            ofsx=0;
            ofsy=0;
          }
          x1=mwd2p((bbox[j  ]+ofsx)*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
          y1=mwd2p((bbox[j+1]+ofsy)*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
          if (j==4) MoveToEx(dc,x1,y1,NULL);
          else LineTo(dc,x1,y1);
        }
      } else {
        if (change==0) {
          x1=mwd2p((bbox[4]+LineX)*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
          y1=mwd2p((bbox[5]+LineY)*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
          x2=mwd2p((bbox[8])*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
          y2=mwd2p((bbox[9])*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
        } else if (change==1) {
          x1=mwd2p((bbox[4])*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
          y1=mwd2p((bbox[5]+LineY)*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
          x2=mwd2p((bbox[8]+LineX)*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
          y2=mwd2p((bbox[9])*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
        } else if (change==2) {
          x1=mwd2p((bbox[4])*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
          y1=mwd2p((bbox[5])*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
          x2=mwd2p((bbox[8]+LineX)*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
          y2=mwd2p((bbox[9]+LineY)*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
        } else if (change==3) {
          x1=mwd2p((bbox[4]+LineX)*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
          y1=mwd2p((bbox[5])*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
          x2=mwd2p((bbox[8])*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
          y2=mwd2p((bbox[9]+LineY)*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
        }
        MoveToEx(dc,x1,y1,NULL);
        LineTo(dc,x2,y1);
        LineTo(dc,x2,y2);
        LineTo(dc,x1,y2);
        LineTo(dc,x1,y1);
      }
	}
  }
  DeleteObject(SelectObject(dc,spen));
  SetROP2(dc,smode);
  restorestdio(&save);
}

void TViewWindow::ShowCrossGauge(HDC dc)
{
  HPEN pen,spen;
  int smode;
  RECT rect;
  COLORREF col;
  int x,y;
  double zoom;

  col=(~GetSysColor(COLOR_GRAYTEXT)) & RGB(255,255,255);
  pen=CreatePen(PS_SOLID,0,col);
  spen=(HPEN)SelectObject(dc,pen);
  smode=SetROP2(dc,R2_XORPEN);
  ::GetClientRect(Handle,&rect);
  zoom=menulocal.PaperZoom/10000.0;
  x=mwd2p(CrossX*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
  y=mwd2p(CrossY*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
  MoveToEx(dc,x,rect.top,NULL);
  LineTo(dc,x,rect.bottom);
  MoveToEx(dc,rect.left,y,NULL);
  LineTo(dc,rect.right,y);
  DeleteObject(SelectObject(dc,spen));
  SetROP2(dc,smode);
}

void TViewWindow::ShowFocusFrame(HDC dc)
{
  int i,j,num;
  struct focuslist **focus;
  struct narray *abbox;
  int bboxnum;
  int *bbox;
  HPEN pen,spen;
  HBRUSH brush,sbrush;
  int smode;
  int x1,y1,x2,y2;
  char *inst;
  struct savedstdio save;
  COLORREF col;
  double zoom;

  ignorestdio(&save);
  col=(~GetSysColor(COLOR_WINDOW)) & RGB(255,255,255);
  pen=CreatePen(PS_DOT,0,col);
  brush=CreateSolidBrush(GetSysColor(COLOR_WINDOW));
  spen=(HPEN)SelectObject(dc,pen);
  sbrush=(HBRUSH)SelectObject(dc,brush);
  smode=SetROP2(dc,R2_XORPEN);
  num=arraynum(focusobj);
  focus=(struct focuslist **)arraydata(focusobj);
  if (num>0) {
	GetFocusFrame(&x1,&y1,&x2,&y2,FrameOfsX,FrameOfsY);
	x1-=5;
	y1-=5;
	x2+=4;
	y2+=4;
	pen=CreatePen(PS_DOT,0,col);
	DeleteObject(SelectObject(dc,pen));
	MoveToEx(dc,x1,y1,NULL);
	LineTo(dc,x1,y2);
	LineTo(dc,x2,y2);
	LineTo(dc,x2,y1);
	LineTo(dc,x1,y1);
	pen=(HPEN)GetStockObject(NULL_PEN);
	DeleteObject(SelectObject(dc,pen));
	Rectangle(dc,x1-6,y1-6,x1+1,y1+1);
	Rectangle(dc,x1-6,y2,x1+1,y2+6+1);
	Rectangle(dc,x2,y1-6,x2+6+1,y1+1);
	Rectangle(dc,x2,y2,x2+6+1,y2+6+1);
  }
  zoom=menulocal.PaperZoom/10000.0;
  if (num>1) {
	pen=CreatePen(PS_DOT,0,col);
	DeleteObject(SelectObject(dc,pen));
	for (i=0;i<num;i++) {
	  if ((inst=chkobjinstoid(focus[i]->obj,focus[i]->oid))!=NULL) {
		_exeobj(focus[i]->obj,"bbox",inst,0,NULL);
		_getobj(focus[i]->obj,"bbox",inst,&abbox);
		bboxnum=arraynum(abbox);
		bbox=(int *)arraydata(abbox);
		if (bboxnum>=4) {
		  x1=mwd2p((bbox[0]+FrameOfsX)*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
		  y1=mwd2p((bbox[1]+FrameOfsY)*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
		  x2=mwd2p((bbox[2]+FrameOfsX)*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
		  y2=mwd2p((bbox[3]+FrameOfsY)*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
		  MoveToEx(dc,x1,y1,NULL);
		  LineTo(dc,x1,y2);
		  LineTo(dc,x2,y2);
		  LineTo(dc,x2,y1);
		  LineTo(dc,x1,y1);
		}
	  }
	}
  } else if (num==1) {
	i=0;
	if ((inst=chkobjinstoid(focus[i]->obj,focus[i]->oid))!=NULL) {
	  _exeobj(focus[i]->obj,"bbox",inst,0,NULL);
	  _getobj(focus[i]->obj,"bbox",inst,&abbox);
	  bboxnum=arraynum(abbox);
	  bbox=(int *)arraydata(abbox);
	  for (j=4;j<bboxnum;j+=2) {
		x1=mwd2p((bbox[j  ]+FrameOfsX)*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
		y1=mwd2p((bbox[j+1]+FrameOfsY)*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
		Rectangle(dc,x1-3,y1-3,x1+3+1,y1+3+1);
	  }
	}
  }
  DeleteObject(SelectObject(dc,spen));
  DeleteObject(SelectObject(dc,sbrush));
  SetROP2(dc,smode);
  restorestdio(&save);
}

void TViewWindow::GetFocusFrame(int *minx,int *miny,int *maxx,int *maxy,int ofsx,int ofsy)
{
  int x1,y1,x2,y2;
  double zoom;

  GetLargeFrame(&x1,&y1,&x2,&y2);
  zoom=menulocal.PaperZoom/10000.0;
  *minx=mwd2p((x1+ofsx)*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
  *miny=mwd2p((y1+ofsy)*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
  *maxx=mwd2p((x2+ofsx)*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
  *maxy=mwd2p((y2+ofsy)*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
}

void TViewWindow::AddList(struct objlist *obj,char *inst)
{
  int addi;
  struct objlist *aobj;
  char *ainst;
  char *afield;
  int i,j,po,num,oid,id,id2;
  struct objlist **objlist;
  struct objlist *obj2;
  char *inst2;
  char *field,**objname;
  struct narray *draw,drawrable;

  aobj=obj;
  ainst=inst;
  afield="draw";
  addi=-1;
  _getobj(obj,"id",inst,&id);
  draw=&(menulocal.drawrable);
  num=arraynum(draw);
  if (num==0) {
    arrayinit(&drawrable,sizeof(char *));
    menuadddrawrable(chkobject("draw"),&drawrable);
    draw=&drawrable;
    num=arraynum(draw);
  }
  if ((objlist=(struct objlist **)memalloc(sizeof(struct objlist *)*num))==NULL) return;
  po=0;
  for (i=0;i<num;i++) {
    objname=(char **)arraynget(draw,i);
    objlist[i]=chkobject(*objname);
    if (objlist[i]==obj) po=i;
  }
  i=1;
  j=0;
  while ((obj2=GRAgetlist(menulocal.GC,&oid,&field,i))!=NULL) {
   for (;j<num;j++)
     if (objlist[j]==obj2) break;
   if (j==po) {
     inst2=chkobjinstoid(obj2,oid);
     _getobj(obj2,"id",inst2,&id2);
     if (id2>id) {
       addi=i;
       memfree(objlist);
       mw_inslist(menulocal.obj,menulocal.inst,aobj,ainst,afield,addi);
       if (draw!=&(menulocal.drawrable)) arraydel2(draw);
       return;
     }
   } else if (j>po) {
     addi=i;
     memfree(objlist);
     mw_inslist(menulocal.obj,menulocal.inst,aobj,ainst,afield,addi);
     if (draw!=&(menulocal.drawrable)) arraydel2(draw);
     return;
   }
   i++;
  }
  addi=i;
  memfree(objlist);
  mw_inslist(menulocal.obj,menulocal.inst,aobj,ainst,afield,addi);
  if (draw!=&(menulocal.drawrable)) arraydel2(draw);
}

void TViewWindow::DelList(struct objlist *obj,char *inst)
{
  int i,oid,oid2;
  struct objlist *obj2;
  char *field;

  _getobj(obj,"oid",inst,&oid);
  i=0;
  while ((obj2=GRAgetlist(menulocal.GC,&oid2,&field,i))!=NULL) {
    if ((obj2==obj) && (oid==oid2))
      mw_dellist(menulocal.obj,menulocal.inst,i);
    i++;
  }
}

void TViewWindow::AddInvalidateRect(struct objlist *obj,char *inst)
{
  RECT brect;
  struct narray *abbox;
  int bboxnum,*bbox;
  double zoom;
  HRGN rgn;

  _exeobj(obj,"bbox",inst,0,NULL);
  _getobj(obj,"bbox",inst,&abbox);
  bboxnum=arraynum(abbox);
  bbox=(int *)arraydata(abbox);
  if (bboxnum>=4) {
    zoom=menulocal.PaperZoom/10000.0;
    brect.left=mwd2p(bbox[0]*zoom+menulocal.LeftMargin)-Scroller->XPos+cx-7;
    brect.top=mwd2p(bbox[1]*zoom+menulocal.TopMargin)-Scroller->YPos+cy-7;
    brect.right=mwd2p(bbox[2]*zoom+menulocal.LeftMargin)-Scroller->XPos+cx+7;
    brect.bottom=mwd2p(bbox[3]*zoom+menulocal.TopMargin)-Scroller->YPos+cy+7;
    if (region==NULL) region=CreateRectRgnIndirect(&brect);
    else {
      rgn=CreateRectRgnIndirect(&brect);
      CombineRgn(region,region,rgn,RGN_OR);
      DeleteObject(rgn);
    }
//    ::InvalidateRect(Handle,&brect,FALSE);
  }
}

void TViewWindow::CheckGrid(int ofs,UINT modKeys,int *x,int *y,double *zoom)
{
  int offset;
  int grid;

  if ((modKeys & MK_CONTROL) && (!ofs)) {
    if ((x!=NULL) && (y!=NULL)) {
      if (abs(*x)>abs(*y)) *y=0;
      else *x=0;
    }
  }
  grid=mwlocal->grid;
  if (!(modKeys & MK_SHIFT)) {
    if (ofs) offset=grid/2;
    else offset=0;
    if (x!=NULL) *x=((*x+offset)/grid)*grid;
    if (y!=NULL) *y=((*y+offset)/grid)*grid;
    if (zoom!=NULL) *zoom=nround(*zoom*grid)/((double )grid);
  }
}

void TViewWindow::EvLButtonDown(UINT modKeys,TPoint& point)
{
  HDC dc;
  int minx,miny,maxx,maxy;
  int x1,y1,vx1,vx2,vy1,vy2;
  double cc,nn;
  struct pointslist *po;
  double zoom,zoom2;
  int i,j;
  struct narray *abbox;
  int bboxnum;
  int *bbox;
  char *inst;
  struct focuslist *focus;
  TRect rect;
  int vdpi;
  int selnum,sel,movenum,iline;
  struct narray *move,*movex,*movey;
  struct objlist *fileobj,*aobjx,*aobjy;
  char *argv[3];
  double dx,dy;
  int ax,ay,anum;
  struct narray iarray;
  char *axis;

  Scroller->AutoMode=TRUE;
  if (menulock || globallock) return;
  GetUpdateRect(rect,FALSE);
  if ((rect.right!=0) || (rect.left!=0) || (rect.top!=0) || (rect.bottom!=0)) return;
  zoom=menulocal.PaperZoom/10000.0;
  MouseX1=MouseX2=(mwp2d(point.x-cx+Scroller->XPos)-menulocal.LeftMargin)/zoom;
  MouseY1=MouseY2=(mwp2d(point.y-cy+Scroller->YPos)-menulocal.TopMargin)/zoom;
  MouseMode=MOUSENONE;
  dc=GetDC(Handle);
  if (MoveData) {
    if ((fileobj=chkobject("file"))!=NULL) {
      selnum=arraynum(&sellist);
      for (i=0;i<selnum;i++) {
        sel=*(int *)arraynget(&sellist,i);
        getobj(fileobj,"axis_x",evallist[sel].id,0,NULL,&axis);
        arrayinit(&iarray,sizeof(int));
        if (getobjilist(axis,&aobjx,&iarray,FALSE,NULL)) ax=-1;
        else {
          anum=arraynum(&iarray);
          if (anum<1) ax=-1;
          else ax=*(int *)arraylast(&iarray);
          arraydel(&iarray);
        }
        getobj(fileobj,"axis_y",evallist[sel].id,0,NULL,&axis);
        arrayinit(&iarray,sizeof(int));
        if (getobjilist(axis,&aobjy,&iarray,FALSE,NULL)) ay=-1;
        else {
          anum=arraynum(&iarray);
          if (anum<1) ay=-1;
          else ay=*(int *)arraylast(&iarray);
          arraydel(&iarray);
        }
        if ((ax!=-1) && (ax!=-1)) {
          argv[0]=(char *)&MouseX1;
          argv[1]=(char *)&MouseY1;
          argv[2]=NULL;
          if ((getobj(aobjx,"coordinate",ax,2,argv,&dx)!=-1)
           && (getobj(aobjy,"coordinate",ay,2,argv,&dy)!=-1)) {
            getobj(fileobj,"move_data",evallist[sel].id,0,NULL,&move);
            getobj(fileobj,"move_data_x",evallist[sel].id,0,NULL,&movex);
            getobj(fileobj,"move_data_y",evallist[sel].id,0,NULL,&movey);
            if (move==NULL) {
              move=arraynew(sizeof(int));
              putobj(fileobj,"move_data",evallist[sel].id,move);
            }
            if (movex==NULL) {
              movex=arraynew(sizeof(double));
              putobj(fileobj,"move_data_x",evallist[sel].id,movex);
            }
            if (movey==NULL) {
              movey=arraynew(sizeof(double));
              putobj(fileobj,"move_data_y",evallist[sel].id,movey);
            }
            movenum=arraynum(move);
            if (arraynum(movex)<movenum) {
              for (j=movenum-1;j>=arraynum(movex);j--) {
                arrayndel(move,j);
              }
              movenum=arraynum(movex);
            }
            if (arraynum(movey)<movenum) {
              for (j=movenum-1;j>=arraynum(movey);j--) {
                arrayndel(move,j);
                arrayndel(movex,j);
              }
              movenum=arraynum(movey);
            }
            for (j=0;j<movenum;j++) {
              iline=*(int *)arraynget(move,j);
              if (iline==evallist[sel].line) break;
            }
            if (j==movenum) {
              arrayadd(move,&(evallist[sel].line));
              arrayadd(movex,&dx);
              arrayadd(movey,&dy);
              NgraphApp->Changed=TRUE;
            } else {
              arrayput(move,&(evallist[sel].line),j);
              arrayput(movex,&dx,j);
              arrayput(movey,&dy,j);
              NgraphApp->Changed=TRUE;
            }
          }
        }
      }
      MessageBox("Data points are moved.","Confirm",MB_OK|MB_ICONINFORMATION);
    }
    arraydel(&sellist);
    MoveData=FALSE;
    Capture=FALSE;
    SetCursor(NULL,IDC_ARROW);
  } else if ((Mode==PointB) || (Mode==LegendB) || (Mode==AxisB)) {
    Capture=TRUE;
    if (arraynum(focusobj)!=0) {
      GetFocusFrame(&minx,&miny,&maxx,&maxy,0,0);
      if ((minx-10<=point.x) && (point.x<=minx-4)
       && (miny-10<=point.y) && (point.y<=miny-4)) {
        GetLargeFrame(&RefX2,&RefY2,&RefX1,&RefY1);
        MouseMode=MOUSEZOOM1;
        SetCursor(NULL,IDC_SIZENWSE);
      } else if ((maxx+4<=point.x) && (point.x<=maxx+10)
       && (miny-10<=point.y) && (point.y<=miny-4)) {
        GetLargeFrame(&RefX1,&RefY2,&RefX2,&RefY1);
        MouseMode=MOUSEZOOM2;
        SetCursor(NULL,IDC_SIZENESW);
      } else if ((maxx+4<=point.x) && (point.x<=maxx+10)
       && (maxy+4<=point.y) && (point.y<=maxy+10)) {
        GetLargeFrame(&RefX1,&RefY1,&RefX2,&RefY2);
        MouseMode=MOUSEZOOM3;
        SetCursor(NULL,IDC_SIZENWSE);
      } else if ((minx-10<=point.x) && (point.x<=minx-4)
       && (maxy+4<=point.y) && (point.y<=maxy+10)) {
        GetLargeFrame(&RefX2,&RefY1,&RefX1,&RefY2);
        MouseMode=MOUSEZOOM4;
        SetCursor(NULL,IDC_SIZENESW);
      } else {
        focus=*(struct focuslist **)arraynget(focusobj,0);
        if ((arraynum(focusobj)==1)
        && ((inst=chkobjinstoid(focus->obj,focus->oid))!=NULL)) {
          _exeobj(focus->obj,"bbox",inst,0,NULL);
          _getobj(focus->obj,"bbox",inst,&abbox);
          bboxnum=arraynum(abbox);
          bbox=(int *)arraydata(abbox);
          for (j=4;j<bboxnum;j+=2) {
            x1=mwd2p(bbox[j]*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
            y1=mwd2p(bbox[j+1]*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
            if ((x1-3<=point.x) && (point.x<=x1+3)
             && (y1-3<=point.y) && (point.y<=y1+3)) break;
          }
          if (j<bboxnum) {
            MouseMode=MOUSECHANGE;
            ChangePoint=(j-4)/2;
            ShowFocusFrame(dc);
            ShowFrame=FALSE;
            SetCursor(NULL,IDC_CROSS);
            ShowLine=TRUE;
            LineX=LineY=0;
            ShowFocusLine(dc,ChangePoint);
          }
        }
      }
      if (MouseMode==MOUSENONE) {
        if ((minx<=point.x) && (point.x<=maxx)
         && (miny<=point.y) && (point.y<=maxy)) MouseMode=MOUSEDRAG;
        else {
          ShowFocusFrame(dc);
          ShowFrame=FALSE;
          arraydel2(focusobj);
        }
      } else if ((MOUSEZOOM1<=MouseMode) && (MouseMode<=MOUSEZOOM4)) {
        ShowFocusFrame(dc);
        ShowFrame=FALSE;
        MouseDX=RefX2-MouseX1;
        MouseDY=RefY2-MouseY1;
        vx1=MouseX1;
        vy1=MouseY1;
        vx1-=RefX1-MouseDX;
        vy1-=RefY1-MouseDY;
        vx2=(RefX2-RefX1);
        vy2=(RefY2-RefY1);
        cc=vx1*vx2+vy1*vy2;
        nn=vx2*vx2+vy2*vy2;
        if ((nn==0) || (cc<0)) {
          zoom2=0;
        } else {
          zoom2=cc/nn;
        }
        CheckGrid(FALSE,modKeys,NULL,NULL,&zoom2);
        NgraphApp->SetZoom(zoom2);
        vx1=RefX1+vx2*zoom2;
        vy1=RefY1+vy2*zoom2;
        MouseX1=RefX1;
        MouseY1=RefY1;
        MouseX2=vx1;
        MouseY2=vy1;
        ShowFrameRect(dc);
        ShowRect=TRUE;
      }
    }
    if (arraynum(focusobj)==0) {
      MouseMode=MOUSEPOINT;
      ShowRect=TRUE;
    }
  } else if ((Mode==TrimB) || (Mode==DataB) || (Mode==EvalB)) {
    Capture=TRUE;
    MouseMode=MOUSEPOINT;
    ShowRect=TRUE;
  } else if ((Mode==MarkB) || (Mode==TextB)) {
    if (!Capture) {
      x1=MouseX1;
      y1=MouseY1;
      CheckGrid(TRUE,modKeys,&x1,&y1,NULL);
      if ((po=(struct pointslist *)memalloc(sizeof(struct pointslist)))!=NULL) {
         po->x=x1;
         po->y=y1;
         arrayadd(points,&po);
      }
      ShowPoints(dc);
      Capture=TRUE;
    }
  } else if (Mode==ZoomB) {
    if (modKeys & MK_SHIFT) {
      Scroller->XPos-=(cx-point.x);
      Scroller->YPos-=(cy-point.y);
      ChangeDPI();
    } else if (getobj(menulocal.obj,"dpi",0,0,NULL,&vdpi)!=-1) {
      if (modKeys & MK_CONTROL) {
        if (nround(vdpi/sqrt(2))>=20) {
          vdpi=nround(vdpi/sqrt(2));
          if (putobj(menulocal.obj,"dpi",0,&vdpi)!=-1) {
            Scroller->XPos-=(cx-point.x);
            Scroller->YPos-=(cy-point.y);
            ChangeDPI();
          }
        } else MessageBeep(MB_ICONASTERISK);
      } else {
        if (nround(vdpi*sqrt(2))<=620) {
          vdpi=nround(vdpi*sqrt(2));
          if (putobj(menulocal.obj,"dpi",0,&vdpi)!=-1) {
            Scroller->XPos-=(cx-point.x);
            Scroller->YPos-=(cy-point.y);
            ChangeDPI();
          }
        } else MessageBeep(MB_ICONASTERISK);
      }
    }
  } else {
    if (!Capture) {
      x1=MouseX1;
      y1=MouseY1;
      CheckGrid(TRUE,modKeys,&x1,&y1,NULL);
      if ((po=(struct pointslist *)memalloc(sizeof(struct pointslist)))!=NULL) {
         po->x=x1;
         po->y=y1;
         arrayadd(points,&po);
      }
      if ((po=(struct pointslist *)memalloc(sizeof(struct pointslist)))!=NULL) {
         po->x=x1;
         po->y=y1;
         arrayadd(points,&po);
      }
      ShowPoints(dc);
      Capture=TRUE;
    }
  }
  ReleaseDC(Handle,dc);
  TMyWindow::EvLButtonDown(modKeys,point);
  if (Capture) SetCapture();
  else ReleaseCapture();
}

void TViewWindow::EvRButtonDown(UINT modKeys,TPoint& point)
{
  HDC dc;
  int num;
  struct pointslist *po;
  double zoom;
  TRect rect;
  TPoint point2;
  int vdpi;
  struct focuslist *focus;
  struct objlist *obj;

  Scroller->AutoMode=TRUE;
  if (menulock || globallock) return;
  if (MoveData) {
    arraydel(&sellist);
    MoveData=FALSE;
    Capture=FALSE;
    SetCursor(NULL,IDC_ARROW);
    MessageBox("Moving data points is canceled.","Confirm",MB_OK|MB_ICONINFORMATION);
  } else if (Capture && ((Mode==LineB) || (Mode==CurveB) || (Mode==PolyB))) {
    zoom=menulocal.PaperZoom/10000.0;
    dc=GetDC(Handle);
    num=arraynum(points);
    if (num>0) {
      ShowPoints(dc);
      arrayndel2(points,num-1);
      if (num<=2) {
        arraydel2(points);
        Capture=FALSE;
      } else {
        po=*(struct pointslist **)arraylast(points);
        if (po!=NULL) {
          MouseX1=(mwp2d(Scroller->XPos+point.x-cx)-menulocal.LeftMargin)/zoom;
          MouseY1=(mwp2d(Scroller->YPos+point.y-cy)-menulocal.TopMargin)/zoom;
          po->x=MouseX1;
          po->y=MouseY1;
          CheckGrid(TRUE,modKeys,&(po->x),&(po->y),NULL);
        }
        ShowPoints(dc);
      }
    }
    ReleaseDC(Handle,dc);
  } else if (Mode==ZoomB) {
    if (modKeys & MK_SHIFT) {
      Scroller->XPos-=(cx-point.x);
      Scroller->YPos-=(cy-point.y);
      ChangeDPI();
    } else if (getobj(menulocal.obj,"dpi",0,0,NULL,&vdpi)!=-1) {
      if (modKeys & MK_CONTROL) {
        if (nround(vdpi*sqrt(2))<=620) {
          vdpi=nround(vdpi*sqrt(2));
          if (putobj(menulocal.obj,"dpi",0,&vdpi)!=-1) {
            Scroller->XPos-=(cx-point.x);
            Scroller->YPos-=(cy-point.y);
            ChangeDPI();
          }
        } else MessageBeep(MB_ICONASTERISK);
      } else {
        if (nround(vdpi/sqrt(2))>=20) {
          vdpi=nround(vdpi/sqrt(2));
          if (putobj(menulocal.obj,"dpi",0,&vdpi)!=-1) {
            Scroller->XPos-=(cx-point.x);
            Scroller->YPos-=(cy-point.y);
            ChangeDPI();
          }
        } else MessageBeep(MB_ICONASTERISK);
      }
    }
  } else if (MouseMode==MOUSENONE) {
    PopupMenu.EnableMenuItem(CM_DELETEK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_UPDATEK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_COPYK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_TOPK,MF_GRAYED);
    PopupMenu.EnableMenuItem(CM_LASTK,MF_GRAYED);
    if ((Mode==PointB) || (Mode==LegendB) || (Mode==AxisB)) {
      num=arraynum(focusobj);
      if (num>0) {
        PopupMenu.EnableMenuItem(CM_DELETEK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_UPDATEK,MF_ENABLED);
        PopupMenu.EnableMenuItem(CM_COPYK,MF_ENABLED);
      }
      if (num==1) {
        focus=*(struct focuslist **)arraynget(focusobj,0);
        obj=focus->obj;
        if (chkobjchild(chkobject("legend"),obj)) {
          PopupMenu.EnableMenuItem(CM_TOPK,MF_ENABLED);
          PopupMenu.EnableMenuItem(CM_LASTK,MF_ENABLED);
        }
      }  
    }
    if (ShowCross) PopupMenu.CheckMenuItem(CM_CROSSK,MF_CHECKED);
    else PopupMenu.CheckMenuItem(CM_CROSSK,MF_UNCHECKED);
    GetWindowRect(rect);
    point2.x=point.x+rect.left;
    point2.y=point.y+rect.top;
    PopupMenu.TrackPopupMenu(TPM_LEFTBUTTON | TPM_RIGHTBUTTON,point2,0,Handle);
  }
  TMyWindow::EvRButtonDown(modKeys,point);
  if (Capture) SetCapture();
  else ReleaseCapture();
}

void TViewWindow::ShowFrameRect(HDC dc)
{
  HPEN spen,pen;
  int smode;
  COLORREF col;
  int x1,y1,x2,y2;
  double zoom;

  zoom=menulocal.PaperZoom/10000.0;
  col=(~GetSysColor(COLOR_WINDOW)) & RGB(255,255,255);
  pen=CreatePen(PS_DOT,0,col);
  spen=(HPEN)SelectObject(dc,pen);
  smode=SetROP2(dc,R2_XORPEN);
  if ((MouseX1!=MouseX2) || (MouseY1!=MouseY2)) {
    x1=mwd2p(MouseX1*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
    y1=mwd2p(MouseY1*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
    x2=mwd2p(MouseX2*zoom+menulocal.LeftMargin)-Scroller->XPos+cx;
    y2=mwd2p(MouseY2*zoom+menulocal.TopMargin)-Scroller->YPos+cy;
    MoveToEx(dc,x1,y1,NULL);
    LineTo(dc,x1,y2);
    LineTo(dc,x2,y2);
    LineTo(dc,x2,y1);
    LineTo(dc,x1,y1);
  }
  SetROP2(dc,smode);
  DeleteObject(SelectObject(dc,spen));
}

void TViewWindow::EvMouseMove(UINT modKeys, TPoint& point)
{
  HDC dc;
  struct pointslist *po;
  int x,y,dx,dy,vx1,vx2,vy1,vy2;
  double cc,nn;
  double zoom,zoom2;
  TRect rect;

  if (menulock || globallock) return;
  zoom=menulocal.PaperZoom/10000.0;
  dx=(mwp2d(point.x+Scroller->XPos-cx)-menulocal.LeftMargin)/zoom;
  dy=(mwp2d(point.y+Scroller->YPos-cy)-menulocal.TopMargin)/zoom;
  if ((Mode!=DataB) && (Mode!=EvalB) && (Mode!=ZoomB) && (MouseMode!=MOUSEPOINT)
  && (((Mode!=PointB) && (Mode!=LegendB) && (Mode!=AxisB)) || (MouseMode!=MOUSENONE)))
    CheckGrid(TRUE,modKeys,&dx,&dy,NULL);
  NgraphApp->SetCoord(dx,dy);
  NgraphApp->SetPoint(dx,dy);
  dc=GetDC(Handle);
  GetUpdateRect(rect,FALSE);
  if ((rect.left==0) && (rect.right==0) && (rect.top==0) && (rect.bottom==0)) {
    if (ShowCross) ShowCrossGauge(dc);
    CrossX=dx;
    CrossY=dy;
    if (ShowCross) ShowCrossGauge(dc);
  }
  if (Capture) {
    if ((Mode==PointB) || (Mode==LegendB) || (Mode==AxisB)
     || (Mode==TrimB) || (Mode==DataB) || (Mode==EvalB)) {
      if (MouseMode==MOUSEDRAG) {
      	ShowFocusFrame(dc);
        MouseX2=(mwp2d(point.x+Scroller->XPos-cx)-menulocal.LeftMargin)/zoom;
        MouseY2=(mwp2d(point.y+Scroller->YPos-cy)-menulocal.TopMargin)/zoom;
      	x=MouseX2-MouseX1;
      	y=MouseY2-MouseY1;
        CheckGrid(FALSE,modKeys,&x,&y,NULL);
        FrameOfsX=x;
        FrameOfsY=y;
      	ShowFocusFrame(dc);
      } else if ((MOUSEZOOM1<=MouseMode) && (MouseMode<=MOUSEZOOM4)) {
        ShowFrameRect(dc);
        vx1=(mwp2d(point.x-cx+Scroller->XPos)-menulocal.LeftMargin)/zoom;
        vy1=(mwp2d(point.y-cy+Scroller->YPos)-menulocal.TopMargin)/zoom;
        vx1-=RefX1-MouseDX;
        vy1-=RefY1-MouseDY;
        vx2=(RefX2-RefX1);
        vy2=(RefY2-RefY1);
        cc=vx1*vx2+vy1*vy2;
        nn=vx2*vx2+vy2*vy2;
        if ((nn==0) || (cc<0)) {
          zoom2=0;
        } else {
          zoom2=cc/nn;
        }
        if ((Mode!=DataB) && (Mode!=EvalB)) CheckGrid(FALSE,modKeys,NULL,NULL,&zoom2);
        NgraphApp->SetZoom(zoom2);
        vx1=RefX1+vx2*zoom2;
        vy1=RefY1+vy2*zoom2;
        MouseX1=RefX1;
        MouseY1=RefY1;
        MouseX2=vx1;
        MouseY2=vy1;
        ShowFrameRect(dc);
      } else if (MouseMode==MOUSECHANGE) {
      	ShowFocusLine(dc,ChangePoint);
        MouseX2=(mwp2d(point.x+Scroller->XPos-cx)-menulocal.LeftMargin)/zoom;
        MouseY2=(mwp2d(point.y+Scroller->YPos-cy)-menulocal.TopMargin)/zoom;
      	x=MouseX2-MouseX1;
      	y=MouseY2-MouseY1;
        if ((Mode!=DataB) && (Mode!=EvalB)) CheckGrid(FALSE,modKeys,&x,&y,NULL);
        LineX=x;
        LineY=y;
      	ShowFocusLine(dc,ChangePoint);
      } else if (MouseMode==MOUSEPOINT) {
        ShowFrameRect(dc);
        MouseX2=(mwp2d(point.x+Scroller->XPos-cx)-menulocal.LeftMargin)/zoom;
        MouseY2=(mwp2d(point.y+Scroller->YPos-cy)-menulocal.TopMargin)/zoom;
        ShowFrameRect(dc);
      }
    } else {
      ShowPoints(dc);
      if (arraynum(points)!=0) {
    	po=*(struct pointslist **)arraylast(points);
    	if (po!=NULL) {
    	  po->x=dx;
    	  po->y=dy;
    	}
      }
      ShowPoints(dc);
    }
  }
  ReleaseDC(Handle,dc);
  TMyWindow::EvMouseMove(modKeys,point);
//  if (Capture) SetCapture();
//  else ReleaseCapture();
}

void TViewWindow::EvLButtonUp(UINT modKeys, TPoint& point)
{
  int x1,y1,x2,y2,err;
  HDC dc;
  int i,num,dx,dy;
  struct focuslist *focus;
  struct objlist *obj;
  char *inst;
  char *argv[4];
  struct pointslist *po;
  int vx1,vx2,vy1,vy2;
  double cc,nn,zoom,zoom2;
  int zm;
  int axis;

  if (menulock || globallock) return;
  dc=GetDC(Handle);
  zoom=menulocal.PaperZoom/10000.0;
  axis=FALSE;
  if (Capture) {
    if ((Mode==PointB) || (Mode==LegendB) || (Mode==AxisB)
     || (Mode==TrimB) || (Mode==DataB) || (Mode==EvalB)) {
      if (MouseMode==MOUSEDRAG) {
        Capture=FALSE;
        if ((MouseX1!=MouseX2) || (MouseY1!=MouseY2)) {
          ShowFocusFrame(dc);
          ShowFrame=FALSE;
          MouseX2=(mwp2d(point.x+Scroller->XPos-cx)-menulocal.LeftMargin)/zoom;
          MouseY2=(mwp2d(point.y+Scroller->YPos-cy)-menulocal.TopMargin)/zoom;
          dx=MouseX2-MouseX1;
          dy=MouseY2-MouseY1;
          CheckGrid(FALSE,modKeys,&dx,&dy,NULL);
          argv[0]=(char *)&dx;
          argv[1]=(char *)&dy;
          argv[2]=NULL;
          PaintLock=TRUE;
          num=arraynum(focusobj);
          for (i=num-1;i>=0;i--) {
            focus=*(struct focuslist **)arraynget(focusobj,i);
            obj=focus->obj;
            if (obj==chkobject("axis")) axis=TRUE;
            if ((inst=chkobjinstoid(focus->obj,focus->oid))!=NULL) {
              AddInvalidateRect(obj,inst);
              _exeobj(obj,"move",inst,2,argv);
              NgraphApp->Changed=TRUE;
              AddInvalidateRect(obj,inst);
            }
          }
          PaintLock=FALSE;
          FrameOfsX=FrameOfsY=0;
          ShowFrame=TRUE;
          ShowFocusFrame(dc);
          if ((Mode==LegendB) || ((Mode==PointB) && (!axis)))
            allclear=FALSE;
          NgraphApp->Update();
        }
        MouseMode=MOUSENONE;
      } else if ((MOUSEZOOM1<=MouseMode) && (MouseMode<=MOUSEZOOM4)) {
        Capture=FALSE;
        ShowFrameRect(dc);
        ShowRect=FALSE;
        vx1=(mwp2d(point.x-cx+Scroller->XPos)-menulocal.LeftMargin)/zoom;
        vy1=(mwp2d(point.y-cy+Scroller->YPos)-menulocal.TopMargin)/zoom;
        vx1-=RefX1-MouseDX;
        vy1-=RefY1-MouseDY;
        vx2=(RefX2-RefX1);
        vy2=(RefY2-RefY1);
        cc=vx1*vx2+vy1*vy2;
        nn=vx2*vx2+vy2*vy2;
        if ((nn==0) || (cc<0)) {
          zoom2=0;
        } else {
          zoom2=cc/nn;
        }
        if ((Mode!=DataB) && (Mode!=EvalB)) CheckGrid(FALSE,modKeys,NULL,NULL,&zoom2);
        zm=nround(zoom2*10000);
        NgraphApp->ResetZoom();
        argv[0]=(char *)&zm;
        argv[1]=(char *)&RefX1;
        argv[2]=(char *)&RefY1;
        argv[3]=NULL;
        PaintLock=TRUE;
        num=arraynum(focusobj);
        for (i=num-1;i>=0;i--) {
          focus=*(struct focuslist **)arraynget(focusobj,i);
          obj=focus->obj;
          if (obj==chkobject("axis")) axis=TRUE;
          if ((inst=chkobjinstoid(focus->obj,focus->oid))!=NULL) {
            AddInvalidateRect(obj,inst);
            _exeobj(obj,"zooming",inst,3,argv);
            NgraphApp->Changed=TRUE;
            AddInvalidateRect(obj,inst);
          }
        }
        PaintLock=FALSE;
        FrameOfsX=FrameOfsY=0;
        ShowFrame=TRUE;
        ShowFocusFrame(dc);
        if ((Mode==LegendB) || ((Mode==PointB) && (!axis)))
          allclear=FALSE;
        NgraphApp->Update();
        SetCursor(NULL,IDC_ARROW);
        MouseMode=MOUSENONE;
      } else if (MouseMode==MOUSECHANGE) {
        Capture=FALSE;
        ShowFocusLine(dc,ChangePoint);
        ShowLine=FALSE;
        if ((MouseX1!=MouseX2) || (MouseY1!=MouseY2)) {
          MouseX2=(mwp2d(point.x+Scroller->XPos-cx)-menulocal.LeftMargin)/zoom;
          MouseY2=(mwp2d(point.y+Scroller->YPos-cy)-menulocal.TopMargin)/zoom;
          dx=MouseX2-MouseX1;
          dy=MouseY2-MouseY1;
          if ((Mode!=DataB) && (Mode!=EvalB)) CheckGrid(FALSE,modKeys,&dx,&dy,NULL);
          argv[0]=(char *)&ChangePoint;
          argv[1]=(char *)&dx;
          argv[2]=(char *)&dy;
          argv[3]=NULL;
          PaintLock=TRUE;
          focus=*(struct focuslist **)arraynget(focusobj,0);
          obj=focus->obj;
          if (obj==chkobject("axis")) axis=TRUE;
          if ((inst=chkobjinstoid(focus->obj,focus->oid))!=NULL) {
            AddInvalidateRect(obj,inst);
            _exeobj(obj,"change",inst,3,argv);
            NgraphApp->Changed=TRUE;
            AddInvalidateRect(obj,inst);
          }
          PaintLock=FALSE;
          FrameOfsX=FrameOfsY=0;
          ShowFrame=TRUE;
          ShowFocusFrame(dc);
          if ((Mode==LegendB) || ((Mode==PointB) && (!axis)))
            allclear=FALSE;
          NgraphApp->Update();
        } else {
          FrameOfsX=FrameOfsY=0;
          ShowFrame=TRUE;
          ShowFocusFrame(dc);
        }
        SetCursor(NULL,IDC_ARROW);
        MouseMode=MOUSENONE;
      } else if (MouseMode==MOUSEPOINT) {
        Capture=FALSE;
        ShowFrameRect(dc);
        ShowRect=FALSE;
        MouseX2=(mwp2d(point.x+Scroller->XPos-cx)-menulocal.LeftMargin)/zoom;
        MouseY2=(mwp2d(point.y+Scroller->YPos-cy)-menulocal.TopMargin)/zoom;
        x1=MouseX1;
        y1=MouseY1;
        x2=MouseX2;
        y2=MouseY2;
        err=mwp2d(2)/zoom;
        Scroller->AutoMode=FALSE;
        if (Mode==PointB) {
          Match("legend",x1,y1,x2,y2,err);
          Match("axis",x1,y1,x2,y2,err);
          Match("merge",x1,y1,x2,y2,err);
          FrameOfsX=FrameOfsY=0;
          ShowFrame=TRUE;
          ShowFocusFrame(dc);
        } else if (Mode==LegendB) {
          Match("legend",x1,y1,x2,y2,err);
          Match("merge",x1,y1,x2,y2,err);
          FrameOfsX=FrameOfsY=0;
          ShowFrame=TRUE;
          ShowFocusFrame(dc);
        } else if (Mode==AxisB) {
          Match("axis",x1,y1,x2,y2,err);
          FrameOfsX=FrameOfsY=0;
          ShowFrame=TRUE;
          ShowFocusFrame(dc);
        } else if (Mode==TrimB) {
          Trimming(x1,y1,x2,y2);
        } else if (Mode==DataB) {
          if (FileUpdate(x1,y1,x2,y2,err)) NgraphApp->Update();
        } else {
          Evaluate(x1,y1,x2,y2,err);
        }
      }
      MouseMode=MOUSENONE;
    } else if ((Mode==MarkB) || (Mode==TextB)) {
      Capture=FALSE;
      ShowPoints(dc);
      MouseX1=(mwp2d(point.x+Scroller->XPos-cx)-menulocal.LeftMargin)/zoom;
      MouseY1=(mwp2d(point.y+Scroller->YPos-cy)-menulocal.TopMargin)/zoom;
      x1=MouseX1;
      y1=MouseY1;
      CheckGrid(TRUE,modKeys,&x1,&y1,NULL);
      num=arraynum(points);
      if (num>=1) {
        po=*(struct pointslist **)arraynget(points,0);
        po->x=x1;
        po->y=y1;
      }
      ShowPoints(dc);
      if (arraynum(points)==1) {
        Scroller->AutoMode=FALSE;
        EvLButtonDblClk(modKeys,point);
      }
    } else if ((Mode!=MarkB) && (Mode!=TextB)) {
      ShowPoints(dc);
      MouseX1=(mwp2d(point.x+Scroller->XPos-cx)-menulocal.LeftMargin)/zoom;
      MouseY1=(mwp2d(point.y+Scroller->YPos-cy)-menulocal.TopMargin)/zoom;
      x1=MouseX1;
      y1=MouseY1;
      CheckGrid(TRUE,modKeys,&x1,&y1,NULL);
      num=arraynum(points);
      if (num>=2) po=*(struct pointslist **)arraynget(points,num-2);
      if ((num<2) || (po->x!=x1) || (po->y!=y1)) {
        if ((po=(struct pointslist *)memalloc(sizeof(struct pointslist)))!=NULL) {
          po->x=x1;
          po->y=y1;
          arrayadd(points,&po);
        }
      }
      ShowPoints(dc);
      if ((Mode==RectB) || (Mode==ArcB) || (Mode==GaussB)
      || (Mode==FrameB) || (Mode==SectionB) || (Mode==CrossB) || (Mode==SingleB)) {
        if (arraynum(points)==3) {
          Capture=FALSE;
          Scroller->AutoMode=FALSE;
          EvLButtonDblClk(modKeys,point);
        }
      }
    }
  }
  ReleaseDC(Handle,dc);
  TMyWindow::EvLButtonUp(modKeys,point);
  if (Capture) SetCapture();
  else ReleaseCapture();
}

void TViewWindow::EvLButtonDblClk(UINT modKeys,TPoint& point)
{
  int i,id,num;
  LegendArrowDialog *dlgll;
  LegendCurveDialog *dlglc;
  LegendPolyDialog *dlglp;
  LegendRectDialog *dlglr;
  LegendArcDialog *dlglar;
  LegendMarkDialog *dlglm;
  LegendTextDialog *dlglt;
  LegendGaussDialog *dlglg;
  SectionDialog *dlgsc;
  CrossDialog *dlgcs;
  AxisDialog *dlgax;
  struct objlist *obj,*obj2;
  char *inst;
  int ret;
  HDC dc;
  int x1,y1,x2,y2,x,y,rx,ry;
  struct pointslist *po,**pdata;
  struct narray *parray;
  int idx,idy,idu,idr,idg,lenx,leny,oidx,oidy;
  double fx1,fy1;
  int dir;
  struct narray group;
  int type;
  char *argv[2];
  char *ref;

  if (menulock || globallock) return;
  if ((Mode==PointB) || (Mode==LegendB) || (Mode==AxisB)) {
    Capture=FALSE;
    ViewUpdate();
  } else if ((Mode!=TrimB) && (Mode!=DataB) && (Mode!=EvalB)) {
    dc=GetDC(Handle);
    if ((Mode==MarkB) || (Mode==TextB)) {
      Capture=FALSE;
      num=arraynum(points);
      if (Mode==MarkB) obj=chkobject("mark");
      else obj=chkobject("text");
      if (obj!=NULL) {
        if ((id=newobj(obj))>=0) {
          if (num>=1) {
            po=*(struct pointslist **)arraynget(points,0);
            x1=po->x;
            y1=po->y;
          }
          inst=chkobjinst(obj,id);
          _putobj(obj,"x",inst,&x1);
          _putobj(obj,"y",inst,&y1);
          PaintLock=TRUE;
          if (Mode==MarkB) {
            dlglm=new LegendMarkDialog(this,DIALOG_LEGENDMARK,obj,id);
            ret=dlglm->Execute();
            delete dlglm;
          } else {
            dlglt=new LegendTextDialog(this,DIALOG_LEGENDTEXT,obj,id);
            ret=dlglt->Execute();
            delete dlglt;
          }
          if ((ret==IDDELETE) || (ret==IDCANCEL)) delobj(obj,id);
          else {
            AddList(obj,inst);
            AddInvalidateRect(obj,inst);
            NgraphApp->Changed=TRUE;
          }
          PaintLock=FALSE;
        }
      }
      ShowPoints(dc);
      arraydel2(points);
      allclear=FALSE;
    } else if ((Mode==LineB) || (Mode==CurveB) || (Mode==PolyB)) {
      Capture=FALSE;
      num=arraynum(points);
      if (num>=3) {
        if (Mode==LineB) obj=chkobject("line");
        else if (Mode==CurveB) obj=chkobject("curve");
        else if (Mode==PolyB) obj=chkobject("polygon");
        if (obj!=NULL) {
          if ((id=newobj(obj))>=0) {
            inst=chkobjinst(obj,id);
            parray=arraynew(sizeof(int));
            for (i=0;i<num-1;i++) {
              po=*(struct pointslist **)arraynget(points,i);
              arrayadd(parray,&(po->x));
              arrayadd(parray,&(po->y));
            }
            _putobj(obj,"points",inst,parray);
            PaintLock=TRUE;
            if (Mode==LineB) {
              dlgll=new LegendArrowDialog(this,DIALOG_LEGENDARROW,obj,id);
              ret=dlgll->Execute();
              delete dlgll;
            } else if (Mode==CurveB) {
              dlglc=new LegendCurveDialog(this,DIALOG_LEGENDCURVE,obj,id);
              ret=dlglc->Execute();
              delete dlglc;
            } else if (Mode==PolyB) {
              dlglp=new LegendPolyDialog(this,DIALOG_LEGENDPOLY,obj,id);
              ret=dlglp->Execute();
              delete dlglp;
            }
            if ((ret==IDDELETE) || (ret==IDCANCEL)) delobj(obj,id);
            else {
              AddList(obj,inst);
              AddInvalidateRect(obj,inst);
              NgraphApp->Changed=TRUE;
            }
            PaintLock=FALSE;
          }
        }
      }
      ShowPoints(dc);
      arraydel2(points);
      allclear=FALSE;
    } else if ((Mode==RectB) || (Mode==ArcB)) {
      Capture=FALSE;
      num=arraynum(points);
      pdata=(struct pointslist **)arraydata(points);
      if (num>=3) {
        if (Mode==RectB) obj=chkobject("rectangle");
        else if (Mode==ArcB) obj=chkobject("arc");
        if (obj!=NULL) {
          if ((id=newobj(obj))>=0) {
            inst=chkobjinst(obj,id);
            x1=pdata[0]->x;
            y1=pdata[0]->y;
            x2=pdata[1]->x;
            y2=pdata[1]->y;
            PaintLock=TRUE;
            if (Mode==RectB) {
              _putobj(obj,"x1",inst,&x1);
              _putobj(obj,"y1",inst,&y1);
              _putobj(obj,"x2",inst,&x2);
              _putobj(obj,"y2",inst,&y2);
              dlglr=new LegendRectDialog(this,DIALOG_LEGENDRECT,obj,id);
              ret=dlglr->Execute();
              delete dlglr;
            } else if (Mode==ArcB) {
              x=(x1+x2)/2;
              y=(y1+y2)/2;
              rx=abs(x1-x);
              ry=abs(y1-y);
              _putobj(obj,"x",inst,&x);
              _putobj(obj,"y",inst,&y);
              _putobj(obj,"rx",inst,&rx);
              _putobj(obj,"ry",inst,&ry);
              dlglar=new LegendArcDialog(this,DIALOG_LEGENDARC,obj,id);
              ret=dlglar->Execute();
              delete dlglar;
            }
            if ((ret==IDDELETE) || (ret==IDCANCEL)) delobj(obj,id);
            else {
              AddList(obj,inst);
              AddInvalidateRect(obj,inst);
              NgraphApp->Changed=TRUE;
            }
            PaintLock=FALSE;
          }
        }
      }
      ShowPoints(dc);
      arraydel2(points);
      allclear=FALSE;
    } else if (Mode==GaussB) {
      Capture=FALSE;
      num=arraynum(points);
      pdata=(struct pointslist **)arraydata(points);
      if (num>=3) {
        obj=chkobject("curve");
        if (obj!=NULL) {
          if ((id=newobj(obj))>=0) {
            inst=chkobjinst(obj,id);
            x1=pdata[0]->x;
            y1=pdata[0]->y;
            x2=pdata[1]->x;
            y2=pdata[1]->y;
            if (x1>x2) {
              x=x1; x1=x2; x2=x;
            }
            if (y1>y2) {
              y=y1; y1=y2; y2=y;
            }
            PaintLock=TRUE;
            if ((x1!=x2) && (y1!=y2)) {
              dlglg=new LegendGaussDialog(this,DIALOG_LEGENDGAUSS,obj,id,x1,y1,x2-x1,y2-y1);
              ret=dlglg->Execute();
              delete dlglg;
              if (ret!=IDOK) delobj(obj,id);
              else {
                AddList(obj,inst);
                AddInvalidateRect(obj,inst);
                NgraphApp->Changed=TRUE;
              }
            }
            PaintLock=FALSE;
          }
        }
      }
      ShowPoints(dc);
      arraydel2(points);
      allclear=FALSE;
    } else if (Mode==SingleB) {
      Capture=FALSE;
      num=arraynum(points);
      pdata=(struct pointslist **)arraydata(points);
      if (num>=3) {
        obj=chkobject("axis");
        if (obj!=NULL) {
          if ((id=newobj(obj))>=0) {
            inst=chkobjinst(obj,id);
            x1=pdata[0]->x;
            y1=pdata[0]->y;
            x2=pdata[1]->x;
            y2=pdata[1]->y;
            fx1=x2-x1;
            fy1=y2-y1;
            lenx=nround(sqrt(fx1*fx1+fy1*fy1));
            if (fx1==0) {
              if (fy1>=0) dir=27000;
              else dir=9000;
            } else {
              dir=nround(atan(-fy1/fx1)/MPI*18000);
              if (fx1<0) dir+=18000;
              if (dir<0) dir+=36000;
              if (dir>=36000) dir-=36000;
            }
            inst=chkobjinst(obj,id);
            _putobj(obj,"x",inst,&x1);
            _putobj(obj,"y",inst,&y1);
            _putobj(obj,"length",inst,&lenx);
            _putobj(obj,"direction",inst,&dir);
            dlgax=new AxisDialog(this,DIALOG_AXIS,obj,id,TRUE);
            ret=dlgax->Execute();
            delete dlgax;
            if ((ret==IDDELETE) || (ret==IDCANCEL)) {
              delobj(obj,id);
            } else NgraphApp->Changed=TRUE;
            AddList(obj,inst);
          }
        }
      }
      ShowPoints(dc);
      arraydel2(points);
      allclear=TRUE;
    } else if ((Mode==FrameB) || (Mode==SectionB) || (Mode==CrossB)) {
      Capture=FALSE;
      num=arraynum(points);
      pdata=(struct pointslist **)arraydata(points);
      if (num>=3) {
        obj=chkobject("axis");
        obj2=chkobject("axisgrid");
        if (obj!=NULL) {
          x1=pdata[0]->x;
          y1=pdata[0]->y;
          x2=pdata[1]->x;
          y2=pdata[1]->y;
          lenx=abs(x1-x2);
          leny=abs(y1-y2);
          x1=min(x1,x2);
          y1=max(y1,y2);
          idx=newobj(obj);
          idy=newobj(obj);
          if (Mode!=CrossB) {
            idu=newobj(obj);
            idr=newobj(obj);
            arrayinit(&group,sizeof(int));
            if (Mode==FrameB) type=1;
            else type=2;
            arrayadd(&group,&type);
            arrayadd(&group,&idx);
            arrayadd(&group,&idy);
            arrayadd(&group,&idu);
            arrayadd(&group,&idr);
            arrayadd(&group,&x1);
            arrayadd(&group,&y1);
            arrayadd(&group,&lenx);
            arrayadd(&group,&leny);
            argv[0]=(char *)&group;
            argv[1]=NULL;
            exeobj(obj,"default_grouping",idr,1,argv);
            arraydel(&group);
          } else {
            arrayinit(&group,sizeof(int));
            type=3;
            arrayadd(&group,&type);
            arrayadd(&group,&idx);
            arrayadd(&group,&idy);
            arrayadd(&group,&x1);
            arrayadd(&group,&y1);
            arrayadd(&group,&lenx);
            arrayadd(&group,&leny);
            argv[0]=(char *)&group;
            argv[1]=NULL;
            exeobj(obj,"default_grouping",idx,1,argv);
            arraydel(&group);
          }
          if ((Mode==SectionB) && (obj2!=NULL)) {
            idg=newobj(obj2);
            if (idg>=0) {
              getobj(obj,"oid",idx,0,NULL,&oidx);
              if ((ref=(char *)memalloc(15))!=NULL) {
                sprintf(ref,"axis:^%d",oidx);
                putobj(obj2,"axis_x",idg,ref);
              }
              getobj(obj,"oid",idy,0,NULL,&oidy);
              if ((ref=(char *)memalloc(15))!=NULL) {
                sprintf(ref,"axis:^%d",oidy);
                putobj(obj2,"axis_y",idg,ref);
              }
            }
          } else idg=-1;
          if (Mode==FrameB) {
            dlgsc=new SectionDialog(this,DIALOG_SECTION,x1,y1,lenx,leny,obj,idx,idy,idu,idr,obj2,&idg,FALSE);
            ret=dlgsc->Execute();
            delete dlgsc;
          } else if (Mode==SectionB) {
            dlgsc=new SectionDialog(this,DIALOG_SECTION,x1,y1,lenx,leny,obj,idx,idy,idu,idr,obj2,&idg,TRUE);
            ret=dlgsc->Execute();
            delete dlgsc;
          } else if (Mode==CrossB) {
            dlgcs=new CrossDialog(this,DIALOG_CROSS,x1,y1,lenx,leny,obj,idx,idy);
            ret=dlgcs->Execute();
            delete dlgcs;
          }
          if ((ret==IDDELETE) || (ret==IDCANCEL)) {
            if (Mode!=CrossB) {
              delobj(obj,idr);
              delobj(obj,idu);
            }
            delobj(obj,idy);
            delobj(obj,idx);
            if ((idg!=-1) && (obj2!=NULL)) delobj(obj2,idg);
          } else {
            if ((inst=chkobjinst(obj,idx))!=NULL) AddList(obj,inst);
            if ((inst=chkobjinst(obj,idy))!=NULL) AddList(obj,inst);
            if (Mode!=CrossB) {
              if ((inst=chkobjinst(obj,idu))!=NULL) AddList(obj,inst);
              if ((inst=chkobjinst(obj,idr))!=NULL) AddList(obj,inst);
            }
            if ((idg!=-1) && (obj2!=NULL)) {
              if ((inst=chkobjinst(obj2,idg))!=NULL) AddList(obj2,inst);
            }
            NgraphApp->Changed=TRUE;
          }
        }
      }
      ShowPoints(dc);
      arraydel2(points);
      allclear=TRUE;
    }
    ReleaseDC(Handle,dc);
    NgraphApp->Update();
  }
  TMyWindow::EvLButtonDblClk(modKeys,point);
  if (Capture) SetCapture();
  else ReleaseCapture();
}

void TViewWindow::ViewUpdate()
{
  int i,id,id2,did,num;
  struct focuslist *focus;
  LegendRectDialog *dlglr;
  LegendArcDialog *dlglar;
  LegendMarkDialog *dlglm;
  LegendTextDialog *dlglt;
  LegendArrowDialog *dlgll;
  LegendCurveDialog *dlglc;
  LegendPolyDialog *dlglp;
  SectionDialog *dlgsc;
  CrossDialog *dlgcs;
  AxisDialog *dlgax;
  MergeDialog *dlgmerge;
  struct objlist *obj,*dobj,*aobj;
  char *inst,*inst2,*dinst,*dfield;
  int ret,section;
  int x1,y1;
  int aid,idx,idy,idu,idr,idg,j,lenx,leny;
  int findX,findY,findU,findR,findG;
  char *axisx,*axisy;
  int matchx,matchy;
  struct narray iarray,*sarray;
  int snum;
  char **sdata;
  HDC dc;
  TRect urect;
  char type;
  char *group,*group2;
  int  axis;

  if (menulock || globallock) return;
  dc=GetDC(Handle);
  ShowFocusFrame(dc);
  ShowFrame=FALSE;
  num=arraynum(focusobj);
  axis=FALSE;
  PaintLock=TRUE;
//  ::LockWindowUpdate(Handle);
  for (i=num-1;i>=0;i--) {
    focus=*(struct focuslist **)arraynget(focusobj,i);
    if ((focus!=NULL) && ((inst=chkobjinstoid(focus->obj,focus->oid))!=NULL)) {
      obj=focus->obj;
      _getobj(obj,"id",inst,&id);
      ret=IDCANCEL;
      if (obj==chkobject("axis")) {
        axis=TRUE;
        _getobj(obj,"group",inst,&group);
        if ((group!=NULL) && (group[0]!='a')) {
          findX=findY=findU=findR=findG=FALSE;
          type=group[0];
          for (j=0;j<=id;j++) {
            inst2=chkobjinst(obj,j);
            _getobj(obj,"group",inst2,&group2);
            _getobj(obj,"id",inst2,&id2);
            if ((group2!=NULL) && (group2[0]==type)) {
              if (strcmp(group+2,group2+2)==0) {
                if (group2[1]=='X') {
                  findX=TRUE;
                  idx=id2;
                } else if (group2[1]=='Y') {
                  findY=TRUE;
                  idy=id2;
                } else if (group2[1]=='U') {
                  findU=TRUE;
                  idu=id2;
                } else if (group2[1]=='R') {
                  findR=TRUE;
                  idr=id2;
                }
              }
            }
          }
          if (((type=='s') || (type=='f')) && findX && findY
          && !_getobj(menulocal.obj,"_list",menulocal.inst,&sarray)
          && ((snum=arraynum(sarray))>=0)) {
            sdata=(char **)arraydata(sarray);
            for (j=1;j<snum;j++) {
              if (((dobj=getobjlist(sdata[j],&did,&dfield,NULL))!=NULL)
              && (dobj==chkobject("axisgrid"))) {
                if ((dinst=chkobjinstoid(dobj,did))!=NULL) {
                  _getobj(dobj,"axis_x",dinst,&axisx);
                  _getobj(dobj,"axis_y",dinst,&axisy);
                  matchx=matchy=FALSE;
                  if (axisx!=NULL) {
                    arrayinit(&iarray,sizeof(int));
                    if (!getobjilist(axisx,&aobj,&iarray,FALSE,NULL)) {
                      if ((arraynum(&iarray)>=1) && (obj==aobj)
                      && (*(int *)arraylast(&iarray)==idx)) matchx=TRUE;
                    }
                    arraydel(&iarray);
                  }
                  if (axisy!=NULL) {
                    arrayinit(&iarray,sizeof(int));
                    if (!getobjilist(axisy,&aobj,&iarray,FALSE,NULL)) {
                      if ((arraynum(&iarray)>=1) && (obj==aobj)
                      && (*(int *)arraylast(&iarray)==idy)) matchy=TRUE;
                    }
                    arraydel(&iarray);
                  }
                  if (matchx && matchy) {
                    findG=TRUE;
                    _getobj(dobj,"id",dinst,&idg);
                    break;
                  }
                }
              }
            }
          }
          if (((type=='s') || (type=='f')) && findX && findY && findU && findR) {
            if (!findG) {
              dobj=chkobject("axisgrid");
              idg=-1;
            }
            if (type=='s') section=TRUE;
            else section=FALSE;
            getobj(obj,"y",idx,0,NULL,&y1);
            getobj(obj,"x",idy,0,NULL,&x1);
            getobj(obj,"y",idu,0,NULL,&leny);
            getobj(obj,"x",idr,0,NULL,&lenx);
            leny=y1-leny;
            lenx=lenx-x1;
            dlgsc=new SectionDialog(this,DIALOG_SECTION,x1,y1,lenx,leny,obj,idx,idy,idu,idr,dobj,&idg,section);
            ret=dlgsc->Execute();
            delete dlgsc;
            if (ret==IDDELETE) {
              AxisDel(id);
              arrayndel2(focusobj,i);
            }
            if (!findG) {
              if (idg!=-1) {
                if ((dinst=chkobjinst(dobj,idg))!=NULL) AddList(dobj,dinst);
              }
            }
            if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
          } else if ((type=='c') && findX && findY) {
            getobj(obj,"x",idx,0,NULL,&x1);
            getobj(obj,"y",idy,0,NULL,&y1);
            getobj(obj,"length",idx,0,NULL,&lenx);
            getobj(obj,"length",idy,0,NULL,&leny);
            dlgcs=new CrossDialog(this,DIALOG_CROSS,x1,y1,lenx,leny,obj,idx,idy);
            ret=dlgcs->Execute();
            delete dlgcs;
            if (ret==IDDELETE) {
              AxisDel(aid);
              arrayndel2(focusobj,i);
            }
            if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
          }
        } else {
          dlgax=new AxisDialog(this,DIALOG_AXIS,obj,id,TRUE);
          ret=dlgax->Execute();
          delete dlgax;
          if (ret==IDDELETE) {
            AxisDel(id);
            arrayndel2(focusobj,i);
          }
          if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
        }
      } else {
        AddInvalidateRect(obj,inst);
        if (obj==chkobject("line")) {
          dlgll=new LegendArrowDialog(this,DIALOG_LEGENDARROW,obj,id);
          ret=dlgll->Execute();
          delete dlgll;
        } else if (obj==chkobject("curve")) {
          dlglc=new LegendCurveDialog(this,DIALOG_LEGENDCURVE,obj,id);
          ret=dlglc->Execute();
          delete dlglc;
        } else if (obj==chkobject("polygon")) {
          dlglp=new LegendPolyDialog(this,DIALOG_LEGENDPOLY,obj,id);
          ret=dlglp->Execute();
          delete dlglp;
        } else if (obj==chkobject("rectangle")) {
          dlglr=new LegendRectDialog(this,DIALOG_LEGENDRECT,obj,id);
          ret=dlglr->Execute();
          delete dlglr;
        } else if (obj==chkobject("arc")) {
          dlglar=new LegendArcDialog(this,DIALOG_LEGENDARC,obj,id);
          ret=dlglar->Execute();
          delete dlglar;
        } else if (obj==chkobject("mark")) {
          dlglm=new LegendMarkDialog(this,DIALOG_LEGENDMARK,obj,id);
          ret=dlglm->Execute();
          delete dlglm;
        } else if (obj==chkobject("text")) {
          dlglt=new LegendTextDialog(this,DIALOG_LEGENDTEXT,obj,id);
          ret=dlglt->Execute();
          delete dlglt;
        } else if (obj==chkobject("merge")) {
          dlgmerge=new MergeDialog(this,DIALOG_MERGE,obj,id);
          ret=dlgmerge->Execute();
          delete dlgmerge;
        }
        if (ret==IDDELETE) {
          delobj(obj,id);
        }
        if (ret==IDOK) AddInvalidateRect(obj,inst);
        if (ret!=IDCANCEL) NgraphApp->Changed=TRUE;
      }
    }
  }
  PaintLock=FALSE;
//  ::LockWindowUpdate(NULL);
  if ((Mode==LegendB) || ((Mode==PointB) && (!axis))) allclear=FALSE;
  NgraphApp->Update();
  ShowFocusFrame(dc);
  ShowFrame=TRUE;
  ReleaseDC(Handle,dc);
}

void TViewWindow::EvKeyUp(UINT key,UINT repeatCount,UINT flags)
{
  HDC dc;
  int num,i,dx,dy;
  struct focuslist *focus;
  char *inst;
  struct objlist *obj;
  char *argv[4];
  int axis;

  if (menulock || globallock) return;
  switch (key) {
  case VK_SHIFT:
  case VK_CONTROL:
    if (Mode==ZoomB) SetCursor(NgraphApp->GetApplication(),CURSOR_ZOOMIN);
    break;
  case VK_DOWN: case VK_UP: case VK_LEFT: case VK_RIGHT:
    if (MouseMode==MOUSEDRAG) {
      dc=GetDC(Handle);
      ShowFocusFrame(dc);
      dx=FrameOfsX;
      dy=FrameOfsY;
      argv[0]=(char *)&dx;
      argv[1]=(char *)&dy;
      argv[2]=NULL;
      num=arraynum(focusobj);
      axis=FALSE;
      PaintLock=TRUE;
      for (i=num-1;i>=0;i--) {
        focus=*(struct focuslist **)arraynget(focusobj,i);
        obj=focus->obj;
        if (obj==chkobject("axis")) axis=TRUE;
        if ((inst=chkobjinstoid(focus->obj,focus->oid))!=NULL) {
          AddInvalidateRect(obj,inst);
          _exeobj(obj,"move",inst,2,argv);
          NgraphApp->Changed=TRUE;
          AddInvalidateRect(obj,inst);
        }
      }
      PaintLock=FALSE;
      FrameOfsX=FrameOfsY=0;
      ShowFrame=TRUE;
      ShowFocusFrame(dc);
      ReleaseDC(Handle,dc);
      if ((Mode==LegendB) || ((Mode==PointB) && (!axis))) allclear=FALSE;
      NgraphApp->Update();
      MouseMode=MOUSENONE;
    }
    break;
  default:
    break;
  }
}

void TViewWindow::EvKeyDown(UINT key,UINT repeatCount,UINT flags)
{
  HDC dc;
  int dx,dy,mv;
  double zoom;
  TPoint point;
  TRect rect;
  struct focuslist *focus;
  struct objlist *obj;
  int num;

  if (menulock || globallock) return;
  switch (key) {
  case VK_DELETE:
    ViewDelete();
    break;
  case VK_RETURN:
    ViewUpdate();
    break;
  case VK_INSERT:
    ViewCopy();
    break;
  case VK_HOME:
    ViewTop();
    break;
  case VK_END:
    ViewLast();
    break;
  case VK_DOWN: case VK_UP: case VK_LEFT: case VK_RIGHT:
    if (((MouseMode==MOUSENONE) || (MouseMode==MOUSEDRAG))
     && ((Mode==PointB) || (Mode==LegendB) || (Mode==AxisB))) {
      dc=GetDC(Handle);
      zoom=menulocal.PaperZoom/10000.0;
      ShowFocusFrame(dc);
      if (GetAsyncKeyState(VK_SHIFT) & 0x8000) mv=mwlocal->grid/10;
      else mv=mwlocal->grid;
      if (key==VK_DOWN) {
        dx=0;
        dy=mv;
      } else if (key==VK_UP) {
        dx=0;
        dy=-mv;
      } else if (key==VK_RIGHT) {
        dx=mv;
        dy=0;
      } else if (key==VK_LEFT) {
        dx=-mv;
        dy=0;
      }
      FrameOfsX+=dx/zoom;
      FrameOfsY+=dy/zoom;
      ShowFocusFrame(dc);
      ReleaseDC(Handle,dc);
      MouseMode=MOUSEDRAG;
    }
    break;
  case VK_SHIFT:
    if (Mode==ZoomB) SetCursor(NgraphApp->GetApplication(),CURSOR_CENTER);
    break;
  case VK_CONTROL:
    if (Mode==ZoomB) SetCursor(NgraphApp->GetApplication(),CURSOR_ZOOMOUT);
    break;
  case VK_TAB:
    ViewCross();
    break;
  case 'P':
    if (GetAsyncKeyState(VK_CONTROL) & 0x8000) {
      if (!MoveData && !Capture && (MouseMode==MOUSENONE)) {
        PopupMenu.EnableMenuItem(CM_DELETEK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_UPDATEK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_COPYK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_TOPK,MF_GRAYED);
        PopupMenu.EnableMenuItem(CM_LASTK,MF_GRAYED);
        if ((Mode==PointB) || (Mode==LegendB) || (Mode==AxisB)) {
          num=arraynum(focusobj);
          if (num>0) {
            PopupMenu.EnableMenuItem(CM_DELETEK,MF_ENABLED);
            PopupMenu.EnableMenuItem(CM_UPDATEK,MF_ENABLED);
            PopupMenu.EnableMenuItem(CM_COPYK,MF_ENABLED);
          }
          if (num==1) {
            focus=*(struct focuslist **)arraynget(focusobj,0);
            obj=focus->obj;
            if (chkobjchild(chkobject("legend"),obj)) {
              PopupMenu.EnableMenuItem(CM_TOPK,MF_ENABLED);
              PopupMenu.EnableMenuItem(CM_LASTK,MF_ENABLED);
            }
          }
        }
        if (ShowCross) PopupMenu.CheckMenuItem(CM_CROSSK,MF_CHECKED);
        else PopupMenu.CheckMenuItem(CM_CROSSK,MF_UNCHECKED);
        GetWindowRect(rect);
        point.x=rect.left;
        point.y=rect.top;
        PopupMenu.TrackPopupMenu(TPM_LEFTBUTTON | TPM_RIGHTBUTTON,point,0,Handle);
      }
    }
    break;
  case VK_SPACE:
    if (GetAsyncKeyState(VK_CONTROL) & 0x8000) {
      if (!MoveData && !Capture && (MouseMode==MOUSENONE)) {
        GetWindowRect(rect);
        point.x=rect.left;
        point.y=rect.top;
        PopupMenu2.TrackPopupMenu(TPM_LEFTBUTTON | TPM_RIGHTBUTTON,point,0,Handle);
      }
    } else Draw(FALSE);
    break;
  default:
    NgraphApp->EvKeyDown2(key,repeatCount,flags);
    break;
  }
  TMyWindow::EvKeyDown(key,repeatCount,flags);
}

void TViewWindow::ViewDelete()
{
  int i,id,num;
  struct focuslist *focus;
  struct objlist *obj;
  char *inst;
  HDC dc;
  int axis;

  if ((MouseMode==MOUSENONE) && ((Mode==PointB) || (Mode==LegendB) || (Mode==AxisB))) {
    dc=GetDC(Handle);
    ShowFocusFrame(dc);
    ShowFrame=FALSE;
    axis=FALSE;
    PaintLock=TRUE;
    num=arraynum(focusobj);
    for (i=num-1;i>=0;i--) {
      focus=*(struct focuslist **)arraynget(focusobj,i);
      obj=focus->obj;
      if ((inst=chkobjinstoid(obj,focus->oid))!=NULL) {
        AddInvalidateRect(obj,inst);
        DelList(obj,inst);
        _getobj(obj,"id",inst,&id);
        if (obj==chkobject("axis")) {
          AxisDel(id);
          axis=TRUE;
        } else delobj(obj,id);
        NgraphApp->Changed=TRUE;
      }
    }
    PaintLock=FALSE;
    if ((Mode==LegendB) || ((Mode==PointB) && (!axis))) allclear=FALSE;
    if (num!=0) NgraphApp->Update();
  }
}

void TViewWindow::ViewTop()
{
  int id,num;
  struct focuslist *focus;
  struct objlist *obj;
  char *inst;

  if ((MouseMode==MOUSENONE) && ((Mode==PointB) || (Mode==LegendB) || (Mode==AxisB))) {
    num=arraynum(focusobj);
    if (num==1) {
      focus=*(struct focuslist **)arraynget(focusobj,0);
      obj=focus->obj;
      if (chkobjchild(chkobject("legend"),obj)) {
        if ((inst=chkobjinstoid(obj,focus->oid))!=NULL) {
          DelList(obj,inst);
          _getobj(obj,"id",inst,&id);
          movetopobj(obj,id);
          AddList(obj,inst);
          NgraphApp->Changed=TRUE;
          allclear=TRUE;
          NgraphApp->Update();
        }
      }
    }
  }
}

void TViewWindow::ViewLast()
{
  int id,num;
  struct focuslist *focus;
  struct objlist *obj;
  char *inst;

  if ((MouseMode==MOUSENONE) && ((Mode==PointB) || (Mode==LegendB) || (Mode==AxisB))) {
    num=arraynum(focusobj);
    if (num==1) {
      focus=*(struct focuslist **)arraynget(focusobj,0);
      obj=focus->obj;
      if (chkobjchild(chkobject("legend"),obj)) {
        if ((inst=chkobjinstoid(obj,focus->oid))!=NULL) {
          DelList(obj,inst);
          _getobj(obj,"id",inst,&id);
          movelastobj(obj,id);
          AddList(obj,inst);
          NgraphApp->Changed=TRUE;
          allclear=TRUE;
          NgraphApp->Update();
        }
      }
    }
  }
}

void ncopyobj(struct objlist *obj,int id1,int id2)
{
  int j;
  char *field;
  int perm,type;

  for (j=0;j<chkobjfieldnum(obj);j++) {
    field=chkobjfieldname(obj,j);
    perm=chkobjperm(obj,field);
    type=chkobjfieldtype(obj,field);
    if (((perm&NREAD)!=0) && ((perm&NWRITE)!=0) && (type<NVFUNC))
      copyobj(obj,field,id1,id2);
  }
}

void TViewWindow::ViewCopy()
{
  int i,j,id,id2,did,num;
  struct focuslist *focus;
  struct objlist *obj,*dobj,*aobj;
  char *inst,*inst2,*dinst,*dfield;
  HDC dc;
  int findX,findY,findU,findR,findG;
  char *axisx,*axisy;
  int matchx,matchy;
  struct narray iarray,*sarray;
  int snum;
  char **sdata;
  int oidx,oidy;
  int idx,idy,idu,idr,idg;
  int idx2,idy2,idu2,idr2,idg2;
  char type;
  char *group,*group2;
  int tp;
  struct narray agroup;
  char *argv[2];
  int axis;

  if ((MouseMode==MOUSENONE) && ((Mode==PointB) || (Mode==LegendB) || (Mode==AxisB))) {
    dc=GetDC(Handle);
    ShowFocusFrame(dc);
    ShowFrame=FALSE;
    axis=FALSE;
    PaintLock=TRUE;
    num=arraynum(focusobj);
    for (i=0;i<num;i++) {
      focus=*(struct focuslist **)arraynget(focusobj,i);
      if ((focus!=NULL) && ((inst=chkobjinstoid(focus->obj,focus->oid))!=NULL)) {
        obj=focus->obj;
        _getobj(obj,"id",inst,&id);
        if (obj==chkobject("axis")) {
          axis=TRUE;
          _getobj(obj,"group",inst,&group);
          if ((group!=NULL) && (group[0]!='a')) {
            findX=findY=findU=findR=findG=FALSE;
            type=group[0];
            for (j=0;j<=id;j++) {
              inst2=chkobjinst(obj,j);
              _getobj(obj,"group",inst2,&group2);
              _getobj(obj,"id",inst2,&id2);
              if ((group2!=NULL) && (group2[0]==type)) {
                if (strcmp(group+2,group2+2)==0) {
                  if (group2[1]=='X') {
                    findX=TRUE;
                    idx=id2;
                  } else if (group2[1]=='Y') {
                    findY=TRUE;
                    idy=id2;
                  } else if (group2[1]=='U') {
                    findU=TRUE;
                    idu=id2;
                  } else if (group2[1]=='R') {
                    findR=TRUE;
                    idr=id2;
                  }
                }
              }
            }
            if (((type=='s') || (type=='f')) && findX && findY
            && !_getobj(menulocal.obj,"_list",menulocal.inst,&sarray)
            && ((snum=arraynum(sarray))>=0)) {
              sdata=(char **)arraydata(sarray);
              for (j=1;j<snum;j++) {
                if (((dobj=getobjlist(sdata[j],&did,&dfield,NULL))!=NULL)
                && (dobj==chkobject("axisgrid"))) {
                  if ((dinst=chkobjinstoid(dobj,did))!=NULL) {
                    _getobj(dobj,"axis_x",dinst,&axisx);
                    _getobj(dobj,"axis_y",dinst,&axisy);
                    matchx=matchy=FALSE;
                    if (axisx!=NULL) {
                      arrayinit(&iarray,sizeof(int));
                      if (!getobjilist(axisx,&aobj,&iarray,FALSE,NULL)) {
                        if ((arraynum(&iarray)>=1) && (obj==aobj)
                        && (*(int *)arraylast(&iarray)==idx)) matchx=TRUE;
                      }
                      arraydel(&iarray);
                    }
                    if (axisy!=NULL) {
                      arrayinit(&iarray,sizeof(int));
                      if (!getobjilist(axisy,&aobj,&iarray,FALSE,NULL)) {
                        if ((arraynum(&iarray)>=1) && (obj==aobj)
                        && (*(int *)arraylast(&iarray)==idy)) matchy=TRUE;
                      }
                      arraydel(&iarray);
                    }
                    if (matchx && matchy) {
                      findG=TRUE;
                      _getobj(dobj,"id",dinst,&idg);
                      break;
                    }
                  }
                }
              }
            }
            if (((type=='s') || (type=='f')) && findX && findY && findU && findR) {
              if ((idx2=newobj(obj))>=0) {
                ncopyobj(obj,idx2,idx);
                inst2=chkobjinst(obj,idx2);
                _getobj(obj,"oid",inst2,&oidx);
                AddList(obj,inst2);
                AddInvalidateRect(obj,inst2);
                NgraphApp->Changed=TRUE;
              } else {
                AddInvalidateRect(obj,inst);
              }
              if ((idy2=newobj(obj))>=0) {
                ncopyobj(obj,idy2,idy);
                inst2=chkobjinst(obj,idy2);
                _getobj(obj,"oid",inst2,&oidy);
                AddList(obj,inst2);
                AddInvalidateRect(obj,inst2);
                NgraphApp->Changed=TRUE;
              } else {
                AddInvalidateRect(obj,inst);
              }
              if ((idu2=newobj(obj))>=0) {
                ncopyobj(obj,idu2,idu);
                inst2=chkobjinst(obj,idu2);
                if ((idx2>=0) && ((axisx=(char *)memalloc(13))!=NULL)) {
                  sprintf(axisx,"axis:^%d",oidx);
                  putobj(obj,"reference",idu2,axisx);
                }
                AddList(obj,inst2);
                AddInvalidateRect(obj,inst2);
                NgraphApp->Changed=TRUE;
              } else {
                AddInvalidateRect(obj,inst);
              }
              if ((idr2=newobj(obj))>=0) {
                ncopyobj(obj,idr2,idr);
                inst2=chkobjinst(obj,idr2);
                if ((idy2>=0) && ((axisy=(char *)memalloc(13))!=NULL)) {
                  sprintf(axisy,"axis:^%d",oidy);
                  putobj(obj,"reference",idr2,axisy);
                }
                arrayinit(&agroup,sizeof(int));
                if (type=='f') tp=1;
                else tp=2;
                arrayadd(&agroup,&tp);
                arrayadd(&agroup,&idx2);
                arrayadd(&agroup,&idy2);
                arrayadd(&agroup,&idu2);
                arrayadd(&agroup,&idr2);
                argv[0]=(char *)&agroup;
                argv[1]=NULL;
                exeobj(obj,"grouping",idr2,1,argv);
                arraydel(&agroup);
                _getobj(obj,"oid",inst2,&(focus->oid));
                AddList(obj,inst2);
                AddInvalidateRect(obj,inst2);
                NgraphApp->Changed=TRUE;
              } else {
                AddInvalidateRect(obj,inst);
              }
              if (findG) {
                dobj=chkobject("axisgrid");
                if ((idg2=newobj(dobj))>=0) {
                  ncopyobj(dobj,idg2,idg);
                  inst2=chkobjinst(dobj,idg2);
                  if ((idx2>=0) && (idu2>=0) && ((axisx=(char *)memalloc(13))!=NULL)) {
                    sprintf(axisx,"axis:^%d",oidx);
                    putobj(dobj,"axis_x",idg2,axisx);
                  }
                  if ((idy2>=0) && (idr2>=0) && ((axisy=(char *)memalloc(13))!=NULL)) {
                    sprintf(axisy,"axis:^%d",oidy);
                    putobj(dobj,"axis_y",idg2,axisy);
                  }
                  AddList(dobj,inst2);
                  NgraphApp->Changed=TRUE;
                }
              }
            } else if ((type=='c') && findX && findY) {
              if ((idx2=newobj(obj))>=0) {
                ncopyobj(obj,idx2,idx);
                inst2=chkobjinst(obj,idx2);
                _getobj(obj,"oid",inst2,&oidx);
                AddList(obj,inst2);
                AddInvalidateRect(obj,inst2);
                NgraphApp->Changed=TRUE;
              } else {
                AddInvalidateRect(obj,inst);
              }
              if ((idy2=newobj(obj))>=0) {
                ncopyobj(obj,idy2,idy);
                inst2=chkobjinst(obj,idy2);
                _getobj(obj,"oid",inst2,&oidy);
                arrayinit(&agroup,sizeof(int));
                tp=3;
                arrayadd(&agroup,&tp);
                arrayadd(&agroup,&idx2);
                arrayadd(&agroup,&idy2);
                argv[0]=(char *)&agroup;
                argv[1]=NULL;
                exeobj(obj,"grouping",idy2,1,argv);
                arraydel(&agroup);
                focus->oid=oidy;
                AddList(obj,inst2);
                AddInvalidateRect(obj,inst2);
                NgraphApp->Changed=TRUE;
              } else {
                AddInvalidateRect(obj,inst);
              }
              if ((idx2>=0) && (idy2>=0)) {
                if ((axisy=(char *)memalloc(13))!=NULL) {
                  sprintf(axisy,"axis:^%d",oidy);
                  putobj(obj,"adjust_axis",idx2,axisy);
                }
                if ((axisx=(char *)memalloc(13))!=NULL) {
                  sprintf(axisx,"axis:^%d",oidx);
                  putobj(obj,"adjust_axis",idy2,axisx);
                }
              }
            }
          } else {
            if ((id2=newobj(obj))>=0) {
              ncopyobj(obj,id2,id);
              inst2=chkobjinst(obj,id2);
              _getobj(obj,"oid",inst2,&(focus->oid));
              AddList(obj,inst2);
              AddInvalidateRect(obj,inst2);
              NgraphApp->Changed=TRUE;
            } else {
              AddInvalidateRect(obj,inst);
            }
          }
        } else {
          if ((id2=newobj(obj))>=0) {
            ncopyobj(obj,id2,id);
            inst2=chkobjinst(obj,id2);
            _getobj(obj,"oid",inst2,&(focus->oid));
            AddList(obj,inst2);
            AddInvalidateRect(obj,inst2);
            NgraphApp->Changed=TRUE;
          } else {
            AddInvalidateRect(obj,inst);
          }
        }
      }
    }
    PaintLock=FALSE;
    if ((Mode==LegendB) || ((Mode==PointB) && (!axis))) allclear=FALSE;
    NgraphApp->Update();
    ShowFocusFrame(dc);
    ShowFrame=TRUE;
    ReleaseDC(Handle,dc);
  }
}

void TViewWindow::ViewCross()
{
  HDC dc;

  dc=GetDC(Handle);
  if (ShowCross) {
    ShowCrossGauge(dc);
    ShowCross=FALSE;
  } else {
    ShowCrossGauge(dc);
    ShowCross=TRUE;
  }
  ReleaseDC(Handle,dc);
}

void TViewWindow::EvSize(UINT sizeType, TSize& size)
{
  ChangeDPI();
  TMyWindow::EvSize(sizeType,size);
}

void TViewWindow::MakeRuler(HDC dc,struct mwlocal *mwlocal)
{
  int Width,Height;
  HPEN orgpen,pen;
  int x,y,len;
  int smode;

  Width=menulocal.PaperWidth;
  Height=menulocal.PaperHeight;
//  smode=SetROP2(dc,R2_XORPEN);
  smode=SetROP2(dc,R2_COPYPEN);
  pen=CreatePen(PS_SOLID,0,RGB(128,128,128));
  orgpen=(HPEN)SelectObject(dc,pen);
  BeginPath(dc);
  MoveToEx(dc,mwd2px(0),mwd2py(0),NULL);
  LineTo(dc,mwd2px(Width)-1,mwd2py(0));
  LineTo(dc,mwd2px(Width)-1,mwd2py(Height)-1);
  LineTo(dc,mwd2px(0),mwd2py(Height)-1);
  LineTo(dc,mwd2px(0),mwd2py(0));
  CloseFigure(dc);
  EndPath(dc);
  StrokePath(dc);
  if (mwlocal->ruler) {
    pen=CreatePen(PS_SOLID,1,RGB(128,128,128));
    DeleteObject(SelectObject(dc,pen));
    for (x=500;x<Width;x+=500) {
      if (!(x%10000)) {
        pen=CreatePen(PS_SOLID,1,RGB(255,0,0));
        DeleteObject(SelectObject(dc,pen));
      }
      if (!(x%10000)) len=225;
      else if (!(x%1000)) len=150;
      else len=75;
      MoveToEx(dc,mwd2px(x),mwd2py(0),NULL);
      LineTo(dc,mwd2px(x),mwd2py(len));
      MoveToEx(dc,mwd2px(x),mwd2py(Height)-1,NULL);
      LineTo(dc,mwd2px(x),mwd2py(Height-len)-1);
      if (!(x%10000)) {
        pen=CreatePen(PS_SOLID,1,RGB(128,128,128));
        DeleteObject(SelectObject(dc,pen));
      }
    }
    for (y=500;y<Height;y+=500) {
      if (!(y%10000)) {
        pen=CreatePen(PS_SOLID,1,RGB(255,0,0));
        DeleteObject(SelectObject(dc,pen));
      }
      if (!(y%10000)) len=225;
      else if (!(y%1000)) len=150;
      else len=75;
      MoveToEx(dc,mwd2px(0),mwd2py(y),NULL);
      LineTo(dc,mwd2px(len),mwd2py(y));
      MoveToEx(dc,mwd2px(Width)-1,mwd2py(y),NULL);
      LineTo(dc,mwd2px(Width-len)-1,mwd2py(y));
      if (!(y%10000)) {
        pen=CreatePen(PS_SOLID,1,RGB(128,128,128));
        DeleteObject(SelectObject(dc,pen));
      }
    }
  }
  DeleteObject(SelectObject(dc,orgpen));
  SetROP2(dc,smode);
}

void TViewWindow::EvPaint()
{
  PAINTSTRUCT ps;
  HDC dc;
  RECT rc;
  HRGN rgn;
  struct mwlocal mwsave;
  struct savedstdio save;
  HBRUSH brush;

  if (!::GetUpdateRect(Handle,&rc,FALSE) && (region==NULL)) return;
  if (::GetUpdateRect(Handle,&rc,FALSE)) {
    dc=::BeginPaint(Handle,&ps);
    ::EndPaint(Handle,&ps);
    if (region==NULL) region=CreateRectRgnIndirect(&rc);
    else {
      rgn=CreateRectRgnIndirect(&rc);
      CombineRgn(region,region,rgn,RGN_OR);
      DeleteObject(rgn);
    }
  }
  if (globallock) return;
  dc=GetDC(Handle);
  if (!PaintLock) {
    brush=CreateSolidBrush(RGB(mwlocal->bgR,mwlocal->bgG,mwlocal->bgB));
    FillRgn(dc,region,brush);
    DeleteObject(brush);
  }
  SelectClipRgn(dc,region);
  if (chkobjinstoid(menulocal.GRAobj,menulocal.GRAoid)!=NULL) {
    mwsaveDC(dc,Handle,-cx+Scroller->XPos,-cy+Scroller->YPos,
             &mwsave,-1,FALSE,region);
    MakeRuler(dc,mwlocal);
    if (mwlocal->autoredraw) {
      ignorestdio(&save);
      NgraphApp->SetStatusBar("Redrawing.");
      mw_redraw(menulocal.obj,menulocal.inst);
      NgraphApp->ResetStatusBar();
      restorestdio(&save);
    }
    mwrestoreDC(&mwsave);
    if (ShowFrame) ShowFocusFrame(dc);
    ShowPoints(dc);
    if (ShowLine) ShowFocusLine(dc,ChangePoint);
    if (ShowRect) ShowFrameRect(dc);
    if (ShowCross) ShowCrossGauge(dc);
    mwlocal->scrollx=-cx+Scroller->XPos;
    mwlocal->scrolly=-cy+Scroller->YPos;
    SetScrollPos(SB_HORZ,Scroller->XPos,TRUE);
    SetScrollPos(SB_VERT,Scroller->YPos,TRUE);
  }
  ReleaseDC(Handle,dc);
  if (!PaintLock) {
    DeleteObject(region);
    region=NULL;
  }
}

void TViewWindow::UnFocus()
{
  HDC dc;

  if (arraynum(focusobj)!=0) {
    if (Handle!=NULL) {
      dc=GetDC(Handle);
      ShowFocusFrame(dc);
      ReleaseDC(Handle,dc);
    }
    arraydel2(focusobj);
  }
  ShowFrame=FALSE;
  if (arraynum(points)!=0) {
    if (Handle!=NULL) {
      dc=GetDC(Handle);
      ShowPoints(dc);
      ReleaseDC(Handle,dc);
    }
    arraydel2(points);
  }
}

void TViewWindow::OpenGDC()
{
  if (mwlocal->DC!=NULL) return;
  mwlocal->hWnd=Handle;
  mwlocal->DC=GetDC(Handle);
  mwlocal->MetaFile=FALSE;
  mwlocal->style=PS_SOLID;
  mwlocal->width=1;
  mwlocal->Col=RGB(0,0,0);
  mwlocal->dashlist=NULL;
  mwlocal->dashn=0;
  mwlocal->dashlen=0;
  mwlocal->dashi=0;
  mwlocal->dotf=TRUE;
  mwlocal->x0=0;
  mwlocal->y0=0;
  mwlocal->linetonum=0;
  mwlocal->cpx=0;
  mwlocal->cpy=0;
  mwlocal->pixel_dot=mwlocal->windpi/25.4/100;
  mwlocal->fontalias=NULL;
  mwlocal->loadfontf=FALSE;
  mwlocal->ThePen=(HPEN)GetStockObject(BLACK_PEN);
  mwlocal->TheBrush=(HBRUSH)GetStockObject(BLACK_BRUSH);
  mwlocal->TheFont=(HFONT)GetStockObject(SYSTEM_FONT);
  mwlocal->OrgPen=(HPEN)SelectObject(mwlocal->DC,mwlocal->ThePen);
  mwlocal->OrgBrush=(HBRUSH)SelectObject(mwlocal->DC,mwlocal->TheBrush);
  mwlocal->OrgFont=(HFONT)SelectObject(mwlocal->DC,mwlocal->TheFont);
  mwlocal->offsetx=0;
  mwlocal->offsety=0;
  mwlocal->scrollx=0;
  mwlocal->scrolly=0;
  mwlocal->region=NULL;
}

void TViewWindow::ReopenGDC()
{
  if (mwlocal->DC!=NULL) {
    if (mwlocal->linetonum!=0) {
      EndPath(mwlocal->DC);
      StrokePath(mwlocal->DC);
      mwlocal->linetonum=0;
    }
    if (Handle!=NULL) ReleaseDC(mwlocal->hWnd,mwlocal->DC);
  }
  memfree(mwlocal->fontalias);
  memfree(mwlocal->dashlist);
  mwlocal->fontalias=NULL;
  mwlocal->dashlist=NULL;
  mwlocal->MetaFile=FALSE;
  mwlocal->style=PS_SOLID;
  mwlocal->width=1;
  mwlocal->Col=RGB(0,0,0);
  mwlocal->dashlist=NULL;
  mwlocal->dashn=0;
  mwlocal->dashlen=0;
  mwlocal->dashi=0;
  mwlocal->dotf=TRUE;
  mwlocal->x0=0;
  mwlocal->y0=0;
  mwlocal->linetonum=0;
  mwlocal->cpx=0;
  mwlocal->cpy=0;
  mwlocal->pixel_dot=mwlocal->windpi/25.4/100;
  mwlocal->fontalias=NULL;
  mwlocal->loadfontf=FALSE;
  mwlocal->hWnd=Handle;
  mwlocal->DC=GetDC(Handle);
  SelectObject(mwlocal->DC,mwlocal->ThePen);
  SelectObject(mwlocal->DC,mwlocal->TheBrush);
  SelectObject(mwlocal->DC,mwlocal->TheFont);
  mwlocal->offsetx=0;
  mwlocal->offsety=0;
  if (mwlocal->region!=NULL) DeleteObject(mwlocal->region);
}


void TViewWindow::CloseGDC()
{
  if (mwlocal->DC==NULL) return;
  if (mwlocal->linetonum!=0) {
    EndPath(mwlocal->DC);
    StrokePath(mwlocal->DC);
    mwlocal->linetonum=0;
  }
  SelectObject(mwlocal->DC,mwlocal->OrgPen);
  SelectObject(mwlocal->DC,mwlocal->OrgBrush);
  SelectObject(mwlocal->DC,mwlocal->OrgFont);
  if (Handle!=NULL) ReleaseDC(mwlocal->hWnd,mwlocal->DC);
  mwlocal->DC=NULL;
  mwlocal->hWnd=NULL;
  DeleteObject(mwlocal->ThePen);
  DeleteObject(mwlocal->TheBrush);
  DeleteObject(mwlocal->TheFont);
  mwlocal->ThePen=NULL;
  mwlocal->TheBrush=NULL;
  mwlocal->TheFont=NULL;
  memfree(mwlocal->fontalias);
  memfree(mwlocal->dashlist);
  mwlocal->fontalias=NULL;
  mwlocal->dashlist=NULL;
  mwlocal->linetonum=0;
  UnFocus();
}

void TViewWindow::ChangeScroller()
{
  int width,height;

  width=mwd2p(menulocal.PaperWidth);
  height=mwd2p(menulocal.PaperHeight);
  Scroller->SetPageSize();
  Scroller->SetRange(width+Scroller->XPage,height+Scroller->YPage);
  Scroller->XPos=width/2;
  Scroller->YPos=height/2;
  SetScrollPos(SB_HORZ,Scroller->XPos,TRUE);
  SetScrollPos(SB_VERT,Scroller->YPos,TRUE);
}

void TViewWindow::Update(int clear)
{
  int i,num;
  struct focuslist **focus;

  if (chkobjinstoid(menulocal.GRAobj,menulocal.GRAoid)==NULL) {
    CloseGDC();
    CloseGRA();
    OpenGRA();
    OpenGDC();
    ChangeScroller();
    ChangeDPI();
  }
  CheckPage();
  num=arraynum(focusobj);
  focus=(struct focuslist **)arraydata(focusobj);
  for (i=num-1;i>=0;i--) {
    if (chkobjoid(focus[i]->obj,focus[i]->oid)==-1) arrayndel2(focusobj,i);
  }
  if (allclear) Invalidate();
  else RedrawWindow(NULL,region,RDW_INTERNALPAINT);
  allclear=TRUE;
}

void TViewWindow::ChangeDPI()
{
  int width,height,i;
  TRect rect;
  double ratex,ratey,xrange,yrange;
  struct objlist *obj;
  int num;
  struct narray *array;
  char *inst;

  width=mwd2p(menulocal.PaperWidth);
  height=mwd2p(menulocal.PaperHeight);
  xrange=Scroller->XRange-Scroller->XPage;
  if (xrange<=0) ratex=0;
  else ratex=Scroller->XPos/xrange;
  yrange=Scroller->YRange-Scroller->YPage;
  if (yrange<=0) ratey=0;
  else ratey=Scroller->YPos/yrange;
  Scroller->SetPageSize();
  Scroller->SetRange(width+Scroller->XPage,height+Scroller->YPage);
  Scroller->XPos=(Scroller->XRange-Scroller->XPage)*ratex;
  Scroller->YPos=(Scroller->YRange-Scroller->YPage)*ratey;
  if (Scroller->XPos>width) Scroller->XPos=width;
  if (Scroller->XPos<0) Scroller->XPos=0;
  if (Scroller->YPos>height) Scroller->YPos=height;
  if (Scroller->YPos<0) Scroller->YPos=0;
  SetScrollPos(SB_HORZ,Scroller->XPos,TRUE);
  SetScrollPos(SB_VERT,Scroller->YPos,TRUE);
  GetClientRect(rect);
  cx=(rect.right-rect.left)/2;
  cy=(rect.bottom-rect.top)/2;
  mwlocal->scrollx=-cx+Scroller->XPos;
  mwlocal->scrolly=-cy+Scroller->YPos;
  if ((obj=chkobject("text"))!=NULL) {
    num=chkobjlastinst(obj);
    for (i=0;i<=num;i++) {
      inst=chkobjinst(obj,i);
      _getobj(obj,"bbox",inst,&array);
      arrayfree(array);
      _putobj(obj,"bbox",inst,NULL);
    }
  }
  Invalidate();
}

void TViewWindow::Draw(int SelectFile)
{
  int SShowFrame,SShowLine,SShowRect,SShowCross;
  HDC dc;

  if (SelectFile && !SetFileHiden((TWindow *)this)) return;
  FitClear();
  FileAutoScale();
  AdjustAxis();
  NgraphApp->SetStatusBar("Drawing.");
  dc=GetDC(Handle);
  if (ShowFrame) ShowFocusFrame(dc);
  if (ShowLine) ShowFocusLine(dc,ChangePoint);
  if (ShowRect) ShowFrameRect(dc);
  if (ShowCross) ShowCrossGauge(dc);
  ReleaseDC(Handle,dc);
  SShowFrame=ShowFrame;
  SShowLine=ShowLine;
  SShowRect=ShowRect;
  SShowCross=ShowCross;
  ShowFrame=ShowLine=ShowRect=ShowCross=FALSE;
  ReopenGDC();
  if (chkobjinstoid(menulocal.GRAobj,menulocal.GRAoid)!=NULL) {
    _exeobj(menulocal.GRAobj,"clear",menulocal.GRAinst,0,NULL);
    _exeobj(menulocal.GRAobj,"draw",menulocal.GRAinst,0,NULL);
    _exeobj(menulocal.GRAobj,"flush",menulocal.GRAinst,0,NULL);
  }
  ShowFrame=SShowFrame;
  ShowLine=SShowLine;
  ShowRect=SShowRect;
  ShowCross=SShowCross;
  dc=GetDC(Handle);
  if (ShowFrame) ShowFocusFrame(dc);
  if (ShowLine) ShowFocusLine(dc,ChangePoint);
  if (ShowRect) ShowFrameRect(dc);
  if (ShowCross) ShowCrossGauge(dc);
  ReleaseDC(Handle,dc);
  NgraphApp->ResetStatusBar();
}

void TViewWindow::Clear()
{
  if (chkobjinstoid(menulocal.GRAobj,menulocal.GRAoid)!=NULL) {
    UnFocus();
    _exeobj(menulocal.GRAobj,"clear",menulocal.GRAinst,0,NULL);
    ReopenGDC();
  }
}

void TViewWindow::Clipboard()
{
  HDC dc,rdc;
  HENHMETAFILE hMF;
  HANDLE MetaPictHD;
  METAFILEPICT *MetaFilePict;
  HMETAFILE HMF;
  METAFILEHEADER MetaHeader;
  WORD *Header;
  int GC;
  struct narray *sarray;
  int snum;
  char **sdata;
  struct GRAbbox bbox;
  struct mwlocal mwsave;
  RECT rect;
  int sx,sy;
  int dpix,dpiy;
  char *file;
  char *pfx,*tmpfile;
  struct objlist *sys;
  DWORD fdwAccess,fdwCreate;
  HANDLE fd1,fd2;
  SECURITY_ATTRIBUTES saAttr1,saAttr2;
  char buf[256];
  unsigned long len2,len3;
  OSVERSIONINFO VersionInformation;
  int iWidthMM,iHeightMM,iWidthPels,iHeightPels,iMMPerPelX,iMMPerPelY;
  struct savedstdio stdio;

  file=NULL;
  if (ClipboardDialog(this,DIALOG_CLIPBOARD,&(menulocal.clipboardEMF),&file,&(menulocal.clipboardDPI)).Execute()==IDOK) {
    if ((sys=getobject("system"))==NULL) return;
    if (getobj(sys,"temp_prefix",0,0,NULL,&pfx)) return;
    if (chkobjinstoid(menulocal.GRAobj,menulocal.GRAoid)==NULL) return;
    NgraphApp->SetStatusBar("Calculating bounding box.");
    GRAinitbbox(&bbox);
    if ((GC=_GRAopencallback(GRAboundingbox,NULL,&bbox))==-1) {
      GRAendbbox(&bbox);
      NgraphApp->ResetStatusBar();
      return;
    }
    GRAinit(GC,menulocal.LeftMargin,menulocal.TopMargin,
             menulocal.PaperWidth,menulocal.PaperHeight,menulocal.PaperZoom);
    if (_getobj(menulocal.obj,"_list",menulocal.inst,&sarray)) {
      _GRAclose(GC);
      GRAendbbox(&bbox);
      NgraphApp->ResetStatusBar();
      return;
    }
    if ((snum=arraynum(sarray))==0) {
      _GRAclose(GC);
      GRAendbbox(&bbox);
      NgraphApp->ResetStatusBar();
      return;
    }
    sdata=(char **)arraydata(sarray);
    ignorestdio(&stdio);
    _GRAredraw(GC,snum,sdata,TRUE,TRUE,-1,NULL,NULL,NULL);
    _GRAclose(GC);
    restorestdio(&stdio);
    GRAendbbox(&bbox);
    sx=(bbox.maxx-bbox.minx)/100;
    sy=(bbox.maxy-bbox.miny)/100;
    bbox.minx-=sx;
    bbox.miny-=sy;
    bbox.maxx+=sx;
    bbox.maxy+=sy;
    if (!bbox.set) {
      NgraphApp->ResetStatusBar();
      return;
    }
    NgraphApp->SetStatusBar("Copying to clipboard.");
    if (!(menulocal.clipboardEMF)) {
      rect.left=0;
      rect.top=0;
      rect.right=(bbox.maxx-bbox.minx);
      rect.bottom=(bbox.maxy-bbox.miny);
      if (file==NULL) {
        MetaPictHD=GlobalAlloc(GMEM_MOVEABLE,sizeof(METAFILEPICT));
        MetaFilePict=(METAFILEPICT *)GlobalLock(MetaPictHD);
        if (MetaFilePict!=NULL) {
          MetaFilePict->mm=MM_ANISOTROPIC;
          MetaFilePict->xExt=rect.right;
          MetaFilePict->yExt=rect.bottom;
        }
        dc=CreateMetaFile(NULL);
        SetWindowExtEx(dc,rect.right*(menulocal.clipboardDPI)/2540,
                          rect.bottom*(menulocal.clipboardDPI)/2540,NULL);
        SetWindowOrgEx(dc,0,0,NULL);
      } else {
        if ((tmpfile=tempnam(NULL,pfx))==NULL) {
          memfree(file);
          return;
        }
        MetaHeader.key=0x9AC6CDD7L;
        MetaHeader.hmf=0;
        MetaHeader.bbox.left=rect.left*(menulocal.clipboardDPI)/2540;
        MetaHeader.bbox.top=rect.top*(menulocal.clipboardDPI)/2540;
        MetaHeader.bbox.right=rect.right*(menulocal.clipboardDPI)/2540;
        MetaHeader.bbox.bottom=rect.bottom*(menulocal.clipboardDPI)/2540;
        MetaHeader.inch=menulocal.clipboardDPI;
        MetaHeader.reserved=0;
        Header=(WORD *)&MetaHeader;
        MetaHeader.checksum= *(Header+0) ^ *(Header+1) ^ *(Header+2) ^ *(Header+3) ^ *(Header+4)
                           ^ *(Header+5) ^ *(Header+6) ^ *(Header+7) ^ *(Header+8) ^ *(Header+9);
        dc=CreateMetaFile(tmpfile);
        Escape(dc,MFCOMMENT,6,"Ngraph",NULL);
        SetMapMode(dc,MM_ANISOTROPIC);
        SetWindowExtEx(dc,rect.right*(menulocal.clipboardDPI)/2540,
                          rect.bottom*(menulocal.clipboardDPI)/2540,NULL);
        SetWindowOrgEx(dc,0,0,NULL);
      }
      mwsaveDC(dc,Handle,bbox.minx*(menulocal.clipboardDPI)/2540,
                         bbox.miny*(menulocal.clipboardDPI)/2540,
                         &mwsave,(menulocal.clipboardDPI),1,NULL);
    } else {
      rect.left=0;
      rect.top=0;
      rect.right=(bbox.maxx-bbox.minx);
      rect.bottom=(bbox.maxy-bbox.miny);
      rdc=GetDC(Handle);
      iWidthMM=GetDeviceCaps(rdc,HORZSIZE);
      iHeightMM=GetDeviceCaps(rdc,VERTSIZE);

      iWidthPels=GetDeviceCaps(rdc,HORZRES);

      iHeightPels=GetDeviceCaps(rdc,VERTRES);
      iMMPerPelX=(iWidthMM*100)/iWidthPels;
      iMMPerPelY=(iHeightMM*100)/iHeightPels;
      dpix=2540/iMMPerPelX;
      dpiy=2540/iMMPerPelY;
      VersionInformation.dwOSVersionInfoSize=sizeof(OSVERSIONINFO);
      GetVersionEx(&VersionInformation);
      if (VersionInformation.dwPlatformId==VER_PLATFORM_WIN32_NT) {
        dc=CreateEnhMetaFile(rdc,file,NULL,"Ngraph\0Graph\0\0");
      } else {
        dc=CreateEnhMetaFile(rdc,file,&rect,"Ngraph\0Graph\0\0");
      }
      ReleaseDC(Handle,rdc);
      SetMapMode(dc,MM_ANISOTROPIC);
      SetWindowExtEx(dc,rect.right*(menulocal.clipboardDPI)/2540,
                        rect.bottom*(menulocal.clipboardDPI)/2540,NULL);
      SetWindowOrgEx(dc,0,0,NULL);
      SetViewportExtEx(dc,rect.right*dpix/2540,rect.bottom*dpix/2540,NULL);
      SetViewportOrgEx(dc,0,0,NULL);
      mwsaveDC(dc,Handle,bbox.minx*(menulocal.clipboardDPI)/2540,
                         bbox.miny*(menulocal.clipboardDPI)/2540,
                         &mwsave,(menulocal.clipboardDPI),2,NULL);
    }
    ignorestdio(&stdio);
    GRAredraw(menulocal.obj,menulocal.inst,TRUE,TRUE);
    mw_flush(menulocal.obj,menulocal.inst);
    restorestdio(&stdio);
    if (file==NULL) {
      ::OpenClipboard(Handle);
      ::EmptyClipboard();
      if (!(menulocal.clipboardEMF)) {
        if (MetaFilePict!=NULL) {
          HMF=CloseMetaFile(dc);
          MetaFilePict->hMF=HMF;
          GlobalUnlock(MetaPictHD);
          ::SetClipboardData(CF_METAFILEPICT,MetaPictHD);
        } else GlobalFree(MetaPictHD);
      } else {
        hMF=CloseEnhMetaFile(dc);
        GlobalUnlock(hMF);
        ::SetClipboardData(CF_ENHMETAFILE,hMF);
      }
      ::CloseClipboard();
    } else {
      if (!(menulocal.clipboardEMF)) {
        DeleteMetaFile(CloseMetaFile(dc));
        saAttr1.nLength = sizeof(SECURITY_ATTRIBUTES);
        saAttr1.bInheritHandle = TRUE;
        saAttr1.lpSecurityDescriptor = NULL;
        fdwAccess=GENERIC_WRITE;
        fdwCreate=CREATE_ALWAYS;
        fd1=CreateFile(file,fdwAccess,FILE_SHARE_READ,&saAttr1,fdwCreate,0,NULL);
        saAttr2.nLength = sizeof(SECURITY_ATTRIBUTES);
        saAttr2.bInheritHandle = TRUE;
        saAttr2.lpSecurityDescriptor = NULL;
        fdwAccess=GENERIC_READ;
        fdwCreate=OPEN_EXISTING;
        fd2=CreateFile(tmpfile,fdwAccess,FILE_SHARE_READ,&saAttr2,fdwCreate,0,NULL);
        WriteFile(fd1,(char *)&MetaHeader,sizeof(METAFILEHEADER),&len2,NULL);
        while (ReadFile(fd2,buf,256,&len2,NULL) && (len2>0)) {
          WriteFile(fd1,buf,len2,&len3,NULL);
        }
        CloseHandle(fd1);
        CloseHandle(fd2);
        unlink(tmpfile);
        free(tmpfile);
      } else {
        DeleteEnhMetaFile(CloseEnhMetaFile(dc));
      }
    }
    mwrestoreDC(&mwsave);
  }
  memfree(file);
  NgraphApp->ResetStatusBar();
}

void TMyWindow::CmViewerDraw()
{
  TViewWindow *win;
  TFileWindow *fwin;
  TAxisWindow *awin;

  if (menulock || globallock) return;
  win=(TViewWindow *)pTViewWindow;
  if (win!=NULL) win->Draw(TRUE);
  fwin=(TFileWindow *)pTFileWindow;
  if (fwin!=NULL) fwin->Update(TRUE);
  awin=(TAxisWindow *)pTAxisWindow;
  if (awin!=NULL) awin->Update(TRUE);
}

void TMyWindow::CmViewerDrawB()
{
  TViewWindow *win;
  TFileWindow *fwin;
  TAxisWindow *awin;

  if (menulock || globallock) return;
  win=(TViewWindow *)pTViewWindow;
  if (win!=NULL) win->Draw(FALSE);
  fwin=(TFileWindow *)pTFileWindow;
  if (fwin!=NULL) fwin->Update(TRUE);
  awin=(TAxisWindow *)pTAxisWindow;
  if (awin!=NULL) awin->Update(TRUE);
}

void TMyWindow::CmViewerClear()
{
  TViewWindow *win;
  TFileWindow *fwin;

  if (menulock || globallock) return;
  win=(TViewWindow *)pTViewWindow;
  if (win!=NULL) win->Clear();
  fwin=(TFileWindow *)pTFileWindow;
  if (fwin!=NULL) fwin->Update(TRUE);
}

void TMyWindow::CmViewerClipboard()
{
  TViewWindow *win;

  if (menulock || globallock) return;
  win=(TViewWindow *)pTViewWindow;
  if (win!=NULL) win->Clipboard();
}


