/* 
 * 
 * This file is part of "Ngraph for Windows".
 * 
 * Copyright (C) 2015, Satoshi ISHIZAKA. isizaka@msa.biglobe.ne.jp
 * 
 * "Ngraph for Windows" is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * "Ngraph for Windows" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

/*
 *
 * calc.cpp
 *
 */

#include <owl/pch.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <io.h>
#include <dir.h>

HINSTANCE hInstance;

extern "C" {
#include "../win/winfopen.h"
}

#include "calc.rh"

#define DIRSEP '/'
#define NSTRLEN 256

#define TRUE  1
#define FALSE 0

char *ScriptFile=NULL;

int niskanji(unsigned char code)
{
  if (((0x81<=code) && (code<=0x9f))
      || ((0xe0<=code) && (code<=0xff))) return TRUE;
  return FALSE;
}

int niskanji2(char *s,int pos)
{
  int k;

  if (pos>=strlen(s)) return FALSE;
  else {
    k=pos;
    do {
      k--;
    } while ((k>=0) && niskanji((unsigned char)s[k]));
    if ((pos-k)%2==1) return FALSE;
    else return TRUE;
  }
}

void changefilename(char *name)
{
  int i;
  char disk[4];
  DWORD csensitive;

  if (name==NULL) return;
  for (i=0;name[i]!='\0';i++)
    if ((name[i]=='\\') && (!niskanji2(name,i))) name[i]=DIRSEP;
  if ((name[0]==DIRSEP) && (name[1]==DIRSEP)) return;
  if (isalpha(name[0]) && (name[1]==':')) {
    name[0]=tolower(name[0]);
    disk[0]=name[0];
    disk[1]=name[1];
    disk[2]='\\';
    disk[3]='\0';
    GetVolumeInformation(disk,NULL,0,NULL,NULL,&csensitive,NULL,0);
  } else {
    GetVolumeInformation(NULL,NULL,0,NULL,NULL,&csensitive,NULL,0);
  }
  if ((csensitive & FS_CASE_SENSITIVE)==0) {
    for (i=0;name[i]!='\0';i++) {
      if (niskanji(name[i])) i++;
      else name[i]=tolower(name[i]);
    }
  }
}

char *nstrnew(void)
{
  char *po;

  po=(char *)malloc(NSTRLEN);
  po[0]='\0';
  return po;
}

char *nstrccat(char *po,char ch)
{
  size_t len,num;

  if (po==NULL) return NULL;
  len=strlen(po);
  num=len/NSTRLEN;
  if (len%NSTRLEN==NSTRLEN-1) po=(char *)realloc(po,NSTRLEN*(num+2));
  po[len]=ch;
  po[len+1]='\0';
  return po;
}

int fgetline(FILE *fp,char **buf)
{
  char *s;
  char ch;

  *buf=NULL;
  ch=fgetc(fp);
  if (ch==EOF) return 1;
  s=nstrnew();
  while (TRUE) {
    if ((ch=='\0') || (ch=='\n') || (ch==EOF)) {
      *buf=s;
      return 0;
    }
    if (ch!='\r') s=nstrccat(s,ch);
    ch=fgetc(fp);
  }
}

int sscanf2(char *buffer,char *format,...)
{
  va_list ap;
  int i,num;
  int *d;
  double *e;
  char *s;
  char *endptr;

  va_start(ap,format);
  s=buffer;
  num=0;
  i=0;
  while (format[i]!='\0') {
    if (format[i]=='d') {
      d=va_arg(ap,int *);
      *d=strtol(s,&endptr,10);
      num++;
      if (endptr[0]=='\0') break;
      s=endptr;
    } else if (format[i]=='e') {
      e=va_arg(ap,double *);
      *e=strtod(s,&endptr);
      num++;
      if (endptr[0]=='\0') break;
      s=endptr;
    }
    i++;
  }
  va_end(ap);
  return num;
}

int printfstderr(char *fmt,...)
{
  int len;
  char buf[1024];
  va_list ap;

  va_start(ap,fmt);
  len=vsprintf(buf,fmt,ap);
  va_end(ap);
  MessageBeep(MB_ICONASTERISK);
  MessageBox(NULL,buf,NULL,MB_OK|MB_TASKMODAL|MB_ICONEXCLAMATION|MB_SETFOREGROUND);
  return len;
}

class MainDialog  : public TDialog {
  public:
    MainDialog(TWindow *parent,TResId resID):TDialog(parent,resID)
    {
    }
    ~MainDialog()
    {
    }
    void SetupItem();
    void LoadClicked();
  protected:
    void SetupWindow();
    void BrowseClicked();
    void CloseWindow(int retVal);
    int SaveData();
    char FileBuf[512];
    DECLARE_RESPONSE_TABLE(MainDialog);
};

DEFINE_RESPONSE_TABLE1(MainDialog,TDialog)
  EV_CHILD_NOTIFY(IDBROWSE,BN_CLICKED,BrowseClicked),
  EV_CHILD_NOTIFY(IDLOAD,BN_CLICKED,LoadClicked),
END_RESPONSE_TABLE;

void MainDialog::SetupWindow()
{
  TDialog::SetupWindow();
}

void MainDialog::SetupItem()
{
  char str[40];
  double min,max;
  int div,incmin,incmax;

  TDialog::SetupWindow();
  SendDlgItemMessage(IDFILE,WM_SETTEXT,0,(LPARAM)"$CALC$.DAT");
  SendDlgItemMessage(IDFORMULA,WM_SETTEXT,0,(LPARAM)"X");
  if (ScriptFile!=NULL) SendDlgItemMessage(IDDATA,BM_SETCHECK,1,0);
  else SendDlgItemMessage(IDDATA,BM_SETCHECK,0,0);
  min=0;
  max=1;
  div=100;
  incmin=TRUE;
  incmax=TRUE;
  sprintf(str,"%.15e",min);
  SendDlgItemMessage(IDMIN,WM_SETTEXT,0,(LPARAM)str);
  sprintf(str,"%.15e",max);
  SendDlgItemMessage(IDMAX,WM_SETTEXT,0,(LPARAM)str);
  sprintf(str,"%d",div);
  SendDlgItemMessage(IDDIV,WM_SETTEXT,0,(LPARAM)str);
  if (incmin) SendDlgItemMessage(IDINCLUDEMIN,BM_SETCHECK,1,0);
  else SendDlgItemMessage(IDINCLUDEMIN,BM_SETCHECK,0,0);
  if (incmax) SendDlgItemMessage(IDINCLUDEMAX,BM_SETCHECK,1,0);
  else SendDlgItemMessage(IDINCLUDEMAX,BM_SETCHECK,0,0);
}

int MainDialog::SaveData()
{
  FILE *fp;
  char *formula,*buf,*endptr;;
  double min,max,x;
  int len,div,incmin,incmax;
  int i,st,ed;

  len=SendDlgItemMessage(IDMIN,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)malloc(len+1))!=NULL) {
    SendDlgItemMessage(IDMIN,WM_GETTEXT,len+1,(LPARAM)buf);
    min=strtod(buf,&endptr);
    if (endptr[0]!='\0') {
      free(buf);
      return 0;
    }
    free(buf);
  } else return 0;
  len=SendDlgItemMessage(IDMAX,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)malloc(len+1))!=NULL) {
    SendDlgItemMessage(IDMAX,WM_GETTEXT,len+1,(LPARAM)buf);
    max=strtod(buf,&endptr);
    if (endptr[0]!='\0') {
      free(buf);
      return 0;
    }
    free(buf);
  } else return 0;
  len=SendDlgItemMessage(IDDIV,WM_GETTEXTLENGTH,0,0);
  if ((buf=(char *)malloc(len+1))!=NULL) {
    SendDlgItemMessage(IDDIV,WM_GETTEXT,len+1,(LPARAM)buf);
    div=strtol(buf,&endptr,10);
    if (endptr[0]!='\0') {
      free(buf);
      return 0;
    }
    free(buf);
  } else return 0;
  if (SendDlgItemMessage(IDINCLUDEMIN,BM_GETCHECK,0,0)) incmin=TRUE;
  else incmin=FALSE;
  if (SendDlgItemMessage(IDINCLUDEMAX,BM_GETCHECK,0,0)) incmax=TRUE;
  else incmax=FALSE;
  len=SendDlgItemMessage(IDFORMULA,WM_GETTEXTLENGTH,0,0);
  if ((formula=(char *)malloc(len+1))!=NULL) {
    SendDlgItemMessage(IDFORMULA,WM_GETTEXT,len+1,(LPARAM)formula);
  } else return 0;
  fp=fopen(FileBuf,"wt");
  fprintf(fp,"-s2 -mxX -my%s\n",formula);
  fprintf(fp,"%cCALC.NSC %.15e %.15e %d %d %d\n",'%',min,max,div,incmin,incmax);
  if (incmin) st=0;
  else st=1;
  if (incmax) ed=div;
  else ed=div-1;
  for (i=st;i<=ed;i++) {
    x=min+(max-min)/div*i;
    fprintf(fp,"%.15e %.15e\n",x,x);
  }
  fclose(fp);
  return 1;
}

void MainDialog::CloseWindow(int retVal)
{
  char *buf;
  FILE *fp;
  int len;


  if (retVal==IDOK) {
    SendDlgItemMessage(IDFILE,WM_GETTEXT,512,(LPARAM)FileBuf);
    if (FileBuf[0]=='\0') return;
    if (access(FileBuf,04)==0) {
      MessageBeep(MB_ICONASTERISK);
      buf=(char *)malloc(strlen(FileBuf)+60);
      if (buf!=NULL) {
        sprintf(buf,
                 "`%s'\n\nFile exists. Can I overwrite?",
                 FileBuf);
        if (MessageBox(buf,"Making data file",MB_ICONQUESTION|MB_YESNO)!=IDYES) {
          free(buf);
          return;
        }
        free(buf);
      } else {
        if (MessageBox(
                 "File exists. Can I overwrite?",
        "Making data file",MB_ICONQUESTION|MB_YESNO)!=IDYES)
         return;
      }
    }
    if (SaveData()) {
      if ((ScriptFile!=NULL) &&SendDlgItemMessage(IDDATA,BM_GETCHECK,0,0)) {
        fp=fopen(ScriptFile,"wt");
        fprintf(fp,"new file\n");
        fprintf(fp,"file::file='%s'\n",FileBuf);
        changefilename(FileBuf);
        fprintf(fp,"file::file='%s'\n",FileBuf);
        fprintf(fp,"file::head_skip=2\n",FileBuf);
        fprintf(fp,"file::type=line\n",FileBuf);
        len=SendDlgItemMessage(IDFORMULA,WM_GETTEXTLENGTH,0,0);
        if ((buf=(char *)malloc(len+1))!=NULL) {
          SendDlgItemMessage(IDFORMULA,WM_GETTEXT,len+1,(LPARAM)buf);
          fprintf(fp,"file::math_y='%s'\n",buf);
          free(buf);
        }
        fclose(fp);
      }
    }
  }
  TDialog::CloseWindow(retVal);
}

void MainDialog::BrowseClicked()
{
  FileBuf[0]='\0';
  if (nGetSaveFileName(Handle,"data file",NULL,NULL,
      FileBuf,512,"All files (*.*)\0*.*\0Data files (*.dat)\0*.dat\0",FALSE)==IDOK) {
    SendDlgItemMessage(IDFILE,WM_SETTEXT,0,(LPARAM)FileBuf);
  }
}

void MainDialog::LoadClicked()
{
  FILE *fd;
  char *buf;
  int i;
  char *po,*s;
  double min,max;
  int div,incmin,incmax;
  char str[40];

  SendDlgItemMessage(IDFILE,WM_GETTEXT,512,(LPARAM)FileBuf);
  if (FileBuf[0]=='\0') return;
  if ((fd=fopen(FileBuf,"rt"))==NULL) return;
  if (fgetline(fd,&buf)!=0) {
    fclose(fd);
    return;
  }
  po=buf;
  while (po[0]!='\0') {
    for (;(po[0]!='\0') && (strchr(" \x09",po[0])!=NULL);po++);
    if (po[0]=='-') {
      switch (po[1]) {
      case 'm':
        if (po[2]=='y') {
          for (i=3;(po[i]!='\0') && (strchr(" \x09",po[i])==NULL);i++);
          if (i>3) {
            if ((s=(char *)malloc(i-2))!=NULL) {
              strncpy(s,po+3,i-3);
              s[i-3]='\0';
              SendDlgItemMessage(IDFORMULA,WM_SETTEXT,0,(LPARAM)s);
              free(s);
            }
          }
          po+=i;
        } else {
          for (i=3;(po[i]!='\0') && (strchr(" \x09",po[i])==NULL);i++);
          po+=i;
        }
        break;
      default:
        for (i=1;(po[i]!='\0') && (strchr(" \x09",po[i])==NULL);i++);
        po+=i;
        break;
      }
    } else {
      for (i=1;(po[i]!='\0') && (strchr(" \x09",po[i])==NULL);i++);
      po+=i;
    }
  }
  free(buf);
  if (fgetline(fd,&buf)!=0) {
    fclose(fd);
    return;
  }
  if (strncmp("%CALC.NSC",buf,9)==0) {
    if (sscanf2(buf+9,"%le%le%d%d%d",
    (double *)&min,(double *)&max,(int *)&div,(int *)&incmin,(int *)&incmax)==5) {
      sprintf(str,"%.15e",min);
      SendDlgItemMessage(IDMIN,WM_SETTEXT,0,(LPARAM)str);
      sprintf(str,"%.15e",max);
      SendDlgItemMessage(IDMAX,WM_SETTEXT,0,(LPARAM)str);
      sprintf(str,"%d",div);
      SendDlgItemMessage(IDDIV,WM_SETTEXT,0,(LPARAM)str);
      if (incmin) SendDlgItemMessage(IDINCLUDEMIN,BM_SETCHECK,1,0);
      else SendDlgItemMessage(IDINCLUDEMIN,BM_SETCHECK,0,0);
      if (incmax) SendDlgItemMessage(IDINCLUDEMAX,BM_SETCHECK,1,0);
      else SendDlgItemMessage(IDINCLUDEMAX,BM_SETCHECK,0,0);
    }
  }
  free(buf);
  fclose(fd);
}

class TMyApp : public TApplication {
  public:
    TMyApp(int argc,char **argv):TApplication() {
      Argc=argc;
      Argv=argv;
    }
  protected:
    void InitMainWindow();
    void InitInstance();
    MainDialog *dlg;
    int Argc;
    char **Argv;
};

void TMyApp::InitInstance()
{
  int i;

  TApplication::InitInstance();
  PumpWaitingMessages();

  for (i=1;(i<Argc) && (Argv[i][0]=='-');i++) {
    switch (Argv[i][1]) {
    default:
      printfstderr("unknown option `%s'.",Argv[i]);
      BreakMessageLoop=TRUE;
      return;
    }
  }
  if (i<Argc) ScriptFile=Argv[i];
  else ScriptFile=NULL;
  dlg->SetupItem();
  dlg->LoadClicked();
}

void TMyApp::InitMainWindow()
{
  TFrameWindow *frame;
  dlg=new MainDialog(0,DIALOG_MAIN);
  frame=new TFrameWindow(0, "CALC",dlg,TRUE);
  frame->Attr.Style=WS_VISIBLE | WS_SYSMENU | WS_CAPTION | WS_DLGFRAME;
  SetMainWindow(frame);
  GetMainWindow()->SetIcon(this,ICON_1);
}


int OwlMain(int argc,char **argv)
{
  hInstance=_hInstance;
  TMyApp(argc,argv).Run();
  return 0;
}


