/* 
 * 
 * This file is part of "Ngraph for Windows".
 * 
 * Copyright (C) 2015, Satoshi ISHIZAKA. isizaka@msa.biglobe.ne.jp
 * 
 * "Ngraph for Windows" is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * "Ngraph for Windows" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

/*
 *
 * clip2gra.cpp
 *
 */

#include <owl/pch.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <io.h>
#include <dir.h>

HINSTANCE hInstance;

extern "C" {
#include "../win/winfopen.h"
}

#include "clip2gra.rh"

#define MPI 3.1415926535897932385
#define GRAF "%Ngraph GRAF"
#define LIBDIR "/usr/local/lib/Ngraph"
#define CONF "Ngraph.ini"
#define G2WINCONF "[winmenu]"
#define DIRSEP '/'
#define CONFSEP "/"
#define NSTRLEN 256
#define LINETOLIMIT 500


#define TRUE  1
#define FALSE 0

#define FONT_ROMAN 0
#define FONT_BOLD 1
#define FONT_ITALIC 2
#define FONT_BOLDITALIC 3
#define FONT_UNKNOWN 4

struct fontmap;

struct fontmap {
  char *fontalias;
  char *fontname;
  int charset;
  int italic;
  struct fontmap *next;
};

char *ScriptFile=NULL;
int winextx=0;
int winexty=0;
int winorgx=0;
int winorgy=0;
int winwidth=0;
int winheight=0;
int winext=FALSE;
int winorg=FALSE;
int view=FALSE;
char *libdir,*homedir;
int Style,Width;
COLORREF Col,TCol;
double dot_pixelx=1;
double dot_pixely=1;
int PolyFillMode=ALTERNATE;
int cpx2=0;
int cpx=0;
int cpy2=0;
int cpy=0;
struct fontmap *fontmaproot;
char *fontalias=NULL;
int space,fontdir,fontpt;
int debug=FALSE;

int printfstderr(char *fmt,...)
{
  int len;
  char buf[1024];
  va_list ap;

  va_start(ap,fmt);
  len=vsprintf(buf,fmt,ap);
  va_end(ap);
  MessageBeep(MB_ICONASTERISK);
  MessageBox(NULL,buf,NULL,MB_OK|MB_TASKMODAL|MB_ICONEXCLAMATION|MB_SETFOREGROUND);
  return len;
}

int niskanji(unsigned char code)
{
  if (((0x81<=code) && (code<=0x9f))
   || ((0xe0<=code) && (code<=0xff))) return TRUE;
  return FALSE;
}

int niskanji2(char *s,int pos)
{
  int k;

  if (pos>=strlen(s)) return FALSE;
  else {
    k=pos;
    do {
      k--;
    } while ((k>=0) && niskanji((unsigned char)s[k]));
    if ((pos-k)%2==1) return FALSE;
    else return TRUE;
  }
}

char *nstrnew(void)
{
  char *po;

  po=(char *)malloc(NSTRLEN);
  po[0]='\0';
  return po;
}

char *nstrccat(char *po,char ch)
{
  size_t len,num;

  if (po==NULL) return NULL;
  len=strlen(po);
  num=len/NSTRLEN;
  if (len%NSTRLEN==NSTRLEN-1) po=(char *)realloc(po,NSTRLEN*(num+2));
  po[len]=ch;
  po[len+1]='\0';
  return po;
}

void changefilename(char *name)
{
  int i;
  char disk[4];
  DWORD csensitive;

  if (name==NULL) return;
  for (i=0;name[i]!='\0';i++)
    if ((name[i]=='\\') && (!niskanji2(name,i))) name[i]=DIRSEP;
  if ((name[0]==DIRSEP) && (name[1]==DIRSEP)) return;
  if (isalpha(name[0]) && (name[1]==':')) {
    name[0]=tolower(name[0]);
    disk[0]=name[0];
    disk[1]=name[1];
    disk[2]='\\';
    disk[3]='\0';
    GetVolumeInformation(disk,NULL,0,NULL,NULL,&csensitive,NULL,0);
  } else {
    GetVolumeInformation(NULL,NULL,0,NULL,NULL,&csensitive,NULL,0);
  }
  if ((csensitive & FS_CASE_SENSITIVE)==0) {
    for (i=0;name[i]!='\0';i++) {
      if (niskanji(name[i])) i++;
      else name[i]=tolower(name[i]);
    }
  }
}

int fgetline(FILE *fp,char **buf)
{
  char *s;
  char ch;

  *buf=NULL;
  ch=fgetc(fp);
  if (ch==EOF) return 1;
  s=nstrnew();
  while (TRUE) {
    if ((ch=='\0') || (ch=='\n') || (ch==EOF)) {
      *buf=s;
      return 0;
    }
    if (ch!='\r') s=nstrccat(s,ch);
    ch=fgetc(fp);
  }
}

FILE *openconfig(char *section)
{
  char *s,*buf,*homeconf,*libconf;
  FILE *fp;
  struct stat homestat,libstat;

  homeconf=(char *)malloc(strlen(homedir)+strlen(CONFSEP)+strlen(CONF)+1);
  strcpy(homeconf,homedir);
  if ((strlen(homedir)>0) && (homeconf[strlen(homedir)-1]=='/'))
    homeconf[strlen(homedir)-1]='\0';
  strcat(homeconf,CONFSEP);
  strcat(homeconf,CONF);
  libconf=(char *)malloc(strlen(libdir)+strlen(CONFSEP)+strlen(CONF)+1);
  strcpy(libconf,libdir);
  if ((strlen(libdir)>0) && (libconf[strlen(libdir)-1]=='/'))
    libconf[strlen(libdir)-1]='\0';
  strcat(libconf,CONFSEP);
  strcat(libconf,CONF);
  if (access(homeconf,04)==0) {
    if (stat(homeconf,&homestat)!=0) {
      free(homeconf);
      homeconf=NULL;
    }
  } else {
    free(homeconf);
    homeconf=NULL;
  }
  if (access(libconf,04)==0) {
    if (stat(libconf,&libstat)!=0) {
      free(libconf);
      libconf=NULL;
    }
  } else {
    free(libconf);
    libconf=NULL;
  }
  if (homeconf!=NULL) {
    if (libconf==NULL) s=homeconf;
    else if (homestat.st_mtime>=libstat.st_mtime) {
      s=homeconf;
      free(libconf);
    } else {
      s=libconf;
      free(homeconf);
    }
  } else if (libconf!=NULL) s=libconf;
  else return NULL;
  if ((fp=fopen(s,"rt"))==NULL) return NULL;
  free(s);
  while (fgetline(fp,&buf)==0) {
    if (strcmp(buf,section)==0) {
      free(buf);
      return fp;
    }
    free(buf);
  }
  fclose(fp);
  return NULL;
}

char *getitok(char **s,int *len,char *ifs)
{
  char *po,*spo;
  int i;

  if (*s==NULL) return NULL;
  po=*s;
  for (i=0;(po[i]!='\0') && (strchr(ifs,po[i])!=NULL);i++);
  if (po[i]=='\0') {
    *len=0;
    return NULL;
  }
  spo=po+i;
  for (;(po[i]!='\0') && (strchr(ifs,po[i])==NULL);i++);
  *s+=i;
  *len=*s-spo;
  return spo;
}

char *getitok2(char **s,int *len,char *ifs)
{
  char *po,*s2;

  if ((s2=getitok(s,len,ifs))==NULL) return NULL;
  po=(char *)malloc(*len+1);
  strncpy(po,s2,*len);
  po[*len]='\0';
  return po;
}

char *getconfig(FILE *fp,char **val)
{
  char *s,*tok,*buf;
  int len;

  while (TRUE) {
    if (fgetline(fp,&buf)!=0) return NULL;
    else {
      if (buf[0]=='[') {
        free(buf);
        return NULL;
      } else {
        s=buf;
        if ((tok=getitok2(&s,&len," \x09=,"))!=NULL) {
          for (;(s[0]!='\0') && (strchr(" \x09=,",s[0])!=NULL);s++);
          *val=(char *)malloc(strlen(s)+1);
          strcpy(*val,s);
          free(buf);
          return tok;
        }
        free(buf);
        free(tok);
      }
    }
  }
}

void closeconfig(FILE *fp)
{
  fclose(fp);
}

int round(double x)
{
  int ix;
  double dx;

  if (x>INT_MAX) return INT_MAX;
  else if (x<INT_MIN) return INT_MIN;
  else {
    ix=(int )x;
    dx=x-ix;
    if (dx>=0.5) return ix+1;
    else if (dx<=-0.5) return ix-1;
    else return ix;
  }
}

int loadconfig()
{
  FILE *fp;
  char *tok,*str,*s2;
  char *f1,*f2,*f3,*f4;
  struct fontmap *fcur,*fnew;
  int len;

  if ((fp=openconfig(G2WINCONF))==NULL) return 0;
  fcur=fontmaproot;
  while ((tok=getconfig(fp,&str))!=NULL) {
    s2=str;
    if (strcmp(tok,"font_map")==0) {
      f1=getitok2(&s2,&len," \x09,");
      f3=getitok2(&s2,&len," \x09,");
      f4=getitok2(&s2,&len," \x09,");
      for (;(s2[0]!='\0') && (strchr(" \x09,",s2[0])!=NULL);s2++);
      f2=getitok2(&s2,&len,"");
      if ((f1!=NULL) && (f2!=NULL) && (f3!=NULL) && (f4!=NULL)) {
        if ((fnew=(fontmap *)malloc(sizeof(struct fontmap)))==NULL) {
          free(tok);
          free(f1);
          free(f2);
          free(f3);
          free(f4);
          closeconfig(fp);
          return 1;
        }
        if (fcur==NULL) fontmaproot=fnew;
        else fcur->next=fnew;
        fcur=fnew;
        fcur->next=NULL;
        fcur->fontalias=f1;
        fcur->fontname=f2;
        if (strcmp(f4,"roman")==0) fcur->italic=FONT_ROMAN;
        else if (strcmp(f4,"italic")==0) fcur->italic=FONT_ITALIC;
        else if (strcmp(f4,"bold")==0) fcur->italic=FONT_BOLD;
        else if (strcmp(f4,"bold_italic")==0) fcur->italic=FONT_BOLDITALIC;
        else fcur->italic=FONT_UNKNOWN;
        if (strcmp(f3,"shiftjis")==0) fcur->charset=SHIFTJIS_CHARSET;
        else if (strcmp(f3,"symbol")==0) fcur->charset=SYMBOL_CHARSET;
        else if (strcmp(f3,"ansi")==0) fcur->charset=ANSI_CHARSET;
        else if (strcmp(f3,"oem")==0) fcur->charset=OEM_CHARSET;
        else if (strcmp(f3,"hangeul")==0) fcur->charset=HANGEUL_CHARSET;
        else if (strcmp(f3,"chinesebig5")==0) fcur->charset=CHINESEBIG5_CHARSET;
        else fcur->charset=DEFAULT_CHARSET;
        free(f3);
        free(f4);
      } else {
        free(f1);
        free(f2);
        free(f3);
        free(f4);
      }
    }
    free(tok);
    free(str);
  }
  closeconfig(fp);
  return 0;
}

class WinStatic : public TStatic {
  public:
    WinStatic(TWindow *parent,int resId) : TStatic(parent,resId) {};
    ~WinStatic() {};
    HDC GetWinDC();
    void ReleaseWinDC(HDC dc) { ReleaseDC(Handle,dc); };
  protected:
    void EvPaint();
    DECLARE_RESPONSE_TABLE(WinStatic);
};

DEFINE_RESPONSE_TABLE1(WinStatic,TStatic)
  EV_WM_PAINT,
END_RESPONSE_TABLE;

HDC WinStatic::GetWinDC()
{
  HDC dc;
  RECT rect;

  ::GetClientRect(Handle,&rect);
  dc=GetDC(Handle);
  SetMapMode(dc,MM_ANISOTROPIC);
  SetViewportExtEx(dc,rect.right-rect.left,rect.bottom-rect.top,NULL);
  return dc;
}

void WinStatic::EvPaint()
{
  HANDLE MetaPictHD;
  METAFILEPICT *MetaFilePict;
  HMETAFILE HMF;
  PAINTSTRUCT ps;
  HDC dc;
  RECT rect;
  HBRUSH brush;

  dc=::BeginPaint(Handle,&ps);
  ::GetClientRect(Handle,&rect);
  brush=CreateSolidBrush(RGB(255,255,255));
  FillRect(dc,&rect,brush);
  DeleteObject(brush);
  if (IsClipboardFormatAvailable(CF_METAFILEPICT)) {
    ::OpenClipboard(Handle);
    MetaPictHD=GetClipboardData(CF_METAFILEPICT);
    MetaFilePict=(LPMETAFILEPICT)GlobalLock(MetaPictHD);
    HMF=MetaFilePict->hMF;
    SetMapMode(dc,MM_ANISOTROPIC);
    SetViewportExtEx(dc,rect.right-rect.left,rect.bottom-rect.top,NULL);
    PlayMetaFile(dc,HMF);
    GlobalUnlock(MetaPictHD);
    ::CloseClipboard();
  }
  ::EndPaint(Handle,&ps);
}

int p2dx(int r)
{
  return round(r*dot_pixelx);
}

int p2dy(int r)
{
  return round(r*dot_pixely);
}

int CALLBACK EnumMetaFileProc(HDC hdc,HANDLETABLE FAR *lpht,METARECORD FAR *lpmr,
                              int cObj,LPARAM lParam)
{
  FILE *fp;
  HGDIOBJ hgdiobj;
  LOGPEN lPen;
  LOGBRUSH lBrush;
  LOGFONT lFont;
  WORD rdFunction;
  short int *rdParm;
  int i,j,k;
  int stangle,edangle,angle,rx,ry,cx,cy;
  double tx,ty;
  int count,pcount,pmode;
  struct fontmap *fcur;
  int italic,pt,dir;
  char *font;
  char *s;

  fp=(FILE *)lParam;
  rdFunction=lpmr->rdFunction;
  rdParm=(short int *)lpmr->rdParm;
  switch (rdFunction) {
//SetWindowOrg	0x020B
  case 0x020B:
    winorgx=rdParm[1];
    winorgy=rdParm[0];
    fprintf(fp,"V 5 %d %d 0 0 0\n",p2dx(winorgx),p2dy(winorgy));
    break;
//OffsetWindowOrg	0x020F
  case 0x020F:
    winorgx+=rdParm[1];
    winorgy+=rdParm[0];
    fprintf(fp,"V 5 %d %d 0 0 0\n",p2dx(winorgx),p2dy(winorgy));
    break;
//SetWindowExt	0x020C
  case 0x020C:
    winextx=rdParm[1];
    winexty=rdParm[0];
    dot_pixelx=((double )winwidth)/winextx;
    dot_pixely=((double )winheight)/winexty;
    break;
//ScaleWindowExt	0x0400
  case 0x0400:
    winextx=winextx*rdParm[3]/rdParm[2];
    winexty=winexty*rdParm[1]/rdParm[0];
    dot_pixelx=((double )winwidth)/winextx;
    dot_pixely=((double )winheight)/winexty;
    break;
//MoveTo	0x0214
  case 0x0214:
    cpx2=cpx=rdParm[1];
    cpy2=cpy=rdParm[0];
    fprintf(fp,"M 2 %d %d\n",p2dx(rdParm[1]),p2dy(rdParm[0]));
    break;
//LineTo	0x0213
  case 0x0213:
    hgdiobj=GetCurrentObject(hdc,OBJ_PEN);
    if (GetObject(hgdiobj,sizeof(LOGPEN),(LPSTR)&lPen)!=0) {
      if (Col!=lPen.lopnColor) {
        Col=lPen.lopnColor;
        fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
      }
      if ((Style!=lPen.lopnStyle) || (Width!=lPen.lopnWidth.x)) {
        Style=lPen.lopnStyle;
        Width=lPen.lopnWidth.x;
        if (Style==PS_SOLID) fprintf(fp,"A 5 0 %d 1 1 1000\n",p2dx(Width));
        else if (Style==PS_DASH) fprintf(fp,"A 7 2 %d 1 1 1000 450 150\n",p2dx(Width));
        else if (Style==PS_DOT) fprintf(fp,"A 7 2 %d 1 1 1000 150 150\n",p2dx(Width));
        else if (Style==PS_DASHDOT) fprintf(fp,"A 9 4 %d 1 1 1000 450 150 150 150\n",p2dx(Width));
        else if (Style==PS_DASHDOTDOT) fprintf(fp,"A 11 6 %d 1 1 1000 450 150 150 150 150 150\n",p2dx(Width));
        else fprintf(fp,"A 5 0 0 1 1 1000\n");
      }
    }
    if ((cpx!=cpx2) || (cpy!=cpy2)) {
      fprintf(fp,"M 2 %d %d\n",p2dx(cpx),p2dy(cpy));
    }
    cpx2=cpx=rdParm[1];
    cpy2=cpy=rdParm[0];
    fprintf(fp,"T 2 %d %d\n",p2dx(rdParm[1]),p2dy(rdParm[0]));
    break;
//SetPixel	0x041F
  case 0x041F:
    if (Col!=rdParm[0]) {
      Col=lPen.lopnColor;
      fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
    }
    fprintf(fp,"P 2 %d %d\n",p2dx(rdParm[2]),p2dy(rdParm[1]));
    break;
//SetTextColor	0x0209
  case 0x0209:
    TCol=*(COLORREF *)(rdParm+0);
    break;
//RoundRect	0x061C
//Rectangle	0x041B
  case 0x061c:
  case 0x041b:
    hgdiobj=GetCurrentObject(hdc,OBJ_BRUSH);
    if (GetObject(hgdiobj,sizeof(LOGBRUSH),(LPSTR)&lBrush)!=0) {
      if (lBrush.lbStyle==BS_SOLID) {
        if (Col!=lBrush.lbColor) {
          Col=lBrush.lbColor;
          fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
        }
        fprintf(fp,"B 5 %d %d %d %d 1\n",p2dx(rdParm[3]),p2dy(rdParm[2]),
                                          p2dx(rdParm[1]),p2dy(rdParm[0]));
      }
    }
    hgdiobj=GetCurrentObject(hdc,OBJ_PEN);
    if (GetObject(hgdiobj,sizeof(LOGPEN),(LPSTR)&lPen)!=0) {
      if (Col!=lPen.lopnColor) {
        Col=lPen.lopnColor;
        fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
      }
      if ((lPen.lopnStyle!=PS_NULL) && (lPen.lopnStyle!=PS_INSIDEFRAME)) {
        if ((Style!=lPen.lopnStyle) || (Width!=lPen.lopnWidth.x)) {
          Style=lPen.lopnStyle;
          Width=lPen.lopnWidth.x;
          if (Style==PS_SOLID) fprintf(fp,"A 5 0 %d 1 1 1000\n",p2dx(Width));
          else if (Style==PS_DASH) fprintf(fp,"A 7 2 %d 1 1 1000 450 150\n",p2dx(Width));
          else if (Style==PS_DOT) fprintf(fp,"A 7 2 %d 1 1 1000 150 150\n",p2dx(Width));
          else if (Style==PS_DASHDOT) fprintf(fp,"A 9 4 %d 1 1 1000 450 150 150 150\n",p2dx(Width));
          else if (Style==PS_DASHDOTDOT) fprintf(fp,"A 11 6 %d 1 1 1000 450 150 150 150 150 150\n",p2dx(Width));
        }
        fprintf(fp,"B 5 %d %d %d %d 0\n",p2dx(rdParm[3]),p2dy(rdParm[2]),
                                         p2dx(rdParm[1]),p2dy(rdParm[0]));
      }
    }
    break;
//PatBlt	0x061D
  case 0x061d:
    hgdiobj=GetCurrentObject(hdc,OBJ_BRUSH);
    if (GetObject(hgdiobj,sizeof(LOGBRUSH),(LPSTR)&lBrush)!=0) {
      if (lBrush.lbStyle==BS_SOLID) {
        if (Col!=lBrush.lbColor) {
          Col=lBrush.lbColor;
          fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
        }
        fprintf(fp,"B 5 %d %d %d %d 1\n",p2dx(rdParm[5]),p2dy(rdParm[4]),
                p2dx(rdParm[5])+p2dx(rdParm[3]),p2dy(rdParm[4])+p2dy(rdParm[2]));
      }
    }
    break;
//Arc	0x0817
  case 0x0817:
    cx=(p2dx(rdParm[7])+p2dx(rdParm[5]))/2;
    cy=(p2dy(rdParm[6])+p2dy(rdParm[4]))/2;
    rx=abs(p2dx(rdParm[7])-p2dx(rdParm[5]))/2;
    ry=abs(p2dy(rdParm[6])-p2dy(rdParm[4]))/2;
    ty=-((double )(p2dy(rdParm[2])-cy))/ry;
    tx= ((double )(p2dy(rdParm[3])-cx))/rx;
    if (tx==0) {
      if (ty>0) stangle=9000;
      else stangle=27000;
    } else {
      stangle=round(atan(ty/tx)*18000/MPI);
      if (tx<0) stangle-=18000;
    }
    ty=-((double )(p2dy(rdParm[0])-cy))/ry;
    tx= ((double )(p2dy(rdParm[1])-cx))/rx;
    if (tx==0) {
      if (ty>0) edangle=9000;
      else edangle=27000;
    } else {
      edangle=round(atan(ty/tx)*18000/MPI);
      if (tx<0) edangle-=18000;
    }
    if (stangle<0) stangle+=36000;
    if (edangle<0) edangle+=36000;
    if (edangle<stangle) edangle+=36000;
    angle=edangle-stangle;
    hgdiobj=GetCurrentObject(hdc,OBJ_PEN);
    if (GetObject(hgdiobj,sizeof(LOGPEN),(LPSTR)&lPen)!=0) {
      if (Col!=lPen.lopnColor) {
        Col=lPen.lopnColor;
        fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
      }
      if ((lPen.lopnStyle!=PS_NULL) && (lPen.lopnStyle!=PS_INSIDEFRAME)) {
        if ((Style!=lPen.lopnStyle) || (Width!=lPen.lopnWidth.x)) {
          Style=lPen.lopnStyle;
          Width=lPen.lopnWidth.x;
          if (Style==PS_SOLID) fprintf(fp,"A 5 0 %d 1 1 1000\n",p2dx(Width));
          else if (Style==PS_DASH) fprintf(fp,"A 7 2 %d 1 1 1000 450 150\n",p2dx(Width));
          else if (Style==PS_DOT) fprintf(fp,"A 7 2 %d 1 1 1000 150 150\n",p2dx(Width));
          else if (Style==PS_DASHDOT) fprintf(fp,"A 9 4 %d 1 1 1000 450 150 150 150\n",p2dx(Width));
          else if (Style==PS_DASHDOTDOT) fprintf(fp,"A 11 6 %d 1 1 1000 450 150 150 150 150 150\n",p2dx(Width));
        }
        fprintf(fp,"C 7 %d %d %d %d %d %d 0\n",cx,cy,rx,ry,stangle,angle);
      }
    }
    break;
//Chord	0x0830
  case 0x0830:
    cx=(p2dx(rdParm[7])+p2dx(rdParm[5]))/2;
    cy=(p2dy(rdParm[6])+p2dy(rdParm[4]))/2;
    rx=abs(p2dx(rdParm[7])-p2dx(rdParm[5]))/2;
    ry=abs(p2dy(rdParm[6])-p2dy(rdParm[4]))/2;
    ty=-((double )(p2dy(rdParm[2])-cy))/ry;
    tx= ((double )(p2dy(rdParm[3])-cx))/rx;
    if (tx==0) {
      if (ty>0) stangle=9000;
      else stangle=27000;
    } else {
      stangle=round(atan(ty/tx)*18000/MPI);
      if (tx<0) stangle-=18000;
    }
    ty=-((double )(p2dy(rdParm[0])-cy))/ry;
    tx= ((double )(p2dy(rdParm[1])-cx))/rx;
    if (tx==0) {
      if (ty>0) edangle=9000;
      else edangle=27000;
    } else {
      edangle=round(atan(ty/tx)*18000/MPI);
      if (tx<0) edangle-=18000;
    }
    if (stangle<0) stangle+=36000;
    if (edangle<0) edangle+=36000;
    if (edangle<stangle) edangle+=36000;
    angle=edangle-stangle;
    hgdiobj=GetCurrentObject(hdc,OBJ_BRUSH);
    if (GetObject(hgdiobj,sizeof(LOGBRUSH),(LPSTR)&lBrush)!=0) {
      if (lBrush.lbStyle==BS_SOLID) {
        if (Col!=lBrush.lbColor) {
          Col=lBrush.lbColor;
          fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
        }
        fprintf(fp,"C 7 %d %d %d %d %d %d 2\n",cx,cy,rx,ry,stangle,angle);
      }
    }
    hgdiobj=GetCurrentObject(hdc,OBJ_PEN);
    if (GetObject(hgdiobj,sizeof(LOGPEN),(LPSTR)&lPen)!=0) {
      if (Col!=lPen.lopnColor) {
        Col=lPen.lopnColor;
        fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
      }
      if ((lPen.lopnStyle!=PS_NULL) && (lPen.lopnStyle!=PS_INSIDEFRAME)) {
        if ((Style!=lPen.lopnStyle) || (Width!=lPen.lopnWidth.x)) {
          Style=lPen.lopnStyle;
          Width=lPen.lopnWidth.x;
          if (Style==PS_SOLID) fprintf(fp,"A 5 0 %d 1 1 1000\n",p2dx(Width));
          else if (Style==PS_DASH) fprintf(fp,"A 7 2 %d 1 1 1000 450 150\n",p2dx(Width));
          else if (Style==PS_DOT) fprintf(fp,"A 7 2 %d 1 1 1000 150 150\n",p2dx(Width));
          else if (Style==PS_DASHDOT) fprintf(fp,"A 9 4 %d 1 1 1000 450 150 150 150\n",p2dx(Width));
          else if (Style==PS_DASHDOTDOT) fprintf(fp,"A 11 6 %d 1 1 1000 450 150 150 150 150 150\n",p2dx(Width));
        }
        fprintf(fp,"C 7 %d %d %d %d %d %d 0\n",cx,cy,rx,ry,stangle,angle);
        fprintf(fp,"M 2 %d %d\n",cx+rx*round(cos(stangle*MPI/18000)),
                                 cy-ry*round(sin(stangle*MPI/18000)));
        fprintf(fp,"T 2 %d %d\n",cx+rx*round(cos(edangle*MPI/18000)),
                                 cy-ry*round(sin(edangle*MPI/18000)));
      }
    }
    break;
//Ellipse	0x0418
  case 0x0418:
    cx=(p2dx(rdParm[3])+p2dx(rdParm[1]))/2;
    cy=(p2dy(rdParm[2])+p2dy(rdParm[0]))/2;
    rx=abs(p2dx(rdParm[3])-p2dx(rdParm[1]))/2;
    ry=abs(p2dy(rdParm[2])-p2dy(rdParm[0]))/2;
    hgdiobj=GetCurrentObject(hdc,OBJ_BRUSH);
    if (GetObject(hgdiobj,sizeof(LOGBRUSH),(LPSTR)&lBrush)!=0) {
      if (lBrush.lbStyle==BS_SOLID) {
        if (Col!=lBrush.lbColor) {
          Col=lBrush.lbColor;
          fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
        }
        fprintf(fp,"C 7 %d %d %d %d 0 36000 1\n",cx,cy,rx,ry);
      }
    }
    hgdiobj=GetCurrentObject(hdc,OBJ_PEN);
    if (GetObject(hgdiobj,sizeof(LOGPEN),(LPSTR)&lPen)!=0) {
      if (Col!=lPen.lopnColor) {
        Col=lPen.lopnColor;
        fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
      }
      if ((lPen.lopnStyle!=PS_NULL) && (lPen.lopnStyle!=PS_INSIDEFRAME)) {
        if ((Style!=lPen.lopnStyle) || (Width!=lPen.lopnWidth.x)) {
          Style=lPen.lopnStyle;
          Width=lPen.lopnWidth.x;
          if (Style==PS_SOLID) fprintf(fp,"A 5 0 %d 1 1 1000\n",p2dx(Width));
          else if (Style==PS_DASH) fprintf(fp,"A 7 2 %d 1 1 1000 450 150\n",p2dx(Width));
          else if (Style==PS_DOT) fprintf(fp,"A 7 2 %d 1 1 1000 150 150\n",p2dx(Width));
          else if (Style==PS_DASHDOT) fprintf(fp,"A 9 4 %d 1 1 1000 450 150 150 150\n",p2dx(Width));
          else if (Style==PS_DASHDOTDOT) fprintf(fp,"A 11 6 %d 1 1 1000 450 150 150 150 150 150\n",p2dx(Width));
        }
        fprintf(fp,"C 7 %d %d %d %d 0 36000 0\n",cx,cy,rx,ry,stangle,angle);
      }
    }
    break;
//Pie	0x081A
  case 0x081A:
    cx=(p2dx(rdParm[7])+p2dx(rdParm[5]))/2;
    cy=(p2dy(rdParm[6])+p2dy(rdParm[4]))/2;
    rx=abs(p2dx(rdParm[7])-p2dx(rdParm[5]))/2;
    ry=abs(p2dy(rdParm[6])-p2dy(rdParm[4]))/2;
    ty=-((double )(p2dy(rdParm[2])-cy))/ry;
    tx= ((double )(p2dy(rdParm[3])-cx))/rx;
    if (tx==0) {
      if (ty>0) stangle=9000;
      else stangle=27000;
    } else {
      stangle=round(atan(ty/tx)*18000/MPI);
      if (tx<0) stangle-=18000;
    }
    ty=-((double )(p2dy(rdParm[0])-cy))/ry;
    tx= ((double )(p2dy(rdParm[1])-cx))/rx;
    if (tx==0) {
      if (ty>0) edangle=9000;
      else edangle=27000;
    } else {
      edangle=round(atan(ty/tx)*18000/MPI);
      if (tx<0) edangle-=18000;
    }
    if (stangle<0) stangle+=36000;
    if (edangle<0) edangle+=36000;
    if (edangle<stangle) edangle+=36000;
    angle=edangle-stangle;
    hgdiobj=GetCurrentObject(hdc,OBJ_BRUSH);
    if (GetObject(hgdiobj,sizeof(LOGBRUSH),(LPSTR)&lBrush)!=0) {
      if (lBrush.lbStyle==BS_SOLID) {
        if (Col!=lBrush.lbColor) {
          Col=lBrush.lbColor;
          fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
        }
        fprintf(fp,"C 7 %d %d %d %d %d %d 1\n",cx,cy,rx,ry,stangle,angle);
      }
    }
    hgdiobj=GetCurrentObject(hdc,OBJ_PEN);
    if (GetObject(hgdiobj,sizeof(LOGPEN),(LPSTR)&lPen)!=0) {
      if (Col!=lPen.lopnColor) {
        Col=lPen.lopnColor;
        fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
      }
      if ((lPen.lopnStyle!=PS_NULL) && (lPen.lopnStyle!=PS_INSIDEFRAME)) {
        if ((Style!=lPen.lopnStyle) || (Width!=lPen.lopnWidth.x)) {
          Style=lPen.lopnStyle;
          Width=lPen.lopnWidth.x;
          if (Style==PS_SOLID) fprintf(fp,"A 5 0 %d 1 1 1000\n",p2dx(Width));
          else if (Style==PS_DASH) fprintf(fp,"A 7 2 %d 1 1 1000 450 150\n",p2dx(Width));
          else if (Style==PS_DOT) fprintf(fp,"A 7 2 %d 1 1 1000 150 150\n",p2dx(Width));
          else if (Style==PS_DASHDOT) fprintf(fp,"A 9 4 %d 1 1 1000 450 150 150 150\n",p2dx(Width));
          else if (Style==PS_DASHDOTDOT) fprintf(fp,"A 11 6 %d 1 1 1000 450 150 150 150 150 150\n",p2dx(Width));
        }
        fprintf(fp,"C 7 %d %d %d %d %d %d 0\n",cx,cy,rx,ry,stangle,angle);
        fprintf(fp,"M 2 %d %d\n",cx+rx*round(cos(stangle*MPI/18000)),
                                 cy-ry*round(sin(stangle*MPI/18000)));
        fprintf(fp,"T 2 %d %d\n",cx,cy);
        fprintf(fp,"T 2 %d %d\n",cx+rx*round(cos(edangle*MPI/18000)),
                                 cy-ry*round(sin(edangle*MPI/18000)));
      }
    }
    break;
//SetPolyFillMode	0x0106
  case 0x0106:
    PolyFillMode=rdParm[0];
    break;
//Polygon
  case 0x0324:
    count=rdParm[0];
    hgdiobj=GetCurrentObject(hdc,OBJ_BRUSH);
    if (GetObject(hgdiobj,sizeof(LOGBRUSH),(LPSTR)&lBrush)!=0) {
      if (lBrush.lbStyle==BS_SOLID) {
        if (Col!=lBrush.lbColor) {
          Col=lBrush.lbColor;
          fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
        }
        if (PolyFillMode==ALTERNATE) pmode=1;
        else pmode=2;
        fprintf(fp,"D %d %d %d",count*2+2,count,pmode);
        j=1;
        for (i=0;i<count;i++) {
          fprintf(fp," %d %d",p2dx(rdParm[j]),p2dy(rdParm[j+1]));
          j+=2;
        }
        fprintf(fp,"\n");
      }
    }
    hgdiobj=GetCurrentObject(hdc,OBJ_PEN);
    if (GetObject(hgdiobj,sizeof(LOGPEN),(LPSTR)&lPen)!=0) {
      if (Col!=lPen.lopnColor) {
        Col=lPen.lopnColor;
        fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
      }
      if ((lPen.lopnStyle!=PS_NULL) && (lPen.lopnStyle!=PS_INSIDEFRAME)) {
        if ((Style!=lPen.lopnStyle) || (Width!=lPen.lopnWidth.x)) {
          Style=lPen.lopnStyle;
          Width=lPen.lopnWidth.x;
          if (Style==PS_SOLID) fprintf(fp,"A 5 0 %d 1 1 1000\n",p2dx(Width));
          else if (Style==PS_DASH) fprintf(fp,"A 7 2 %d 1 1 1000 450 150\n",p2dx(Width));
          else if (Style==PS_DOT) fprintf(fp,"A 7 2 %d 1 1 1000 150 150\n",p2dx(Width));
          else if (Style==PS_DASHDOT) fprintf(fp,"A 9 4 %d 1 1 1000 450 150 150 150\n",p2dx(Width));
          else if (Style==PS_DASHDOTDOT) fprintf(fp,"A 11 6 %d 1 1 1000 450 150 150 150 150 150\n",p2dx(Width));
        }
        fprintf(fp,"D %d %d 0",count*2+2,count);
        j=1;
        for (i=0;i<count;i++) {
          fprintf(fp," %d %d",p2dx(rdParm[j]),p2dy(rdParm[j+1]));
          j+=2;
        }
        fprintf(fp,"\n");
      }
    }
    break;
//PolyPolygon
  case 0x0538:
    pcount=rdParm[0];
    hgdiobj=GetCurrentObject(hdc,OBJ_BRUSH);
    if (GetObject(hgdiobj,sizeof(LOGBRUSH),(LPSTR)&lBrush)!=0) {
      if (lBrush.lbStyle==BS_SOLID) {
        if (Col!=lBrush.lbColor) {
          Col=lBrush.lbColor;
          fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
        }
        if (PolyFillMode==ALTERNATE) pmode=1;
        else pmode=2;
        k=pcount+1;
        for (j=0;j<pcount;j++) {
          count=rdParm[j+1];
          fprintf(fp,"D %d %d %d",count*2+2,count,pmode);
          for (i=0;i<count;i++) {
            fprintf(fp," %d %d",p2dx(rdParm[k]),p2dy(rdParm[k+1]));
            k+=2;
          }
          fprintf(fp,"\n");
        }
      }
    }
    hgdiobj=GetCurrentObject(hdc,OBJ_PEN);
    if (GetObject(hgdiobj,sizeof(LOGPEN),(LPSTR)&lPen)!=0) {
      if (Col!=lPen.lopnColor) {
        Col=lPen.lopnColor;
        fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
      }
      if ((lPen.lopnStyle!=PS_NULL) && (lPen.lopnStyle!=PS_INSIDEFRAME)) {
        if ((Style!=lPen.lopnStyle) || (Width!=lPen.lopnWidth.x)) {
          Style=lPen.lopnStyle;
          Width=lPen.lopnWidth.x;
          if (Style==PS_SOLID) fprintf(fp,"A 5 0 %d 1 1 1000\n",p2dx(Width));
          else if (Style==PS_DASH) fprintf(fp,"A 7 2 %d 1 1 1000 450 150\n",p2dx(Width));
          else if (Style==PS_DOT) fprintf(fp,"A 7 2 %d 1 1 1000 150 150\n",p2dx(Width));
          else if (Style==PS_DASHDOT) fprintf(fp,"A 9 4 %d 1 1 1000 450 150 150 150\n",p2dx(Width));
          else if (Style==PS_DASHDOTDOT) fprintf(fp,"A 11 6 %d 1 1 1000 450 150 150 150 150 150\n",p2dx(Width));
        }
        k=pcount+1;
        for (j=0;j<pcount;j++) {
          count=rdParm[j+1];
          fprintf(fp,"D %d %d %d",count*2+2,count,pmode);
          for (i=0;i<count;i++) {
            fprintf(fp," %d %d",p2dx(rdParm[k]),p2dy(rdParm[k+1]));
            k+=2;
          }
          fprintf(fp,"\n");
        }
      }
    }
    break;
//Polyline
  case 0x0325:
    count=rdParm[0];
    hgdiobj=GetCurrentObject(hdc,OBJ_PEN);
    if (GetObject(hgdiobj,sizeof(LOGPEN),(LPSTR)&lPen)!=0) {
      if (Col!=lPen.lopnColor) {
        Col=lPen.lopnColor;
        fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
      }
      if ((lPen.lopnStyle!=PS_NULL) && (lPen.lopnStyle!=PS_INSIDEFRAME)) {
        if ((Style!=lPen.lopnStyle) || (Width!=lPen.lopnWidth.x)) {
          Style=lPen.lopnStyle;
          Width=lPen.lopnWidth.x;
          if (Style==PS_SOLID) fprintf(fp,"A 5 0 %d 1 1 1000\n",p2dx(Width));
          else if (Style==PS_DASH) fprintf(fp,"A 7 2 %d 1 1 1000 450 150\n",p2dx(Width));
          else if (Style==PS_DOT) fprintf(fp,"A 7 2 %d 1 1 1000 150 150\n",p2dx(Width));
          else if (Style==PS_DASHDOT) fprintf(fp,"A 9 4 %d 1 1 1000 450 150 150 150\n",p2dx(Width));
          else if (Style==PS_DASHDOTDOT) fprintf(fp,"A 11 6 %d 1 1 1000 450 150 150 150 150 150\n",p2dx(Width));
        }
        j=1;
        fprintf(fp,"M 2 %d %d\n",p2dx(rdParm[j]),p2dy(rdParm[j+1]));
        j+=2;
        for (i=1;i<count;i++) {
          fprintf(fp,"T 2 %d %d\n",p2dx(rdParm[j]),p2dy(rdParm[j+1]));
          j+=2;
        }
      }
    }
    break;
//SetTextCharacterExtra	0x0108
  case 0x0108:
    space=rdParm[0];
    break;
//TextOut
  case 0x0521:
    if (Col!=TCol) {
      Col=TCol;
      fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
    }
    hgdiobj=GetCurrentObject(hdc,OBJ_FONT);
    if (GetObject(hgdiobj,sizeof(LOGFONT),(LPSTR)&lFont)!=0) {
      if (lFont.lfWeight>=550) {
        if (lFont.lfItalic) italic=FONT_BOLDITALIC;
        else italic=FONT_BOLD;
      } else {
        if (lFont.lfItalic) italic=FONT_ITALIC;
        else italic=FONT_ROMAN;
      }
      font=NULL;
      fcur=fontmaproot;
      while (fcur!=NULL) {
        if ((strcmp(fcur->fontname,lFont.lfFaceName)==0)
        && (italic==fcur->italic)) {
          font=fcur->fontalias;
          break;
        }
        fcur=fcur->next;
      }
      if (font==NULL) {
        fprintf(fp,"FHelvetica\n");
      } else if (font!=fontalias) {
        fprintf(fp,"F%s\n",font);
        fontalias=font;
      }
      pt=abs(lFont.lfHeight);
      dir=lFont.lfOrientation;
      if ((pt!=fontpt) || (dir!=fontdir)) {
        fprintf(fp,"H 3 %d %d %d\n",round(p2dx(pt)*72.0/25.4),
                                    round(p2dx(space)*72.0/25.4),
                                    dir*10);
        fontpt=pt;
        fontdir=dir;
      }
      fprintf(fp,"M 2 %d %d\n",p2dx(rdParm[3]),p2dy(rdParm[2]));
      cpx2=rdParm[3];
      cpy2=rdParm[2];
      fprintf(fp,"S");
      s=(char *)(rdParm+1);
      for (i=0;i<rdParm[2];i++) fprintf(fp,"%c",s[i]);
      fprintf(fp,"\n");
    }
    break;
//ExtTextOut
  case 0x0A32:
    if (Col!=TCol) {
      Col=TCol;
      fprintf(fp,"G 3 %d %d %d\n",GetRValue(Col),GetGValue(Col),GetBValue(Col));
    }
    hgdiobj=GetCurrentObject(hdc,OBJ_FONT);
    if (GetObject(hgdiobj,sizeof(LOGFONT),(LPSTR)&lFont)!=0) {
      if (lFont.lfWeight>=550) {
        if (lFont.lfItalic) italic=FONT_BOLDITALIC;
        else italic=FONT_BOLD;
      } else {
        if (lFont.lfItalic) italic=FONT_ITALIC;
        else italic=FONT_ROMAN;
      }
      font=NULL;
      fcur=fontmaproot;
      while (fcur!=NULL) {
        if ((strcmp(fcur->fontname,lFont.lfFaceName)==0)
        && (italic==fcur->italic)) {
          font=fcur->fontalias;
          break;
        }
        fcur=fcur->next;
      }
      if (font==NULL) {
        fprintf(fp,"FHelvetica\n");
      } else if (font!=fontalias) {
        fprintf(fp,"F%s\n",font);
        fontalias=font;
      }
      pt=abs(lFont.lfHeight);
      dir=lFont.lfOrientation;
      if ((pt!=fontpt) || (dir!=fontdir)) {
        fprintf(fp,"H 3 %d %d %d\n",round(p2dx(pt)*72.0/25.4),
                                    round(p2dx(space)*72.0/25.4),
                                    dir*10);
        fontpt=pt;
        fontdir=dir;
      }
      fprintf(fp,"M 2 %d %d\n",p2dx(rdParm[1]),p2dy(rdParm[0]));
      cpx2=rdParm[1];
      cpy2=rdParm[0];
      fprintf(fp,"S");
      if (rdParm[3]==0) s=(char *)(rdParm+4);
      else s=(char *)(rdParm+8);
      count=rdParm[2];
      for (i=0;i<count;i++) fprintf(fp,"%c",s[i]);
      fprintf(fp,"\n");
    }
    break;

//CreateBrushIndirect  0x02fc
//CreateFontIndirect 0x02fb
//CreatePenIndirect 0x02fa
//CreateRegion
//CreatePalette
//CreatePatternBrush 0x0142
//DeleteObject 0x01f0
//SelectObject 0x012d
//SaveDC	0x001E
//RestoreDC	0x0127
//SelectClipRgn
//SelectPalette  0x0234
  case 0x02fc:
  case 0x02fb:
  case 0x02fa:
  case 0x0142:
  case 0x01f0:
  case 0x012d:
  case 0x001e:
  case 0x0127:
  case 0x0234:
    break;
  default:
//SetViewportExt	0x020E
//SetViewportOrg	0x020D
//ScaleViewportExt	0x0412
//OffsetViewportOrg	0x0211
//SetTextJustification	0x020A
//SetStretchBltMode	0x0107
//SetMapMode	0x0103
//SetROP2	0x0104
//RealizePalette (3.0$B0J9_(B)	0x0035
//ResizePalette (3.0$B0J9_(B)	0x0139
//ExcludeClipRect	0x0415
//IntersectClipRect	0x0416
//OffsetClipRgn	0x0220
//SetBkMode	0x0102
//SetMapperFlags	0x0231
//FloodFill	0x0419
//SetTextAlign	0x012E
//SetBkColor	0x0201
//AnimatePalette 0x0463
//BitBlt  0x0940
//Escape
//SetDIBitsToDevice 0x0d33
//SetPaletteEntries
//StretchBlt  0x0b23
//StretchDIBits  0x0f43
    if (debug) fprintf(fp,"%cunsupported: %x\n",'%',rdFunction);
    break;
  }
  PlayMetaFileRecord(hdc,lpht,lpmr,cObj);
  return 1;
}

int CALLBACK EnumMetaFileExtProc(HDC hdc,HANDLETABLE FAR *lpht,METARECORD FAR *lpmr,
                              int cObj,LPARAM lParam)
{
  WORD rdFunction;
  WORD *rdParm;

  rdFunction=lpmr->rdFunction;
  rdParm=lpmr->rdParm;
  switch (rdFunction) {
//SetWindowExt	0x020C
  case 0x020C:
    winextx=rdParm[1];
    winexty=rdParm[0];
    winext=TRUE;
    break;
//SetWindowOrg	0x020B
//  case 0x020B:
//    winorgx=rdParm[1];
//    winorgy=rdParm[0];
//    winorg=TRUE;
//    break;
//SetViewportExt	0x020E
//SetViewportOrg	0x020D
//ScaleViewportExt	0x0412
  case 0x020E:
  case 0x020D:
  case 0x0412:
    view=TRUE;
  default:
    break;
  }
  if ((winext && winorg) || view) return 0;
  return 1;
}


class MainDialog  : public TDialog {
  public:
    MainDialog(TWindow *parent,TResId resID):TDialog(parent,resID)
    {
      Win=new WinStatic(this,500);
    }
    ~MainDialog()
    {
      delete Win;
    }
    void SetupItem();
  protected:
    void BrowseClicked();
    void CloseWindow(int retVal);
    int SaveGRA();
    char FileBuf[512];
    WinStatic *Win;
    DECLARE_RESPONSE_TABLE(MainDialog);
};

DEFINE_RESPONSE_TABLE1(MainDialog,TDialog)
  EV_CHILD_NOTIFY(IDBROWSE,BN_CLICKED,BrowseClicked),
END_RESPONSE_TABLE;

void MainDialog::SetupItem()
{
  TDialog::SetupWindow();
  if (ScriptFile!=NULL) SendDlgItemMessage(IDMERGE,BM_SETCHECK,1,0);
  else SendDlgItemMessage(IDMERGE,BM_SETCHECK,0,0);
}

int MainDialog::SaveGRA()
{
  HANDLE MetaPictHD;
  METAFILEPICT *MetaFilePict;
  HMETAFILE HMF;
  HDC dc;
  FILE *fp;
  double sc;

  if (!IsClipboardFormatAvailable(CF_METAFILEPICT)) {
    printfstderr("No Windows Metafile Picture in Clipboard.");
    return 0;
  }
  ::OpenClipboard(Handle);
  MetaPictHD=GetClipboardData(CF_METAFILEPICT);
  MetaFilePict=(LPMETAFILEPICT)GlobalLock(MetaPictHD);
  HMF=MetaFilePict->hMF;
  dc=Win->GetWinDC();
  EnumMetaFile(dc,HMF,EnumMetaFileExtProc,NULL);
  Win->ReleaseWinDC(dc);
  if ((winextx==0) || (winexty==0)
   || view || (MetaFilePict->mm!=MM_ANISOTROPIC)) {
    printfstderr("This Windows Metafile Picture is NOT scalable.");
    GlobalUnlock(MetaPictHD);
    ::CloseClipboard();
    return 0;
  }
  if ((MetaFilePict->xExt==0) || (MetaFilePict->yExt==0)) {
    winwidth=10000;
    winheight=10000;
  } else if ((MetaFilePict->xExt<0) || (MetaFilePict->yExt<0)) {
    sc=((double )MetaFilePict->xExt)/MetaFilePict->yExt;
    winwidth=10000*sc;
    winheight=10000/sc;
  } else {
    winwidth=MetaFilePict->xExt;
    winheight=MetaFilePict->yExt;
  }
  dot_pixelx=((double )winwidth)/winextx;
  dot_pixely=((double )winheight)/winexty;
  fp=fopen(FileBuf,"wt");
  fprintf(fp,"%cNgraph GRAF\n",'%');
  fprintf(fp,"%cCreator: CLIP2GRA\n",'%');
  fprintf(fp,"I 5 0 0 %d %d 10000\n",winwidth,winheight);
  fprintf(fp,"G 3 0 0 0\n");
  TCol=Col=RGB(0,0,0);
  fprintf(fp,"A 5 0 1 1 1 1000\n");
  Style=PS_SOLID;
  Width=1;
  dc=Win->GetWinDC();
  EnumMetaFile(dc,HMF,EnumMetaFileProc,(LPARAM)fp);
  Win->ReleaseWinDC(dc);
  fprintf(fp,"E 0\n");
  fclose(fp);
  GlobalUnlock(MetaPictHD);
  ::CloseClipboard();
  return 1;
}

void MainDialog::CloseWindow(int retVal)
{
  char *buf,s[20],*endptr;
  int val;
  FILE *fp;


  if (retVal==IDOK) {
    SendDlgItemMessage(IDFILE,WM_GETTEXT,512,(LPARAM)FileBuf);
    if (FileBuf[0]=='\0') return;
    if (access(FileBuf,04)==0) {
      MessageBeep(MB_ICONASTERISK);
      buf=(char *)malloc(strlen(FileBuf)+60);
      if (buf!=NULL) {
        sprintf(buf,
                 "`%s'\n\nFile exists. Can I overwrite?",
                 FileBuf);
        if (MessageBox(buf,"Making GRA file",MB_ICONQUESTION|MB_YESNO)!=IDYES) {
          free(buf);
          return;
        }
        free(buf);
      } else {
        if (MessageBox(
                 "File exists. Can I overwrite?",
        "Making GRA file",MB_ICONQUESTION|MB_YESNO)!=IDYES)
         return;
      }
    }
    if (SaveGRA()) {
      if ((ScriptFile!=NULL) && SendDlgItemMessage(IDMERGE,BM_GETCHECK,0,0)){
        fp=fopen(ScriptFile,"wt");
        fprintf(fp,"new merge\n");
        changefilename(FileBuf);
        fprintf(fp,"merge::file='%s'\n",FileBuf);
        SendDlgItemMessage(IDLEFT,WM_GETTEXT,20,(LPARAM)s);
        if (s[0]!='\0') {
          val=strtol(s,&endptr,10);
          if (endptr[0]=='\0') fprintf(fp,"merge::left_margin=%d\n",val);
        }
        SendDlgItemMessage(IDTOP,WM_GETTEXT,20,(LPARAM)s);
        if (s[0]!='\0') {
          val=strtol(s,&endptr,10);
          if (endptr[0]=='\0') fprintf(fp,"merge::top_margin=%d\n",val);
        }
        SendDlgItemMessage(IDZOOM,WM_GETTEXT,20,(LPARAM)s);
        if (s[0]!='\0') {
          val=strtol(s,&endptr,10);
          if (endptr[0]=='\0') fprintf(fp,"merge::zoom=%d\n",val);
        }
        fclose(fp);
      }
    }
  }
  TDialog::CloseWindow(retVal);
}

void MainDialog::BrowseClicked()
{
  FileBuf[0]='\0';
  if (nGetSaveFileName(Handle,"GRA file","gra",NULL,
      FileBuf,512,"GRA file (*.gra)\0*.gra\0",FALSE)==IDOK) {
    SendDlgItemMessage(IDFILE,WM_SETTEXT,0,(LPARAM)FileBuf);
  }
}

class TMyApp : public TApplication {
  public:
    TMyApp(int argc,char **argv):TApplication() {
      Argc=argc;
      Argv=argv;
    }
  protected:
    void InitMainWindow();
    void InitInstance();
    MainDialog *dlg;
    int Argc;
    char **Argv;
};

void TMyApp::InitInstance()
{
  int i;

  TApplication::InitInstance();
  PumpWaitingMessages();

  for (i=1;(i<Argc) && (Argv[i][0]=='-');i++) {
    switch (Argv[i][1]) {
    case 'd':
      debug=TRUE;
      break;
    default:
      printfstderr("unknown option `%s'.",Argv[i]);
      BreakMessageLoop=TRUE;
      return;
    }
  }
  if (i<Argc) ScriptFile=Argv[i];
  else ScriptFile=NULL;
  dlg->SetupItem();
}

void TMyApp::InitMainWindow()
{
  TFrameWindow *frame;
  dlg=new MainDialog(0,DIALOG_MAIN);
  frame=new TFrameWindow(0, "CLIP2GRA",dlg,TRUE);
  frame->Attr.Style=WS_VISIBLE | WS_SYSMENU | WS_CAPTION | WS_DLGFRAME;
  SetMainWindow(frame);
  GetMainWindow()->SetIcon(this,ICON_1);
}


int OwlMain(int argc,char **argv)
{
  int i;
  char *lib,*home;

  if ((lib=getenv("NGRAPHLIB"))!=NULL) {
    if ((libdir=(char *)malloc(strlen(lib)+1))==NULL) exit(1);
    strcpy(libdir,lib);
  } else {
    for (i=strlen(argv[0]);(argv[0][i]!='\\') && (i>0);i--);
    if ((libdir=(char *)malloc(i+1))==NULL) exit(1);
    strncpy(libdir,argv[0],i);
    libdir[i]='\0';
  }
  if ((home=getenv("NGRAPHHOME"))!=NULL) {
    if ((homedir=(char *)malloc(strlen(home)+1))==NULL) exit(1);
    strcpy(homedir,home);
  } else if ((home=getenv("HOME"))!=NULL) {
    if ((homedir=(char *)malloc(strlen(home)+1))==NULL) exit(1);
    strcpy(homedir,home);
  } else if ((home=getenv("Home"))!=NULL) {
    if ((homedir=(char *)malloc(strlen(home)+1))==NULL) exit(1);
    strcpy(homedir,home);
  } else {
    if ((homedir=(char *)malloc(strlen(libdir)+1))==NULL) exit(1);
    strcpy(homedir,libdir);
  }

  loadconfig();

  hInstance=_hInstance;
  TMyApp(argc,argv).Run();

  free(libdir);
  free(homedir);
  return 0;
}


