/* 
 * 
 * This file is part of "Ngraph for Windows".
 * 
 * Copyright (C) 2015, Satoshi ISHIZAKA. isizaka@msa.biglobe.ne.jp
 * 
 * "Ngraph for Windows" is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * "Ngraph for Windows" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

/*
 *
 * legend.cpp
 *
 */

#include <owl/pch.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <io.h>
#include <dir.h>

HINSTANCE hInstance;

#include "legend.rh"

#define CONF "Ngraph.ini"
#define TEXTCONF "[text]"
#define FONTCONF "[winmenu]"
#define DIRSEP '/'
#define CONFSEP "/"
#define NSTRLEN 256

#define TRUE  1
#define FALSE 0

#define FONT_ROMAN 0
#define FONT_BOLD 1
#define FONT_ITALIC 2
#define FONT_BOLDITALIC 3
#define FONT_UNKNOWN 4

struct filepar;

struct filepar {
  char *file;
  int x,y;
  char *type;
  int mark_type,mark_size,line_width;
  char *line_style;
  int R,G,B,R2,G2,B2;
  char *mathx,*mathy;
  int show;
  char *caption;
  int mix;
  struct filepar *next;
};

struct fontmap;

struct fontmap {
  char *fontalias;
  char *fontname;
  int charset;
  int italic;
  struct fontmap *next;
};

char *ScriptFile=NULL;
char *InputFile=NULL;
char *libdir,*homedir;

struct fontmap *fontmaproot;
int R,G,B,pt,space,script,direction;
char *font,*jfont;
int mix,type,caption,frame;
int posx,posy,posw;

struct filepar *fileparroot;
TColor custColors[16];

char *nstrnew(void)
{
  char *po;

  po=(char *)malloc(NSTRLEN);
  po[0]='\0';
  return po;
}

char *nstrccat(char *po,char ch)
{
  size_t len,num;

  if (po==NULL) return NULL;
  len=strlen(po);
  num=len/NSTRLEN;
  if (len%NSTRLEN==NSTRLEN-1) po=(char *)realloc(po,NSTRLEN*(num+2));
  po[len]=ch;
  po[len+1]='\0';
  return po;
}

char *getbasename(char *name)
{
  char *s;
  int i;

  if (name==NULL) return NULL;
  for (i=strlen(name);(name[i]!='/') && (name[i]!='\\') && (name[i]!=':') && (i!=0);i--);
  if ((name[i]=='/') || (name[i]=='\\') || (name[i]==':')) i++;
  if ((s=(char *)malloc(strlen(name)-i+1))==NULL) return NULL;
  strcpy(s,name+i);
  return s;
}

int fgetline(FILE *fp,char **buf)
{
  char *s;
  char ch;

  *buf=NULL;
  ch=fgetc(fp);
  if (ch==EOF) return 1;
  s=nstrnew();
  while (TRUE) {
    if ((ch=='\0') || (ch=='\n') || (ch==EOF)) {
      *buf=s;
      return 0;
    }
    if (ch!='\r') s=nstrccat(s,ch);
    ch=fgetc(fp);
  }
}

FILE *openconfig(char *section)
{
  char *s,*buf,*homeconf,*libconf;
  FILE *fp;
  struct stat homestat,libstat;

  homeconf=(char *)malloc(strlen(homedir)+strlen(CONFSEP)+strlen(CONF)+1);
  strcpy(homeconf,homedir);
  if ((strlen(homedir)>0) && (homeconf[strlen(homedir)-1]=='/'))
    homeconf[strlen(homedir)-1]='\0';
  strcat(homeconf,CONFSEP);
  strcat(homeconf,CONF);
  libconf=(char *)malloc(strlen(libdir)+strlen(CONFSEP)+strlen(CONF)+1);
  strcpy(libconf,libdir);
  if ((strlen(libdir)>0) && (libconf[strlen(libdir)-1]=='/'))
    libconf[strlen(libdir)-1]='\0';
  strcat(libconf,CONFSEP);
  strcat(libconf,CONF);
  if (access(homeconf,04)==0) {
    if (stat(homeconf,&homestat)!=0) {
      free(homeconf);
      homeconf=NULL;
    }
  } else {
    free(homeconf);
    homeconf=NULL;
  }
  if (access(libconf,04)==0) {
    if (stat(libconf,&libstat)!=0) {
      free(libconf);
      libconf=NULL;
    }
  } else {
    free(libconf);
    libconf=NULL;
  }
  if (homeconf!=NULL) {
    if (libconf==NULL) s=homeconf;
    else if (homestat.st_mtime>=libstat.st_mtime) {
      s=homeconf;
      free(libconf);
    } else {
      s=libconf;
      free(homeconf);
    }
  } else if (libconf!=NULL) s=libconf;
  else return NULL;
  if ((fp=fopen(s,"rt"))==NULL) return NULL;
  free(s);
  while (fgetline(fp,&buf)==0) {
    if (strcmp(buf,section)==0) {
      free(buf);
      return fp;
    }
    free(buf);
  }
  fclose(fp);
  return NULL;
}

char *getitok(char **s,int *len,char *ifs)
{
  char *po,*spo;
  int i;

  if (*s==NULL) return NULL;
  po=*s;
  for (i=0;(po[i]!='\0') && (strchr(ifs,po[i])!=NULL);i++);
  if (po[i]=='\0') {
    *len=0;
    return NULL;
  }
  spo=po+i;
  for (;(po[i]!='\0') && (strchr(ifs,po[i])==NULL);i++);
  *s+=i;
  *len=*s-spo;
  return spo;
}

char *getitok2(char **s,int *len,char *ifs)
{
  char *po,*s2;

  if ((s2=getitok(s,len,ifs))==NULL) return NULL;
  po=(char *)malloc(*len+1);
  strncpy(po,s2,*len);
  po[*len]='\0';
  return po;
}

char *getconfig(FILE *fp,char **val)
{
  char *s,*tok,*buf;
  int len;

  while (TRUE) {
    if (fgetline(fp,&buf)!=0) return NULL;
    else {
      if (buf[0]=='[') {
        free(buf);
        return NULL;
      } else {
        s=buf;
        if ((tok=getitok2(&s,&len," \x09=,"))!=NULL) {
          for (;(s[0]!='\0') && (strchr(" \x09=,",s[0])!=NULL);s++);
          *val=(char *)malloc(strlen(s)+1);
          strcpy(*val,s);
          free(buf);
          return tok;
        }
        free(buf);
        free(tok);
      }
    }
  }
}

void closeconfig(FILE *fp)
{
  fclose(fp);
}

int fontloadconfig()
{
  FILE *fp;
  char *tok,*str,*s2;
  char *f1,*f2,*f3,*f4;
  struct fontmap *fcur,*fnew;
  int len;

  if ((fp=openconfig(FONTCONF))==NULL) return 0;
  fcur=fontmaproot;
  while ((tok=getconfig(fp,&str))!=NULL) {
    s2=str;
    if (strcmp(tok,"font_map")==0) {
      f1=getitok2(&s2,&len," \x09,");
      f3=getitok2(&s2,&len," \x09,");
      f4=getitok2(&s2,&len," \x09,");
      for (;(s2[0]!='\0') && (strchr(" \x09,",s2[0])!=NULL);s2++);
      f2=getitok2(&s2,&len,"");
      if ((f1!=NULL) && (f2!=NULL) && (f3!=NULL) && (f4!=NULL)) {
        if ((fnew=(fontmap *)malloc(sizeof(struct fontmap)))==NULL) {
          free(tok);
          free(f1);
          free(f2);
          free(f3);
          free(f4);
          closeconfig(fp);
          return 1;
        }
        if (fcur==NULL) fontmaproot=fnew;
        else fcur->next=fnew;
        fcur=fnew;
        fcur->next=NULL;
        fcur->fontalias=f1;
        fcur->fontname=f2;
        if (strcmp(f4,"roman")==0) fcur->italic=FONT_ROMAN;
        else if (strcmp(f4,"italic")==0) fcur->italic=FONT_ITALIC;
        else if (strcmp(f4,"bold")==0) fcur->italic=FONT_BOLD;
        else if (strcmp(f4,"bold_italic")==0) fcur->italic=FONT_BOLDITALIC;
        else fcur->italic=FONT_UNKNOWN;
        if (strcmp(f3,"shiftjis")==0) fcur->charset=SHIFTJIS_CHARSET;
        else if (strcmp(f3,"symbol")==0) fcur->charset=SYMBOL_CHARSET;
        else if (strcmp(f3,"ansi")==0) fcur->charset=ANSI_CHARSET;
        else if (strcmp(f3,"oem")==0) fcur->charset=OEM_CHARSET;
        else if (strcmp(f3,"hangeul")==0) fcur->charset=HANGEUL_CHARSET;
        else if (strcmp(f3,"chinesebig5")==0) fcur->charset=CHINESEBIG5_CHARSET;
        else fcur->charset=DEFAULT_CHARSET;
        free(f3);
        free(f4);
      } else {
        free(f1);
        free(f2);
        free(f3);
        free(f4);
      }
    }
    free(tok);
    free(str);
  }
  closeconfig(fp);
  return 0;
}

int textloadconfig()
{
  FILE *fp;
  char *tok,*str,*s2;
  char *f1;
  int val;
  char *endptr;
  int len;

  if ((fp=openconfig(TEXTCONF))==NULL) return 0;
  while ((tok=getconfig(fp,&str))!=NULL) {
    s2=str;
    if (strcmp(tok,"R")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      R=val;
      free(f1);
    } else if (strcmp(tok,"G")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      G=val;
      free(f1);
    } else if (strcmp(tok,"B")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      B=val;
      free(f1);
    } else if (strcmp(tok,"pt")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      pt=val;
      free(f1);
    } else if (strcmp(tok,"space")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      space=val;
      free(f1);
    } else if (strcmp(tok,"direction")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      direction=val;
      free(f1);
    } else if (strcmp(tok,"script_size")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      script=val;
      free(f1);
    } else if (strcmp(tok,"font")==0) {
      f1=getitok2(&s2,&len,"");
      free(font);
      font=f1;
    } else if (strcmp(tok,"jfont")==0) {
      f1=getitok2(&s2,&len,"");
      free(jfont);
      jfont=f1;
    }
    free(tok);
    free(str);
  }
  closeconfig(fp);
  return 0;
}

int sscanf2(char *buffer,char *format,...)
{
  va_list ap;
  int i,num;
  int *d;
  double *e;
  char *s;
  char *endptr;

  va_start(ap,format);
  s=buffer;
  num=0;
  i=0;
  while (format[i]!='\0') {
    if (format[i]=='d') {
      d=va_arg(ap,int *);
      *d=strtol(s,&endptr,10);
      num++;
      if (endptr[0]=='\0') break;
      s=endptr;
    } else if (format[i]=='e') {
      e=va_arg(ap,double *);
      *e=strtod(s,&endptr);
      num++;
      if (endptr[0]=='\0') break;
      s=endptr;
    }
    i++;
  }
  va_end(ap);
  return num;
}

int printfstderr(char *fmt,...)
{
  int len;
  char buf[1024];
  va_list ap;

  va_start(ap,fmt);
  len=vsprintf(buf,fmt,ap);
  va_end(ap);
  MessageBeep(MB_ICONASTERISK);
  MessageBox(NULL,buf,NULL,MB_OK|MB_TASKMODAL|MB_ICONEXCLAMATION|MB_SETFOREGROUND);
  return len;
}

class ColorButton : public TButton {
  public:
    ColorButton(TWindow *parent,int resId) : TButton(parent,resId) {}
    void SetColor(int r,int g,int b);
    void GetColor(int *r,int *g,int *b);
  protected:
    int R,G,B;
    void EvDrawItem(UINT ctrlId,DRAWITEMSTRUCT far& draw);
    void BNClicked();
    DECLARE_RESPONSE_TABLE(ColorButton);
};

DEFINE_RESPONSE_TABLE1(ColorButton,TButton)
  EV_NOTIFY_AT_CHILD(BN_CLICKED,BNClicked),
  EV_WM_DRAWITEM,
END_RESPONSE_TABLE;

void ColorButton::EvDrawItem(UINT ctrlId,DRAWITEMSTRUCT far& draw)
{
  int i,shift;
  HBRUSH brush;
  RECT rect;

  TButton::EvDrawItem(ctrlId,draw);
  brush=CreateSolidBrush(GetSysColor(COLOR_MENU));
  rect.left=draw.rcItem.left;
  rect.top=draw.rcItem.top;
  rect.right=draw.rcItem.right;
  rect.bottom=draw.rcItem.bottom;
  FillRect(draw.hDC,&rect,brush);
  DeleteObject(brush);
  if (draw.itemState & ODS_SELECTED) {
    brush=CreateSolidBrush(GetSysColor(COLOR_3DSHADOW));
    shift=1;
  } else {
    brush=CreateSolidBrush(GetSysColor(COLOR_3DHILIGHT));
    shift=0;
  }
  rect.left=draw.rcItem.left+1;
  rect.top=draw.rcItem.top+1;
  rect.right=draw.rcItem.left+2;
  rect.bottom=draw.rcItem.bottom;
  FillRect(draw.hDC,&rect,brush);
  rect.left=draw.rcItem.left+1;
  rect.top=draw.rcItem.top+1;
  rect.right=draw.rcItem.right;
  rect.bottom=draw.rcItem.top+2;
  FillRect(draw.hDC,&rect,brush);
  DeleteObject(brush);

  if (draw.itemState & ODS_DISABLED) {
    brush=CreateSolidBrush(GetSysColor(COLOR_GRAYTEXT));
  } else {
    brush=CreateSolidBrush(GetSysColor(COLOR_WINDOWTEXT));
  }
  rect.left=draw.rcItem.left+4+shift;
  rect.top=draw.rcItem.top+4+shift;
  rect.right=draw.rcItem.right-4+shift;
  rect.bottom=draw.rcItem.bottom-4+shift;
  FillRect(draw.hDC,&rect,brush);
  DeleteObject(brush);
  brush=CreateSolidBrush(RGB(R,G,B));
  rect.left=draw.rcItem.left+6+shift;
  rect.top=draw.rcItem.top+6+shift;
  rect.right=draw.rcItem.right-6+shift;
  rect.bottom=draw.rcItem.bottom-6+shift;
  FillRect(draw.hDC,&rect,brush);
  DeleteObject(brush);
  brush=CreateSolidBrush(GetSysColor(COLOR_3DSHADOW));
  for (i=1;i<3-shift;i++) {
    rect.left=draw.rcItem.right-i;
    rect.top=draw.rcItem.top+i;
    rect.right=draw.rcItem.right-i+1;
    rect.bottom=draw.rcItem.bottom;
    FillRect(draw.hDC,&rect,brush);
  }
  for (i=1;i<3-shift;i++) {
    rect.left=draw.rcItem.left+i;
    rect.top=draw.rcItem.bottom-i;
    rect.right=draw.rcItem.right;
    rect.bottom=draw.rcItem.bottom-i+1;
    FillRect(draw.hDC,&rect,brush);
  }
  DeleteObject(brush);
}

void ColorButton::BNClicked()
{
  TColor Color(RGB(R,G,B));

  TChooseColorDialog::TData choose;
  choose.Flags=CC_RGBINIT;
  choose.Color=Color;
  choose.CustColors=custColors;
  if (TChooseColorDialog(this,choose).Execute()==IDOK) {
    R=choose.Color.Red();
    G=choose.Color.Green();
    B=choose.Color.Blue();
  }
}

void ColorButton::SetColor(int r,int g,int b)
{
  R=r;
  G=g;
  B=b;
}

void ColorButton::GetColor(int *r,int *g,int *b)
{
  *r=R;
  *g=G;
  *b=B;
}


class FontDialog  : public TDialog {
  public:
    FontDialog(TWindow *parent,TResId resID);
    ~FontDialog();
  protected:
    ColorButton *Col;
    void SetupWindow();
    void CloseWindow(int retVal);
};

FontDialog::FontDialog(TWindow *parent,TResId resID):TDialog(parent,resID)
{
  Col=new ColorButton(this,IDC_COL);
}

FontDialog::~FontDialog()
{
  delete Col;
}

void FontDialog::SetupWindow()
{
  char buf[256];
  int j,selfont;
  struct fontmap *fcur;

  TDialog::SetupWindow();
  SendDlgItemMessage(IDC_PT,CB_RESETCONTENT,0,0);
  SendDlgItemMessage(IDC_PT,CB_INSERTSTRING,-1,(LPARAM)"1200");
  SendDlgItemMessage(IDC_PT,CB_INSERTSTRING,-1,(LPARAM)"1600");
  SendDlgItemMessage(IDC_PT,CB_INSERTSTRING,-1,(LPARAM)"1800");
  SendDlgItemMessage(IDC_PT,CB_INSERTSTRING,-1,(LPARAM)"2000");
  SendDlgItemMessage(IDC_PT,CB_INSERTSTRING,-1,(LPARAM)"2400");
  SendDlgItemMessage(IDC_PT,CB_INSERTSTRING,-1,(LPARAM)"3000");
  sprintf(buf,"%d",pt);
  SetDlgItemText(IDC_PT,buf);
  sprintf(buf,"%d",space);
  SetDlgItemText(IDC_SPACE,buf);
  sprintf(buf,"%d",script);
  SetDlgItemText(IDC_SCRIPT,buf);
  SendDlgItemMessage(IDC_FONT,CB_RESETCONTENT,0,0);
  fcur=fontmaproot;
  j=0;
  selfont=-1;
  while (fcur!=NULL) {
    if (fcur->charset!=SHIFTJIS_CHARSET) {
      SendDlgItemMessage(IDC_FONT,CB_INSERTSTRING,-1,(LPARAM)fcur->fontalias);
      if ((font!=NULL) && (strcmp(font,fcur->fontalias)==0)) selfont=j;
      j++;
    }
    fcur=fcur->next;
  }
  if (selfont!=-1) SendDlgItemMessage(IDC_FONT,CB_SETCURSEL,selfont,0);
  SendDlgItemMessage(IDC_JFONT,CB_RESETCONTENT,0,0);
  fcur=fontmaproot;
  j=0;
  selfont=-1;
  while (fcur!=NULL) {
    if (fcur->charset==SHIFTJIS_CHARSET) {
      SendDlgItemMessage(IDC_JFONT,CB_INSERTSTRING,-1,(LPARAM)fcur->fontalias);
      if ((jfont!=NULL) && (strcmp(jfont,fcur->fontalias)==0)) selfont=j;
      j++;
    }
    fcur=fcur->next;
  }
  if (selfont!=-1) SendDlgItemMessage(IDC_JFONT,CB_SETCURSEL,selfont,0);
  Col->SetColor(R,G,B);
}

void FontDialog::CloseWindow(int retVal)
{
  char *buf,*endptr;
  int val;
  int len;

  if (retVal==IDOK) {
    len=SendDlgItemMessage(IDC_PT,WM_GETTEXTLENGTH,0,0);
    if ((buf=(char *)malloc(len+1))!=NULL) {
      GetDlgItemText(IDC_PT,buf,len+1);
      val=strtol(buf,&endptr,10);
      if (endptr[0]=='\0') pt=val;
      free(buf);
    }
    len=SendDlgItemMessage(IDC_SPACE,WM_GETTEXTLENGTH,0,0);
    if ((buf=(char *)malloc(len+1))!=NULL) {
      GetDlgItemText(IDC_SPACE,buf,len+1);
      val=strtol(buf,&endptr,10);
      if (endptr[0]=='\0') space=val;
      free(buf);
    }
    len=SendDlgItemMessage(IDC_SCRIPT,WM_GETTEXTLENGTH,0,0);
    if ((buf=(char *)malloc(len+1))!=NULL) {
      GetDlgItemText(IDC_SCRIPT,buf,len+1);
      val=strtol(buf,&endptr,10);
      if (endptr[0]=='\0') script=val;
      free(buf);
    }
    len=SendDlgItemMessage(IDC_FONT,WM_GETTEXTLENGTH,0,0);
    if ((buf=(char *)malloc(len+1))!=NULL) {
      GetDlgItemText(IDC_FONT,buf,len+1);
      free(font);
      font=buf;
    }
    len=SendDlgItemMessage(IDC_JFONT,WM_GETTEXTLENGTH,0,0);
    if ((buf=(char *)malloc(len+1))!=NULL) {
      GetDlgItemText(IDC_JFONT,buf,len+1);
      free(jfont);
      jfont=buf;
    }
  }
  Col->GetColor(&R,&G,&B);
  TDialog::CloseWindow(retVal);
}

class MainDialog  : public TDialog {
  public:
    MainDialog(TWindow *parent,TResId resID):TDialog(parent,resID) {}
    void SetupItem();
    void SetupItem2();
  protected:
    void SetupWindow();
    int Sel;
    void MakeScript();
    void MakeScript2(FILE *fp,struct filepar *fcur,int gx,int gy,int height);
    void FontClicked();
    void MixClicked();
    void SelChange();
    void CloseWindow(int retVal);
    DECLARE_RESPONSE_TABLE(MainDialog);
};

DEFINE_RESPONSE_TABLE1(MainDialog,TDialog)
  EV_CHILD_NOTIFY(IDC_FONTB,BN_CLICKED,FontClicked),
  EV_CHILD_NOTIFY(IDC_MIX,BN_CLICKED,MixClicked),
  EV_CHILD_NOTIFY(IDC_FILE,LBN_SELCHANGE,SelChange),
END_RESPONSE_TABLE;

void MainDialog::SetupWindow()
{
  TDialog::SetupWindow();
  SetupItem2();
}

void MainDialog::SetupItem()
{
  char buf[256],*s;
  struct filepar *fcur;

  SendDlgItemMessage(IDC_FILE,LB_RESETCONTENT,0,0);
  fcur=fileparroot;
  while (fcur!=NULL) {
    if ((fcur->mix==-1) || (!mix)) {
      s=getbasename(fcur->file);
      sprintf(buf,"%2d %2d %.128s",fcur->x,fcur->y,s);
      free(s);
      SendDlgItemMessage(IDC_FILE,LB_INSERTSTRING,-1,(LPARAM)buf);
    }
    fcur=fcur->next;
  }
  Sel=-1;
}

void MainDialog::SetupItem2()
{
  char buf[256];

  if (mix) SendDlgItemMessage(IDC_MIX,BM_SETCHECK,1,0);
  else SendDlgItemMessage(IDC_MIX,BM_SETCHECK,0,0);
  if (type) SendDlgItemMessage(IDC_TYPE,BM_SETCHECK,1,0);
  else SendDlgItemMessage(IDC_TYPE,BM_SETCHECK,0,0);
  if (caption) SendDlgItemMessage(IDC_CAPTION,BM_SETCHECK,1,0);
  else SendDlgItemMessage(IDC_CAPTION,BM_SETCHECK,0,0);
  if (frame) SendDlgItemMessage(IDC_FRAME,BM_SETCHECK,1,0);
  else SendDlgItemMessage(IDC_FRAME,BM_SETCHECK,0,0);
  sprintf(buf,"%d",posx);
  SetDlgItemText(IDC_PX,buf);
  sprintf(buf,"%d",posy);
  SetDlgItemText(IDC_PY,buf);
  sprintf(buf,"%d",posw);
  SetDlgItemText(IDC_PWIDTH,buf);
}

void MainDialog::CloseWindow(int retVal)
{
  char *buf,*endptr;
  int val;
  int len;

  if (retVal==IDOK) {
    mix=SendDlgItemMessage(IDC_MIX,BM_GETCHECK,0,0);
    type=SendDlgItemMessage(IDC_TYPE,BM_GETCHECK,0,0);
    caption=SendDlgItemMessage(IDC_CAPTION,BM_GETCHECK,0,0);
    frame=SendDlgItemMessage(IDC_FRAME,BM_GETCHECK,0,0);
    len=SendDlgItemMessage(IDC_PX,WM_GETTEXTLENGTH,0,0);
    if ((buf=(char *)malloc(len+1))!=NULL) {
      GetDlgItemText(IDC_PX,buf,len+1);
      val=strtol(buf,&endptr,10);
      if (endptr[0]=='\0') posx=val;
      free(buf);
    }
    len=SendDlgItemMessage(IDC_PY,WM_GETTEXTLENGTH,0,0);
    if ((buf=(char *)malloc(len+1))!=NULL) {
      GetDlgItemText(IDC_PY,buf,len+1);
      val=strtol(buf,&endptr,10);
      if (endptr[0]=='\0') posy=val;
      free(buf);
    }
    len=SendDlgItemMessage(IDC_PWIDTH,WM_GETTEXTLENGTH,0,0);
    if ((buf=(char *)malloc(len+1))!=NULL) {
      GetDlgItemText(IDC_PWIDTH,buf,len+1);
      val=strtol(buf,&endptr,10);
      if (endptr[0]=='\0') posw=val;
      free(buf);
    }
    SelChange();
    MakeScript();
  }
  TDialog::CloseWindow(retVal);
}

void MainDialog::SelChange()
{
  struct filepar *fcur;
  int i,len;
  char *buf;

  if (Sel!=-1) {
    i=-1;
    fcur=fileparroot;
    while (fcur!=NULL) {
      if ((fcur->mix==-1) || (!mix)) i++;
      if (i==Sel) break;
      fcur=fcur->next;
    }
    if ((i==Sel) && (fcur!=NULL)) {
      fcur->show=SendDlgItemMessage(IDC_SHOW,BM_GETCHECK,0,0);
      len=SendDlgItemMessage(IDC_CAPTIONTEXT,WM_GETTEXTLENGTH,0,0);
      if ((buf=(char *)malloc(len+1))!=NULL) {
        GetDlgItemText(IDC_CAPTIONTEXT,buf,len+1);
        free(fcur->caption);
        fcur->caption=buf;
      }
    }
  }
  Sel=SendDlgItemMessage(IDC_FILE,LB_GETCURSEL,0,0);
  i=-1;
  fcur=fileparroot;
  while (fcur!=NULL) {
    if ((fcur->mix==-1) || (!mix)) i++;
    if (i==Sel) break;
    fcur=fcur->next;
  }
  if ((i==Sel) && (fcur!=NULL)) {
    if (fcur->show) SendDlgItemMessage(IDC_SHOW,BM_SETCHECK,1,0);
    else SendDlgItemMessage(IDC_SHOW,BM_SETCHECK,0,0);
    if ((fcur->caption==NULL) && (fcur->file!=NULL))
      SetDlgItemText(IDC_CAPTIONTEXT,fcur->file);
    else
      SetDlgItemText(IDC_CAPTIONTEXT,fcur->caption);
  }
}

void MainDialog::MixClicked()
{
  SelChange();
  mix=SendDlgItemMessage(IDC_MIX,BM_GETCHECK,0,0);
  SetupItem();
  SelChange();
}

void MainDialog::MakeScript2(FILE *fp,struct filepar *fcur,int gx,int gy,int height)
{
  if (fcur->type==NULL) return;
  if ((strcmp(fcur->type,"line")==0)
   || (strcmp(fcur->type,"polygon")==0)
   || (strcmp(fcur->type,"curve")==0)
   || (strcmp(fcur->type,"diagonal")==0)
   || (strcmp(fcur->type,"errorbar_x")==0)
   || (strcmp(fcur->type,"errorbar_y")==0)
   || (strcmp(fcur->type,"staircase_x")==0)
   || (strcmp(fcur->type,"staircase_y")==0)
   || (strcmp(fcur->type,"fit")==0)) {
    fprintf(fp,"new line\n");
    fprintf(fp,"line::points='%d %d %d %d'\n",gx,gy+height*2/3,gx+posw,gy+height*2/3);
    fprintf(fp,"line::width=%d\n",fcur->line_width);
    if (fcur->line_style!=NULL)
      fprintf(fp,"line::style='%s'\n",fcur->line_style);
    fprintf(fp,"line::R=%d\n",fcur->R);
    fprintf(fp,"line::G=%d\n",fcur->G);
    fprintf(fp,"line::B=%d\n",fcur->B);
  } else if (strcmp(fcur->type,"arrow")==0) {
    fprintf(fp,"new line\n");
    fprintf(fp,"line::points='%d %d %d %d'\n",gx,gy+height*2/3,gx+posw,gy+height*2/3);
    fprintf(fp,"line::width=%d\n",fcur->line_width);
    if (fcur->line_style!=NULL)
      fprintf(fp,"line::style='%s'\n",fcur->line_style);
    fprintf(fp,"line::R=%d\n",fcur->R);
    fprintf(fp,"line::G=%d\n",fcur->G);
    fprintf(fp,"line::B=%d\n",fcur->B);
    fprintf(fp,"line::arrow=end\n");
  } else if ((strcmp(fcur->type,"rectangle")==0)
   || (strcmp(fcur->type,"rectangle_fill")==0)
   || (strcmp(fcur->type,"rectangle_solid_fill")==0)
   || (strcmp(fcur->type,"bar_x")==0)
   || (strcmp(fcur->type,"bar_y")==0)
   || (strcmp(fcur->type,"bar_fill_x")==0)
   || (strcmp(fcur->type,"bar_fill_y")==0)
   || (strcmp(fcur->type,"bar_solid_fill_x")==0)
   || (strcmp(fcur->type,"bar_solid_fill_y")==0)) {
    fprintf(fp,"new rectangle\n");
    fprintf(fp,"rectangle::x1=%d\n",gx);
    fprintf(fp,"rectangle::y1=%d\n",gy+height*2/3-height/2);
    fprintf(fp,"rectangle::x2=%d\n",gx+posw);
    fprintf(fp,"rectangle::y2=%d\n",gy+height*2/3+height/2);
    if ((strcmp(fcur->type,"rectangle")==0)
     || (strcmp(fcur->type,"bar_x")==0)
     || (strcmp(fcur->type,"bar_y")==0)) {
      fprintf(fp,"rectangle::fill=false\n");
      fprintf(fp,"rectangle::R=%d\n",fcur->R);
      fprintf(fp,"rectangle::G=%d\n",fcur->G);
      fprintf(fp,"rectangle::B=%d\n",fcur->B);
    } else if ((strcmp(fcur->type,"rectangle_fill")==0)
     || (strcmp(fcur->type,"bar_fill_x")==0)
     || (strcmp(fcur->type,"bar_fill_y")==0)) {
      fprintf(fp,"rectangle::fill=true\n");
      fprintf(fp,"rectangle::frame=true\n");
      fprintf(fp,"rectangle::R=%d\n",fcur->R2);
      fprintf(fp,"rectangle::G=%d\n",fcur->G2);
      fprintf(fp,"rectangle::B=%d\n",fcur->B2);
      fprintf(fp,"rectangle::R2=%d\n",fcur->R);
      fprintf(fp,"rectangle::G2=%d\n",fcur->G);
      fprintf(fp,"rectangle::B2=%d\n",fcur->B);
    } else if ((strcmp(fcur->type,"rectangle_solid_fill")==0)
     || (strcmp(fcur->type,"bar_solid_fill_x")==0)
     || (strcmp(fcur->type,"bar_solid_fill_y")==0)) {
      fprintf(fp,"rectangle::fill=true\n");
      fprintf(fp,"rectangle::R=%d\n",fcur->R);
      fprintf(fp,"rectangle::G=%d\n",fcur->G);
      fprintf(fp,"rectangle::B=%d\n",fcur->B);
    }
    fprintf(fp,"rectangle::width=%d\n",fcur->line_width);
    if (fcur->line_style!=NULL)
      fprintf(fp,"rectangle::style='%s'\n",fcur->line_style);
  } else if (strcmp(fcur->type,"mark")==0) {
    fprintf(fp,"new mark\n");
    fprintf(fp,"mark::x=%d\n",gx+posw/2);
    fprintf(fp,"mark::y=%d\n",gy+height*2/3);
    fprintf(fp,"mark::size=%d\n",fcur->mark_size);
    fprintf(fp,"mark::type=%d\n",fcur->mark_type);
    fprintf(fp,"mark::width=%d\n",fcur->line_width);
    if (fcur->line_style!=NULL)
      fprintf(fp,"mark::style='%s'\n",fcur->line_style);
    fprintf(fp,"mark::R=%d\n",fcur->R);
    fprintf(fp,"mark::G=%d\n",fcur->G);
    fprintf(fp,"mark::B=%d\n",fcur->B);
    fprintf(fp,"mark::R2=%d\n",fcur->R2);
    fprintf(fp,"mark::G2=%d\n",fcur->G2);
    fprintf(fp,"mark::B2=%d\n",fcur->B2);
  }
}

void MainDialog::MakeScript()
{
  FILE *fp;
  struct filepar *fcur,*fcur2;
  int i;
  int gx,gy,height,len;
  char *s;

  if (ScriptFile==NULL) return;
  fp=fopen(ScriptFile,"wt");
  i=0;
  fcur=fileparroot;
  height=(int )(pt*25.4/72.0);
  len=0;
  gy=posy;
  if (frame) {
    fprintf(fp,"new int name:textlen\n");
    fprintf(fp,"new int name:texttot\n");
    fprintf(fp,"new iarray name:textbbox\n");
  }
  while (fcur!=NULL) {
    if (((fcur->mix==-1) || (!mix)) && (fcur->show)) {
      gx=posx;
      if (type) {
        MakeScript2(fp,fcur,gx,gy,height);
        fcur2=fcur->next;
        while (fcur2!=NULL) {
          if ((fcur2->mix==i) && (mix)) MakeScript2(fp,fcur2,gx,gy,height);
          fcur2=fcur2->next;
        }
        len=posw+height/2;
      }
      if (caption) {
        fprintf(fp,"new text\n");
        if ((fcur->caption==NULL) && (fcur->file!=NULL)) {
          s=getbasename(fcur->file);
          fprintf(fp,"text::text='%s'\n",s);
          free(s);
          fprintf(fp,"text::raw=true\n");
        } else if (fcur->caption!=NULL)
          fprintf(fp,"text::text='%s'\n",fcur->caption);
        fprintf(fp,"text::x=%d\n",gx+len);
        fprintf(fp,"text::y=%d\n",gy+height);
        fprintf(fp,"text::pt=%d\n",pt);
        if (font!=NULL) fprintf(fp,"text::font='%s'\n",font);
        if (jfont!=NULL) fprintf(fp,"text::jfont='%s'\n",jfont);
        fprintf(fp,"text::space=%d\n",space);
        fprintf(fp,"text::script_size=%d\n",script);
        fprintf(fp,"text::R=%d\n",R);
        fprintf(fp,"text::G=%d\n",G);
        fprintf(fp,"text::B=%d\n",B);
        if (frame) {
          fprintf(fp,"iarray:textbbox:@=${text::bbox}\n");
          fprintf(fp,"int:textlen:@=\"${iarray:textbbox:get:2}-${iarray:textbbox:get:0}\"\n");
          fprintf(fp,"if [ \"${int:texttot:@}\" -lt \"${int:textlen:@}\" ]; then\n");
          fprintf(fp,"int:texttot:@=${int:textlen:@}\n");
          fprintf(fp,"fi\n");
        }
      }
      gy+=height*1.2;
    }
    i++;
    fcur=fcur->next;
  }
  if (frame) {
    fprintf(fp,"new rectangle\n");
    fprintf(fp,"rectangle::x1=%d\n",posx-height/4);
    fprintf(fp,"rectangle::y1=%d\n",posy);
    fprintf(fp,"rectangle::x2=%d+${int:texttot:@}\n",posx+len+3*height/4);
    fprintf(fp,"rectangle::y2=%d\n",gy+height/2);
    fprintf(fp,"rectangle::R=0\n");
    fprintf(fp,"rectangle::G=0\n");
    fprintf(fp,"rectangle::B=0\n");
    fprintf(fp,"rectangle::fill=true\n");
    fprintf(fp,"new rectangle\n");
    fprintf(fp,"rectangle::x1=%d\n",posx-height/2);
    fprintf(fp,"rectangle::y1=%d\n",posy-height/4);
    fprintf(fp,"rectangle::x2=%d+${int:texttot:@}\n",posx+len+height/2);
    fprintf(fp,"rectangle::y2=%d\n",gy+height/4);
    fprintf(fp,"rectangle::R=255\n");
    fprintf(fp,"rectangle::G=255\n");
    fprintf(fp,"rectangle::B=255\n");
    fprintf(fp,"rectangle::R2=0\n");
    fprintf(fp,"rectangle::G2=0\n");
    fprintf(fp,"rectangle::B2=0\n");
    fprintf(fp,"rectangle::fill=true\n");
    fprintf(fp,"rectangle::frame=true\n");
    fprintf(fp,"del int:textlen\n");
    fprintf(fp,"del int:texttot\n");
    fprintf(fp,"del iarray:textbbox\n");
  }
  fclose(fp);
}

void MainDialog::FontClicked()
{
  FontDialog *dlg;

  dlg=new FontDialog(this,DIALOG_FONT);
  dlg->Execute();
  delete dlg;
}

class TMyApp : public TApplication {
  public:
    TMyApp(int argc,char **argv):TApplication() {
      Argc=argc;
      Argv=argv;
    }
  protected:
    void InitMainWindow();
    void InitInstance();
    int LoadFilePar();
    MainDialog *dlg;
    int Argc;
    char **Argv;
    char *filename;
};

int TMyApp::LoadFilePar()
{
  FILE *fp;
  char *s,*endptr;
  int i,j,num;
  struct filepar *filepar,*fcur,*fcur2;

  if (InputFile==NULL) return FALSE;
  if ((fp=fopen(InputFile,"rt"))==NULL) {
    printfstderr("file open `%s'.",InputFile);
    return FALSE;
  }
  if (fgetline(fp,&s)!=0) {
    printfstderr("illegal parameter in `%s'.",InputFile);
    fclose(fp);
    return FALSE;
  }
  num=strtol(s,&endptr,10);
  if (endptr[0]!='\0') {
    printfstderr("illegal parameter in `%s'.",InputFile);
    fclose(fp);
    return FALSE;
  }
  free(s);
  fcur=fileparroot;
  for (i=0;i<num;i++) {
    if ((filepar=(struct filepar *)malloc(sizeof(struct filepar)))!=NULL) {
      filepar->next=NULL;
      if (fgetline(fp,&s)==0) filepar->file=s;
      else filepar->file=NULL;
      if (fgetline(fp,&s)==0) filepar->x=strtol(s,&endptr,10);
      else filepar->x=1;
      free(s);
      if (fgetline(fp,&s)==0) filepar->y=strtol(s,&endptr,10);
      else filepar->y=2;
      free(s);
      if (fgetline(fp,&s)==0) filepar->type=s;
      else filepar->type=NULL;
      if (fgetline(fp,&s)==0) filepar->mark_type=strtol(s,&endptr,10);
      else filepar->mark_type=-1;
      free(s);
      if (fgetline(fp,&s)==0) filepar->mark_size=strtol(s,&endptr,10);
      else filepar->mark_size=0;
      free(s);
      if (fgetline(fp,&s)==0) filepar->line_width=strtol(s,&endptr,10);
      else filepar->line_width=0;
      free(s);
      if (fgetline(fp,&s)==0) filepar->line_style=s;
      else filepar->line_style=NULL;
      if (fgetline(fp,&s)==0) filepar->R=strtol(s,&endptr,10);
      else filepar->R=0;
      free(s);
      if (fgetline(fp,&s)==0) filepar->G=strtol(s,&endptr,10);
      else filepar->G=0;
      free(s);
      if (fgetline(fp,&s)==0) filepar->B=strtol(s,&endptr,10);
      else filepar->B=0;
      free(s);
      if (fgetline(fp,&s)==0) filepar->R2=strtol(s,&endptr,10);
      else filepar->R2=0;
      free(s);
      if (fgetline(fp,&s)==0) filepar->G2=strtol(s,&endptr,10);
      else filepar->G2=0;
      free(s);
      if (fgetline(fp,&s)==0) filepar->B2=strtol(s,&endptr,10);
      else filepar->B2=0;
      free(s);
      if (fgetline(fp,&s)==0) filepar->mathx=s;
      else filepar->mathx=NULL;
      if (fgetline(fp,&s)==0) filepar->mathy=s;
      else filepar->mathy=NULL;
      filepar->show=TRUE;
      filepar->caption=NULL;
      filepar->mix=-1;
      j=0;
      fcur2=fileparroot;
      while (fcur2!=NULL) {
        if ((fcur2->file!=NULL) && (filepar->file!=NULL) && (strcmp(fcur2->file,filepar->file)==0)) {
          if ((fcur2->x==filepar->x) && (fcur2->y==filepar->y)) {
            if ((fcur2->mathx==NULL) && (filepar->mathx==NULL)
             || (fcur2->mathx!=NULL) && (filepar->mathx!=NULL) && (strcmp(fcur2->mathx,filepar->mathx)==0)) {
              if ((fcur2->mathy==NULL) && (filepar->mathy==NULL)
               || (fcur2->mathy!=NULL) && (filepar->mathy!=NULL) && (strcmp(fcur2->mathy,filepar->mathy)==0)) {
                filepar->mix=j;
                break;
              }
            }
          }
        }
        j++;
        fcur2=fcur2->next;
      }
      if (fcur==NULL) fileparroot=filepar;
      else fcur->next=filepar;
      fcur=filepar;
    }
  }
  fclose(fp);
  return TRUE;
}

void TMyApp::InitInstance()
{
  int i,val;
  char *endptr;

  TApplication::InitInstance();
  PumpWaitingMessages();
  for (i=1;(i<Argc) && (Argv[i][0]=='-');i++) {
    switch (Argv[i][1]) {
    case 'x':
      if (i+1<Argc) {
        val=strtol(Argv[i+1],&endptr,10);
        if (endptr[0]=='\0') {
          posx=val;
          i++;
        }
      }
      break;
    case 'y':
      if (i+1<Argc) {
        val=strtol(Argv[i+1],&endptr,10);
        if (endptr[0]=='\0') {
          posy=val;
          i++;
        }
      }
      break;
    case 'w':
      if (i+1<Argc) {
        val=strtol(Argv[i+1],&endptr,10);
        if (endptr[0]=='\0') {
          posw=val;
          i++;
        }
      }
      break;
    default:
      printfstderr("unknown option `%s'.",Argv[i]);
      BreakMessageLoop=TRUE;
      return;
    }
  }
  if (i+1>=Argc) {
    printfstderr("Usage: LEGEND [option] INPUT SCRIPT");
    BreakMessageLoop=TRUE;
    return;
  }
  InputFile=Argv[i];
  ScriptFile=Argv[i+1];
  if (!LoadFilePar()) {
    BreakMessageLoop=TRUE;
    return;
  }
  dlg->SetupItem();
  dlg->SetupItem2();
}

void TMyApp::InitMainWindow()
{
  TFrameWindow *frame;
  dlg=new MainDialog(0,DIALOG_MAIN);
  frame=new TFrameWindow(0, "Legend",dlg,TRUE);
  frame->Attr.Style=WS_VISIBLE | WS_SYSMENU | WS_CAPTION | WS_DLGFRAME;
  SetMainWindow(frame);
  GetMainWindow()->SetIcon(this,ICON_1);
}


int OwlMain(int argc,char **argv)
{
  int i;
  char *lib,*home;

  if ((lib=getenv("NGRAPHLIB"))!=NULL) {
    if ((libdir=(char *)malloc(strlen(lib)+1))==NULL) exit(1);
    strcpy(libdir,lib);
  } else {
    for (i=strlen(argv[0]);(argv[0][i]!='\\') && (i>0);i--);
    if ((libdir=(char *)malloc(i+1))==NULL) exit(1);
    strncpy(libdir,argv[0],i);
    libdir[i]='\0';
  }
  if ((home=getenv("NGRAPHHOME"))!=NULL) {
    if ((homedir=(char *)malloc(strlen(home)+1))==NULL) exit(1);
    strcpy(homedir,home);
  } else if ((home=getenv("HOME"))!=NULL) {
    if ((homedir=(char *)malloc(strlen(home)+1))==NULL) exit(1);
    strcpy(homedir,home);
  } else if ((home=getenv("Home"))!=NULL) {
    if ((homedir=(char *)malloc(strlen(home)+1))==NULL) exit(1);
    strcpy(homedir,home);
  } else {
    if ((homedir=(char *)malloc(strlen(libdir)+1))==NULL) exit(1);
    strcpy(homedir,libdir);
  }

  fontmaproot=NULL;
  R=G=B=0;
  pt=2000;
  space=0;
  script=7000;
  direction=0;

  if (((font=(char *)malloc(strlen("Helvetica")+1))==NULL)
  || ((jfont=(char *)malloc(strlen("Gothic")+1))==NULL)) {
    free(font);
    free(jfont);
  }
  strcpy(font,"Helvetica");
  strcpy(jfont,"Gothic");

  fontloadconfig();
  textloadconfig();

  mix=1;
  type=1;
  caption=1;
  frame=1;
  posx=5000;
  posy=5000;
  posw=3000;

  for (i=0;i<16;i++) custColors[i]=RGB(127,127,127);

  fileparroot=NULL;

  hInstance=_hInstance;
  TMyApp(argc,argv).Run();

  return 0;
}


