/* 
 * 
 * This file is part of "Ngraph for Windows".
 * 
 * Copyright (C) 2015, Satoshi ISHIZAKA. isizaka@msa.biglobe.ne.jp
 * 
 * "Ngraph for Windows" is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * "Ngraph for Windows" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

/*
 *
 * gra2.cpp
 *
 */

#include <owl/pch.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <io.h>
#include <dir.h>

#define MPI 3.1415926535897932385
#define GRAF "%Ngraph GRAF"
#define CONF "gra2win.ini"
#define G2WINCONF "[gra2win]"
#define DIRSEP '/'
#define CONFSEP "/"
#define NSTRLEN 256
#define LINETOLIMIT 500
#define VERSION "1.03.21"

#include "gra2.rh"

#define TRUE  1
#define FALSE 0

#define FONT_ROMAN 0
#define FONT_BOLD 1
#define FONT_ITALIC 2
#define FONT_BOLDITALIC 3
#define FONT_UNKNOWN 4

#define ANSI2_CHARSET -1
struct fontmap;

struct fontmap {
  char *fontalias;
  char *fontname;
  int charset;
  int italic;
  struct fontmap *next;
};


struct fontmap *fontmaproot;

#define PREVIEW 0
#define CLIPBOARD 1
#define METAFILE 2

int MetaFile=0;
int output=PREVIEW;
char *metafile=NULL;
HENHMETAFILE hMF=NULL;
int PaperWidth=0;
int PaperHeight=0;
int windpi,winheight,winwidth,winbgR,winbgG,winbgB;
char *libdir,*homedir;
int scrollx,scrolly;
char *title;
HDC DDC=NULL;
HPEN ThePen,OrgPen;
HBRUSH TheBrush,OrgBrush;
HFONT TheFont,OrgFont;
COLORREF Col;
LOGFONT IDFont;
int style,width,dashn;
DWORD *dashlist;
int x0,y0;
int dashi,dotf;
double dashlen;
double pixel_dot;
int offsetx,offsety;
int cpx,cpy;
int loadfontf;
char *fontalias;
int charset;
double fontspace,fontcos,fontsin;
int linetonum;
int minus_hyphen;

int niskanji(unsigned int code)
{
  if (((0x81<=code) && (code<=0x9f))
   || ((0xe0<=code) && (code<=0xff))) return TRUE;
  return FALSE;
}

unsigned int njms2jis(unsigned int code)
{
  unsigned char dh,dl;

  dh=code >> 8;
  dl=code & 0xff;
  if (dh<=0x9f) dh-=0x70;
  else dh-=0xb0;
  dh=dh<<1;
  if (dl>=0x9f) dl-=0x7e;
  else {
    dh--;
    if (dl>=0x80) dl-=0x20;
    else dl-=0x1f;
  }
  return ((unsigned int)(dh << 8))+dl;
}

char *nstrnew(void)
{
  char *po;

  po=(char *)malloc(NSTRLEN);
  po[0]='\0';
  return po;
}

char *nstrccat(char *po,char ch)
{
  size_t len,num;

  if (po==NULL) return NULL;
  len=strlen(po);
  num=len/NSTRLEN;
  if (len%NSTRLEN==NSTRLEN-1) po=(char *)realloc(po,NSTRLEN*(num+2));
  po[len]=ch;
  po[len+1]='\0';
  return po;
}

int fgetline(FILE *fp,char **buf)
{
  char *s;
  char ch;

  *buf=NULL;
  ch=fgetc(fp);
  if (ch==EOF) return 1;
  s=nstrnew();
  while (TRUE) {
    if ((ch=='\0') || (ch=='\n') || (ch==EOF)) {
      *buf=s;
      return 0;
    }
    if (ch!='\r') s=nstrccat(s,ch);
    ch=fgetc(fp);
  }
}

FILE *openconfig(char *section)
{
  char *s,*buf,*homeconf,*libconf;
  FILE *fp;
  struct stat homestat,libstat;

  homeconf=(char *)malloc(strlen(homedir)+strlen(CONFSEP)+strlen(CONF)+1);
  strcpy(homeconf,homedir);
  if ((strlen(homedir)>0) && (homeconf[strlen(homedir)-1]=='/'))
    homeconf[strlen(homedir)-1]='\0';
  strcat(homeconf,CONFSEP);
  strcat(homeconf,CONF);
  libconf=(char *)malloc(strlen(libdir)+strlen(CONFSEP)+strlen(CONF)+1);
  strcpy(libconf,libdir);
  if ((strlen(libdir)>0) && (libconf[strlen(libdir)-1]=='/'))
    libconf[strlen(libdir)-1]='\0';
  strcat(libconf,CONFSEP);
  strcat(libconf,CONF);
  if (access(homeconf,04)==0) {
    if (stat(homeconf,&homestat)!=0) {
      free(homeconf);
      homeconf=NULL;
    }
  } else {
    free(homeconf);
    homeconf=NULL;
  }
  if (access(libconf,04)==0) {
    if (stat(libconf,&libstat)!=0) {
      free(libconf);
      libconf=NULL;
    }
  } else {
    free(libconf);
    libconf=NULL;
  }
  if (homeconf!=NULL) {
    if (libconf==NULL) s=homeconf;
    else if (homestat.st_mtime>=libstat.st_mtime) {
      s=homeconf;
      free(libconf);
    } else {
      s=libconf;
      free(homeconf);
    }
  } else if (libconf!=NULL) s=libconf;
  else return NULL;
  if ((fp=fopen(s,"rt"))==NULL) return NULL;
  free(s);
  while (fgetline(fp,&buf)==0) {
    if (strcmp(buf,section)==0) {
      free(buf);
      return fp;
    }
    free(buf);
  }
  fclose(fp);
  return NULL;
}

char *getitok(char **s,int *len,char *ifs)
{
  char *po,*spo;
  int i;

  if (*s==NULL) return NULL;
  po=*s;
  for (i=0;(po[i]!='\0') && (strchr(ifs,po[i])!=NULL);i++);
  if (po[i]=='\0') {
    *len=0;
    return NULL;
  }
  spo=po+i;
  for (;(po[i]!='\0') && (strchr(ifs,po[i])==NULL);i++);
  *s+=i;
  *len=*s-spo;
  return spo;
}

char *getitok2(char **s,int *len,char *ifs)
{
  char *po,*s2;

  if ((s2=getitok(s,len,ifs))==NULL) return NULL;
  po=(char *)malloc(*len+1);
  strncpy(po,s2,*len);
  po[*len]='\0';
  return po;
}

char *getconfig(FILE *fp,char **val)
{
  char *s,*tok,*buf;
  int len;

  while (TRUE) {
    if (fgetline(fp,&buf)!=0) return NULL;
    else {
      if (buf[0]=='[') {
        free(buf);
        return NULL;
      } else {
        s=buf;
        if ((tok=getitok2(&s,&len," \x09=,"))!=NULL) {
          for (;(s[0]!='\0') && (strchr(" \x09=,",s[0])!=NULL);s++);
          *val=(char *)malloc(strlen(s)+1);
          strcpy(*val,s);
          free(buf);
          return tok;
        }
        free(buf);
        free(tok);
      }
    }
  }
}

void closeconfig(FILE *fp)
{
  fclose(fp);
}

int getintpar(char *s,int num,int cpar[])
{
  int i,pos1,pos2;
  char s2[256];
  char *endptr;

  pos1=0;
  for (i=0;i<num;i++) {
	while ((s[pos1]!='\0') && (strchr(" \x09,",s[pos1])!=NULL)) pos1++;
	if (s[pos1]=='\0') return FALSE;
	pos2=0;
	while ((s[pos1]!='\0') && (strchr(" \x09,",s[pos1])==NULL)) {
	  s2[pos2]=s[pos1];
	  pos2++;
	  pos1++;
	}
	s2[pos2]='\0';
	cpar[i]=strtol(s2,&endptr,10);
	if (endptr[0]!='\0') return FALSE;
  }
  return TRUE;
}

int GRAinput(char *s,void (*draw)(char code,int *cpar,char *cstr))
{
  int pos,i;
  int num;
  char code;
  int *cpar;
  char *cstr;

  code='\0';
  cpar=NULL;
  cstr=NULL;
  for (i=0;s[i]!='\0';i++)
	if (strchr("\n\r",s[i])!=NULL) {
	  s[i]='\0';
	  break;
	}
  pos=0;
  while ((s[pos]==' ') || (s[pos]=='\x09')) pos++;
  if (s[pos]=='\0') return TRUE;
  if (strchr("IE%VAGOMNLTCBPRDFHSK",s[pos])==NULL) return FALSE;
  code=s[pos];
  if (strchr("%FSK",code)==NULL) {
	if (!getintpar(s+pos+1,1,&num)) return FALSE;
    num++;
    cpar=(int *)malloc(sizeof(int)*num);
    if (!getintpar(s+pos+1,num,cpar)) goto errexit;
  } else {
    cpar=(int *)malloc(sizeof(int));
    cpar[0]=-1;
    cstr=(char *)malloc(strlen(s)-pos);
    strcpy(cstr,s+pos+1);
  }
  draw(code,cpar,cstr);
  free(cpar);
  free(cstr);
  return TRUE;

errexit:
  free(cpar);
  free(cstr);
  return FALSE;
}

int round(double x)
{
  int ix;
  double dx;

  if (x>INT_MAX) return INT_MAX;
  else if (x<INT_MIN) return INT_MIN;
  else {
    ix=(int )x;
    dx=x-ix;
    if (dx>=0.5) return ix+1;
    else if (dx<=-0.5) return ix-1;
    else return ix;
  }
}

char *bfontalias=NULL;
int bloadfont=FALSE;
int boffsetx=0;
int boffsety=0;
int bminx=INT_MAX;
int bminy=INT_MAX;
int bmaxx=INT_MIN;
int bmaxy=INT_MIN;
int bposx=0;
int bposy=0;
int bpt=0;
int bspc=0;
int bdir=0;
int blinew=0;
int bclip=TRUE;
int bclipsizex=21000;
int bclipsizey=29700;

void setbbminmax(int x1,int y1,int x2,int y2,int lw)
{
  int x,y;

  if (x1>x2) {
    x=x1; x1=x2; x2=x;
  }
  if (y1>y2) {
    y=y1; y1=y2; y2=y;
  }
  if (lw) {
    x1-=blinew;
    y1-=blinew;
    x2+=blinew;
    y2+=blinew;
  }
  if (!bclip || !((x2<0) || (y2<0) || (x1>bclipsizex) || (y1>bclipsizey))) {
    if (bclip) {
      if (x1<0) x1=0;
      if (y1<0) y1=0;
      if (x2>bclipsizex) x2=bclipsizex;
      if (y2>bclipsizey) y1=bclipsizey;
    }
    x1+=boffsetx;
    x2+=boffsetx;
    y1+=boffsety;
    y2+=boffsety;
    if (x1<bminx) bminx=x1;
    if (y1<bminy) bminy=y1;
    if (x2>bmaxx) bmaxx=x2;
    if (y2>bmaxy) bmaxy=y2;
  }
}

int charwidth(unsigned c,char *font,int pt)
{
  char ch[3];
  double size;
  char *fontname;
  struct fontmap *fcur;
  HFONT TmpFont,OrgFont;
  LOGFONT IDFont;
  SIZE lpsize;
  HDC DC;
  int charset,italic;

  ch[0]=(c & 0xff);
  ch[1]=(c & 0xff00) >> 8;
  ch[2]='\0';
  size=pt/72.0*25.4;
  fcur=fontmaproot;
  fontname=NULL;
  while (fcur!=NULL) {
    if (strcmp(font,fcur->fontalias)==0) {
      fontname=fcur->fontname;
      charset=fcur->charset;
      italic=fcur->italic;
      break;
    }
    fcur=fcur->next;
  }
  if (fontname==NULL) {
    return round(size*0.600);
  } else {
    if ((charset==ANSI_CHARSET) && minus_hyphen) {
      if (ch[0]=='-') ch[0]='\x96';
      else if (ch[0]=='\x96') ch[0]='-';
    }
    IDFont.lfHeight=-size;
    IDFont.lfWidth=0;
    IDFont.lfEscapement=0;
    IDFont.lfOrientation=0;
    IDFont.lfUnderline=0;
    IDFont.lfStrikeOut=0;
    IDFont.lfPitchAndFamily=0;
    if (charset==ANSI2_CHARSET) IDFont.lfCharSet=ANSI_CHARSET;
    else IDFont.lfCharSet=charset;
    IDFont.lfOutPrecision=0;
    IDFont.lfClipPrecision=0;
    IDFont.lfQuality=0;
    if (italic==FONT_ROMAN) {
      IDFont.lfWeight=400;
      IDFont.lfItalic=FALSE;
    } else if (italic==FONT_ITALIC) {
      IDFont.lfWeight=400;
      IDFont.lfItalic=TRUE;
    } else if (italic==FONT_BOLD) {
      IDFont.lfWeight=700;
      IDFont.lfItalic=FALSE;
    } else if (italic==FONT_BOLDITALIC) {
      IDFont.lfWeight=700;
      IDFont.lfItalic=TRUE;
    } else {
      IDFont.lfWeight=0;
      IDFont.lfItalic=0;
    }
    strcpy(IDFont.lfFaceName,fontname);
    if ((TmpFont=CreateFontIndirect(&IDFont))!=NULL) {
      DC=GetDC(HWND_DESKTOP);
      OrgFont=(HFONT)SelectObject(DC,TmpFont);
      SetTextAlign(DC,TA_BASELINE | TA_LEFT);
      if (IDFont.lfCharSet==SHIFTJIS_CHARSET)
        GetTextExtentPoint32(DC,ch,2,&lpsize);
      else
        GetTextExtentPoint32(DC,ch,1,&lpsize);
      SelectObject(DC,OrgFont);
      DeleteObject(TmpFont);
      ReleaseDC(HWND_DESKTOP,DC);
      return lpsize.cx;
    } else return round(size*0.600);
  }
}

int charascent(char *font,int pt)
{
  double size;
  char *fontname;
  struct fontmap *fcur;
  HFONT TmpFont,OrgFont;
  LOGFONT IDFont;
  TEXTMETRIC tm;
  HDC DC;
  int charset,italic;

  size=pt/72.0*25.4;
  fcur=fontmaproot;
  fontname=NULL;
  while (fcur!=NULL) {
    if (strcmp(font,fcur->fontalias)==0) {
      fontname=fcur->fontname;
      charset=fcur->charset;
      italic=fcur->italic;
      break;
    }
    fcur=fcur->next;
  }
  if (fontname==NULL) {
    return round(size*0.563);
  } else {
    IDFont.lfHeight=-size;
    IDFont.lfWidth=0;
    IDFont.lfEscapement=0;
    IDFont.lfOrientation=0;
    IDFont.lfUnderline=0;
    IDFont.lfStrikeOut=0;
    IDFont.lfPitchAndFamily=0;
    if (charset==ANSI2_CHARSET) IDFont.lfCharSet=ANSI_CHARSET;
    else IDFont.lfCharSet=charset;
    IDFont.lfOutPrecision=0;
    IDFont.lfClipPrecision=0;
    IDFont.lfQuality=0;
    if (italic==FONT_ROMAN) {
      IDFont.lfWeight=400;
      IDFont.lfItalic=FALSE;
    } else if (italic==FONT_ITALIC) {
      IDFont.lfWeight=400;
      IDFont.lfItalic=TRUE;
    } else if (italic==FONT_BOLD) {
      IDFont.lfWeight=700;
      IDFont.lfItalic=FALSE;
    } else if (italic==FONT_BOLDITALIC) {
      IDFont.lfWeight=700;
      IDFont.lfItalic=TRUE;
    } else {
      IDFont.lfWeight=0;
      IDFont.lfItalic=0;
    }
    strcpy(IDFont.lfFaceName,fontname);
    if ((TmpFont=CreateFontIndirect(&IDFont))!=NULL) {
      DC=GetDC(HWND_DESKTOP);
      OrgFont=(HFONT)SelectObject(DC,TmpFont);
      GetTextMetrics(DC,&tm);
      SelectObject(DC,OrgFont);
      DeleteObject(TmpFont);
      ReleaseDC(HWND_DESKTOP,DC);
      return tm.tmAscent;
    } else {
      return round(size*0.563);
    }
  }
}

int chardescent(char *font,int pt)
{
  double size;
  char *fontname;
  struct fontmap *fcur;
  HFONT TmpFont,OrgFont;
  LOGFONT IDFont;
  TEXTMETRIC tm;
  HDC DC;
  int charset,italic;

  size=pt/72.0*25.4;
  fcur=fontmaproot;
  fontname=NULL;
  while (fcur!=NULL) {
    if (strcmp(font,fcur->fontalias)==0) {
      fontname=fcur->fontname;
      charset=fcur->charset;
      italic=fcur->italic;
      break;
    }
    fcur=fcur->next;
  }
  if (fontname==NULL) {
    return round(size*0.250);
  } else {
    IDFont.lfHeight=-size;
    IDFont.lfWidth=0;
    IDFont.lfEscapement=0;
    IDFont.lfOrientation=0;
    IDFont.lfUnderline=0;
    IDFont.lfStrikeOut=0;
    IDFont.lfPitchAndFamily=0;
    if (charset==ANSI2_CHARSET) IDFont.lfCharSet=ANSI_CHARSET;
    else IDFont.lfCharSet=charset;
    IDFont.lfOutPrecision=0;
    IDFont.lfClipPrecision=0;
    IDFont.lfQuality=0;
    if (italic==FONT_ROMAN) {
      IDFont.lfWeight=400;
      IDFont.lfItalic=FALSE;
    } else if (italic==FONT_ITALIC) {
      IDFont.lfWeight=400;
      IDFont.lfItalic=TRUE;
    } else if (italic==FONT_BOLD) {
      IDFont.lfWeight=700;
      IDFont.lfItalic=FALSE;
    } else if (italic==FONT_BOLDITALIC) {
      IDFont.lfWeight=700;
      IDFont.lfItalic=TRUE;
    } else {
      IDFont.lfWeight=0;
      IDFont.lfItalic=0;
    }
    strcpy(IDFont.lfFaceName,fontname);
    if ((TmpFont=CreateFontIndirect(&IDFont))!=NULL) {
      DC=GetDC(HWND_DESKTOP);
      OrgFont=(HFONT)SelectObject(DC,TmpFont);
      GetTextMetrics(DC,&tm);
      SelectObject(DC,OrgFont);
      DeleteObject(TmpFont);
      ReleaseDC(HWND_DESKTOP,DC);
      return tm.tmDescent;
    } else {
      return round(size*0.25);
    }
  }
}

void getboundingbox(char code,int *cpar,char *cstr)
{
  int i,lw;
  double x,y,csin,ccos;
  int w,h,d,x1,y1,x2,y2,x3,y3,x4,y4;
  int c1,c2;
  char ch;

  switch (code) {
  case 'X': case 'I': case 'E': case '%': case 'G':
    break;
  case 'V':
    boffsetx=cpar[1];
	boffsety=cpar[2];
    bposx=0;
    bposy=0;
    if (cpar[5]==1) bclip=TRUE;
    else bclip=FALSE;
    bclipsizex=cpar[3]-cpar[1];
    bclipsizey=cpar[4]-cpar[2];
    break;
  case 'A':
    blinew=cpar[2]/2;
    break;
  case 'M':
    bposx=cpar[1];
    bposy=cpar[2];
    break;
  case 'N':
    bposx+=cpar[1];
    bposy+=cpar[2];
    break;
  case 'L':
    setbbminmax(cpar[1],cpar[2],cpar[3],cpar[4],TRUE);
    break;
  case 'T':
    setbbminmax(bposx,bposy,cpar[1],cpar[2],TRUE);
    bposx=cpar[1];
    bposy=cpar[2];
    break;
  case 'C':
    if (cpar[7]==0) lw=TRUE;
    else lw=FALSE;
    if (cpar[7]==1) setbbminmax(cpar[1],cpar[2],cpar[1],cpar[2],lw);
    setbbminmax(cpar[1]+(int )(cpar[3]*cos(cpar[5]/18000.0*MPI)),
                cpar[2]-(int )(cpar[4]*sin(cpar[5]/18000.0*MPI)),
                cpar[1]+(int )(cpar[3]*cos((cpar[5]+cpar[6])/18000.0*MPI)),
                cpar[2]-(int )(cpar[4]*sin((cpar[5]+cpar[6])/18000.0*MPI)),lw);
    cpar[6]+=cpar[5];
    cpar[5]-=9000;
    cpar[6]-=9000;
    if ((cpar[5]<0) && (cpar[6]>0))
      setbbminmax(cpar[1],cpar[2]-cpar[4],cpar[1],cpar[2]-cpar[4],lw);
    cpar[5]-=9000;
    cpar[6]-=9000;
    if ((cpar[5]<0) && (cpar[6]>0))
      setbbminmax(cpar[1]-cpar[3],cpar[2],cpar[1]-cpar[3],cpar[2],lw);
    cpar[5]-=9000;
    cpar[6]-=9000;
    if ((cpar[5]<0) && (cpar[6]>0))
      setbbminmax(cpar[1],cpar[2]+cpar[4],cpar[1],cpar[2]+cpar[4],lw);
    cpar[5]-=9000;
    cpar[6]-=9000;
    if ((cpar[5]<0) && (cpar[6]>0))
      setbbminmax(cpar[1]+cpar[3],cpar[2],cpar[1]+cpar[3],cpar[2],lw);
    break;
  case 'B':
    if (cpar[5]==1) lw=FALSE;
    else lw=TRUE;
    setbbminmax(cpar[1],cpar[2],cpar[3],cpar[4],lw);
    break;
  case 'P':
    setbbminmax(cpar[1],cpar[2],cpar[1],cpar[2],FALSE);
    break;
  case 'R':
    for (i=0;i<(cpar[1]-1);i++)
      setbbminmax(cpar[i*2+2],cpar[i*2+3],cpar[i*2+4],cpar[i*2+5],lw);
    break;
  case 'D':
    if (cpar[2]==0) lw=TRUE;
    else lw=FALSE;
    for (i=0;i<(cpar[1]-1);i++)
      setbbminmax(cpar[i*2+3],cpar[i*2+4],cpar[i*2+5],cpar[i*2+6],lw);
    break;
  case 'F':
    free(bfontalias);
    bfontalias=(char *)malloc(strlen(cstr)+1);
    strcpy(bfontalias,cstr);
    break;
  case 'H':
    bpt=cpar[1];
    bspc=cpar[2];
    bdir=cpar[3];
    bloadfont=TRUE;
    break;
  case 'S':
    if (!bloadfont) break;
    csin=sin(bdir/18000.0*MPI);
    ccos=cos(bdir/18000.0*MPI);
    i=0;
    while (i<strlen(cstr)) {
      if ((cstr[i]=='\\') && (cstr[i+1]=='x')) {
        if (toupper(cstr[i+2])>='A') c1=toupper(cstr[i+2])-'A'+10;
        else c1=cstr[i+2]-'0';
        if (toupper(cstr[i+3])>='A') c2=toupper(cstr[i+3])-'A'+10;
        else c2=cstr[i+3]-'0';
        ch=c1*16+c2;
        i+=4;
      } else {
        ch=cstr[i];
        i++;
      }
      w=charwidth((unsigned int)ch,bfontalias,bpt);
      h=charascent(bfontalias,bpt);
      d=chardescent(bfontalias,bpt);
      x=0;
      y=d;
      x1=(int )(bposx+(x*ccos+y*csin));
      y1=(int )(bposy+(-x*csin+y*ccos));
      x=0;
      y=-h;
      x2=(int )(bposx+(x*ccos+y*csin));
      y2=(int )(bposy+(-x*csin+y*ccos));
      x=w;
      y=d;
      x3=(int )(bposx+(x*ccos+y*csin));
      y3=(int )(bposy+(-x*csin+y*ccos));
      x=w;
      y=-h;
      x4=(int )(bposx+(int )(x*ccos+y*csin));
      y4=(int )(bposy+(int )(-x*csin+y*ccos));
      setbbminmax(x1,y1,x4,y4,FALSE);
      setbbminmax(x2,y2,x3,y3,FALSE);
      bposx+=(int )((w+bspc*25.4/72)*ccos);
      bposy-=(int )((w+bspc*25.4/72)*csin);
    }
    break;
  case 'K':
    if (!bloadfont) break;
    csin=sin(bdir/18000.0*MPI);
    ccos=cos(bdir/18000.0*MPI);
    i=0;
    while (i<strlen(cstr)) {
      if (niskanji((unsigned char)cstr[i]) && (cstr[i+1]!='\0')) {
        i+=2;
        w=charwidth((((unsigned char)cstr[i+1])<<8)+(unsigned char)cstr[i],
                       bfontalias,bpt);
        h=charascent(bfontalias,bpt);
        d=chardescent(bfontalias,bpt);
        x=0;
        y=d;
        x1=(int )(bposx+(x*ccos+y*csin));
        y1=(int )(bposy+(-x*csin+y*ccos));
        x=0;
        y=-h;
        x2=(int )(bposx+(x*ccos+y*csin));
        y2=(int )(bposy+(-x*csin+y*ccos));
        x=w;
        y=d;
        x3=(int )(bposx+(x*ccos+y*csin));
        y3=(int )(bposy+(-x*csin+y*ccos));
        x=w;
        y=-h;
        x4=(int )(bposx+(int )(x*ccos+y*csin));
        y4=(int )(bposy+(int )(-x*csin+y*ccos));
        setbbminmax(x1,y1,x4,y4,FALSE);
        setbbminmax(x2,y2,x3,y3,FALSE);
        bposx+=(int )((w+bspc*25.4/72)*ccos);
        bposy-=(int )((w+bspc*25.4/72)*csin);
      } else i++;
    }
    break;
  }
}

void moveto(HDC DC,int x,int y)
{
  dashlen=0;
  dashi=0;
  dotf=TRUE;
  x0=x;
  y0=y;
  MoveToEx(DC,x0,y0,NULL);
}

void lineto(HDC DC,int x,int y)
{
  double dd,len,len2;
  double dx,dy;
  int gx,gy,gx1,gy1,gx2,gy2;

  gx1=x0;
  gy1=y0;
  gx2=x;
  gy2=y;
  if (dashn==0) LineTo(DC,gx2,gy2);
  else {
    dx=(gx2-gx1);
    dy=(gy2-gy1);
    len2=len=sqrt(dx*dx+dy*dy);
    while (len2>((dashlist)[dashi]-dashlen)) {
      dd=(len-len2+(dashlist)[dashi]-dashlen)/len;
      gx=gx1+round(dx*dd);
      gy=gy1+round(dy*dd);
      if (dotf) LineTo(DC,gx,gy);
      else MoveToEx(DC,gx,gy,NULL);
      dotf=dotf ? FALSE : TRUE;
      len2-=((dashlist)[dashi]-dashlen);
      dashlen=0;
      dashi++;
      if (dashi>=dashn) {
        dashi=0;
        dotf=TRUE;
      }
    }
    if (dotf) LineTo(DC,gx2,gy2);
    dashlen+=len2;
  }
  x0=x;
  y0=y;
}

int dot2pixel(int r)
{
  return round(r*pixel_dot);
}

int dot2pixelx(int x)
{
  return round(x*pixel_dot+offsetx-scrollx);
}

int dot2pixely(int y)
{
  return round(y*pixel_dot+offsety-scrolly);
}

int pixel2dot(int r)
{
  return round(r/pixel_dot);
}

void MakeRuler(HDC dc)
{
  int Width,Height;
  HPEN orgpen,pen;
  int smode;

  SelectClipRgn(dc,NULL);
  Width=PaperWidth;
  Height=PaperHeight;
  smode=SetROP2(dc,R2_XORPEN);
  pen=CreatePen(PS_SOLID,0,RGB(128,128,128));
  orgpen=(HPEN)SelectObject(dc,pen);
  BeginPath(dc);
  MoveToEx(dc,dot2pixel(0)-scrollx,dot2pixel(0)-scrolly,NULL);
  LineTo(dc,dot2pixel(Width)-scrollx,dot2pixel(0)-scrolly);
  LineTo(dc,dot2pixel(Width)-scrollx,dot2pixel(Height)-scrolly);
  LineTo(dc,dot2pixel(0)-scrollx,dot2pixel(Height)-scrolly);
  LineTo(dc,dot2pixel(0)-scrollx,dot2pixel(0)-scrolly);
  CloseFigure(dc);
  EndPath(dc);
  StrokePath(dc);
  DeleteObject(SelectObject(dc,orgpen));
  SetROP2(dc,smode);
}

void draw(char code,int *cpar,char *cstr)
{
  char ch[1],*s,*sw;
  HPEN TmpPen;
  HBRUSH TmpBrush;
  HFONT TmpFont;
  HRGN hClip;
  HDC DC,refDC;
  unsigned int cap,join;
  int i,R,G,B;
  double Theta1,Theta2;
  POINT *Points;
  LOGBRUSH lBrush;
  SIZE lpsize;
  int italic;
  char *fontname;
  struct fontmap *fcur;
  double fontsize,fontdir;
  double x0,y0,fontwidth;
  RECT rect;
  OSVERSIONINFO VersionInformation;

  DC=DDC;
  if (DC==NULL) return;
  if (linetonum!=0) {
	if ((code!='T') || (linetonum>=LINETOLIMIT)) {
     if (MetaFile!=1) {
       EndPath(DC);
	    StrokePath(DC);
     }
	  linetonum=0;
	}
  }
  switch (code) {
  case 'I':
    PaperWidth=cpar[3];
    PaperHeight=cpar[4];
    break;
  case 'E': case '%': case 'X':
    break;
  case 'V':
    offsetx=dot2pixel(cpar[1]);
    offsety=dot2pixel(cpar[2]);
    cpx=0;
    cpy=0;
    if (!MetaFile) {
      if (cpar[5]==1) {
        hClip=CreateRectRgn(dot2pixel(cpar[1])-scrollx,
                            dot2pixel(cpar[2])-scrolly,
                            dot2pixel(cpar[3])-scrollx,
                            dot2pixel(cpar[4])-scrolly);
        SelectClipRgn(DC,hClip);
        DeleteObject(hClip);
      } else {
        SelectClipRgn(DC,NULL);
      }
    }
    break;
   case 'A':
    if (cpar[1]==0) {
      style=PS_SOLID;
      dashn=0;
      free(dashlist);
      dashlist=NULL;
    } else {
      free(dashlist);
      style=PS_USERSTYLE;
      if ((dashlist=(unsigned long *)malloc(sizeof(int)*cpar[1]))==NULL) break;
      for (i=0;i<cpar[1];i++)
       if ((dashlist[i]=dot2pixel(cpar[6+i]))<=0)
         dashlist[i]=1;
      dashn=cpar[1];
    }
    width=dot2pixel(cpar[2]);
    if (cpar[3]==2) cap=PS_ENDCAP_SQUARE;
    else if (cpar[3]==1) cap=PS_ENDCAP_ROUND;
    else cap=PS_ENDCAP_FLAT;
    if (cpar[4]==2) join=PS_JOIN_BEVEL;
    else if (cpar[4]==1) join=PS_JOIN_ROUND;
    else join=PS_JOIN_MITER;
    style= PS_GEOMETRIC | style | cap | join;
    if (MetaFile!=1) {
      lBrush.lbStyle=BS_SOLID;
      lBrush.lbColor=Col;
      VersionInformation.dwOSVersionInfoSize=sizeof(OSVERSIONINFO);
      GetVersionEx(&VersionInformation);
      if (VersionInformation.dwPlatformId==VER_PLATFORM_WIN32_NT) {
        TmpPen=ExtCreatePen(style,
                            width,&lBrush,
                            dashn,dashlist);
      } else {
        style= PS_GEOMETRIC | PS_SOLID | cap | join;
        TmpPen=ExtCreatePen(style,width,&lBrush,0,NULL);
      }
    } else {
      style=PS_SOLID;
      TmpPen=CreatePen(style,width,Col);
    }
    SelectObject(DC,TmpPen);
    DeleteObject(ThePen);
    ThePen=TmpPen;
    SetMiterLimit(DC,cpar[5]/100,NULL);
    break;
  case 'G':
    R=cpar[1];
    if (R>255) R=255;
    else if (R<0) R=0;
    G=cpar[2];
    if (G>255) G=255;
    else if (G<0) G=0;
    B=cpar[3];
    if (B>255) B=255;
    else if (B<0) B=0;
    Col=RGB(R,G,B);
    if (MetaFile!=1) {
      lBrush.lbStyle=BS_SOLID;
      lBrush.lbColor=Col;
      if (style & PS_USERSTYLE) {
        TmpPen=ExtCreatePen(style,
                            width,&lBrush,
                            dashn,dashlist);
      } else {
        TmpPen=ExtCreatePen(style,
                            width,&lBrush,0,NULL);
      }
    } else {
      TmpPen=CreatePen(style,width,Col);
    }
    SelectObject(DC,TmpPen);
    DeleteObject(ThePen);
    ThePen=TmpPen;
    TmpBrush=CreateSolidBrush(Col);
    SelectObject(DC,TmpBrush);
    DeleteObject(TheBrush);
    SetTextColor(DC,Col);
    TheBrush=TmpBrush;
    break;
  case 'M':
    cpx=cpar[1];
    cpy=cpar[2];
    break;
  case 'N':
    cpx+=cpar[1];
    cpy+=cpar[2];
    break;
  case 'L':
    if (MetaFile!=1) BeginPath(DC);
    if ((dashn!=0) && ((style & PS_USERSTYLE)==0)) {
      moveto(DC,dot2pixelx(cpar[1]),dot2pixely(cpar[2]));
      lineto(DC,dot2pixelx(cpar[3]),dot2pixely(cpar[4]));
    } else {
      MoveToEx(DC,dot2pixelx(cpar[1]),dot2pixely(cpar[2]),NULL);
      LineTo(DC,dot2pixelx(cpar[3]),dot2pixely(cpar[4]));
    }
    if (MetaFile!=1) {
      EndPath(DC);
      StrokePath(DC);
    }
    break;
  case 'T':
    if (linetonum==0) {
      if (MetaFile!=1) BeginPath(DC);
      if ((dashn!=0) && ((style & PS_USERSTYLE)==0))
        moveto(DC,dot2pixelx(cpx),dot2pixely(cpy));
      else
        MoveToEx(DC,dot2pixelx(cpx),dot2pixely(cpy),NULL);
    }
    if ((dashn!=0) && ((style & PS_USERSTYLE)==0))
      lineto(DC,dot2pixelx(cpar[1]),dot2pixely(cpar[2]));
    else
      LineTo(DC,dot2pixelx(cpar[1]),dot2pixely(cpar[2]));
    linetonum++;
    cpx=cpar[1];
    cpy=cpar[2];
    break;
  case 'C':
    if (cpar[6]==36000) {
      if (cpar[7]==0) {
        if (MetaFile!=1) TmpBrush=(HBRUSH)GetStockObject(NULL_BRUSH);
        else TmpBrush=OrgBrush;
        SelectObject(DC,TmpBrush);
        Ellipse(DC,
               dot2pixelx(cpar[1]-cpar[3]),
               dot2pixely(cpar[2]-cpar[4]),
               dot2pixelx(cpar[1]+cpar[3]),
               dot2pixely(cpar[2]+cpar[4]));
        SelectObject(DC,TheBrush);
        if (MetaFile!=1) DeleteObject(TmpBrush);
      } else {
        if (MetaFile!=1) TmpPen=(HPEN)GetStockObject(NULL_PEN);
        else TmpPen=OrgPen;
        SelectObject(DC,TmpPen);
        Ellipse(DC,
               dot2pixelx(cpar[1]-cpar[3]),
               dot2pixely(cpar[2]-cpar[4]),
               dot2pixelx(cpar[1]+cpar[3]),
               dot2pixely(cpar[2]+cpar[4]));
        SelectObject(DC,ThePen);
        if (MetaFile!=1) DeleteObject(TmpPen);
      }
    } else {
      Theta1=cpar[5]*MPI/18000.0;
      Theta2=Theta1+cpar[6]*MPI/18000.0;
      if (cpar[7]==0) {
        Arc(DC,dot2pixelx(cpar[1]-cpar[3]),
               dot2pixely(cpar[2]-cpar[4]),
               dot2pixelx(cpar[1]+cpar[3]),
               dot2pixely(cpar[2]+cpar[4]),
               dot2pixelx(cpar[1]+round(cpar[3]*cos(Theta1)))-1,
               dot2pixely(cpar[2]-round(cpar[4]*sin(Theta1)))-1,
               dot2pixelx(cpar[1]+round(cpar[3]*cos(Theta2)))-1,
               dot2pixely(cpar[2]-round(cpar[4]*sin(Theta2)))-1);
      } else {
        if (MetaFile!=1) TmpPen=(HPEN)GetStockObject(NULL_PEN);
        else TmpPen=OrgPen;
        SelectObject(DC,TmpPen);
        if (cpar[7]==1) {
          if ((dot2pixel(cpar[3])<2) && (dot2pixel(cpar[4])<2)) {
            SetPixel(DC,dot2pixelx(cpar[1]),dot2pixely(cpar[2]),Col);
          }
          Pie(DC,dot2pixelx(cpar[1]-cpar[3]),
                 dot2pixely(cpar[2]-cpar[4]),
                 dot2pixelx(cpar[1]+cpar[3]),
                 dot2pixely(cpar[2]+cpar[4]),
                 dot2pixelx(cpar[1]+round(cpar[3]*cos(Theta1)))-1,
                 dot2pixely(cpar[2]-round(cpar[4]*sin(Theta1)))-1,
                 dot2pixelx(cpar[1]+round(cpar[3]*cos(Theta2)))-1,
                 dot2pixely(cpar[2]-round(cpar[4]*sin(Theta2)))-1);
        } else {
          Chord(DC,dot2pixelx(cpar[1]-cpar[3]),
                   dot2pixely(cpar[2]-cpar[4]),
                   dot2pixelx(cpar[1]+cpar[3]),
                   dot2pixely(cpar[2]+cpar[4]),
                   dot2pixelx(cpar[1]+round(cpar[3]*cos(Theta1)))-1,
                   dot2pixely(cpar[2]-round(cpar[4]*sin(Theta1)))-1,
                   dot2pixelx(cpar[1]+round(cpar[3]*cos(Theta2)))-1,
                   dot2pixely(cpar[2]-round(cpar[4]*sin(Theta2)))-1);
        }
        SelectObject(DC,ThePen);
        if (MetaFile!=1) DeleteObject(TmpPen);
      }
    }
    break;
  case 'B':
    if (cpar[5]==0) {
      if (MetaFile!=1) TmpBrush=(HBRUSH)GetStockObject(NULL_BRUSH);
      else TmpBrush=OrgBrush;
      SelectObject(DC,TmpBrush);
      Rectangle(DC,dot2pixelx(cpar[1]),dot2pixely(cpar[2]),
                   dot2pixelx(cpar[3]),dot2pixely(cpar[4]));
      SelectObject(DC,TheBrush);
      if (MetaFile!=1) DeleteObject(TmpBrush);
    } else {
      rect.left=dot2pixelx(cpar[1]);
      rect.top=dot2pixely(cpar[2]);
      rect.right=dot2pixelx(cpar[3]);
      rect.bottom=dot2pixely(cpar[4]);
      FillRect(DC,&rect,TheBrush);
    }
    break;
  case 'P':
    SetPixel(DC,dot2pixelx(cpar[1]),dot2pixely(cpar[2]),Col);
    break;
  case 'R':
    if (cpar[1]==0) break;
    if (MetaFile!=1) BeginPath(DC);
    MoveToEx(DC,dot2pixelx(cpar[2]),dot2pixely(cpar[3]),NULL);
    for (i=1;i<cpar[1];i++) {
      LineTo(DC,dot2pixelx(cpar[i*2+2]),dot2pixely(cpar[i*2+3]));
    }
    if (MetaFile!=1) {
      EndPath(DC);
      StrokePath(DC);
    }
    break;
  case 'D':
    if (cpar[1]==0) break;
    if ((Points=(POINT *)malloc(sizeof(POINT)*cpar[1]))==NULL) break;
    for (i=0;i<cpar[1];i++) {
      Points[i].x=dot2pixelx(cpar[i*2+3]);
      Points[i].y=dot2pixely(cpar[i*2+4]);
    }
    if (cpar[2]==0) {
      if (MetaFile!=1) TmpBrush=(HBRUSH)GetStockObject(NULL_BRUSH);
      else TmpBrush=OrgBrush;
      SelectObject(DC,TmpBrush);
      if (MetaFile!=1) BeginPath(DC);
      Polygon(DC,Points,cpar[1]);
      if (MetaFile!=1) {
        CloseFigure(DC);
        EndPath(DC);
        StrokePath(DC);
      }
      SelectObject(DC,TheBrush);
      if (MetaFile!=1) DeleteObject(TmpBrush);
    } else {
      if (MetaFile!=1) TmpPen=(HPEN)GetStockObject(NULL_PEN);
      else TmpPen=OrgPen;
      SelectObject(DC,TmpPen);
      if (cpar[2]==1) {
        SetPolyFillMode(DC,ALTERNATE);
        if (MetaFile!=1) BeginPath(DC);
        Polygon(DC,Points,cpar[1]);
        if (MetaFile!=1) {
          CloseFigure(DC);
          EndPath(DC);
          FillPath(DC);
        }
      } else {
        SetPolyFillMode(DC,WINDING);
        if (MetaFile!=1) BeginPath(DC);
        Polygon(DC,Points,cpar[1]);
        if (MetaFile!=1) {
          CloseFigure(DC);
          EndPath(DC);
          FillPath(DC);
        }
      }
      SelectObject(DC,ThePen);
      if (MetaFile!=1) DeleteObject(TmpPen);
    }
    free(Points);
    break;
  case 'F':
    free(fontalias);
    if ((fontalias=(char *)malloc(strlen(cstr)+1))==NULL) break;
    strcpy(fontalias,cstr);
    break;
  case 'H':
    loadfontf=FALSE;
    fontspace=cpar[2]/72.0*25.4;
    fontsize=cpar[1]/72.0*25.4;
    fontdir=cpar[3]/100.0;
    fontsin=sin(fontdir/180*MPI);
    fontcos=cos(fontdir/180*MPI);
    fcur=fontmaproot;
    fontname=NULL;
    while (fcur!=NULL) {
      if (strcmp(fontalias,fcur->fontalias)==0) {
        fontname=fcur->fontname;
        charset=fcur->charset;
        italic=fcur->italic;
        break;
      }
      fcur=fcur->next;
    }
    if (fontname==NULL) {
      loadfontf=FALSE;
      break;
    }
    IDFont.lfHeight=-round(pixel_dot*fontsize);
    IDFont.lfWidth=0;
    IDFont.lfEscapement=IDFont.lfOrientation=round(fontdir*10);
    IDFont.lfUnderline=0;
    IDFont.lfStrikeOut=0;
    IDFont.lfPitchAndFamily=0;
    IDFont.lfCharSet=charset;
    IDFont.lfOutPrecision=0;
    IDFont.lfClipPrecision=0;
    IDFont.lfQuality=0;
    if (italic==FONT_ROMAN) {
      IDFont.lfWeight=400;
      IDFont.lfItalic=FALSE;
    } else if (italic==FONT_ITALIC) {
      IDFont.lfWeight=400;
      IDFont.lfItalic=TRUE;
    } else if (italic==FONT_BOLD) {
      IDFont.lfWeight=700;
      IDFont.lfItalic=FALSE;
    } else if (italic==FONT_BOLDITALIC) {
      IDFont.lfWeight=700;
      IDFont.lfItalic=TRUE;
    } else {
      IDFont.lfWeight=0;
      IDFont.lfItalic=0;
    }
    strcpy(IDFont.lfFaceName,fontname);
    if ((TmpFont=CreateFontIndirect(&IDFont))!=NULL) {
      SelectObject(DC,TmpFont);
      DeleteObject(TheFont);
      TheFont=TmpFont;
      SetTextAlign(DC,TA_BASELINE | TA_LEFT);
      SetTextCharacterExtra(DC,dot2pixel(fontspace));
      SetBkMode(DC,TRANSPARENT);
      loadfontf=TRUE;
    }
    IDFont.lfEscapement=IDFont.lfOrientation=0;
    break;
  case 'S':
    if (!loadfontf) break;
    if ((s=nstrnew())==NULL) break;
    i=0;
    while (i<strlen(cstr)) {
      if (cstr[i]=='\\') {
        if (cstr[i+1]=='x') {
          if (toupper(cstr[i+2])>='A') ch[0]=toupper(cstr[i+2])-'A'+10;
          else ch[0]=cstr[i+2]-'0';
          if (toupper(cstr[i+3])>='A') ch[0]=ch[0]*16+toupper(cstr[i+3])-'A'+10;
          else ch[0]=ch[0]*16+cstr[i+3]-'0';
          i+=4;
        } else if (cstr[i+1]=='\\') {
          ch[0]=cstr[i+1];
          i+=2;
        } else {
          ch[0]='\0';
          i++;
        }
      } else {
        ch[0]=cstr[i];
        i++;
      }
      s=nstrccat(s,ch[0]);
    }
    if (s==NULL) break;
    x0=cpx;
    y0=cpy;
    if ((charset==ANSI_CHARSET) && (!minus_hyphen)) {
      if ((sw=(char *)malloc(strlen(s)*2))!=NULL) {
        for (i=0;i<strlen(s);i++) {
          if (s[i]=='-') {
            sw[2*i]='\x12';
            sw[2*i+1]='\x22';
          } else {
            sw[2*i]=s[i];
            sw[2*i+1]='\0';
          }
        }
        ExtTextOutW(DC,dot2pixelx(round(x0)),dot2pixely(round(y0)),NULL,NULL,(wchar_t *)sw,strlen(s),NULL);
        refDC=GetDC(HWND_DESKTOP);
        TmpFont=CreateFontIndirect(&IDFont);
        TmpFont=(HFONT)SelectObject(refDC,TmpFont);
        SetTextAlign(refDC,TA_BASELINE | TA_LEFT);
        SetTextCharacterExtra(refDC,dot2pixel(fontspace));
        GetTextExtentPoint32W(refDC,(wchar_t *)sw,strlen(s),&lpsize);
        DeleteObject(SelectObject(refDC,TmpFont));
        ReleaseDC(HWND_DESKTOP,refDC);
      }
      free(sw);
    } else {
      if ((charset==ANSI_CHARSET) && minus_hyphen) {
        if (ch[0]=='-') ch[0]='\x96';
        else if (ch[0]=='\x96') ch[0]='-';
      }
      ExtTextOut(DC,dot2pixelx(round(x0)),
                 dot2pixely(round(y0)),NULL,NULL,s,strlen(s),NULL);
      refDC=GetDC(HWND_DESKTOP);
      TmpFont=CreateFontIndirect(&IDFont);
      TmpFont=(HFONT)SelectObject(refDC,TmpFont);
      SetTextAlign(refDC,TA_BASELINE | TA_LEFT);
      SetTextCharacterExtra(refDC,dot2pixel(fontspace));
      GetTextExtentPoint32(refDC,s,strlen(s),&lpsize);
      DeleteObject(SelectObject(refDC,TmpFont));
      ReleaseDC(HWND_DESKTOP,refDC);
    }
    free(s);
    fontwidth=pixel2dot(lpsize.cx);
    x0+=fontwidth*fontcos;
    y0-=fontwidth*fontsin;
    cpx=round(x0);
    cpy=round(y0);
    break;
  case 'K':
    if (!loadfontf) break;
    x0=cpx;
    y0=cpy;
    ExtTextOut(DC,dot2pixelx(round(x0)),
               dot2pixely(round(y0)),NULL,NULL,cstr,strlen(cstr),NULL);
    refDC=GetDC(HWND_DESKTOP);
    TmpFont=CreateFontIndirect(&IDFont);
    TmpFont=(HFONT)SelectObject(refDC,TmpFont);
    SetTextAlign(refDC,TA_BASELINE | TA_LEFT);
    SetTextCharacterExtra(refDC,dot2pixel(fontspace));
	GetTextExtentPoint32(refDC,cstr,strlen(cstr),&lpsize);
    DeleteObject(SelectObject(refDC,TmpFont));
    ReleaseDC(HWND_DESKTOP,refDC);
    fontwidth=pixel2dot(lpsize.cx);
    x0+=fontwidth*fontcos;
    y0-=fontwidth*fontsin;
    cpx=round(x0);
    cpy=round(y0);
    break;
  default: break;
  }
}

int printfstderr(char *fmt,...)
{
  int len;
  char buf[1024];
  va_list ap;

  va_start(ap,fmt);
  len=vsprintf(buf,fmt,ap);
  va_end(ap);
  MessageBeep(MB_ICONASTERISK);
  MessageBox(NULL,buf,"GRA2 version "VERSION,MB_OK|MB_TASKMODAL|MB_ICONEXCLAMATION|MB_SETFOREGROUND);
  return len;
}

int loadconfig()
{
  FILE *fp;
  char *tok,*str,*s2;
  char *f1,*f2,*f3,*f4;
  struct fontmap *fcur,*fnew;
  int len,val;
  char *endptr;

  if ((fp=openconfig(G2WINCONF))==NULL) return 0;
  fcur=fontmaproot;
  while ((tok=getconfig(fp,&str))!=NULL) {
    s2=str;
    if (strcmp(tok,"font_map")==0) {
      f1=getitok2(&s2,&len," \x09,");
      f3=getitok2(&s2,&len," \x09,");
      f4=getitok2(&s2,&len," \x09,");
      for (;(s2[0]!='\0') && (strchr(" \x09,",s2[0])!=NULL);s2++);
      f2=getitok2(&s2,&len,"");
      if ((f1!=NULL) && (f2!=NULL) && (f3!=NULL) && (f4!=NULL)) {
        if ((fnew=(fontmap *)malloc(sizeof(struct fontmap)))==NULL) {
          free(tok);
          free(f1);
          free(f2);
          free(f3);
          free(f4);
          closeconfig(fp);
          return 1;
        }
        if (fcur==NULL) fontmaproot=fnew;
        else fcur->next=fnew;
        fcur=fnew;
        fcur->next=NULL;
        fcur->fontalias=f1;
        fcur->fontname=f2;
        if (strcmp(f4,"roman")==0) fcur->italic=FONT_ROMAN;
        else if (strcmp(f4,"italic")==0) fcur->italic=FONT_ITALIC;
        else if (strcmp(f4,"bold")==0) fcur->italic=FONT_BOLD;
        else if (strcmp(f4,"bold_italic")==0) fcur->italic=FONT_BOLDITALIC;
        else fcur->italic=FONT_UNKNOWN;
        if (strcmp(f3,"shiftjis")==0) fcur->charset=SHIFTJIS_CHARSET;
        else if (strcmp(f3,"symbol")==0) fcur->charset=SYMBOL_CHARSET;
        else if (strcmp(f3,"ansi")==0) fcur->charset=ANSI_CHARSET;
        else if (strcmp(f3,"ansi2")==0) fcur->charset=ANSI2_CHARSET;
        else if (strcmp(f3,"oem")==0) fcur->charset=OEM_CHARSET;
        else if (strcmp(f3,"hangeul")==0) fcur->charset=HANGEUL_CHARSET;
        else if (strcmp(f3,"chinesebig5")==0) fcur->charset=CHINESEBIG5_CHARSET;
        else fcur->charset=DEFAULT_CHARSET;
        free(f3);
        free(f4);
      } else {
        free(f1);
        free(f2);
        free(f3);
        free(f4);
      }
    } else if (strcmp(tok,"win_dpi")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      if (endptr[0]=='\0')  windpi=abs(val);
      free(f1);
    } else if (strcmp(tok,"win_width")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      if ((endptr[0]=='\0') && (val!=0)) winwidth=val;
      free(f1);
    } else if (strcmp(tok,"win_height")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      if ((endptr[0]=='\0') && (val!=0)) winheight=val;
      free(f1);
    } else if (strcmp(tok,"backgroundR")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      if (endptr[0]=='\0') winbgR=val;
      free(f1);
    } else if (strcmp(tok,"backgroundG")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      if (endptr[0]=='\0') winbgG=val;
      free(f1);
    } else if (strcmp(tok,"backgroundB")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      if (endptr[0]=='\0') winbgB=val;
      free(f1);
    } else if (strcmp(tok,"minus_hyphen")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      if (endptr[0]=='\0') minus_hyphen=val;
      free(f1);
    }
    free(tok);
    free(str);
  }
  closeconfig(fp);
  return 0;
}

class TMyWindow  : public TWindow {
  public:
    TMyWindow(TWindow *parent=0);
    void PrintPage(HDC DC,char *FileName,int scx,int scy);
    TFrameWindow *Frame;
    void EvPaint();
  protected:
    void EvSize(UINT sizeType, TSize& size);
    void SetupWindow();
  DECLARE_RESPONSE_TABLE(TMyWindow);
};

DEFINE_RESPONSE_TABLE1(TMyWindow, TWindow)
  EV_WM_PAINT,
  EV_WM_SIZE,
END_RESPONSE_TABLE;

TMyWindow::TMyWindow(TWindow *parent)
{
  Init(parent,0,0);
  Attr.Style |= WS_VSCROLL | WS_HSCROLL;
  Scroller = new TScroller(this,1,1,0,0);
  Scroller->XLine=50;
  Scroller->YLine=50;
  Scroller->TrackMode=TRUE;
  Scroller->AutoMode=TRUE;
}

void TMyWindow::SetupWindow()
{
  TWindow::SetupWindow();
  SetBkgndColor(RGB(winbgR,winbgG,winbgB));
}

void TMyWindow::EvSize(UINT sizeType, TSize& size)
{
  TWindow::EvSize(sizeType,size);
  if (Scroller->XPos<0) Scroller->XPos=0;
  if (Scroller->YPos<0) Scroller->YPos=0;
  if (Scroller->XPos>Scroller->XRange) Scroller->XPos=Scroller->XRange;
  if (Scroller->YPos>Scroller->YRange) Scroller->YPos=Scroller->YRange;
  SetScrollPos(SB_HORZ,Scroller->XPos,TRUE);
  SetScrollPos(SB_VERT,Scroller->YPos,TRUE);
  scrollx=Scroller->XPos;
  scrolly=Scroller->YPos;
  Invalidate();
}

void TMyWindow::EvPaint()
{
  PAINTSTRUCT ps;
  HDC dc;
  RECT rc;
  ENHMETAHEADER emh;
  RECT mfrect;

  if (::GetUpdateRect(Handle,&rc,FALSE)) {
    scrollx=Scroller->XPos;
    scrolly=Scroller->YPos;
    SetScrollPos(SB_HORZ,Scroller->XPos,TRUE);
    SetScrollPos(SB_VERT,Scroller->YPos,TRUE);
    dc=::BeginPaint(*this,&ps);
    if (hMF!=NULL) {
      GetEnhMetaFileHeader(hMF,sizeof(ENHMETAHEADER),&emh);
      mfrect.left=emh.rclBounds.left-scrollx;
      mfrect.top=emh.rclBounds.top-scrolly;
      mfrect.right=emh.rclBounds.right-scrollx;
      mfrect.bottom=emh.rclBounds.bottom-scrolly;
      PlayEnhMetaFile(dc,hMF,&mfrect);
      MakeRuler(dc);
    }
    ::EndPaint(Handle,&ps);
  }
}

void TMyWindow::PrintPage(HDC DC,char *FileName,int scx,int scy)
{
  FILE *fp;
  char *buf;
  LOGBRUSH lBrush;

  DDC=DC;
  style=PS_SOLID;
  width=1;
  Col=RGB(0,0,0);
  dashlist=NULL;
  dashn=0;
  dashlen=0;
  dashi=0;
  dotf=TRUE;
  x0=0;
  y0=0;
  offsetx=0;
  offsety=0;
  scrollx=scx;
  scrolly=scy;
  cpx=0;
  cpy=0;
  pixel_dot=windpi/25.4/100;
  fontalias=NULL;
  loadfontf=FALSE;
  linetonum=0;
  if (!MetaFile) {
    ThePen=(HPEN)GetStockObject(BLACK_PEN);
    TheBrush=(HBRUSH)GetStockObject(BLACK_BRUSH);
    TheFont=(HFONT)GetStockObject(SYSTEM_FONT);
    OrgPen=(HPEN)SelectObject(DDC,ThePen);
    OrgBrush=(HBRUSH)SelectObject(DDC,TheBrush);
    OrgFont=(HFONT)SelectObject(DDC,TheFont);
  } else {
    ThePen=NULL;
    TheBrush=NULL;
    TheFont=NULL;
    OrgPen=CreatePen(PS_NULL,0,RGB(0,0,0));
    lBrush.lbStyle=BS_NULL;
    OrgBrush=CreateBrushIndirect(&lBrush);
    OrgFont=NULL;
    SelectObject(DDC,OrgPen);
    SelectObject(DDC,OrgBrush);
  }
  if ((fp=fopen(FileName,"rt"))!=NULL) {
    if (fgetline(fp,&buf)!=1) {
      if (strcmp(buf,GRAF)==0) {
        free(buf);
        while (fgetline(fp,&buf)!=1) {
          if (!GRAinput(buf,draw)) {
            printfstderr("illegal GRA format.");
            fclose(fp);
            free(buf);
            goto error;
          }
          free(buf);
        }
        fclose(fp);
      } else free(buf);
    }
  }
error:
  if (!MetaFile) {
    SelectObject(DDC,OrgPen);
    SelectObject(DDC,OrgBrush);
    SelectObject(DDC,OrgFont);
    DeleteObject(ThePen);
    DeleteObject(TheBrush);
    DeleteObject(TheFont);
  } else {
    SelectObject(DDC,(HPEN)GetStockObject(BLACK_PEN));
    SelectObject(DDC,(HBRUSH)GetStockObject(BLACK_BRUSH));
    SelectObject(DDC,(HFONT)GetStockObject(SYSTEM_FONT));
    DeleteObject(ThePen);
    DeleteObject(TheBrush);
    DeleteObject(TheFont);
    DeleteObject(OrgPen);
    DeleteObject(OrgBrush);
  }
  Scroller->SetRange(dot2pixel(PaperWidth),dot2pixel(PaperHeight));
}

class TMyApp : public TApplication {
  public:
    TMyApp(int argc,char **argv):TApplication() {
      Argc=argc;
      Argv=argv;
    }
  protected:
    void InitMainWindow();
    void InitInstance();
    TMyWindow *win;
    int Argc;
    char **Argv;
};

void TMyApp::InitInstance()
{
  int i,sx,sy;
  char *filename;
  FILE *fp;
  char *buf,buf2[256];
  HDC DC;
  HDC rdc;
  HENHMETAFILE hMF2;
  HANDLE MetaPictHD;
  METAFILEPICT *MetaFilePict;
  HMETAFILE HMF;
  METAFILEHEADER MetaHeader;
  int dpi2,dpix,dpiy;
  WORD *Header;
  RECT rect;
  char *tmpfile;
  DWORD fdwAccess,fdwCreate;
  HANDLE fd1,fd2;
  SECURITY_ATTRIBUTES saAttr1,saAttr2;
  unsigned long len2,len3;
  OSVERSIONINFO VersionInformation;

  TApplication::InitInstance();
  dpi2=576;
  for (i=1;(i<Argc) && (Argv[i][0]=='-');i++) {
    switch (Argv[i][1]) {
    case 'o':
      if (((i+1)<Argc) && (Argv[i+1][0]!='-')) {
        i++;
      }
      break;
    case 'd':
      if (((i+1)<Argc) && (Argv[i+1][0]!='-')) {
        dpi2=atoi(Argv[i+1]);
        if (dpi2<96) dpi2=96;
        if (dpi2>2540) dpi2=2540;
        i++;
      }
      break;
    case 'c':
      output=CLIPBOARD;
      if (Argv[i][2]=='e') MetaFile=2;
      else MetaFile=1;
      break;
    case 'm':
      if (((i+1)<Argc) && (Argv[i+1][0]!='-')) {
        metafile=Argv[i+1];
        output=METAFILE;
        if (Argv[i][2]=='e') MetaFile=2;
        else MetaFile=1;
        i++;
      }
      break;
    default:
      printfstderr("unknown option `%s'.\n"
                   "Usage: GRA2 [-c] [-ce] [-m metafile] [-me enhanced metafile] [-d dpi] GRA-file\n"
                   "-c  copy to clipboad as metafile picture\n"
                   "-ce copy to clipboad as enhanced metafile picture\n"
                   "-m  save as metafile\n"
                   "-me save as enhanced metafile\n"
                   "-d  resolution of metafile",Argv[i]);
      BreakMessageLoop=TRUE;
      return;
    }
  }
  if (i>=Argc) {
    printfstderr("Usage: GRA2 [-c] [-ce] [-m metafile] [-me enhanced metafile] [-d dpi] GRA-file\n"
                 "-c  copy to clipboad as metafile picture\n"
                 "-ce copy to clipboad as enhanced metafile picture\n"
                 "-m  save as metafile\n"
                 "-me save as enhanced metafile\n"
                 "-d  resolution of metafile");
    BreakMessageLoop=TRUE;
    return;
  }

  filename=Argv[i];
  if ((fp=fopen(filename,"rt"))==NULL) {
    printfstderr("file not found `%s'.",filename);
    BreakMessageLoop=TRUE;
    return;
  }
  if (fgetline(fp,&buf)==1) {
    printfstderr("illegal GRA format.");
    BreakMessageLoop=TRUE;
    return;
  }
  if (strcmp(buf,GRAF)!=0) {
    printfstderr("illegal GRA format.");
    free(buf);
    BreakMessageLoop=TRUE;
    return;
  }
  free(buf);
  fclose(fp);
  if (output==PREVIEW) {
    DC=CreateEnhMetaFile(NULL,NULL,NULL,NULL);
    win->PrintPage(DC,filename,0,0);
    hMF=CloseEnhMetaFile(DDC);
  } else {
    BreakMessageLoop=TRUE;
    windpi=dpi2;

    if ((fp=fopen(filename,"rt"))!=NULL) {
      if (fgetline(fp,&buf)!=1) {
        if (strcmp(buf,GRAF)==0) {
          free(buf);
          while (fgetline(fp,&buf)!=1) {
            if (!GRAinput(buf,getboundingbox)) {
              printfstderr("illegal GRA format.");
              fclose(fp);
              free(buf);
              return;
            }
            free(buf);
          }
          fclose(fp);
        } else free(buf);
      }
    }
    sx=(bmaxx-bminx)/100;
    sy=(bmaxy-bminy)/100;
    bminx-=sx;
    bminy-=sy;
    bmaxx+=sx;
    bmaxy+=sy;
    rect.left=0;
    rect.top=0;
    rect.right=bmaxx-bminx;
    rect.bottom=bmaxy-bminy;
    if (MetaFile==1) {
      if (metafile==NULL) {
        MetaPictHD=GlobalAlloc(GMEM_MOVEABLE,sizeof(METAFILEPICT));
        MetaFilePict=(METAFILEPICT *)GlobalLock(MetaPictHD);
        if (MetaFilePict!=NULL) {
          MetaFilePict->mm=MM_ANISOTROPIC;
          MetaFilePict->xExt=rect.right;
          MetaFilePict->yExt=rect.bottom;
        }
        DC=CreateMetaFile(NULL);
        SetWindowExtEx(DC,rect.right*windpi/2540,rect.bottom*windpi/2540,NULL);
        SetWindowOrgEx(DC,0,0,NULL);
      } else {
        if ((tmpfile=tempnam(NULL,"GR2"))==NULL) return;
        MetaHeader.key=0x9AC6CDD7L;
        MetaHeader.hmf=0;
        MetaHeader.bbox.left=rect.left*windpi/2540;
        MetaHeader.bbox.top=rect.top*windpi/2540;
        MetaHeader.bbox.right=rect.right*windpi/2540;
        MetaHeader.bbox.bottom=rect.bottom*windpi/2540;
        MetaHeader.inch=windpi;
        MetaHeader.reserved=0;
        Header=(WORD *)&MetaHeader;
        MetaHeader.checksum= *(Header+0) ^ *(Header+1) ^ *(Header+2) ^ *(Header+3) ^ *(Header+4)
                           ^ *(Header+5) ^ *(Header+6) ^ *(Header+7) ^ *(Header+8) ^ *(Header+9);
        DC=CreateMetaFile(tmpfile);
        Escape(DC,MFCOMMENT,6,"Ngraph",NULL);
        SetMapMode(DC,MM_ANISOTROPIC);
        SetWindowExtEx(DC,rect.right*windpi/2540,rect.bottom*windpi/2540,NULL);
        SetWindowOrgEx(DC,0,0,NULL);
      }
    } else {
      rdc=GetDC(HWND_DESKTOP);
      dpix=GetDeviceCaps(rdc,LOGPIXELSX);
      dpiy=GetDeviceCaps(rdc,LOGPIXELSY);
      VersionInformation.dwOSVersionInfoSize=sizeof(OSVERSIONINFO);
      GetVersionEx(&VersionInformation);
      if (VersionInformation.dwPlatformId==VER_PLATFORM_WIN32_NT) {
        DC=CreateEnhMetaFile(rdc,metafile,NULL,"Ngraph\0Graph\0\0");
      } else {
        DC=CreateEnhMetaFile(rdc,metafile,&rect,"Ngraph\0Graph\0\0");
      }
      ReleaseDC(HWND_DESKTOP,rdc);
      SetMapMode(DC,MM_ANISOTROPIC);
      SetWindowExtEx(DC,rect.right*windpi/2540,rect.bottom*windpi/2540,NULL);
      SetWindowOrgEx(DC,0,0,NULL);
      SetViewportExtEx(DC,rect.right*dpix/2540,rect.bottom*dpiy/2540,NULL);
      SetViewportOrgEx(DC,0,0,NULL);
    }
    win->PrintPage(DC,filename,bminx*windpi/2540,bminy*windpi/2540);
    if (metafile==NULL) {
      ::OpenClipboard(HWND_DESKTOP);
      ::EmptyClipboard();
      if (MetaFile==1) {
        if (MetaFilePict!=NULL) {
          HMF=CloseMetaFile(DC);
          MetaFilePict->hMF=HMF;
          GlobalUnlock(MetaPictHD);
          ::SetClipboardData(CF_METAFILEPICT,MetaPictHD);
        } else GlobalFree(MetaPictHD);
      } else {
        hMF2=CloseEnhMetaFile(DC);
        GlobalUnlock(hMF2);
        ::SetClipboardData(CF_ENHMETAFILE,hMF2);
      }
      ::CloseClipboard();
    } else {
      if (MetaFile==1) {
        DeleteMetaFile(CloseMetaFile(DC));
        saAttr1.nLength = sizeof(SECURITY_ATTRIBUTES);
        saAttr1.bInheritHandle = TRUE;
        saAttr1.lpSecurityDescriptor = NULL;
        fdwAccess=GENERIC_WRITE;
        fdwCreate=CREATE_ALWAYS;
        fd1=CreateFile(metafile,fdwAccess,FILE_SHARE_READ,&saAttr1,fdwCreate,0,NULL);
        saAttr2.nLength = sizeof(SECURITY_ATTRIBUTES);
        saAttr2.bInheritHandle = TRUE;
        saAttr2.lpSecurityDescriptor = NULL;
        fdwAccess=GENERIC_READ;
        fdwCreate=OPEN_EXISTING;
        fd2=CreateFile(tmpfile,fdwAccess,FILE_SHARE_READ,&saAttr2,fdwCreate,0,NULL);
        WriteFile(fd1,(char *)&MetaHeader,sizeof(METAFILEHEADER),&len2,NULL);
        while (ReadFile(fd2,buf2,256,&len2,NULL) && (len2>0)) {
          WriteFile(fd1,buf2,len2,&len3,NULL);
        }
        CloseHandle(fd1);
        CloseHandle(fd2);
        unlink(tmpfile);
        free(tmpfile);
      } else {
        DeleteEnhMetaFile(CloseEnhMetaFile(DC));
      }
    }
  }
}

void TMyApp::InitMainWindow()
{
  win=new TMyWindow;
  win->Frame=new TFrameWindow(0,"GRA2",win);
  win->Frame->Attr.X=CW_USEDEFAULT;
  win->Frame->Attr.Y=CW_USEDEFAULT;
  win->Frame->Attr.W=winwidth;
  win->Frame->Attr.H=winheight;
  SetMainWindow(win->Frame);
  GetMainWindow()->SetIcon(this,ICON_1);
  GetMainWindow()->SetIconSm(this,ICON_1);
}

int OwlMain(int argc,char **argv)
{
  int i;
  char *lib,*home;

  if ((lib=getenv("NGRAPHLIB"))!=NULL) {
    if ((libdir=(char *)malloc(strlen(lib)+1))==NULL) exit(1);
    strcpy(libdir,lib);
  } else {
    for (i=strlen(argv[0]);(argv[0][i]!='\\') && (i>0);i--);
    if ((libdir=(char *)malloc(i+1))==NULL) exit(1);
    strncpy(libdir,argv[0],i);
    libdir[i]='\0';
  }
  if ((home=getenv("NGRAPHHOME"))!=NULL) {
    if ((homedir=(char *)malloc(strlen(home)+1))==NULL) exit(1);
    strcpy(homedir,home);
  } else if ((home=getenv("HOME"))!=NULL) {
    if ((homedir=(char *)malloc(strlen(home)+1))==NULL) exit(1);
    strcpy(homedir,home);
  } else if ((home=getenv("Home"))!=NULL) {
    if ((homedir=(char *)malloc(strlen(home)+1))==NULL) exit(1);
    strcpy(homedir,home);
  } else {
    if ((homedir=(char *)malloc(strlen(libdir)+1))==NULL) exit(1);
    strcpy(homedir,libdir);
  }
  winwidth=0;
  winheight=0;
  windpi=70;
  winbgR=GetSysColor(COLOR_WINDOW) & 255;
  winbgG=(GetSysColor(COLOR_WINDOW) >> 8) & 255;
  winbgB=(GetSysColor(COLOR_WINDOW) >> 16) & 255;
  minus_hyphen=TRUE;
  loadconfig();
  if (windpi<1) windpi=70;
  if (windpi>2540) windpi=2540;
  if (winwidth<1) winwidth=CW_USEDEFAULT;
  if (winwidth>10000) winwidth=CW_USEDEFAULT;
  if (winheight<1) winheight=CW_USEDEFAULT;
  if (winheight>10000) winheight=CW_USEDEFAULT;

  TMyApp(argc,argv).Run();

  free(libdir);
  free(homedir);
  return 0;
}


