/* 
 * 
 * This file is part of "Ngraph for Windows".
 * 
 * Copyright (C) 2015, Satoshi ISHIZAKA. isizaka@msa.biglobe.ne.jp
 * 
 * "Ngraph for Windows" is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * "Ngraph for Windows" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

/*
 *
 * gra2gra.cpp
 *
 */

#include <owl/pch.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <io.h>
#include <dir.h>

#include "gra2gra.rh"

#define NAME "GRA2GRA"
#define VER "1.00.01"
#define GRAF "%Ngraph GRAF"
#define GRATMP "$GRA2GRA.GRA"
#define NSTRLEN 256

#define TRUE  1
#define FALSE 0

FILE *fp;
char *tmp;

int mergetop=0;
int mergeleft=0;
int mergezoom=10000;
int mergefont=0;
int mergept=0;
int mergesp=0;
int mergedir=0;

int oldFR=0;
int oldFG=0;
int oldFB=0;
int oldBR=0;
int oldBG=0;
int oldBB=0;

int printfstderr(char *fmt,...)
{
  int len;
  char buf[1024];
  va_list ap;

  va_start(ap,fmt);
  len=vsprintf(buf,fmt,ap);
  va_end(ap);
  MessageBeep(MB_ICONASTERISK);
  MessageBox(NULL,buf,NULL,MB_OK|MB_TASKMODAL|MB_ICONEXCLAMATION|MB_SETFOREGROUND);
  return len;
}

int GRAinputdraw(char code,int *cpar,char *cstr)
{
  int i;

  if (code=='I') {
    fprintf(fp,"%s\n",GRAF);
    fprintf(fp,"%%Creator: %s version %s\n",NAME,VER);
  }
  fputc(code,fp);
  if (cpar[0]==-1) {
    for (i=0;cstr[i]!='\0';i++)
      fputc(cstr[i],fp);
  } else {
    fprintf(fp,",%d",cpar[0]);
    for (i=1;i<=cpar[0];i++)
      fprintf(fp,",%d",cpar[i]);
  }
  fputc('\n',fp);
  return 0;
}

char *fonttbl[21]={
   "Times","TimesBold","TimesItalic","TimesBoldItalic",
   "Helvetica","HelveticaBold","HelveticaOblique","HelveticaBoldOblique",
   "Mincho","Mincho","Mincho","Mincho",
   "Gothic","Gothic","Gothic","Gothic",
   "Courier","CourierBold","CourierItalic","CourierBoldItalic",
   "Symbol"};

struct greektbltype {
  unsigned int jis,symbol;
};

struct greektbltype greektable[48]={
 {0x21,0101}, {0x22,0102}, {0x23,0107}, {0x24,0104}, {0x25,0105}, {0x26,0132},
 {0x27,0110}, {0x28,0121}, {0x29,0111}, {0x2A,0113}, {0x2B,0114}, {0x2C,0115},
 {0x2D,0116}, {0x2E,0130}, {0x2F,0117},
 {0x30,0120}, {0x31,0122}, {0x32,0123}, {0x33,0124}, {0x34,0241}, {0x35,0106},
 {0x36,0103}, {0x37,0131}, {0x38,0127},
 {0x41,0141}, {0x42,0142}, {0x43,0147}, {0x44,0144}, {0x45,0145}, {0x46,0172},
 {0x47,0150}, {0x48,0161}, {0x49,0151}, {0x4A,0153}, {0x4B,0154}, {0x4C,0155},
 {0x4D,0156}, {0x4E,0170}, {0x4F,0157}, {0x50,0160}, {0x51,0162}, {0x52,0163},
 {0x53,0164}, {0x54,0165}, {0x55,0146}, {0x56,0143}, {0x57,0171}, {0x58,0167}};

unsigned int njms2jis(unsigned int code)
{
  unsigned char dh,dl;

  dh=code >> 8;
  dl=code & 0xff;
  if (dh<=0x9f) dh-=0x70;
  else dh-=0xb0;
  dh=dh<<1;
  if (dl>=0x9f) dl-=0x7e;
  else {
    dh--;
    if (dl>=0x80) dl-=0x20;
    else dl-=0x1f;
  }
  return ((unsigned int)(dh << 8))+dl;
}

int round(double x)
{
  int ix;
  double dx;

  if (x>INT_MAX) return INT_MAX;
  else if (x<INT_MIN) return INT_MIN;
  else {
    ix=(int )x;
    dx=x-ix;
    if (dx>=0.5) return ix+1;
    else if (dx<=-0.5) return ix-1;
    else return ix;
  }
}

int getintpar(char *s,int num,int cpar[])
{
  int i,pos1,pos2;
  char s2[256];
  char *endptr;

  pos1=0;
  for (i=0;i<num;i++) {
    while ((s[pos1]!='\0') && (strchr(" \x09,",s[pos1])!=NULL)) pos1++;
    if (s[pos1]=='\0') return FALSE;
    pos2=0;
    while ((s[pos1]!='\0') && (strchr(" \x09,",s[pos1])==NULL)) {
      s2[pos2]=s[pos1];
      pos2++;
      pos1++;
    }
    s2[pos2]='\0';
    cpar[i]=strtol(s2,&endptr,10);
    if (endptr[0]!='\0') return FALSE;
  }
  return TRUE;
}

int GRAinputold(char *s,int greek)
{
  int pos,num,i,j,k,h,m;
  char code,code2;
  int cpar[50],cpar2[50];
  char cstr[256],cstr2[256],*po;
  int col,B,R,G;
  unsigned int jiscode;
  int greekin,len;

  code='\0';
  for (i=0;s[i]!='\0';i++)
    if (strchr("\n\r",s[i])!=NULL) {
      s[i]='\0';
      break;
    }
  pos=0;
  while ((s[pos]==' ') || (s[pos]=='\x09')) pos++;
  if (s[pos]=='\0') return TRUE;
  if (strchr("IEX%VAOMNLTCBPDFSK",s[pos])==NULL) return FALSE;
  code=s[pos];
  if (strchr("%SK",code)==NULL) {
    if (!getintpar(s+pos+1,1,&num)) return FALSE;
    num++;
    if (!getintpar(s+pos+1,num,cpar)) return FALSE;
  } else {
    cpar[0]=-1;
    if ((po=strchr(s+pos+1,','))==NULL) return FALSE;
    if ((po=strchr(po+1,','))==NULL) return FALSE;
    strcpy(cstr,po+1);
    j=0;
    for (i=0;cstr[i]!='\0';i++) {
      if (cstr[i]=='\\') {
        if ((cstr[i+1]!='n') && (cstr[i+1]!='r')) {
          cstr[j]=cstr[i+1];
          j++;
        }
        i++;
      } else {
        cstr[j]=cstr[i];
        j++;
      }
    }
    if ((j>0) && (cstr[j-1]==',')) j--;
    cstr[j]='\0';
  }
  switch (code) {
  case 'X':
    break;
  case '%': case 'S':
    GRAinputdraw(code,cpar,cstr);
    break;
  case 'K':
    greekin=FALSE;
    j=0;
    len=strlen(cstr);
    for (i=0;TRUE;i+=2) {
      if (cstr[i]!='\0')
        jiscode=njms2jis(((unsigned char)cstr[i] << 8)
                         +(unsigned char)cstr[i+1]);
      if (!greekin) {
        if ((i>=len) || (((jiscode>>8)==0x26) && greek)) {
          if (i!=j) {
            code2='K';
            cpar2[0]=-1;
            strncpy(cstr2,cstr+j,i-j);
            cstr2[i-j]='\0';
            GRAinputdraw(code2,cpar2,cstr2);
            j=i;
          }
          greekin=TRUE;
        }
      } else {
        if ((i>=len) || ((jiscode>>8)!=0x26)) {
          if (i!=j) {
            code2='F';
            cpar2[0]=-1;
            GRAinputdraw(code2,cpar2,fonttbl[20]);
            code2='H';
            cpar2[0]=3;
            cpar2[1]=mergept;
            cpar2[2]=mergesp;
            cpar2[3]=mergedir;
            GRAinputdraw(code2,cpar2,cstr2);
            code2='S';
            cpar2[0]=-1;
            m=0;
            for (k=j;k<i;k+=2) {
              jiscode=njms2jis(((unsigned char)cstr[k] << 8)
                               +(unsigned char)cstr[k+1]);
              for (h=0;h<48;h++) if (greektable[h].jis==(jiscode & 0xff))
                break;
              if (h!=48) cstr2[m++]=greektable[h].symbol;
            }
            cstr2[m]='\0';
            GRAinputdraw(code2,cpar2,cstr2);
            j=i;
            code2='F';
            cpar2[0]=-1;
            GRAinputdraw(code2,cpar2,fonttbl[mergefont]);
            code2='H';
            cpar2[0]=3;
            cpar2[1]=mergept;
            cpar2[2]=mergesp;
            cpar2[3]=mergedir;
            GRAinputdraw(code2,cpar2,cstr2);
          }
          greekin=FALSE;
        }
      }
      if (i>=len) break;
    }
    break;
  case 'I':
    cpar[0]=5;
    cpar[5]=round(cpar[3]/2.1);
    cpar[3]=21000;
    cpar[4]=29700;
    GRAinputdraw(code,cpar,cstr);
    break;
  case 'E': case 'M': case 'N': case 'L': case 'T': case 'P':
    GRAinputdraw(code,cpar,cstr);
    break;
  case 'V':
    if (cpar[0]==4) {
      cpar[0]=5;
      cpar[5]=0;
    }
    GRAinputdraw(code,cpar,cstr);
    break;
  case 'A':
    col=cpar[3];
    oldFB=(col & 1)*256;
    oldFG=(col & 2)*128;
    oldFR=(col & 4)*64;
    code2='G';
    cpar2[0]=3;
    cpar2[1]=oldFR;
    cpar2[2]=oldFG;
    cpar2[3]=oldFB;
    GRAinputdraw(code2,cpar2,cstr);
    cpar[0]=5;
    cpar[3]=cpar[4];
    cpar[4]=0;
    cpar[5]=1000;
    GRAinputdraw(code,cpar,cstr);
    break;
  case 'O':
    col=cpar[1];
    oldBB=(col & 1)*256;
    oldBG=(col & 2)*128;
    oldBR=(col & 4)*64;
    break;
  case 'C':
    cpar2[7]=cpar[4];
    if (cpar[0]>=5) cpar[4]=cpar[5];
    else cpar[4]=cpar[3];
    if (cpar[0]==7) {
	  cpar[5]=cpar[6]*10;
	  cpar[6]=cpar[7]*10-cpar[6]*10;
      if (cpar[6]<0) cpar[6]+=36000;
    } else {
      cpar[5]=0;
	  cpar[6]=36000;
    }
    cpar[7]=cpar2[7];
    cpar[0]=7;
    if (cpar[7]==1) {
      code2='G';
      cpar2[0]=3;
      cpar2[1]=oldBR;
      cpar2[2]=oldBG;
      cpar2[3]=oldBB;
      GRAinputdraw(code2,cpar2,cstr);
	}
    GRAinputdraw(code,cpar,cstr);
    if (cpar[7]==1) {
      code2='G';
      cpar2[0]=3;
      cpar2[1]=oldFR;
      cpar2[2]=oldFG;
      cpar2[3]=oldFB;
      GRAinputdraw(code2,cpar2,cstr);
    }
    break;
  case 'B':
    if (cpar[5]==1) {
      code2='G';
      cpar2[0]=3;
      cpar2[1]=oldBR;
      cpar2[2]=oldBG;
      cpar2[3]=oldBB;
      GRAinputdraw(code2,cpar2,cstr);
    }
    GRAinputdraw(code,cpar,cstr);
    if (cpar[5]==1) {
      code2='G';
      cpar2[0]=3;
      cpar2[1]=oldFR;
      cpar2[2]=oldFG;
      cpar2[3]=oldFB;
      GRAinputdraw(code2,cpar2,cstr);
    }
    break;
  case 'D':
    if (cpar[2]==1) {
      code2='G';
      cpar2[0]=3;
      cpar2[1]=oldBR;
      cpar2[2]=oldBG;
      cpar2[3]=oldBB;
      GRAinputdraw(code2,cpar2,cstr);
    }
    GRAinputdraw(code,cpar,cstr);
    if (cpar[2]==1) {
      code2='G';
      cpar2[0]=3;
      cpar2[1]=oldFR;
      cpar2[2]=oldFG;
      cpar2[3]=oldFB;
      GRAinputdraw(code2,cpar2,cstr);
    }
    break;
  case 'F':
    code2='F';
    cpar2[0]=-1;
    mergefont=cpar[1]*4+cpar[2];
    GRAinputdraw(code2,cpar2,fonttbl[cpar[1]*4+cpar[2]]);
    if (cpar[6]==1) cpar[6]=9000;
    else if (cpar[6]<0) cpar[6]=abs(cpar[6]);
    code2='H';
    cpar2[0]=3;
    cpar2[1]=cpar[3]*100;
    cpar2[2]=cpar[4]*100;
    cpar2[3]=cpar[6];
    mergept=cpar2[1];
    mergesp=cpar2[2];
    mergedir=cpar2[3];
    GRAinputdraw(code2,cpar2,cstr);
    col=cpar[5];
    B=(col & 1)*256;
    G=(col & 2)*128;
    R=(col & 4)*64;
    code2='G';
    cpar2[0]=3;
    cpar2[1]=R;
    cpar2[2]=G;
    cpar2[3]=B;
    GRAinputdraw(code2,cpar2,cstr);
    break;
  }
  return TRUE;
}

char *nstrnew(void)
{
  char *po;

  po=(char *)malloc(NSTRLEN);
  po[0]='\0';
  return po;
}

char *nstrccat(char *po,char ch)
{
  size_t len,num;

  if (po==NULL) return NULL;
  len=strlen(po);
  num=len/NSTRLEN;
  if (len%NSTRLEN==NSTRLEN-1) po=(char *)realloc(po,NSTRLEN*(num+2));
  po[len]=ch;
  po[len+1]='\0';
  return po;
}


char *nstrcat(char *po,char *s)
{
  size_t i;

  if (po==NULL) return NULL;
  if (s==NULL) return po;
  for (i=0;s[i]!='\0';i++)
    if ((po=nstrccat(po,s[i]))==NULL) return NULL;
  return po;
}

int fgetline(FILE *fp,char **buf)
{
  char *s;
  char ch;

  *buf=NULL;
  ch=fgetc(fp);
  if (ch==EOF) return 1;
  s=nstrnew();
  while (TRUE) {
    if ((ch=='\0') || (ch=='\n') || (ch==EOF)) {
      *buf=s;
      return 0;
    }
    if (ch!='\r') s=nstrccat(s,ch);
    ch=fgetc(fp);
  }
}

class MainDialog  : public TDialog {
  public:
    MainDialog(TWindow *parent,TResId resID):TDialog(parent,resID) {}
  protected:
    void SetupWindow() {
      TDialog::SetupWindow();
    }
};

class TMyApp : public TApplication {
  public:
    TMyApp(int argc,char **argv):TApplication() {
      Argc=argc;
      Argv=argv;
    }
  protected:
    void InitMainWindow();
    void InitInstance();
    MainDialog *dlg;
    int Argc;
    char **Argv;
};

void TMyApp::InitInstance()
{
  int i,greek;
  char *outfilename;
  char *tmpfilename;
  char *spawn;
  char *filename;
  char *buf;
  FILE *fd;
  int spawnf;
  int spawna;

  TApplication::InitInstance();
  PumpWaitingMessages();

  if (Argc<3) {
    printfstderr("Usage: gra2gra GRA-filename Output-filename [-g] [spawn program]");
    BreakMessageLoop=TRUE;
    return;
  }

  filename=Argv[1];
  outfilename=Argv[2];

  i=3;
  if ((Argc>3) && (strcmp(Argv[3],"-g")==0)) {
    greek=TRUE;
    i++;
  } else greek=FALSE;

  spawnf=FALSE;
  spawna=FALSE;
  spawn=nstrnew();
  for (;i<Argc;i++) {
    spawnf=TRUE;
    spawn=nstrcat(spawn,Argv[i]);
    spawn=nstrccat(spawn,' ');
    if (!spawna) {
      spawn=nstrcat(spawn,"-o ");
      spawn=nstrcat(spawn,outfilename);
      spawn=nstrccat(spawn,' ');
      spawna=TRUE;
    }
    spawnf=TRUE;
  }
  if (spawnf) {
    tmpfilename=(char *)malloc(strlen(GRATMP)+strlen(tmp)+2);
    strcpy(tmpfilename,tmp);
    if ((tmpfilename[strlen(tmpfilename)-1]!='\\')
     && (tmpfilename[strlen(tmpfilename)-1]!='/')) {
      tmpfilename[strlen(tmpfilename)]='\\';
      tmpfilename[strlen(tmpfilename)+1]='\0';
    }
    strcat(tmpfilename,GRATMP);
    spawn=nstrcat(spawn,tmpfilename);
  } else tmpfilename=outfilename;

  if ((fd=fopen(filename,"rt"))==NULL) {
    printfstderr("Open file '%s'.",filename);
    BreakMessageLoop=TRUE;
    return;
  }

  if (fgetline(fd,&buf)==1) {
    printfstderr("illegal GRA format.");
    BreakMessageLoop=TRUE;
    return;
  }
  if (strcmp(" Ngraph GRA file",buf)!=0) {
    printfstderr("illegal GRA format.");
    BreakMessageLoop=TRUE;
    return;
  }
  free(buf);
  if (fgetline(fd,&buf)==1) {
    printfstderr("illegal GRA format.");
    BreakMessageLoop=TRUE;
    return;
  }
  if ((fp=fopen(tmpfilename,"wt"))==NULL) {
    printfstderr("Open file '%s'.",outfilename);
    BreakMessageLoop=TRUE;
    return;
  }
  while (fgetline(fd,&buf)!=1) {
    if (!GRAinputold(buf,greek)) {
      printfstderr("illegal GRA format.");
      BreakMessageLoop=TRUE;
      return;
    }
    free(buf);
  }
  fclose(fd);
  fclose(fp);
  if (spawnf) {
    if (WinExec(spawn,SW_SHOW)<=31)
      printfstderr("Execution: %s",spawn);
  }
  free(spawn);
  BreakMessageLoop=TRUE;
}

void TMyApp::InitMainWindow()
{
  TFrameWindow *frame;
  dlg=new MainDialog(0,DIALOG_MAIN);
  frame=new TFrameWindow(0, "GRA2GRA",dlg,TRUE);
  frame->Attr.Style=WS_VISIBLE | WS_SYSMENU | WS_CAPTION | WS_DLGFRAME;
  SetMainWindow(frame);
  GetMainWindow()->SetIcon(this,ICON_1);
  GetMainWindow()->SetIconSm(this,ICON_1);
}


int OwlMain(int argc,char **argv)
{
  tmp=getenv("TEMP");
  if (tmp==NULL) tmp=getenv("TMP");
  if (tmp==NULL) tmp=".\\";
  TMyApp(argc,argv).Run();
  return 0;
}


