/* 
 * 
 * This file is part of "Ngraph for Windows".
 * 
 * Copyright (C) 2015, Satoshi ISHIZAKA. isizaka@msa.biglobe.ne.jp
 * 
 * "Ngraph for Windows" is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * "Ngraph for Windows" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

/*
 *
 * gra2prnw.cpp
 *
 */

#include <owl/pch.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <io.h>
#include <dir.h>

extern "C" {
#include "../win/winprt.h"
}

HINSTANCE hInstance;

#include "gra2prnw.rh"

#define MPI 3.1415926535897932385
#define GRAF "%Ngraph GRAF"
#define CONF "gra2win.ini"
#define G2WINCONF "[gra2win]"
#define DIRSEP '/'
#define CONFSEP "/"
#define NSTRLEN 256
#define LINETOLIMIT 500


#define TRUE  1
#define FALSE 0

#define FONT_ROMAN 0
#define FONT_BOLD 1
#define FONT_ITALIC 2
#define FONT_BOLDITALIC 3
#define FONT_UNKNOWN 4

#define ANSI2_CHARSET -1

#include <owl\printer.h>

struct fontmap;

struct fontmap {
  char *fontalias;
  char *fontname;
  int charset;
  int italic;
  struct fontmap *next;
};

struct fontmap *fontmaproot;

char *libdir,*homedir;
int scrollx,scrolly;
char *title;
HDC DDC=NULL;
HPEN ThePen;
HBRUSH TheBrush;
HFONT TheFont;
COLORREF Col;
int style,width,dashn;
DWORD *dashlist;
int x0,y0;
int dashi,dotf;
double dashlen;
double pixel_dot;
int offsetx,offsety;
int cpx,cpy;
int loadfontf;
char *fontalias;
int charset;
double fontspace,fontcos,fontsin;
int linetonum;
int minus_hyphen;

int printfstderr(char *fmt,...)
{
  int len;
  char buf[1024];
  va_list ap;

  va_start(ap,fmt);
  len=vsprintf(buf,fmt,ap);
  va_end(ap);
  MessageBeep(MB_ICONASTERISK);
  MessageBox(NULL,buf,NULL,MB_OK|MB_TASKMODAL|MB_ICONEXCLAMATION|MB_SETFOREGROUND);
  return len;
}

int niskanji(unsigned int code)
{
  if (((0x81<=code) && (code<=0x9f))
   || ((0xe0<=code) && (code<=0xff))) return TRUE;
  return FALSE;
}

unsigned int njms2jis(unsigned int code)
{
  unsigned char dh,dl;

  dh=code >> 8;
  dl=code & 0xff;
  if (dh<=0x9f) dh-=0x70;
  else dh-=0xb0;
  dh=dh<<1;
  if (dl>=0x9f) dl-=0x7e;
  else {
    dh--;
    if (dl>=0x80) dl-=0x20;
    else dl-=0x1f;
  }
  return ((unsigned int)(dh << 8))+dl;
}

char *nstrnew(void)
{
  char *po;

  po=(char *)malloc(NSTRLEN);
  po[0]='\0';
  return po;
}

char *nstrccat(char *po,char ch)
{
  size_t len,num;

  if (po==NULL) return NULL;
  len=strlen(po);
  num=len/NSTRLEN;
  if (len%NSTRLEN==NSTRLEN-1) po=(char *)realloc(po,NSTRLEN*(num+2));
  po[len]=ch;
  po[len+1]='\0';
  return po;
}

int fgetline(FILE *fp,char **buf)
{
  char *s;
  char ch;

  *buf=NULL;
  ch=fgetc(fp);
  if (ch==EOF) return 1;
  s=nstrnew();
  while (TRUE) {
    if ((ch=='\0') || (ch=='\n') || (ch==EOF)) {
      *buf=s;
      return 0;
    }
    if (ch!='\r') s=nstrccat(s,ch);
    ch=fgetc(fp);
  }
}

FILE *openconfig(char *section)
{
  char *s,*buf,*homeconf,*libconf;
  FILE *fp;
  struct stat homestat,libstat;

  homeconf=(char *)malloc(strlen(homedir)+strlen(CONFSEP)+strlen(CONF)+1);
  strcpy(homeconf,homedir);
  if ((strlen(homedir)>0) && (homeconf[strlen(homedir)-1]=='/'))
    homeconf[strlen(homedir)-1]='\0';
  strcat(homeconf,CONFSEP);
  strcat(homeconf,CONF);
  libconf=(char *)malloc(strlen(libdir)+strlen(CONFSEP)+strlen(CONF)+1);
  strcpy(libconf,libdir);
  if ((strlen(libdir)>0) && (libconf[strlen(libdir)-1]=='/'))
    libconf[strlen(libdir)-1]='\0';
  strcat(libconf,CONFSEP);
  strcat(libconf,CONF);
  if (access(homeconf,04)==0) {
    if (stat(homeconf,&homestat)!=0) {
      free(homeconf);
      homeconf=NULL;
    }
  } else {
    free(homeconf);
    homeconf=NULL;
  }
  if (access(libconf,04)==0) {
    if (stat(libconf,&libstat)!=0) {
      free(libconf);
      libconf=NULL;
    }
  } else {
    free(libconf);
    libconf=NULL;
  }
  if (homeconf!=NULL) {
    if (libconf==NULL) s=homeconf;
    else if (homestat.st_mtime>=libstat.st_mtime) {
      s=homeconf;
      free(libconf);
    } else {
      s=libconf;
      free(homeconf);
    }
  } else if (libconf!=NULL) s=libconf;
  else return NULL;
  if ((fp=fopen(s,"rt"))==NULL) return NULL;
  free(s);
  while (fgetline(fp,&buf)==0) {
    if (strcmp(buf,section)==0) {
      free(buf);
      return fp;
    }
    free(buf);
  }
  fclose(fp);
  return NULL;
}

char *getitok(char **s,int *len,char *ifs)
{
  char *po,*spo;
  int i;

  if (*s==NULL) return NULL;
  po=*s;
  for (i=0;(po[i]!='\0') && (strchr(ifs,po[i])!=NULL);i++);
  if (po[i]=='\0') {
    *len=0;
    return NULL;
  }
  spo=po+i;
  for (;(po[i]!='\0') && (strchr(ifs,po[i])==NULL);i++);
  *s+=i;
  *len=*s-spo;
  return spo;
}

char *getitok2(char **s,int *len,char *ifs)
{
  char *po,*s2;

  if ((s2=getitok(s,len,ifs))==NULL) return NULL;
  po=(char *)malloc(*len+1);
  strncpy(po,s2,*len);
  po[*len]='\0';
  return po;
}

char *getconfig(FILE *fp,char **val)
{
  char *s,*tok,*buf;
  int len;

  while (TRUE) {
    if (fgetline(fp,&buf)!=0) return NULL;
    else {
      if (buf[0]=='[') {
        free(buf);
        return NULL;
      } else {
        s=buf;
        if ((tok=getitok2(&s,&len," \x09=,"))!=NULL) {
          for (;(s[0]!='\0') && (strchr(" \x09=,",s[0])!=NULL);s++);
          *val=(char *)malloc(strlen(s)+1);
          strcpy(*val,s);
          free(buf);
          return tok;
        }
        free(buf);
        free(tok);
      }
    }
  }
}

void closeconfig(FILE *fp)
{
  fclose(fp);
}

int getintpar(char *s,int num,int cpar[])
{
  int i,pos1,pos2;
  char s2[256];
  char *endptr;

  pos1=0;
  for (i=0;i<num;i++) {
	while ((s[pos1]!='\0') && (strchr(" \x09,",s[pos1])!=NULL)) pos1++;
	if (s[pos1]=='\0') return FALSE;
	pos2=0;
	while ((s[pos1]!='\0') && (strchr(" \x09,",s[pos1])==NULL)) {
	  s2[pos2]=s[pos1];
	  pos2++;
	  pos1++;
	}
	s2[pos2]='\0';
	cpar[i]=strtol(s2,&endptr,10);
	if (endptr[0]!='\0') return FALSE;
  }
  return TRUE;
}

int GRAinput(char *s,void (*draw)(char code,int *cpar,char *cstr))
{
  int pos,i;
  int num;
  char code;
  int *cpar;
  char *cstr;

  code='\0';
  cpar=NULL;
  cstr=NULL;
  for (i=0;s[i]!='\0';i++)
	if (strchr("\n\r",s[i])!=NULL) {
	  s[i]='\0';
	  break;
	}
  pos=0;
  while ((s[pos]==' ') || (s[pos]=='\x09')) pos++;
  if (s[pos]=='\0') return TRUE;
  if (strchr("IE%VAGOMNLTCBPRDFHSK",s[pos])==NULL) return FALSE;
  code=s[pos];
  if (strchr("%FSK",code)==NULL) {
	if (!getintpar(s+pos+1,1,&num)) return FALSE;
    num++;
    cpar=(int *)malloc(sizeof(int)*num);
    if (!getintpar(s+pos+1,num,cpar)) goto errexit;
  } else {
    cpar=(int *)malloc(sizeof(int));
    cpar[0]=-1;
    cstr=(char *)malloc(strlen(s)-pos);
    strcpy(cstr,s+pos+1);
  }
  draw(code,cpar,cstr);
  free(cpar);
  free(cstr);
  return TRUE;

errexit:
  free(cpar);
  free(cstr);
  return FALSE;
}

int round(double x)
{
  int ix;
  double dx;

  if (x>INT_MAX) return INT_MAX;
  else if (x<INT_MIN) return INT_MIN;
  else {
    ix=(int )x;
    dx=x-ix;
    if (dx>=0.5) return ix+1;
    else if (dx<=-0.5) return ix-1;
    else return ix;
  }
}

void moveto(HDC DC,int x,int y)
{
  dashlen=0;
  dashi=0;
  dotf=TRUE;
  x0=x;
  y0=y;
  MoveToEx(DC,x0,y0,NULL);
}

void lineto(HDC DC,int x,int y)
{
  double dd,len,len2;
  double dx,dy;
  int gx,gy,gx1,gy1,gx2,gy2;

  gx1=x0;
  gy1=y0;
  gx2=x;
  gy2=y;
  if (dashn==0) LineTo(DC,gx2,gy2);
  else {
    dx=(gx2-gx1);
    dy=(gy2-gy1);
    len2=len=sqrt(dx*dx+dy*dy);
    while (len2>((dashlist)[dashi]-dashlen)) {
      dd=(len-len2+(dashlist)[dashi]-dashlen)/len;
      gx=gx1+round(dx*dd);
      gy=gy1+round(dy*dd);
      if (dotf) LineTo(DC,gx,gy);
      else MoveToEx(DC,gx,gy,NULL);
      dotf=dotf ? FALSE : TRUE;
      len2-=((dashlist)[dashi]-dashlen);
      dashlen=0;
      dashi++;
      if (dashi>=dashn) {
        dashi=0;
        dotf=TRUE;
      }
    }
    if (dotf) LineTo(DC,gx2,gy2);
    dashlen+=len2;
  }
  x0=x;
  y0=y;
}

int dot2pixel(int r)
{
  return round(r*pixel_dot);
}

int dot2pixelx(int x)
{
  return round(x*pixel_dot+offsetx-scrollx);
}

int dot2pixely(int y)
{
  return round(y*pixel_dot+offsety-scrolly);
}

int pixel2dot(int r)
{
  return round(r/pixel_dot);
}

void draw(char code,int *cpar,char *cstr)
{
  char ch[1],*s,*sw;
  HPEN TmpPen;
  HBRUSH TmpBrush;
  HFONT TmpFont;
  HRGN hClip;
  HDC DC;
  unsigned int cap,join;
  int i,R,G,B;
  double Theta1,Theta2;
  POINT *Points;
  LOGFONT IDFont;
  LOGBRUSH lBrush;
  SIZE lpsize;
  int italic;
  char *fontname;
  struct fontmap *fcur;
  double fontsize,fontdir;
  double x0,y0,fontwidth;
  OSVERSIONINFO VersionInformation;

  DC=DDC;
  if (DC==NULL) return;
  if (linetonum!=0) {
	if ((code!='T') || (linetonum>=LINETOLIMIT)) {
	  EndPath(DC);
	  StrokePath(DC);
	  linetonum=0;
	}
  }
  switch (code) {
  case 'I':
  case 'E': case '%': case 'X':
    break;
  case 'V':
    offsetx=dot2pixel(cpar[1]);
    offsety=dot2pixel(cpar[2]);
    cpx=0;
    cpy=0;
    if (cpar[5]==1) {
      hClip=CreateRectRgn(dot2pixel(cpar[1])-scrollx,
                          dot2pixel(cpar[2])-scrolly,
                          dot2pixel(cpar[3])-scrollx,
                          dot2pixel(cpar[4])-scrolly);
      SelectClipRgn(DC,hClip);
      DeleteObject(hClip);
    } else {
      SelectClipRgn(DC,NULL);
    }
    break;
   case 'A':
    if (cpar[1]==0) {
      style=PS_SOLID;
      dashn=0;
      free(dashlist);
      dashlist=NULL;
    } else {
      free(dashlist);
      style=PS_USERSTYLE;
      if ((dashlist=(unsigned long *)malloc(sizeof(int)*cpar[1]))==NULL) break;
      for (i=0;i<cpar[1];i++)
       if ((dashlist[i]=dot2pixel(cpar[6+i]))<=0)
         dashlist[i]=1;
      dashn=cpar[1];
    }
    width=dot2pixel(cpar[2]);
    if (cpar[3]==2) cap=PS_ENDCAP_SQUARE;
    else if (cpar[3]==1) cap=PS_ENDCAP_ROUND;
    else cap=PS_ENDCAP_FLAT;
    if (cpar[4]==2) join=PS_JOIN_BEVEL;
    else if (cpar[4]==1) join=PS_JOIN_ROUND;
    else join=PS_JOIN_MITER;
    style= PS_GEOMETRIC | style | cap | join;
    lBrush.lbStyle=BS_SOLID;
    lBrush.lbColor=Col;
    VersionInformation.dwOSVersionInfoSize=sizeof(OSVERSIONINFO);
    GetVersionEx(&VersionInformation);
    if (VersionInformation.dwPlatformId==VER_PLATFORM_WIN32_NT) {
      TmpPen=ExtCreatePen(style,
                          width,&lBrush,
                          dashn,dashlist);
    } else {
      style= PS_GEOMETRIC | PS_SOLID | cap | join;
      TmpPen=ExtCreatePen(style,width,&lBrush,0,NULL);
    }
    SelectObject(DC,TmpPen);
    DeleteObject(ThePen);
    SetMiterLimit(DC,cpar[5]/100,NULL);
    ThePen=TmpPen;
    break;
  case 'G':
    R=cpar[1];
    if (R>255) R=255;
    else if (R<0) R=0;
    G=cpar[2];
    if (G>255) G=255;
    else if (G<0) G=0;
    B=cpar[3];
    if (B>255) B=255;
    else if (B<0) B=0;
    Col=RGB(R,G,B);
    lBrush.lbStyle=BS_SOLID;
    lBrush.lbColor=Col;
    if (style & PS_USERSTYLE) {
      TmpPen=ExtCreatePen(style,
                          width,&lBrush,
                          dashn,dashlist);
    } else {
      TmpPen=ExtCreatePen(style,
                          width,&lBrush,0,NULL);
    }
    SelectObject(DC,TmpPen);
    DeleteObject(ThePen);
    ThePen=TmpPen;
    TmpBrush=CreateSolidBrush(Col);
    SelectObject(DC,TmpBrush);
    DeleteObject(TheBrush);
    TheBrush=TmpBrush;
    SetTextColor(DC,Col);
    break;
  case 'M':
    cpx=cpar[1];
    cpy=cpar[2];
    break;
  case 'N':
    cpx+=cpar[1];
    cpy+=cpar[2];
    break;
  case 'L':
    BeginPath(DC);
    if ((dashn!=0) && ((style & PS_USERSTYLE)==0)) {
      moveto(DC,dot2pixelx(cpar[1]),dot2pixely(cpar[2]));
      lineto(DC,dot2pixelx(cpar[3]),dot2pixely(cpar[4]));
    } else {
      MoveToEx(DC,dot2pixelx(cpar[1]),dot2pixely(cpar[2]),NULL);
      LineTo(DC,dot2pixelx(cpar[3]),dot2pixely(cpar[4]));
    }
    EndPath(DC);
    StrokePath(DC);
    break;
  case 'T':
    if (linetonum==0) {
      BeginPath(DC);
      if ((dashn!=0) && ((style & PS_USERSTYLE)==0))
        moveto(DC,dot2pixelx(cpx),dot2pixely(cpy));
      else
        MoveToEx(DC,dot2pixelx(cpx),dot2pixely(cpy),NULL);
    }
    if ((dashn!=0) && ((style & PS_USERSTYLE)==0))
      lineto(DC,dot2pixelx(cpar[1]),dot2pixely(cpar[2]));
    else
      LineTo(DC,dot2pixelx(cpar[1]),dot2pixely(cpar[2]));
    linetonum++;
    cpx=cpar[1];
    cpy=cpar[2];
    break;
  case 'C':
    if (cpar[6]==36000) {
      if (cpar[7]==0) {
        TmpBrush=(HBRUSH)GetStockObject(NULL_BRUSH);
        SelectObject(DC,TmpBrush);
        Ellipse(DC,
               dot2pixelx(cpar[1]-cpar[3]),
               dot2pixely(cpar[2]-cpar[4]),
               dot2pixelx(cpar[1]+cpar[3]),
               dot2pixely(cpar[2]+cpar[4]));
        SelectObject(DC,TheBrush);
        DeleteObject(TmpBrush);
      } else {
        TmpPen=(HPEN)GetStockObject(NULL_PEN);
        SelectObject(DC,TmpPen);
        Ellipse(DC,
               dot2pixelx(cpar[1]-cpar[3]),
               dot2pixely(cpar[2]-cpar[4]),
               dot2pixelx(cpar[1]+cpar[3]),
               dot2pixely(cpar[2]+cpar[4]));
        SelectObject(DC,ThePen);
        DeleteObject(TmpPen);
      }
    } else {
      Theta1=cpar[5]*MPI/18000.0;
      Theta2=Theta1+cpar[6]*MPI/18000.0;
      if (cpar[7]==0) {
        Arc(DC,dot2pixelx(cpar[1]-cpar[3]),
               dot2pixely(cpar[2]-cpar[4]),
               dot2pixelx(cpar[1]+cpar[3]),
               dot2pixely(cpar[2]+cpar[4]),
               dot2pixelx(cpar[1]+round(cpar[3]*cos(Theta1)))-1,
               dot2pixely(cpar[2]-round(cpar[4]*sin(Theta1)))-1,
               dot2pixelx(cpar[1]+round(cpar[3]*cos(Theta2)))-1,
               dot2pixely(cpar[2]-round(cpar[4]*sin(Theta2)))-1);
      } else {
        TmpPen=(HPEN)GetStockObject(NULL_PEN);
        SelectObject(DC,TmpPen);
        if (cpar[7]==1) {
          if ((dot2pixel(cpar[3])<2) && (dot2pixel(cpar[4])<2)) {
            SetPixel(DC,dot2pixelx(cpar[1]),dot2pixely(cpar[2]),Col);
          }
          Pie(DC,dot2pixelx(cpar[1]-cpar[3]),
                 dot2pixely(cpar[2]-cpar[4]),
                 dot2pixelx(cpar[1]+cpar[3]),
                 dot2pixely(cpar[2]+cpar[4]),
                 dot2pixelx(cpar[1]+round(cpar[3]*cos(Theta1)))-1,
                 dot2pixely(cpar[2]-round(cpar[4]*sin(Theta1)))-1,
                 dot2pixelx(cpar[1]+round(cpar[3]*cos(Theta2)))-1,
                 dot2pixely(cpar[2]-round(cpar[4]*sin(Theta2)))-1);
        } else {
          Chord(DC,dot2pixelx(cpar[1]-cpar[3]),
                   dot2pixely(cpar[2]-cpar[4]),
                   dot2pixelx(cpar[1]+cpar[3]),
                   dot2pixely(cpar[2]+cpar[4]),
                   dot2pixelx(cpar[1]+round(cpar[3]*cos(Theta1)))-1,
                   dot2pixely(cpar[2]-round(cpar[4]*sin(Theta1)))-1,
                   dot2pixelx(cpar[1]+round(cpar[3]*cos(Theta2)))-1,
                   dot2pixely(cpar[2]-round(cpar[4]*sin(Theta2)))-1);
        }
        SelectObject(DC,ThePen);
        DeleteObject(TmpPen);
      }
    }
    break;
  case 'B':
    if (cpar[5]==0) {
      TmpBrush=(HBRUSH)GetStockObject(NULL_BRUSH);
      SelectObject(DC,TmpBrush);
      Rectangle(DC,dot2pixelx(cpar[1]),dot2pixely(cpar[2]),
                   dot2pixelx(cpar[3]),dot2pixely(cpar[4]));
      SelectObject(DC,TheBrush);
      DeleteObject(TmpBrush);
    } else {
      TmpPen=(HPEN)GetStockObject(NULL_PEN);
      SelectObject(DC,TmpPen);
      Rectangle(DC,dot2pixelx(cpar[1]),dot2pixely(cpar[2]),
                   dot2pixelx(cpar[3]),dot2pixely(cpar[4]));
      SelectObject(DC,ThePen);
      DeleteObject(TmpPen);
    }
    break;
  case 'P':
    SetPixel(DC,dot2pixelx(cpar[1]),dot2pixely(cpar[2]),Col);
    break;
  case 'R':
    if (cpar[1]==0) break;
    BeginPath(DC);
    MoveToEx(DC,dot2pixelx(cpar[2]),dot2pixely(cpar[3]),NULL);
    for (i=1;i<cpar[1];i++) {
      LineTo(DC,dot2pixelx(cpar[i*2+2]),dot2pixely(cpar[i*2+3]));
    }
    EndPath(DC);
    StrokePath(DC);
    break;
  case 'D':
    if (cpar[1]==0) break;
    if ((Points=(POINT *)malloc(sizeof(POINT)*cpar[1]))==NULL) break;
    for (i=0;i<cpar[1];i++) {
      Points[i].x=dot2pixelx(cpar[i*2+3]);
      Points[i].y=dot2pixely(cpar[i*2+4]);
    }
    if (cpar[2]==0) {
      TmpBrush=(HBRUSH)GetStockObject(NULL_BRUSH);
      SelectObject(DC,TmpBrush);
      BeginPath(DC);
      Polygon(DC,Points,cpar[1]);
      CloseFigure(DC);
      EndPath(DC);
      StrokePath(DC);
      SelectObject(DC,TheBrush);
      DeleteObject(TmpBrush);
    } else {
      TmpPen=(HPEN)GetStockObject(NULL_PEN);
      SelectObject(DC,TmpPen);
      if (cpar[2]==1) {
        SetPolyFillMode(DC,ALTERNATE);
        BeginPath(DC);
        Polygon(DC,Points,cpar[1]);
        CloseFigure(DC);
        EndPath(DC);
        FillPath(DC);
      } else {
        SetPolyFillMode(DC,WINDING);
        BeginPath(DC);
        Polygon(DC,Points,cpar[1]);
        CloseFigure(DC);
        EndPath(DC);
        FillPath(DC);
      }
      SelectObject(DC,ThePen);
      DeleteObject(TmpPen);
    }
    free(Points);
    break;
  case 'F':
    free(fontalias);
    if ((fontalias=(char *)malloc(strlen(cstr)+1))==NULL) break;
    strcpy(fontalias,cstr);
    break;
  case 'H':
    loadfontf=FALSE;
    fontspace=cpar[2]/72.0*25.4;
    fontsize=cpar[1]/72.0*25.4;
    fontdir=cpar[3]/100.0;
    fontsin=sin(fontdir/180*MPI);
    fontcos=cos(fontdir/180*MPI);
    fcur=fontmaproot;
    fontname=NULL;
    while (fcur!=NULL) {
      if (strcmp(fontalias,fcur->fontalias)==0) {
        fontname=fcur->fontname;
        charset=fcur->charset;
        italic=fcur->italic;
        break;
      }
      fcur=fcur->next;
    }
    if (fontname==NULL) {
      loadfontf=FALSE;
      break;
    }
    IDFont.lfHeight=-round(pixel_dot*fontsize);
    IDFont.lfWidth=0;
    IDFont.lfEscapement=IDFont.lfOrientation=round(fontdir*10);
    IDFont.lfUnderline=0;
    IDFont.lfStrikeOut=0;
    IDFont.lfPitchAndFamily=0;
    if (charset==ANSI2_CHARSET) IDFont.lfCharSet=ANSI_CHARSET;
    else IDFont.lfCharSet=charset;
    IDFont.lfOutPrecision=0;
    IDFont.lfClipPrecision=0;
    IDFont.lfQuality=0;
    if (italic==FONT_ROMAN) {
      IDFont.lfWeight=400;
      IDFont.lfItalic=FALSE;
    } else if (italic==FONT_ITALIC) {
      IDFont.lfWeight=400;
      IDFont.lfItalic=TRUE;
    } else if (italic==FONT_BOLD) {
      IDFont.lfWeight=700;
      IDFont.lfItalic=FALSE;
    } else if (italic==FONT_BOLDITALIC) {
      IDFont.lfWeight=700;
      IDFont.lfItalic=TRUE;
    } else {
      IDFont.lfWeight=0;
      IDFont.lfItalic=0;
    }
    strcpy(IDFont.lfFaceName,fontname);
    if ((TmpFont=CreateFontIndirect(&IDFont))!=NULL) {
      SelectObject(DC,TmpFont);
      DeleteObject(TheFont);
      TheFont=TmpFont;
      SetTextAlign(DC,TA_BASELINE | TA_LEFT);
      SetTextCharacterExtra(DC,dot2pixel(fontspace));
      SetBkMode(DC,TRANSPARENT);
      loadfontf=TRUE;
    }
    break;
  case 'S':
    if (!loadfontf) break;
    if ((s=nstrnew())==NULL) break;
    i=0;
    while (i<strlen(cstr)) {
      if (cstr[i]=='\\') {
        if (cstr[i+1]=='x') {
          if (toupper(cstr[i+2])>='A') ch[0]=toupper(cstr[i+2])-'A'+10;
          else ch[0]=cstr[i+2]-'0';
          if (toupper(cstr[i+3])>='A') ch[0]=ch[0]*16+toupper(cstr[i+3])-'A'+10;
          else ch[0]=ch[0]*16+cstr[i+3]-'0';
          i+=4;
        } else if (cstr[i+1]=='\\') {
          ch[0]=cstr[i+1];
          i+=2;
        } else {
          ch[0]='\0';
          i++;
        }
      } else {
        ch[0]=cstr[i];
        i++;
      }
      s=nstrccat(s,ch[0]);
    }
    if (s==NULL) break;
    x0=cpx;
    y0=cpy;
    if ((charset==ANSI_CHARSET) && (!minus_hyphen)) {
      if ((sw=(char *)malloc(strlen(s)*2))!=NULL) {
        for (i=0;i<strlen(s);i++) {
          if (s[i]=='-') {
            sw[2*i]='\x12';
            sw[2*i+1]='\x22';
          } else {
            sw[2*i]=s[i];
            sw[2*i+1]='\0';
          }
        }
        ExtTextOutW(DC,dot2pixelx(round(x0)),dot2pixely(round(y0)),NULL,NULL,(wchar_t *)sw,strlen(s),NULL);
        GetTextExtentPoint32W(DC,(wchar_t *)sw,strlen(s),&lpsize);
      }
      free(sw);
    } else {
      if ((charset==ANSI_CHARSET) && minus_hyphen) {
        if (ch[0]=='-') ch[0]='\x96';
        else if (ch[0]=='\x96') ch[0]='-';
      }
      ExtTextOut(DC,dot2pixelx(round(x0)),
                 dot2pixely(round(y0)),NULL,NULL,s,strlen(s),NULL);
      GetTextExtentPoint32(DC,s,strlen(s),&lpsize);
    }
    free(s);
    fontwidth=pixel2dot(lpsize.cx);
    x0+=fontwidth*fontcos;
    y0-=fontwidth*fontsin;
    cpx=round(x0);
    cpy=round(y0);
    break;
  case 'K':
    if (!loadfontf) break;
    x0=cpx;
    y0=cpy;
    ExtTextOut(DC,dot2pixelx(round(x0)),
               dot2pixely(round(y0)),NULL,NULL,cstr,strlen(cstr),NULL);
    GetTextExtentPoint32(DC,cstr,strlen(cstr),&lpsize);
    fontwidth=pixel2dot(lpsize.cx);
    x0+=fontwidth*fontcos;
    y0-=fontwidth*fontsin;
    cpx=round(x0);
    cpy=round(y0);
    break;
  default: break;
  }
}

int loadconfig()
{
  FILE *fp;
  char *tok,*str,*s2,*endptr;
  char *f1,*f2,*f3,*f4;
  struct fontmap *fcur,*fnew;
  int len,val;

  if ((fp=openconfig(G2WINCONF))==NULL) return 0;
  fcur=fontmaproot;
  while ((tok=getconfig(fp,&str))!=NULL) {
    s2=str;
    if (strcmp(tok,"font_map")==0) {
      f1=getitok2(&s2,&len," \x09,");
      f3=getitok2(&s2,&len," \x09,");
      f4=getitok2(&s2,&len," \x09,");
      for (;(s2[0]!='\0') && (strchr(" \x09,",s2[0])!=NULL);s2++);
      f2=getitok2(&s2,&len,"");
      if ((f1!=NULL) && (f2!=NULL) && (f3!=NULL) && (f4!=NULL)) {
        if ((fnew=(fontmap *)malloc(sizeof(struct fontmap)))==NULL) {
          free(tok);
          free(f1);
          free(f2);
          free(f3);
          free(f4);
          closeconfig(fp);
          return 1;
        }
        if (fcur==NULL) fontmaproot=fnew;
        else fcur->next=fnew;
        fcur=fnew;
        fcur->next=NULL;
        fcur->fontalias=f1;
        fcur->fontname=f2;
        if (strcmp(f4,"roman")==0) fcur->italic=FONT_ROMAN;
        else if (strcmp(f4,"italic")==0) fcur->italic=FONT_ITALIC;
        else if (strcmp(f4,"bold")==0) fcur->italic=FONT_BOLD;
        else if (strcmp(f4,"bold_italic")==0) fcur->italic=FONT_BOLDITALIC;
        else fcur->italic=FONT_UNKNOWN;
        if (strcmp(f3,"shiftjis")==0) fcur->charset=SHIFTJIS_CHARSET;
        else if (strcmp(f3,"symbol")==0) fcur->charset=SYMBOL_CHARSET;
        else if (strcmp(f3,"ansi")==0) fcur->charset=ANSI_CHARSET;
        else if (strcmp(f3,"ansi2")==0) fcur->charset=ANSI2_CHARSET;
        else if (strcmp(f3,"oem")==0) fcur->charset=OEM_CHARSET;
        else if (strcmp(f3,"hangeul")==0) fcur->charset=HANGEUL_CHARSET;
        else if (strcmp(f3,"chinesebig5")==0) fcur->charset=CHINESEBIG5_CHARSET;
        else fcur->charset=DEFAULT_CHARSET;
        free(f3);
        free(f4);
      } else {
        free(f1);
        free(f2);
        free(f3);
        free(f4);
      }
    } else if (strcmp(tok,"minus_hyphen")==0) {
      f1=getitok2(&s2,&len," \x09,");
      val=strtol(f1,&endptr,10);
      if (endptr[0]=='\0') minus_hyphen=val;
      free(f1);
    }
    free(tok);
    free(str);
  }
  closeconfig(fp);
  return 0;
}

class MainDialog  : public TDialog {
  public:
    MainDialog(TWindow *parent,TResId resID):TDialog(parent,resID) {}
  protected:
    void SetupWindow() {
      TDialog::SetupWindow();
    }
};

class TMyPrintout : public TPrintout {
  public:
    TMyPrintout(const char *title,char *filename,int width,int height);
    void EndPrinting();
    void GetDialogInfo(int& minPage, int& maxPage,
                       int& selFromPage, int& selToPage);
    void PrintPage(int page, TRect& rect, unsigned flags);
    void SetBanding(BOOL b) {Banding = b;}
    bool HasPage(int pageNumber) {return pageNumber == 1;}
  protected:
    char *FileName;
    int PaperWidth,PaperHeight;
    BOOL Scale;
};

TMyPrintout::TMyPrintout(const char *title,char *filename,int width,int height)
 : TPrintout(title)
{
  FileName=filename;
  PaperWidth=width;
  PaperHeight=height;
}

void TMyPrintout::PrintPage(int, TRect& , unsigned)
{
  int prevMode;
  TSize oldVExt,oldWExt;
  int dpi;
  FILE *fp;
  char *buf;

  prevMode=DC->SetMapMode(MM_ISOTROPIC);
  dpi=DC->GetDeviceCaps(LOGPIXELSX);
  TSize Ext(round(PaperWidth/2540.0*dpi),round(PaperHeight/2540.0*dpi));
  DC->SetWindowExt(Ext,&oldWExt);
  DC->SetViewportExt(Ext,&oldVExt);
  DDC=HDC(*DC);
  style=PS_SOLID;
  width=1;
  Col=RGB(0,0,0);
  dashlist=NULL;
  dashn=0;
  dashlen=0;
  dashi=0;
  dotf=TRUE;
  x0=0;
  y0=0;
  offsetx=0;
  offsety=0;
  scrollx=0;
  scrolly=0;
  cpx=0;
  cpy=0;
  pixel_dot=dpi/25.4/100;
  fontalias=NULL;
  loadfontf=FALSE;
  ThePen=(HPEN)GetStockObject(BLACK_PEN);
  TheBrush=(HBRUSH)GetStockObject(BLACK_BRUSH);
  TheFont=(HFONT)GetStockObject(SYSTEM_FONT);
  linetonum=0;
  if ((fp=fopen(FileName,"rt"))!=NULL) {
    if (fgetline(fp,&buf)!=1) {
      if (strcmp(buf,GRAF)==0) {
        free(buf);
        while (fgetline(fp,&buf)!=1) {
          if (!GRAinput(buf,draw)) {
            printfstderr("illegal GRA format.");
            goto error;
          }
          free(buf);
        }
        fclose(fp);
      } else free(buf);
    }
  }
error:
  DC->SetWindowExt(oldWExt);
  DC->SetViewportExt(oldVExt);
  DC->SetMapMode(prevMode);
}

void TMyPrintout::GetDialogInfo(int& minPage, int& maxPage,
                                int& selFromPage, int& selToPage)
{
  minPage = 0;
  maxPage = 0;
  selFromPage = selToPage = 0;
}

void TMyPrintout::EndPrinting()
{
}
class TMyApp : public TApplication {
  public:
    TMyApp(int argc,char **argv):TApplication() {
      Argc=argc;
      Argv=argv;
    }
  protected:
    void InitMainWindow();
    void InitInstance();
    MainDialog *dlg;
    int Argc;
    char **Argv;
};

void TMyApp::InitInstance()
{
  int i;
  int Width,Height;
  char *endptr;
  int PageSetup;
  char *filename;
  FILE *fp;
  char *buf;

  TPrinter *Printer;
  TMyPrintout *Printout;
  TPrintDialog::TData *tpd;
  DEVMODE far * dm;

  TApplication::InitInstance();
  PumpWaitingMessages();

  Width=21000;
  Height=29700;
  PageSetup=FALSE;
  for (i=1;(i<Argc) && (Argv[i][0]=='-');i++) {
    switch (Argv[i][1]) {
    case 'o':
      if (((i+1)<Argc) && (Argv[i+1][0]!='-')) {
        i++;
      }
      break;
    case 'w':
      if (((i+1)<Argc) && (Argv[i+1][0]!='-')) {
        Width=strtol(Argv[i+1],&endptr,10);
        i++;
      }
      break;
    case 'h':
      if (((i+1)<Argc) && (Argv[i+1][0]!='-')) {
        Height=strtol(Argv[i+1],&endptr,10);
        i++;
      }
      break;
    case 's':
      PageSetup=TRUE;
      break;
    default:
      printfstderr("unknown option `%s'.",Argv[i]);
      BreakMessageLoop=TRUE;
      return;
    }
  }
  if (i>=Argc) {
    printfstderr("Usage: gra2prnw [-w width] [-h height] [-s] GRA-file\n -w Page width\n -h Page height\n -s Printer setup before print");
    BreakMessageLoop=TRUE;
    return;
  }

  filename=Argv[i];
  if ((fp=fopen(filename,"rt"))==NULL) {
    printfstderr("file not found `%s'.",filename);
    BreakMessageLoop=TRUE;
    return;
  }
  if (fgetline(fp,&buf)==1) {
    printfstderr("illegal GRA format.");
    BreakMessageLoop=TRUE;
    return;
  }
  if (strcmp(buf,GRAF)!=0) {
    printfstderr("illegal GRA format.");
    free(buf);
    BreakMessageLoop=TRUE;
    return;
  }
  free(buf);
  fclose(fp);


  Printer=new TPrinter;
  Printout=new TMyPrintout(filename,filename,Width,Height);
  Printout->SetBanding(FALSE);
  tpd=Printer->GetData();
  //    tpd->Lock();
  dm=(DEVMODE far *)tpd->GetDevMode();
  SetDevMode(dm,&Height,&Width);
  //    tpd->Unlock();

//  TSize Ext(Width/10,Height/10);
//  Printer->SetPageSize(Ext);

  Printer->Print(dlg,*Printout,PageSetup);
  delete Printout;
  delete Printer;

  BreakMessageLoop=TRUE;
}

void TMyApp::InitMainWindow()
{
  TFrameWindow *frame;
  dlg=new MainDialog(0,DIALOG_MAIN);
  frame=new TFrameWindow(0, "GRA2PRNW",dlg,TRUE);
  frame->Attr.Style=WS_VISIBLE | WS_SYSMENU | WS_CAPTION | WS_DLGFRAME;
  SetMainWindow(frame);
  GetMainWindow()->SetIcon(this,ICON_1);
  GetMainWindow()->SetIconSm(this,ICON_1);
}


int OwlMain(int argc,char **argv)
{
  int i;
  char *lib,*home;

  if ((lib=getenv("NGRAPHLIB"))!=NULL) {
    if ((libdir=(char *)malloc(strlen(lib)+1))==NULL) exit(1);
    strcpy(libdir,lib);
  } else {
    for (i=strlen(argv[0]);(argv[0][i]!='\\') && (i>0);i--);
    if ((libdir=(char *)malloc(i+1))==NULL) exit(1);
    strncpy(libdir,argv[0],i);
    libdir[i]='\0';
  }
  if ((home=getenv("NGRAPHHOME"))!=NULL) {
    if ((homedir=(char *)malloc(strlen(home)+1))==NULL) exit(1);
    strcpy(homedir,home);
  } else if ((home=getenv("HOME"))!=NULL) {
    if ((homedir=(char *)malloc(strlen(home)+1))==NULL) exit(1);
    strcpy(homedir,home);
  } else if ((home=getenv("Home"))!=NULL) {
    if ((homedir=(char *)malloc(strlen(home)+1))==NULL) exit(1);
    strcpy(homedir,home);
  } else {
    if ((homedir=(char *)malloc(strlen(libdir)+1))==NULL) exit(1);
    strcpy(homedir,libdir);
  }
  minus_hyphen=TRUE;
  
  loadconfig();

  TMyApp(argc,argv).Run();

  free(libdir);
  free(homedir);
  return 0;
}


